/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffled.c 1.16 1997/06/26 15:57:58 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.2b.
  A demon program to maintain the database and spawn the servers.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>

#include <sys/wait.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#include "wwwoffle.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);
static void sigchild(int signum);
static void sigexit(int signum);


/*+ The server sockets that we listen on +*/
int http_fd=-1,                 /*+ for the HTTP connections. +*/
    wwwoffle_fd=-1;             /*+ for the WWWOFFLE connections. +*/

/*+ Set to 1 when the demon is to shutdown. +*/
static int closedown=0;

/*+ The online / offline status. +*/
int online=0;

/*+ The current number active servers +*/
int n_servers=0,                /*+ in total. +*/
    n_fetch_servers=0;          /*+ fetching a page. +*/

/*+ The wwwoffle client file descriptor when fetching. +*/
int fetch_fd=-1;

/*+ The pids of the servers that are fetching. +*/
int fetch_pids[MAX_FETCH_SERVERS];

/*+ The current status, fetching or not. +*/
int fetching=0;


/*+ The pid of a child that has exited. +*/
static int exit_pid[MAX_SERVERS];

/*+ The status of the child that has exited. +*/
static int exit_status[MAX_SERVERS];


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int i;
 int err;
 struct stat buf;

 /* Parse the command line options */

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-c"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffled: The '-c' option requires a filename.\n"); exit(1);}

       ConfigFile=argv[i];
       continue;
      }

    fprintf(stderr,"wwwoffled: Unknown option '%s'.\n",argv[i]); exit(1);
   }

 /* Initialise things. */

 for(i=0;i<MAX_FETCH_SERVERS;i++)
    fetch_pids[i]=0;

 for(i=0;i<MAX_SERVERS;i++)
    exit_pid[i]=exit_status[i]=0;

 InitErrorHandler("wwwoffled",1);

 if(ReadConfigFile(2))
    PrintMessage(Fatal,"Error in configuration file '%s'.",ConfigFile);

 PrintMessage(Important,"WWWOFFLE Demon Version 1.2b started");
 PrintMessage(Inform,"WWWOFFLE Read Configuration File.");

 http_fd=OpenServerSocket(HTTP_Port);
 if(http_fd==-1)
    PrintMessage(Fatal,"Cannot create HTTP server socket.");

 wwwoffle_fd=OpenServerSocket(WWWOFFLE_Port);
 if(wwwoffle_fd==-1)
    PrintMessage(Fatal,"Cannot create WWWOFFLE server socket.");

 if(stat(SpoolDir,&buf))
   {
    err=mkdir(SpoolDir,0755);
    if(err==-1 && errno!=EEXIST)
       PrintMessage(Fatal,"Cannot create spool directory %s [%!s].",SpoolDir);
    stat(SpoolDir,&buf);
   }

 if(!S_ISDIR(buf.st_mode))
    PrintMessage(Fatal,"The spool directory %s is not a directory.",SpoolDir);

 err=chdir(SpoolDir);
 if(err==-1)
    PrintMessage(Fatal,"Cannot change to spool directory %s [%!s].",SpoolDir);

 sigchild(0);
 sigexit(0);

 /* Loop around waiting for connections. */

 do
   {
    struct timeval tv;
    fd_set readfd;
    int nfds;
    int retval;

    if(http_fd>wwwoffle_fd)
       nfds=http_fd+1;
    else
       nfds=wwwoffle_fd+1;

    FD_ZERO(&readfd);

    FD_SET(wwwoffle_fd,&readfd);

    if(n_servers<MAX_SERVERS)
       FD_SET(http_fd,&readfd);

    tv.tv_sec=10;
    tv.tv_usec=0;

    retval=select(nfds,&readfd,NULL,NULL,&tv);

    if(retval!=-1)
      {
       if(FD_ISSET(wwwoffle_fd,&readfd))
         {
          char *host;
          int port,client;

          client=AcceptConnect(wwwoffle_fd,&host,&port);
          if(client>=0)
            {
             if(IsAllowedConnect(host))
               {
                FILE *file=fdopen(client,"r");

                PrintMessage(Important,"WWWOFFLE connection from %s.",host);

                if(!file)
                  {
                   PrintMessage(Warning,"Unable to fdopen the wwwoffle control socket [%!s].");
                   CloseSocket(client);
                  }
                else
                  {
                   CommandConnect(file);

                   if(fetch_fd!=client)
                      fclose(file);
                  }
               }
             else
               {
                PrintMessage(Warning,"WWWOFFLE connection from %s rejected.",host);
                CloseSocket(client);
               }
            }
         }

       if(FD_ISSET(http_fd,&readfd))
         {
          char *host;
          int port,client;

          client=AcceptConnect(http_fd,&host,&port);
          if(client>=0)
            {
             if(IsAllowedConnect(host))
               {
                PrintMessage(Inform,"HTTP Proxy connection from %s.",host);
                ForkServer(client,1);
               }
             else
                PrintMessage(Warning,"HTTP Proxy connection from %s rejected.",host);

             CloseSocket(client);
            }
         }
      }

    for(i=0;i<MAX_SERVERS;i++)
       if(exit_pid[i]!=0)
         {
          if(exit_status[i]>=0)
             PrintMessage(Inform,"Child wwwoffles exited with status %d (pid=%d).",exit_status[i],exit_pid[i]);
          else
             PrintMessage(Important,"Child wwwoffles terminated by signal %d (pid=%d).",-exit_status[i],exit_pid[i]);
          exit_pid[i]=exit_status[i]=0;
         }

    if(fetch_fd!=-1)
      {
       if(!online)
          fetching=0;

       while(fetching && n_fetch_servers<MAX_FETCH_SERVERS && n_servers<MAX_SERVERS)
          ForkServer(fetch_fd,0);

       if(!fetching && n_fetch_servers==0)
         {
          char *msg="WWWOFFLE No more to fetch.\n";
          write(fetch_fd,msg,strlen(msg));
          close(fetch_fd);
          fetch_fd=-1;

          PrintMessage(Important,"WWWOFFLE Fetch finished.");
         }
      }
   }
 while(!closedown);

 /* Close down and exit. */

 CloseSocket(http_fd);
 CloseSocket(wwwoffle_fd);

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fputs("\n"
       "WWWOFFLED - World Wide Web Offline Explorer (Demon) - Version 1.2b\n"
       "\n",stderr);

 if(verbose)
    fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
          "                             [http://www.gedanken.demon.co.uk/]\n"
          "\n",stderr);

 fputs("Usage: wwwoffled [-h] [-c <config-file>]\n"
       "\n",stderr);

 if(verbose)
    fprintf(stderr,
            "   -h                  : Display this help.\n"
            "\n"
            "   -c <config-file>    : The name of the configuration file to use.\n"
            "                         (Default %s).\n"
            "\n",ConfigFile);

 exit(0);
}


/*++++++++++++++++++++++++++++++++++++++
  The signal handler for the child exiting.

  int signum The signal number.
  ++++++++++++++++++++++++++++++++++++++*/

static void sigchild(int signum)
{
 void (*sig)();
 int pid;
 int status;

 if(signum)
    while((pid=waitpid(-1,&status,WNOHANG))>0)
      {
       int i;
       int exitval=0;

       if(WIFEXITED(status))
          exitval=WEXITSTATUS(status);
       else if(WIFSIGNALED(status))
          exitval=-WTERMSIG(status);

       for(i=0;i<MAX_SERVERS;i++)
          if(exit_pid[i]==0)
            {
             exit_pid[i]=pid;
             exit_status[i]=exitval;
             break;
            }

       n_servers--;

       if(fetch_fd!=-1)
         {
          for(i=0;i<MAX_FETCH_SERVERS;i++)
             if(fetch_pids[i]==pid)
                break;

          if(i!=MAX_FETCH_SERVERS)
            {
             n_fetch_servers--;
             fetch_pids[i]=0;

             if(exitval==3)
                fetching=0;

             if(exitval==4 && online)
                fetching=1;
            }
         }
      }

#if defined(SIGCHLD)
 sig=signal(SIGCHLD,sigchild);
 if(sig==SIG_ERR)
    PrintMessage(Fatal,"Cannot add SIGGHLD handler.");
#else
 sig=signal(SIGCLD,sigchild);
 if(sig==SIG_ERR)
    PrintMessage(Fatal,"Cannot add SIGGLD handler.");
#endif
}


/*++++++++++++++++++++++++++++++++++++++
  The signal handler for the signals to tell us to exit.

  int signum The signal number.
  ++++++++++++++++++++++++++++++++++++++*/

static void sigexit(int signum)
{
 void (*sig)();

 if(signum)
   {
    closedown=1;

    /* Ignore the PrintMessage re-entrancy problem since we are exiting anyway. */
    PrintMessage(Important,"Exit signalled.");
   }

 sig=signal(SIGINT,sigexit);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGINT handler.\n");

 sig=signal(SIGQUIT,sigexit);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGQUIT handler.\n");

 sig=signal(SIGTERM,sigexit);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGTERM handler.\n");
}
