/***************************************
  $Header: /home/amb/wwwoffle/RCS/purge.c 1.11 1997/06/13 19:37:09 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.2a.
  Purge old files from the cache.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>

#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "errors.h"


/* Local functions */
static int PurgeFiles(char *dirname,time_t now,int age,int *bytes_age);

/*+ Set this to 0 for debugging so that nothing is deleted. +*/
#define DO_DELETE 1


/*++++++++++++++++++++++++++++++++++++++
  Purge files from the cache that meet the age criteria.

  int fd the file descriptor of the wwwoffle client.
  ++++++++++++++++++++++++++++++++++++++*/

void PurgeCache(int fd)
{
 time_t now=time(NULL)+60;
 int total_bytes=0,total_dirs=0;
 int i,pass;
 int age_for_size=-1;
 int *nbytes_by_age=(int*)malloc((DefaultPurgeAge+1)*sizeof(int));

 for(i=0;i<=DefaultPurgeAge;i++)
    nbytes_by_age[i]=0;

 for(pass=1;pass<=2;pass++)
   {
    DIR *dir;
    struct dirent* ent;
    char *msg=(char*)malloc(64);

    /* Open the spool directory. */

    dir=opendir(".");
    if(!dir)
      {PrintMessage(Warning,"Cannot open spool directory [%!s]; purge failed.");return;}

    ent=readdir(dir);  /* skip .  */
    if(!ent)
      {PrintMessage(Warning,"Cannot read spool directory [%!s]; purge failed.");closedir(dir);return;}
    ent=readdir(dir);  /* skip .. */

    if(PurgeCacheSize)
      {
       sprintf(msg,"\nPass %d: %s.\n",pass,pass==1?"Checking dates and sizes of files":"Purging files down to specified size");
       write(fd,msg,strlen(msg));
      }

    /* Search through all of the sub directories. */

    write(fd,"\n",1);

    while((ent=readdir(dir)))
      {
       struct stat buf;

       if(lstat(ent->d_name,&buf))
          PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);
       else
          if(S_ISDIR(buf.st_mode) && strcmp(ent->d_name,"outgoing"))
            {
             int bytes=0;
             int age=WhatPurgeAge(ent->d_name);
             char *msg=(char*)malloc(strlen(ent->d_name)+80);

             if(age<0)
                bytes=PurgeFiles(ent->d_name,now,-1,nbytes_by_age);
             else
               {
                if(pass==2)
                   age=age_for_size;
                bytes=PurgeFiles(ent->d_name,now,age,nbytes_by_age);

                if(bytes==-1)
                  {
#if DO_DELETE
                   if(rmdir(ent->d_name))
                      PrintMessage(Warning,"Cannot delete 'empty' directory '%s' [%!s].",ent->d_name);
#endif
                  }
                else
                  {
                   struct utimbuf utbuf;

                   utbuf.actime=buf.st_atime;
                   utbuf.modtime=buf.st_mtime;
                   utime(ent->d_name,&utbuf);
                  }
               }

             if(bytes>0)
                total_bytes+=bytes;
             if(bytes>=0)
                total_dirs++;

             if(age<0)
                sprintf(msg,"Not Purged       %30s ; %5d kB.\n",ent->d_name,(bytes+512)/1024);
             else if(bytes==-1)
                sprintf(msg,"Purged (%2d days) %30s ; empty.\n",age,ent->d_name);
             else
                sprintf(msg,"Purged (%2d days) %30s ; %5d kB.\n",age,ent->d_name,(bytes+512)/1024);
             write(fd,msg,strlen(msg));
             free(msg);
            }
      }

    closedir(dir);

    sprintf(msg,"Total of %d directories ; %d kB.\n",total_dirs,(total_bytes+512)/1024);
    write(fd,msg,strlen(msg));

    if(pass==1)
      {
       write(fd,"\n",1);

       for(total_bytes=i=0;i<=DefaultPurgeAge;i++)
         {
          total_bytes+=nbytes_by_age[i];

          if(PurgeCacheSize && age_for_size<0 && ((total_bytes+512)/1024)>(1024*PurgeCacheSize))
            {
             age_for_size=i;
             sprintf(msg,"Cutoff Age is %2d days for %3d MB.\n",age_for_size,PurgeCacheSize);
             write(fd,msg,strlen(msg));
            }

          if(i==DefaultPurgeAge)
             sprintf(msg,"Total all ages     ; %5d kB.\n",(total_bytes+512)/1024);
          else
             sprintf(msg,"Newer than %2d day%c ; %5d kB.\n",i+1,i?'s':' ',(total_bytes+512)/1024);
          write(fd,msg,strlen(msg));
         }
       total_bytes=0;
      }

    free(msg);

    if(!PurgeCacheSize || age_for_size==-1)
       break;
   }

 write(fd,"\n",1);

 free(nbytes_by_age);
}


/*++++++++++++++++++++++++++++++++++++++
  Delete the file in the current directory that are older than the specified age.

  int PurgeFiles Returns the number of bytes in files that are left.

  char *dirname The name of the directory to purge.

  time_t now The current time (in seconds).

  int age The age that is specified for the directory in days.

  int *bytes_age An array that contains the number of bytes for a given age.
  ++++++++++++++++++++++++++++++++++++++*/

static int PurgeFiles(char *dirname,time_t now,int age,int *bytes_age)
{
 int bytes_left=-1;
 long oldest=0;
 DIR *dir;
 struct dirent* ent;

 if(age>=0)
    oldest=now-age*(24*3600);

 /* Open the spool subdirectory. */

 if(chdir(dirname))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not purged.",dirname);return(1);}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not purged.",dirname);chdir("..");return(1);}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not purged.",dirname);closedir(dir);chdir("..");return(1);}
 ent=readdir(dir);  /* skip .. */

 /* Check all of the files for age, and delete as needed. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",dirname,ent->d_name);
    else
      {
       time_t t;

       if(PurgeUseMTime)
          t=buf.st_mtime;
       else
          t=buf.st_atime;

       if(t>oldest)
         {
          if(bytes_left==-1)
             bytes_left=0;
          bytes_left+=buf.st_size;
          if(age>0)
            {
             int days=(now-t)/(24*3600);
             if(days>DefaultPurgeAge)
                days=DefaultPurgeAge;
             bytes_age[days]+=buf.st_size;
            }
         }
#if DO_DELETE
       else
          if(unlink(ent->d_name))
             PrintMessage(Warning,"Cannot unlink file '%s/%s' [%!s].",dirname,ent->d_name);
#endif
      }
   }

 closedir(dir);
 chdir("..");

 return(bytes_left);
}
