/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffles.c 1.25 1997/03/24 17:36:51 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.1.
  A server to fetch the required pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"



/*+ The mode of operation of the server. +*/
typedef enum _Mode
{
 None,                          /*+ Undecided. +*/
 Real,                          /*+ From server host to cache and client. +*/
 RealNoCache,                   /*+ From server host to client. +*/
 RealRefresh,                   /*+ Refresh the page, forced from index. +*/
 Fetch,                         /*+ From server host to cache. +*/
 Spool,                         /*+ From cache to client. +*/
 SpoolGet,                      /*+ Not in cache so record request in outgoing. +*/
 SpoolRefresh,                  /*+ Refresh the page, forced from index. +*/
 Local                          /*+ A local page e.g. the index. +*/
}
Mode;


/*++++++++++++++++++++++++++++++++++++++
  The main server program.

  int wwwoffles Returns the exit status.

  int online Whether the demon is online or not.

  int browser Set to true if there is a browser.

  int client The file descriptor of the client.
  ++++++++++++++++++++++++++++++++++++++*/

int wwwoffles(int online,int browser,int client)
{
 int                     outgoing     =-1  , spool     =-1  ,server=-1;
 FILE *client_file=NULL,*outgoing_file=NULL,*spool_file=NULL;

 char *request=NULL,*reply=NULL,*url=NULL;
 int replylen=0;

 Mode mode=None;
 int fetch_again=0;

 char *host,*path,*args;
 char *server_host=NULL;
 int server_port=80;

 /* Initialise things. */

 InitErrorHandler("wwwoffles",1);

 signal(SIGPIPE,SIG_IGN);
 signal(SIGINT ,SIG_DFL);
 signal(SIGQUIT,SIG_DFL);
 signal(SIGTERM,SIG_DFL);
#if defined(SIGCHLD)
 signal(SIGCHLD,SIG_DFL);
#else
 signal(SIGCLD ,SIG_DFL);
#endif

 if(online && browser)
    mode=Real;
 else if(online && !browser)
    mode=Fetch;
 else if(!online && browser)
    mode=Spool;
 else
    PrintMessage(Fatal,"Started in a mode that is not allowed.");

 /* Set up the client file. */

 if(client<0)
   {PrintMessage(Warning,"Cannot use client file descriptor %d.",client);exit(1);}

 client_file=fdopen(client,"r");
 if(!client_file)
   {PrintMessage(Warning,"Cannot use fdopen on the client file descriptor.");exit(1);}

 /* Set up the outgoing file. */

 if(mode==Fetch)
   {
    outgoing=OpenOutgoingSpoolFile(1);

    PrintMessage(Debug,"outgoing=%d",outgoing);

    if(outgoing==-1)
      {PrintMessage(Warning,"Cannot open the spooled outgoing request to read.");exit(1);}
    else if(outgoing==-2)
      {PrintMessage(Inform,"No more outgoing requests.");exit(3);}

    outgoing_file=fdopen(outgoing,"r");
    if(!outgoing_file)
      {PrintMessage(Warning,"Cannot use fdopen on the outgoing file descriptor.");exit(1);}
   }

 /* Get the URL from the request. */

 if(mode==Real || mode==Spool)
    request=ParseRequest(client_file,&url);
 else /* mode==Fetch */
    request=ParseRequest(outgoing_file,&url);

 if(!url)
   {
    ServerError(client,request?"Cannot parse the HTTP request":"The HTTP request was empty");
    PrintMessage(Warning,"Could not parse HTTP request (%s).",request?"Parse error":"Empty request");
    exit(1);
   }

 PrintMessage(Inform,"URL = %s",url);

 SplitURL(url,&host,&path,&args);

 /* Change the mode based on the URL as required. */

 if((mode==Real || mode==Spool) && !*host)
   {
    if(!strncmp(path,"outgoing/",9))
      {
       char *copy,*u,*p;

       copy=(char*)malloc(strlen(request)+1);
       p=strstr(request,path)+9;
       u=strstr(request,url); *u=0;
       strcpy(copy,request);
       strcat(copy,"http://");
       strcat(copy,p);
       free(request);
       request=copy;

       copy=(char*)malloc(strlen(url)+1);
       p=strstr(url,path)+9;
       strcpy(copy,"http://");
       strcat(copy,p);
       free(url);
       url=copy;

       SplitURL(url,&host,&path,&args);

       if(!*host || !IsCachedHost(host))
          mode=RealNoCache;
       else if(mode==Real)
         {
          mode=RealRefresh;
          OpenWebpageSpoolFile(-1,host,path,args);
         }
       else if(mode==Spool)
         {
          spool=OpenWebpageSpoolFile(1,host,path,args);
          if(spool!=-1)
            {mode=SpoolRefresh;close(spool);}
          else
             mode=SpoolGet;
         }
      }
    else
       mode=Local;
   }
 else if((mode==Real || mode==Spool) && !IsCachedHost(host))
   {
    mode=RealNoCache;
   }
 else if(mode==Fetch && (!*host || !IsCachedHost(host)))
   {
    PrintMessage(Inform,"A request to fetch a page from a non-cached host is ignored.");
    exit(0);
   }
 else if(mode==Real || mode==Spool || mode==Fetch)
   {
    spool=OpenWebpageSpoolFile(1,host,path,args);

    if(mode==Spool && spool==-1)
       mode=SpoolGet;
    if(mode==Fetch && spool!=-1)
       if(RequestChanges(spool,&request))
          OpenWebpageSpoolFile(-2,host,path,args);
       else
          exit(0);
    if(mode==Real && spool!=-1)
       if(RequestChanges(spool,&request))
          OpenWebpageSpoolFile(-2,host,path,args);
       else
          mode=Spool;
    if(spool!=-1)
       close(spool);
   }

 /* Set up the file descriptor for the spool file. */

 if(mode==Real || mode==Fetch || mode==SpoolGet)
   {
    spool=OpenWebpageSpoolFile(0,host,path,args);

    if(spool==-1)
      {
       if(mode==Real || mode==SpoolGet)
          ServerError(client,"Cannot open the spooled web page to write.");
       PrintMessage(Warning,"Cannot open the spooled web page to write.");
       exit(1);
      }
   }
 else if(mode==Spool)
   {
    spool=OpenWebpageSpoolFile(1,host,path,args);

    if(spool==-1)
      {
       ServerError(client,"Cannot open the spooled web page to read.");
       PrintMessage(Warning,"Cannot open the spooled web page to read.");
       exit(1);
      }
   }

 /* Set up the outgoing file. */

 if(mode==SpoolGet || mode==SpoolRefresh)
   {
    outgoing=OpenOutgoingSpoolFile(0);

    if(outgoing==-1)
      {
       ServerError(client,"Cannot open the outgoing request to write.");
       PrintMessage(Warning,"Cannot open the outgoing request to write.");
       exit(1);
      }
   }

 /* Set up the file descriptor for the server host. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *colon;
    char *proxy=WhichProxy(host);

    if(proxy)
      {
       server_host=(char*)malloc(strlen(proxy)+1);
       strcpy(server_host,proxy);
      }
    else
      {
       server_host=(char*)malloc(strlen(host)+1);
       strcpy(server_host,host);
       MakeRequestNonProxy(url,path,args,&request);
      }

    if((colon=strchr(server_host,':')))
      {
       *colon++=0;
       if(*colon)
          server_port=atoi(colon);
      }

    server=OpenClientSocket(server_host,server_port);

    if(server==-1)
      {
       if(mode==Fetch)
          RemoteHostError(spool,url,server_host,server_port,"could not be connected.");
       if(mode==Real || mode==RealNoCache)
          RemoteHostError(client,url,server_host,server_port,"could not be connected.");

       PrintMessage(Warning,"Cannot open a socket to the remote host %s port %d.",server_host,server_port);
       exit(1);
      }
   }

 /* Write request to remote server. */

 if(mode==Real || mode==RealNoCache)
   {
    int err=write(server,request,strlen(request));

    if(err==-1)
      {
       RemoteHostError(client,url,server_host,server_port,"could not be written to.");
       PrintMessage(Warning,"Failed to write to remote host %s.",server_host);
       exit(1);
      }
   }
 else if(mode==Fetch)
   {
    int err=write(server,request,strlen(request));

    if(err==-1)
      {
       RemoteHostError(spool,url,server_host,server_port,"could not be written to.");
       PrintMessage(Warning,"Failed to write to remote host %s.",server_host);
       exit(1);
      }
    else
      {
       char *buffer=(char*)malloc(strlen(url)+16);
       sprintf(buffer,"Fetching %s\n",url);
       write(client,buffer,strlen(buffer));
       free(buffer);
      }
   }
 else if(mode==SpoolGet || mode==SpoolRefresh)
   {
    int err=write(outgoing,request,strlen(request));

    if(err==-1)
      {
       ServerError(client,"Cannot write the outgoing request.");
       PrintMessage(Warning,"Cannot write the outgoing request.");
       exit(1);
      }
   }

 /* Parse the reply */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    int status=0;

    reply=ParseReply(server,&status,&replylen);

    if(!reply)
      {
       if(mode==Real || mode==RealNoCache)
          RemoteHostError(spool,url,server_host,server_port,"timed out reading the reply.");
       PrintMessage(Warning,"Timed out reading the reply.");
       exit(1);
      }

    if(mode==Fetch && (status==301 || status==302))
      {
       char *new_request;

       PrintMessage(Inform,"Server page has been moved.");

       new_request=RequestMoved(request,reply);
       if(!new_request)
          PrintMessage(Warning,"Cannot parse the reply for the new location.");
       else
         {
          int new_outgoing=OpenOutgoingSpoolFile(0);
          if(new_outgoing==-1)
             PrintMessage(Warning,"Cannot open the new outgoing request to write.");
          else
            {
             write(new_outgoing,new_request,strlen(new_request));
             CloseOutgoingSpoolFile(new_outgoing);
             free(new_request);
             fetch_again++;
            }
         }
      }
    else if(status==304)
      {
       PrintMessage(Inform,"Server page is not newer than the one in cache.");
       if(mode==Fetch)
          exit(0);
       if(mode==Real)
          mode=Spool;
      }

    if(replylen<=256)
       reply=(char*)realloc(reply,257);
   }
 else
    reply=(char*)malloc(257);

 /* Read reply and process it. */

 if(mode==Real || mode==RealNoCache)
   {
    int n=replylen,err=0;

    do
      {
       if(mode==Real)
          write(spool,reply,n);
       err=write(client,reply,n);
      }
    while(err!=-1 && (n=ReadOrTimeout(server,reply,256))>0);

    if(mode==Real && (err==-1 || n<0))
      {
       lseek(spool,0,SEEK_SET);
       ftruncate(spool,0);
       if(err==-1)
         {
          RemoteHostError(spool,url,server_host,server_port,"client disconnected during the transfer.");
          PrintMessage(Warning,"Client disconnected during the transfer.");
          exit(1);
         }
       else
         {
          RemoteHostError(spool,url,server_host,server_port,"timed out during the transfer.");
          PrintMessage(Warning,"Timed out while reading from remote host.");
          exit(1);
         }
      }
   }
 else if(mode==Fetch)
   {
    int n=replylen;

    do
      {
       write(spool,reply,n);
      }
    while((n=ReadOrTimeout(server,reply,256))>0);

    if(n<0)
      {
       lseek(spool,0,SEEK_SET);
       ftruncate(spool,0);
       RemoteHostError(spool,url,server_host,server_port,"timed out during the transfer.");
       PrintMessage(Warning,"Timed out while reading from remote host.");
       exit(1);
      }

    if(FetchImages)
      {
       char **images;

       lseek(spool,0,SEEK_SET);
       spool_file=fdopen(spool,"r");
       if(!spool_file)
         {PrintMessage(Warning,"Cannot use fdopen on the spool file descriptor.");exit(1);}

       if(ParseHTML(spool_file,host,path) && (images=ListImages()))
          for(n=0;images[n];n++)
            {
             char *new_request=RequestURL(images[n],url);
             int new_outgoing=OpenOutgoingSpoolFile(0);

             PrintMessage(Debug,"Image=%s",images[n]);

             if(new_outgoing==-1)
                PrintMessage(Warning,"Cannot open the new outgoing request to write.");
             else
               {
                write(new_outgoing,new_request,strlen(new_request));
                CloseOutgoingSpoolFile(new_outgoing);
                fetch_again++;
               }

             free(new_request);
            }
      }
   }
 else if(mode==Spool)
   {
    char *head="HTTP/1.0 503 WWWOFFLE Remote Host Error\r\n"; /* This line must not be changed (see messages.c). */
    int n=read(spool,reply,64);

    if(!strncmp(reply,head,strlen(head)))
       OpenWebpageSpoolFile(-1,host,path,args);

    if(AddInfoRefresh)
      {
       struct stat buf;
       int status=0;

       sscanf(reply,"%*s %d",&status);

       lseek(spool,0,SEEK_SET);
       spool_file=fdopen(spool,"r");
       if(!spool_file)
         {PrintMessage(Warning,"Cannot use fdopen on the spool file descriptor.");exit(1);}

       if((status>=200 && status<400) && ParseHTML(spool_file,host,path) && !fstat(spool,&buf))
         {
          int position=GetHTMLEnd();
          char *msg=(char*)malloc(strlen(url)+256);

          strcpy(msg,"\n<hr>\n<p align=center>\n");

          if(!args)
             sprintf(&msg[strlen(msg)],
                     "WWWOFFLE - Cached %s - <a href=\"http://%s:%d/outgoing/%s/%s\">[Refresh]</a> - WWWOFFLE",
                     RFC822Date(buf.st_mtime),GetLocalHost(),HTTP_Port,host,path[0]=='/'?"":path);
          else if(*args=='!')
             sprintf(&msg[strlen(msg)],
                     "WWWOFFLE - Cached %s - [Posted] - WWWOFFLE",
                     RFC822Date(buf.st_mtime));
          else
             sprintf(&msg[strlen(msg)],
                     "WWWOFFLE - Cached %s - <a href=\"http://%s:%d/outgoing/%s/%s?%s\">[Refresh]</a> - WWWOFFLE",
                     RFC822Date(buf.st_mtime),GetLocalHost(),HTTP_Port,host,path[0]=='/'?"":path,args);

          strcat(msg,"\n</p>\n<hr>\n");

          fseek(spool_file,0,SEEK_SET);
          while((reply=fgets_realloc(reply,spool_file)))
            {
             if(!strncasecmp(reply,"Content-Length:",15))
                sprintf(reply,"Content-Length: %d\r\n",atoi(&reply[16])+strlen(msg));
             write(client,reply,strlen(reply));
             if(reply[0]=='\r' || reply[0]=='\n')
                break;
            }

          while(position>0 && (n=fread(reply,1,position>256?256:position,spool_file))>0)
            {
             write(client,reply,n);
             position-=n;
            }

          write(client,msg,strlen(msg));

          while((n=fread(reply,1,256,spool_file))>0)
            {
             write(client,reply,n);
            }

          free(msg);
         }
       else
         {
          fseek(spool_file,0,SEEK_SET);
          while((n=fread(reply,1,256,spool_file))>0)
            {
             write(client,reply,n);
            }
         }
      }
    else
       do
         {
          write(client,reply,n);
         }
       while((n=read(spool,reply,256))>0);
   }
 else if(mode==SpoolGet)
   {
    WillGetURL(client,url,0);
    WillGetURL(spool,url,1);
   }
 else if(mode==RealRefresh || mode==SpoolRefresh)
   {
    RefreshRedirect(client,url);
   }
 else /* mode==Local */
   {
    if(!*path && !args)
       WelcomePage(client,PassWord?1:0);
    else if(!strncmp("index/",path,6))
       IndexPage(client,&path[6],args);
    else if(!strncmp("control/",path,8))
       ControlPage(client,&path[8],request);
    else if(!strncmp("refresh/",path,8))
       RefreshPage(client,&path[8],request);
    else
       IllegalPage(client,path,args);
   }

 /* Close down and exit. */

 if(server>=0)
    CloseSocket(server);

 if(outgoing_file)
    fclose(outgoing_file);
 else if(outgoing>=0)
    CloseOutgoingSpoolFile(outgoing);

 if(spool_file)
    fclose(spool_file);
 else if(spool>=0)
    close(spool);

 if(request)
    free(request);

 if(reply)
    free(reply);

 if(client>=0)
    fclose(client_file);

 if(fetch_again)
    return(4);
 else
    return(0);
}
