/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffle.c 1.12 1997/03/24 17:37:27 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.1.
  A user level program to interact with the server.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/
 Config,                        /*+ Tell the server to re-read the configuration file. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Get                            /*+ Tell the server to get pages. +*/
}
Action;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int i;
 int images=0,recursive=0,recursive_any=0;

 Action action=None;

 char *host=NULL;
 int port=0;

 /* Parse the command line options */

 ConfigFile=NULL;

 if(argc==1)
    usage(0);

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-i"))
       {
        if(images)
          {fprintf(stderr,"wwwoffle: Only one '-i' argument may be given.\n"); exit(1);}

        images=1;
        argv[i]=NULL;
        continue;
       }

    if(!strncasecmp(argv[i],"-r",2))
       {
        if(recursive)
          {fprintf(stderr,"wwwoffle: Only one '-r' or '-R' argument may be given.\n"); exit(1);}

        if(argv[i][1]=='R')
           recursive_any=1;

        if(argv[i][2])
           recursive=atoi(&argv[i][2]);
        else
           recursive=1;
        if(recursive==0)
          {fprintf(stderr,"wwwoffle: The '-r' or '-R' argument may only be followed by an non-zero integer.\n"); exit(1);}

        argv[i]=NULL;
        continue;
       }

    if(!strcmp(argv[i],"-p"))
      {
       char *colon;

       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-p' argument requires a hostname and optionally a port number.\n"); exit(1);}

       if(ConfigFile)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       if((colon=strchr(argv[i],':')))
         {
          *colon++=0;

          port=atoi(colon);

          if(port<0 || port>=65536)
            {fprintf(stderr,"wwwoffle: The port number %d '%s' is invalid.\n",port,argv[i]); exit(1);}
         }

       host=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-c"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-c' argument requires a configuration file name.\n"); exit(1);}

       if(host)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       ConfigFile=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-on") || !strcmp(argv[i],"-online"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n");exit(1);}
       action=Online;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-off") || !strcmp(argv[i],"-offline"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n");exit(1);}
       action=Offline;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-fetch"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n");exit(1);}
       action=Fetch;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-config"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n");exit(1);}
       action=Config;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-purge"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n");exit(1);}
       action=Purge;
       argv[i]=NULL;
       continue;
      }

    if(argv[i][0]=='-')
      {
       fprintf(stderr,"wwwoffle: Unknown option '%s'.\n",argv[i]);
       exit(1);
      }
   }

 if(action==None)
    action=Get;
 else
   {
    if(images)
      {fprintf(stderr,"wwwoffle: The '-i' argument is not appropriate here.\n"); exit(1);}

    if(recursive)
      {fprintf(stderr,"wwwoffle: The '-r' or '-R' argument is not appropriate here.\n"); exit(1);}
   }

 /* Initialise things. */

 InitErrorHandler("wwwoffle",0);

 if(ConfigFile)
   {
    if(ReadConfigFile(2))
       PrintMessage(Fatal,"Error in configuration file '%s'.",ConfigFile);

    host=GetLocalHost();
   }

 /* The connections to the WWWOFFLE server. */

 if(action!=Get)
   {
    int socket;
    FILE *server_r,*server_w;
    char *buffer=NULL;

    socket=OpenClientSocket(host?host:"localhost",port?port:WWWOFFLE_Port);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:WWWOFFLE_Port);

    server_r=fdopen(socket,"r");
    server_w=fdopen(socket,"w");

    /* Send the message. */

    if(PassWord)
       fprintf(server_w,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       fprintf(server_w,"WWWOFFLE ONLINE\r\n");
    else if(action==Offline)
       fprintf(server_w,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       fprintf(server_w,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       fprintf(server_w,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       fprintf(server_w,"WWWOFFLE PURGE\r\n");

    fprintf(server_w,"\r\n");
    fflush(server_w);

    while((buffer=fgets_realloc(buffer,server_r)))
       fputs(buffer,stdout);

    fclose(server_w);
    fclose(server_r);
   }

 /* The connections to the http proxy. */

 else /* action==Get */
   {
    for(i=1;i<argc;i++)
       if(argv[i])
         {
          if(!strncmp(argv[i],"http:",5))
            {
             argv[i]+=5;
             if(*argv[i]=='/') argv[i]++;
             if(*argv[i]=='/') argv[i]++;
            }

          RecurseFetchPages(0,argv[i],
                            images,recursive,recursive_any,
                            host?host:"localhost",port?port:HTTP_Port);
         }
   }

 /* exit. */

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fputs("\n"
       "WWWOFFLE - World Wide Web Offline Explorer - Version 1.1\n"
       "\n",stderr);

 if(verbose)
    fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
          "                             [http://www.gedanken.demon.co.uk/]\n"
          "\n",stderr);

 fputs("Usage: wwwoffle -h\n"
       "       wwwoffle -online   [-p <host>[:<port>] | -c <config-file>]\n"
       "       wwwoffle -offline  [-p <host>[:<port>] | -c <config-file>]\n"
       "       wwwoffle -fetch    [-p <host>[:<port>] | -c <config-file>]\n"
       "       wwwoffle -config   [-p <host>[:<port>] | -c <config-file>]\n"
       "       wwwoffle -purge    [-p <host>[:<port>] | -c <config-file>]\n"
       "       wwwoffle <url> ... [-p <host>[:<port>]] [-i] [-r[<depth>] | -R[<depth>]]\n"
       "\n",stderr);

 if(verbose)
    fprintf(stderr,
            "wwwoffle -h          : Display this help.\n"
            "\n"
            "wwwoffle -online     : Indicate to the server that the network is active.\n"
            "                       (Proxy requests will be fetched from remote hosts.)\n"
            "\n"
            "wwwoffle -offline    : Indicate to the server that the network is inactive.\n"
            "                       (Proxy requests will be fetched from cache or recorded.)\n"
            "\n"
            "wwwoffle -fetch      : Force the server to fetch the pages that are recorded.\n"
            "\n"
            "wwwoffle -config     : Force the server to re-read the configuration file.\n"
            "\n"
            "wwwoffle -purge      : Force the server to purge pages from the cache.\n"
            "\n"
            "wwwoffle <url> ...   : Fetch the specified URLs (or record for fetching).\n"
            "\n"
            " -p <host>[:<port>]  : The host name and port number to talk to the demon on.\n"
            "                       (Defaults to localhost for the server and\n"
            "                        %d for server port, %d for http proxy port).\n"
            "\n"
            " -c <config-file>    : The name of the configuration file with the hostname,\n"
            "                       port number and the password (if any).\n"
            "\n"
            " -i                  : Fetch the images included in the specified URLs.\n"
            " -r[<depth>]         : Fetch pages linked to the URLs on the same host and\n"
            "                       their links, in no more than <depth> steps (default 1).\n"
            " -R[<depth>]         : Fetch pages linked to the URLs on any host and\n"
            "                       their links, in no more than <depth> steps (default 1).\n"
            "\n",DEF_WWWOFFLE_PORT,DEF_HTTP_PORT);

 exit(0);
}
