/***************************************
  $Header: /home/amb/wwwoffle/RCS/purge.c 1.8 1997/03/16 17:30:43 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.1.
  Purge old files from the cache.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>

#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "errors.h"


/* Local functions */
static int PurgeFiles(char *dirname,int oldest);


/*++++++++++++++++++++++++++++++++++++++
  Purge files from the cache that meet the age criteria.

  int fd the file descriptor of the wwwoffle client.
  ++++++++++++++++++++++++++++++++++++++*/

void PurgeCache(int fd)
{
 DIR *dir;
 struct dirent* ent;
 time_t now=time(NULL);
 int total_bytes=0,total_dirs=0;
 char *msg;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory [%!s]; purge failed.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory [%!s]; purge failed.");closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Search through all of the sub directories. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);
    else
       if(S_ISDIR(buf.st_mode) && strcmp(ent->d_name,"outgoing"))
         {
          int bytes=0;
          int age=WhatPurgeAge(ent->d_name);
          char *msg=(char*)malloc(strlen(ent->d_name)+80);

          if(age<0)
             bytes=PurgeFiles(ent->d_name,0);
          else
            {
             bytes=PurgeFiles(ent->d_name,now-age*24*3600+60);

             if(!bytes)
               {
                if(rmdir(ent->d_name))
                   PrintMessage(Warning,"Cannot delete 'empty' directory '%s' [%!s].",ent->d_name);
               }
             else
               {
                struct utimbuf utbuf;

                utbuf.actime=buf.st_atime;
                utbuf.modtime=buf.st_mtime;
                utime(ent->d_name,&utbuf);
               }
            }

          total_bytes+=bytes;
          if(bytes)
             total_dirs++;

          if(age<0)
             sprintf(msg,"Not Purged       %30s ; %4d kB.\n",ent->d_name,(bytes+512)/1024);
          else if(bytes==0)
             sprintf(msg,"Purged (%2d days) %30s ; empty.\n",age,ent->d_name);
          else
             sprintf(msg,"Purged (%2d days) %30s ; %4d kB.\n",age,ent->d_name,(bytes+512)/1024);
          write(fd,msg,strlen(msg));
          free(msg);
         }
   }

 closedir(dir);

 msg=(char*)malloc(64);
 sprintf(msg,"Total of %d directories ; %d kB.\n",total_dirs,(total_bytes+512)/1024);
 write(fd,msg,strlen(msg));
 free(msg);
}


/*++++++++++++++++++++++++++++++++++++++
  Delete the file in the current directory that are older than the specified age.

  int PurgeFiles Returns the number of bytes in files that are left.

  char *dirname The name of the directory to purge.

  int oldest The age of the oldest file to leave behind (in seconds).
  ++++++++++++++++++++++++++++++++++++++*/

static int PurgeFiles(char *dirname,int oldest)
{
 int bytes_left=0;
 DIR *dir;
 struct dirent* ent;

 /* Open the spool subdirectory. */

 if(chdir(dirname))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not purged.",dirname);return(1);}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not purged.",dirname);chdir("..");return(1);}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not purged.",dirname);closedir(dir);chdir("..");return(1);}
 ent=readdir(dir);  /* skip .. */

 /* Check all of the files for age, and delete as needed. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",dirname,ent->d_name);
    else
      {
       time_t t;

       if(PurgeUseMTime)
          t=buf.st_mtime;
       else
          t=buf.st_atime;

       if(t>oldest)
          bytes_left+=buf.st_size;
       else
          if(unlink(ent->d_name))
             PrintMessage(Warning,"Cannot unlink file '%s/%s' [%!s].",dirname,ent->d_name);
      }
   }

 closedir(dir);
 chdir("..");

 return(bytes_left);
}
