/***************************************
  $Header: /home/amb/wwwoffle/RCS/parse.c 1.17 1997/03/10 08:41:23 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.1.
  Functions to parse the HTTP requests.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "errors.h"
#include "config.h"


/*++++++++++++++++++++++++++++++++++++++
  Parse the request to the server.

  char *ParseRequest Returns the request.

  FILE *req The file pointer to read the request from.

  char **url Return the url or NULL if it failed.
  ++++++++++++++++++++++++++++++++++++++*/

char *ParseRequest(FILE *req,char **url)
{
 char *request=NULL,*buf=NULL;
 int post=0,n=0,length=-1;

 *url=NULL;

 buf=fgets_realloc(buf,req);

 if(!buf)
    return(NULL);

 *url=(char*)malloc(strlen(buf));

 if(sscanf(buf,"%*s %s",*url)!=1)
   {free(*url);*url=NULL;return(NULL);}

 if(!strncasecmp("POST",buf,4))
    post=1;

 request=(char*)malloc(strlen(buf)+1);
 request[0]=0;

 do
   {
    char *pling;

    if(!strncasecmp("Referer:",buf,8) && (pling=strstr(buf,"?!")))
       strcpy(pling,"\r\n");

    if(!strncasecmp("If-Modified-Since:",buf,18))
       continue;

    if(!strncasecmp("Content-length:",buf,15))
       length=atoi(&buf[15]);

    if(n)
       request=(char*)realloc(request,n+strlen(buf)+1);
    strcpy(&request[n],buf);
    n+=strlen(buf);
   }
 while(buf && buf[0]!='\r' && buf[0]!='\n' && (buf=fgets_realloc(buf,req)));

 if(buf)
    free(buf);

 if(post)
   {
    int m;

    if(length==-1)
      {free(*url);*url=NULL;return(request);}

    if(length)
      {
       request=(char*)realloc(request,n+length+1);
       m=fread(&request[n],1,length,req);

       if(m!=length)
         {free(*url);*url=NULL;return(request);}

       request[n+length]=0;
      }

    *url=(char*)realloc(*url,strlen(*url)+20);
    sprintf(*url+strlen(*url),"?!%08lx.%04lx",MakeHashFromArgs(&request[n]),(unsigned long)length);
   }

 return(request);
}


/*++++++++++++++++++++++++++++++++++++++
  Modify the request to ask for changes since the spooled file.

  int RequestChanges Returns 1 if the file needs changes made.

  int fd The file descriptor of the spooled file.

  char **request The request to modify.
  ++++++++++++++++++++++++++++++++++++++*/

int RequestChanges(int fd,char **request)
{
 struct stat buf;
 char *reply=(char*)malloc(32);
 int status=0;
 int n=read(fd,reply,32);

 if(n)
    sscanf(reply,"%*s %d",&status);
 free(reply);

 if(n==0 || status==0)
   {
    PrintMessage(Debug,"Empty");
    return(0);
   }
 else if(status>=200 && status<400 && !fstat(fd,&buf))
   {
    if((time(NULL)-buf.st_mtime)<3600)
      {
       PrintMessage(Debug,"Too new");
       return(0);
      }
    else
      {
       char *if_mod=(char*)malloc(64);
       char *copy=(char*)malloc(strlen(*request)+64);
       char *eol=strchr(*request,'\n');

       sprintf(if_mod,"If-Modified-Since: %s",RFC822Date(buf.st_mtime));
       if_mod[strlen(if_mod)-1]=0;
       *eol=0; eol++;

       strcpy(copy,*request);
       strcat(copy,"\n");
       strcat(copy,if_mod);
       strcat(copy,"\r\n");
       strcat(copy,eol);

       free(*request);
       free(if_mod);

       *request=copy;
      }
   }

 return(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Create a new request when the reply says that the page has been moved.

  char *RequestMoved Returns the new request.

  char *request The original request.

  char *reply The original reply.
  ++++++++++++++++++++++++++++++++++++++*/

char *RequestMoved(char *request,char *reply)
{
 char *location,*eol;
 char *new;
 int start,stop;

 sscanf(request,"%*s %n%*s%n",&start,&stop);

 location=strstr(reply,"\nLocation:");
 if(!location)
    location=strstr(reply,"\nlocation:");
 if(!location)
    return(NULL);

 location+=10;
 eol=strchr(location,'\n');
 *eol=0;
 new=(char*)malloc(strlen(request)+strlen(location));
 *eol='\n';

 strncpy(new,request,start);
 if(sscanf(location,"%s",&new[start])==0)
   {free(new);return(NULL);}

 while(new[start])
    start++;

 while(request[stop])
    new[start++]=request[stop++];

 new[start]=0;

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Create a new request when the reply says that the page has been moved.

  char *RequestURL Ask for an image.

  char *url The URL to get.

  char *referer The Refering URL.
  ++++++++++++++++++++++++++++++++++++++*/

char *RequestURL(char *url,char *referer)
{
 char *new=(char*)malloc(strlen(url)+strlen(referer)+64);

 sprintf(new,"GET %s HTTP/1.0\r\n"
             "Referer: %s\r\n"
             "Accept: */*\r\n"
             "\r\n",url,referer);

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Change the request from one to a proxy to a normal one.

  char *url The URL that we are trying to get.

  char *path The path we are going to get.

  char *args The arguments to the URL we are going to get.

  char **request The buffer containing the request.
  ++++++++++++++++++++++++++++++++++++++*/

void MakeRequestNonProxy(char *url,char *path,char *args,char **request)
{
 int start,stop;

 sscanf(*request,"%*s %n%*s%n",&start,&stop);

 if(args && *args!='!')
    sprintf(&(*request)[start],"/%s?%s",path,args);
 else
    sprintf(&(*request)[start],"/%s",path);

 while((*request)[start])
    start++;

 while((*request)[stop])
    (*request)[start++]=(*request)[stop++];

 (*request)[start]=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the reply from the server.

  char *ParseReply Return the chunk of data that was read.

  int rep The file descriptor to read the reply from.

  int *status Return the numeric status code.

  int *length The length of the reply.
  ++++++++++++++++++++++++++++++++++++++*/

char *ParseReply(int rep,int *status,int *length)
{
 char *reply=(char*)malloc(65);
 int m,n=0;

 while((m=ReadOrTimeout(rep,&reply[n],64))>0)
   {
    n+=m;
    reply[n]=0;

    if(strstr(reply,"\n\r\n") || strstr(reply,"\n\n"))
       break;

    reply=(char*)realloc(reply,n+65);
   }

 if(m<0)
   {free(reply);return(NULL);}

 if(n==0 || sscanf(reply,"%*s %d",status)!=1)
    *status=0;

 *length=n;

 return(reply);
}


/*++++++++++++++++++++++++++++++++++++++
  Split a URL into a hostname, a path name and an argument list.

  char *url The URL to split.

  char **host Returns a pointer to the hostname.

  char **path Returns a pointer to the path of the file.

  char **args Returns a pointer to the arguments if any to the URL.
  ++++++++++++++++++++++++++++++++++++++*/

void SplitURL(char *url,char **host,char **path,char **args)
{
 int i=0;
 char *copyurl=malloc(strlen(url)+2);

 strcpy(copyurl,url);

 *host=NULL;
 *path=NULL;
 *args=NULL;

 if(!strncasecmp(copyurl,"http:",5))
   {
    i=5;
    if(copyurl[i]=='/') i++;
    if(copyurl[i]=='/') i++;
   }

 *host=&copyurl[i];

 for(;copyurl[i];i++)
    if(copyurl[i]=='/' || copyurl[i]=='?')
       break;

 if(copyurl[i]=='/')
   {
    while(copyurl[i]=='/')
       copyurl[i++]=0;

    *path=&copyurl[i];

    for(;copyurl[i];i++)
       if(copyurl[i]=='?')
          break;
   }

 if(copyurl[i]=='?')
   {
    copyurl[i++]=0;
    *args=&copyurl[i];
   }

 if(IsLocalHost(*host))
    *host="";

 if(!*path || !**path)
    *path="";

 for(i=0;(*host)[i] && (*host)[i]!=':';i++)
    (*host)[i]=tolower((*host)[i]);

 if((*host)[i]==':' && !strcmp(&(*host)[i],":80"))
    (*host)[i]=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Generate a hash value depending on the arguments appended to the URL.

  unsigned long MakeHashFromArgs Returns a long integer to use.

  char *args The arguments.
  ++++++++++++++++++++++++++++++++++++++*/

unsigned long MakeHashFromArgs(char *args)
{
 int i;
 unsigned long hash=0;

 if(args)
    for(i=0;args[i];)
      {
       unsigned long subhash=args[i]&0x7f;

       for(i++;i%25 && args[i];i++)
          subhash=(subhash<<1)^(unsigned long)(args[i]&0x7f);

       hash^=subhash;
      }

 return(hash);
}


/*++++++++++++++++++++++++++++++++++++++
  Convert the time into an RFC 822 compliant date.

  char *RFC822Date Returns a pointer to a fixed string containing the date.

  long t The time.
  ++++++++++++++++++++++++++++++++++++++*/

char *RFC822Date(long t)
{
 static char *week[7]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
 static char *month[12]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
 static char value[32];
 struct tm *tim=gmtime(&t);

 /*             Sun, 06 Nov 1994 08:49:37 GMT    ; RFC 822, updated by RFC 1123 */
 sprintf(value,"%3s, %02d %3s %4d %02d:%02d:%02d GMT",
         week[tim->tm_wday],
         tim->tm_mday,
         month[tim->tm_mon],
         tim->tm_year+1900,
         tim->tm_hour,
         tim->tm_min,
         tim->tm_sec);

 return(value);
}
