/***************************************
  $Header: /home/amb/wwwoffle/RCS/index.c 1.14 1997/03/17 17:08:59 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.1.
  Generate an index of the web pages that are cached in the system.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "errors.h"


/*+ A type to contain the information required to sort the files. +*/
typedef struct _FileIndex
{
 char *name;                    /*+ The name of the file. +*/
 char *type;                    /*+ The type (file extension) of the file. +*/
 long time;                     /*+ The time of the file, +*/
}
FileIndex;

/*+ The method of sorting used. +*/
static char *sorttype[][2]={{"none"  ,"Unsorted"},
                            {"mtime" ,"Modification&nbsp;Time"},
                            {"atime" ,"Access&nbsp;Time"},
                            {"alpha" ,"Alphabetical"},
                            {"type"  ,"File&nbsp;Type"},
                            {"latest","Latest"}};

/*+ An enumerated type for the sort modes. +*/
typedef enum _SortMode
{
 None,                          /*+ No sorting, natural order. +*/
 MTime,                         /*+ Sort by modification time +*/
 ATime,                         /*+ Sort by access time. +*/
 Alpha,                         /*+ Sort Alphabetically. +*/
 Type,                          /*+ Sort by type (file extension). +*/
 Latest,                        /*+ Sort all files in one list. +*/
 NSortModes                     /*+ The number of sort modes. +*/
}
SortMode;


/*+ The list of files. +*/
static FileIndex **files=NULL;
int nfiles=0;


static void IndexRoot(int fd,char *args,SortMode mode);
static void IndexDir(int fd,char *dirname,SortMode mode);
static void IndexLatest(int fd);

static void CorrectIndex(int fd,char *path,char *args);
static void IllegalIndex(int fd,char *path,char *args);

static void add_dir(char *name,struct stat *buf,SortMode mode);
static void add_file(char *name,struct stat *buf,SortMode mode,char *dir);

static int sort_alpha(FileIndex **a,FileIndex **b);
static int sort_type(FileIndex **a,FileIndex **b);
static int sort_time(FileIndex **a,FileIndex **b);


/*++++++++++++++++++++++++++++++++++++++
  Generate an index of the pages that are in the cache.

  int fd The file descriptor to write the output to.

  char *url The URL that specifies the path to generate the index for.
  ++++++++++++++++++++++++++++++++++++++*/

void IndexPage(int fd,char *path,char *args)
{
 char *head=
 "HTTP/1.0 200 WWWOFFLE Index\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Index\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Cache Index</H1>\n"
 "<p align=center>\n";
 char *tail=
 "</BODY>\n"
 "</HTML>\n";
 SortMode sort=NSortModes;
 char *newpath=(char*)malloc(strlen(path)+1);
 char *buffer=(char*)malloc(strlen(path)+64);
 struct stat buf;

 strcpy(newpath,path);

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 if(args)
    for(sort=0;sort<NSortModes;sort++)
       if(!strcmp(args,sorttype[sort][0]))
          break;

 if(strchr(newpath,'/') || !strcmp(newpath,"..") || !strcmp(newpath,".") || !strcmp(newpath,"outgoing") ||
    (*newpath && (stat(newpath,&buf) || !S_ISDIR(buf.st_mode))))
    IllegalIndex(fd,newpath,args);
 else if(sort==NSortModes)
    CorrectIndex(fd,newpath,args);
 else
   {
    SortMode mode;

    write(fd,head,strlen(head));

    for(mode=0;mode<NSortModes;mode++)
      {
       if(*newpath && mode==Latest)
          continue;
       if(mode==sort)
          sprintf(buffer,"[%s]\n",sorttype[mode][1]);
       else
          sprintf(buffer,"<a href=\"/index/%s?%s\">[%s]</a>\n",newpath,sorttype[mode][0],sorttype[mode][1]);
       write(fd,buffer,strlen(buffer));
      }

    if(!*newpath)
      {
       write(fd,"</p>\n<ul>\n",10);

       if(sort==Latest)
          IndexLatest(fd);
       else
          IndexRoot(fd,args,sort);

       write(fd,"</ul>\n",6);

       if(sort==Latest)
         {
          sprintf(buffer,"<a href=\"/index/?none\">[Back to the Hosts Index]</a>\n");
          write(fd,buffer,strlen(buffer));
         }
      }
    else
      {
       sprintf(buffer,"</p>\n<b>http://%s/</b>\n<ul>\n",newpath);
       write(fd,buffer,strlen(buffer));

       IndexDir(fd,newpath,sort);

       sprintf(buffer,"</ul>\n<a href=\"/index/?%s\">[Back to the Hosts Index]</a>\n",args);
       write(fd,buffer,strlen(buffer));
      }

    write(fd,tail,strlen(tail));
   }

 free(newpath);
 free(buffer);
}


/*++++++++++++++++++++++++++++++++++++++
  Index the root of the cache.

  int fd The file descriptor to write to.

  char *args The argument to put in the index for sorting.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexRoot(int fd,char *args,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory [%!s]; index failed.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory [%!s]; index failed.");closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the sub directories. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);
    else if(S_ISDIR(buf.st_mode) && strcmp(ent->d_name,"outgoing"))
       add_dir(ent->d_name,&buf,mode);
   }

 closedir(dir);

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha || mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);

 for(i=0;i<nfiles;i++)
   {
    char *buffer=(char*)malloc(2*strlen(files[i]->name)+strlen(args)+40);

    sprintf(buffer,"<li><a href=\"/index/%s?%s\">%s</a>\n",files[i]->name,args,files[i]->name);
    write(fd,buffer,strlen(buffer));

    free(buffer);
    free(files[i]);
   }

 if(nfiles==0)
   {
    char *msg="<li>No hosts.\n";
    write(fd,msg,strlen(msg));
   }

 free(files);
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of a subdirectory.

  int fd The file descriptor to write to.

  char *dirname The name of the subdirectory.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexDir(int fd,char *dirname,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;

 /* Open the spool subdirectory. */

 if(chdir(dirname))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",dirname);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",dirname);chdir("..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not indexed.",dirname);closedir(dir);chdir("..");return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the file names. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",dirname,ent->d_name);
    else if(S_ISREG(buf.st_mode))
       add_file(ent->d_name,&buf,mode,NULL);
   }

 closedir(dir);

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 for(i=0;i<nfiles;i++)
   {
    char *buffer=(char*)malloc(3*strlen(files[i]->name)+2*strlen(dirname)+128);

    if(strstr(files[i]->name,"?!"))
       sprintf(buffer,"<li>[Posted]&nbsp;&nbsp;&nbsp;<a href=\"http://%s/%s\">%s</a>\n",
               dirname,files[i]->name[0]=='/'?&files[i]->name[1]:files[i]->name,
               files[i]->name);
    else
       sprintf(buffer,"<li><a href=\"/outgoing/%s/%s\">[Refresh]</a>&nbsp;&nbsp;<a href=\"http://%s/%s\">%s</a>\n",
               dirname,files[i]->name[0]=='/'?&files[i]->name[1]:files[i]->name,
               dirname,files[i]->name[0]=='/'?&files[i]->name[1]:files[i]->name,
               files[i]->name);
    write(fd,buffer,strlen(buffer));

    free(buffer);
    free(files[i]);
   }

 if(nfiles==0)
   {
    char *msg="<li>No files on this host.\n";
    write(fd,msg,strlen(msg));
   }

 free(files);
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the latest pages.

  int fd Thefile descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexLatest(int fd)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 time_t now=time(NULL);
 int lastdays=0;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory [%!s]; index failed.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory [%!s]; index failed.");closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the sub directories. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);
    else
       if(S_ISDIR(buf.st_mode) && strcmp(ent->d_name,"outgoing"))
         {
          DIR *dir2;
          struct dirent* ent2;
          struct utimbuf utbuf;

          if(chdir(ent->d_name))
            {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",ent->d_name);continue;}

          dir2=opendir(".");
          if(!dir2)
            {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",ent->d_name);chdir("..");continue;}

          ent2=readdir(dir2);  /* skip .  */
          if(!ent2)
            {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not indexed.",ent->d_name);closedir(dir2);chdir("..");continue;}
          ent2=readdir(dir2);  /* skip .. */

          while((ent2=readdir(dir2)))
            {
             struct stat buf;

             if(stat(ent2->d_name,&buf))
                PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",ent->d_name,ent2->d_name);
             else if(S_ISREG(buf.st_mode) && (now-buf.st_mtime)<=IndexLatestDays*(24*3600))
                add_file(ent2->d_name,&buf,Latest,ent->d_name);
            }

          closedir(dir2);
          chdir("..");

          utbuf.actime=buf.st_atime;
          utbuf.modtime=buf.st_mtime;
          utime(ent->d_name,&utbuf);
         }
   }

 closedir(dir);

 /* Sort the files. */

 qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);

 for(i=0;i<nfiles;i++)
   {
    char *buffer=(char*)malloc(3*strlen(files[i]->name)+128);
    int days=(now-files[i]->time)/(24*3600);

    if(lastdays<days)
      {
       char *msg="</ul>\n";
       write(fd,msg,strlen(msg));
       for(;lastdays<days;lastdays++)
         {
          msg="<hr width=\"50%\">\n";
          write(fd,msg,strlen(msg));
         }
       msg="<ul>\n";
       write(fd,msg,strlen(msg));
      }

    if(strstr(files[i]->name,"?!"))
       sprintf(buffer,"<li>[Posted]&nbsp;&nbsp;&nbsp;<a href=\"http://%s\">%s</a>\n",
               files[i]->name,files[i]->name);
    else
       sprintf(buffer,"<li><a href=\"/outgoing/%s\">[Refresh]</a>&nbsp;&nbsp;<a href=\"http://%s\">%s</a>\n",
               files[i]->name,files[i]->name,files[i]->name);
    write(fd,buffer,strlen(buffer));

    free(buffer);
    free(files[i]);
   }

 if(++lastdays<IndexLatestDays)
   {
    char *msg="</ul>\n";
    write(fd,msg,strlen(msg));
    for(;lastdays<IndexLatestDays;lastdays++)
      {
       msg="<hr width=\"50%\">\n";
       write(fd,msg,strlen(msg));
      }
    msg="<ul>\n";
    write(fd,msg,strlen(msg));
   }

 free(files);
}


/*++++++++++++++++++++++++++++++++++++++
  Redirect the user to the correct index page in case of a bad argument.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified args.
  ++++++++++++++++++++++++++++++++++++++*/

static void CorrectIndex(int fd,char *path,char *args)
{
 char *string=(char*)malloc(2*strlen(sorttype[0][0])+2*strlen(path)+(args?strlen(args):0)+40);
 char *head=
 "HTTP/1.0 301 WWWOFFLE Index Incorrect\r\n"
 "Content-type: text/html\r\n";
 char *head2=
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Index Incorrect\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Index Incorrect</H1>\n"
 "<p align=center>\n"
 "Your request for the index URL\n"
 "<br><b><tt>\n";
 char *middle=
 "\n"
 "</tt></b><br>\n"
 "was incorrect, select the link below for a correct version.\n"
 "<br>\n";
 char *tail=
 "\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,head,strlen(head));
 sprintf(string,"Location: /index/%s?%s\r\n",path,sorttype[0][0]);
 write(fd,string,strlen(string));
 write(fd,head2,strlen(head2));
 if(args)
    sprintf(string,"/index/%s?%s",path,args);
 else
    sprintf(string,"/index/%s",path);
 write(fd,string,strlen(string));
 write(fd,middle,strlen(middle));
 sprintf(string,"<a href=\"/index/%s?%s\">/index/%s?%s</a>",path,sorttype[0][0],path,sorttype[0][0]);
 write(fd,string,strlen(string));
 write(fd,tail,strlen(tail));

 free(string);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the specified index page is illegal.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified args.
  ++++++++++++++++++++++++++++++++++++++*/

static void IllegalIndex(int fd,char *path,char *args)
{
 char *string=(char*)malloc(2*strlen(sorttype[0][0])+2*strlen(path)+(args?strlen(args):0)+40);
 char *head=
 "HTTP/1.0 404 WWWOFFLE Illegal Index\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Illegal Index\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Illegal Index</H1>\n"
 "<p align=center>\n"
 "Your request for the index URL\n"
 "<br><b><tt>\n";
 char *middle=
 "\n"
 "</tt></b><br>\n"
 "is illegal, select the link below for the main index.\n"
 "<br>\n";
 char *tail=
 "\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,head,strlen(head));
 if(args)
    sprintf(string,"/index/%s?%s",path,args);
 else
    sprintf(string,"/index/%s",path);
 write(fd,string,strlen(string));
 write(fd,middle,strlen(middle));
 sprintf(string,"<a href=\"/index/?%s\">/index/?%s</a>",sorttype[0][0],sorttype[0][0]);
 write(fd,string,strlen(string));
 write(fd,tail,strlen(tail));

 free(string);
}


/*++++++++++++++++++++++++++++++++++++++
  Add a directory to the list.

  char *name The name of the directory.

  struct stat *buf The stat result on the file.

  SortMode mode The sort mode.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_dir(char *name,struct stat *buf,SortMode mode)
{
 if(!(nfiles%16))
   {
    if(!files)
       files=(FileIndex**)malloc(16*sizeof(FileIndex*));
    else
       files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
   }
 files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

 files[nfiles]->name=(char*)malloc(strlen(name)+1);
 strcpy(files[nfiles]->name,name);

 files[nfiles]->type="";

 if(mode==MTime)
    files[nfiles]->time=buf->st_mtime;
 else if(mode==ATime)
    files[nfiles]->time=buf->st_atime;

 nfiles++;
}


/*++++++++++++++++++++++++++++++++++++++
  Add a file to the list.

  char *name The name of the file.

  struct stat *buf The stat result on the file.

  SortMode mode The sort mode.

  char *dir The directory that the file is in for the 'Latest' mode.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_file(char *name,struct stat *buf,SortMode mode,char *dir)
{
 int i;
 char *args="";

 for(i=0;name[i];i++)
    if(name[i]==PATH_SEP)
      {
       if(name[i+1]!=PATH_SEP)
          name[i]='/';
       else
          if(name[i+1] && name[i+2]!=PATH_SEP)
            {args=&name[i];break;}
          else
            {name[i]='/';args=&name[i+1];break;}
      }

 if(args!=name)
   {
    if(*args)
      {
       char *old_args=args;
       int afd=open(old_args,O_RDONLY);

       if(afd!=-1)
         {
          int rr,r=0;
          struct utimbuf utbuf;

          args=(char*)malloc(64+1);

          while((rr=read(afd,&args[r],64)))
            {
             r+=rr;
             args=(char*)realloc(args,r+64+1);
            }

          args[r]=0;
          close(afd);

          utbuf.actime=buf->st_atime;
          utbuf.modtime=buf->st_mtime;
          utime(old_args,&utbuf);
         }

       *old_args=0;
      }

    if(!(nfiles%16))
      {
       if(!files)
          files=(FileIndex**)malloc(16*sizeof(FileIndex*));
       else
          files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
      }
    files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

    files[nfiles]->name=(char*)malloc((dir?strlen(dir)+2:0)+strlen(name)+strlen(args)+2);
    if(*args)
      {
       if(dir)
          sprintf(files[nfiles]->name,"%s/%s?%s",dir,*name=='/'?"":name,args);
       else
          sprintf(files[nfiles]->name,"%s?%s",name,args);
      }
    else
      {
       if(dir)
          sprintf(files[nfiles]->name,"%s/%s",dir,*name=='/'?"":name);
       else
          strcpy(files[nfiles]->name,name);
      }

    if(mode==Type)
      {
       files[nfiles]->type="";
       for(i=strlen(name);i>0;i--)
          if(files[nfiles]->name[i]=='.')
             files[nfiles]->type=&files[nfiles]->name[i];
          else if(files[nfiles]->name[i]=='/')
             break;
      }
    else if(mode==MTime || mode==Latest)
       files[nfiles]->time=buf->st_mtime;
    else if(mode==ATime)
       files[nfiles]->time=buf->st_atime;

    nfiles++;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_alpha Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_alpha(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name;
 char *bn=(*b)->name;

 return(strcmp(an,bn));
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into type (file extension) order.

  int sort_type Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_type(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name,*at=(*a)->type;
 char *bn=(*b)->name,*bt=(*b)->type;
 int sort1=strcmp(at,bt);

 if(sort1==0)
    return(strcmp(an,bn));
 else
    return(sort1);
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_time Returns the comparison of the times.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_time(FileIndex **a,FileIndex **b)
{
 long at=(*a)->time;
 long bt=(*b)->time;

 return(bt-at);
}
