/***************************************
  $Header: /home/amb/wwwoffle/RCS/config.c 1.11 1997/03/09 13:24:57 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.1.
  Configuration file management functions.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "config.h"
#include "wwwoffle.h"


#ifndef SPOOL_DIR
#define SPOOL_DIR DEF_SPOOL
#endif

/* Type definitions */

/*+ The type of value to expect for a value. +*/
typedef enum _ConfigType
{
 Fixed,                         /*+ When the left hand side is fixed. +*/
 None,                          /*+ When there is no right hand side. +*/

 PortNumber,                    /*+ For port numbers (>0). +*/
 AgeDays,                       /*+ An age in days (can be -ve). +*/
 Boolean,                       /*+ A boolean response yes/no 1/0 true/false. +*/

 String,                        /*+ For an arbitrary string (no whitespace). +*/
 DirName,                       /*+ For directory name values. +*/
 HostAndPort,                   /*+ For host name and port numbers. +*/
 Host                           /*+ For host names. +*/
}
ConfigType;

/*+ A description of an entry in a section of the config file. +*/
typedef struct _Entry
{
 char *name;                    /*+ The name of the entry. +*/
 void *pointer;                 /*+ A pointer to the value of the entry. +*/
 ConfigType left_type;          /*+ The type of the left side of the equals sign. +*/
 ConfigType right_type;         /*+ The type of the right side of the equals sign. +*/
}
Entry;

/*+ A description of a section in the config file. +*/
typedef struct _Section
{
 char *name;                    /*+ The name of the section. +*/
 Entry *entries;                /*+ The entries in the section (NULL terminated). +*/
}
Section;

/*+ A keyed entry for a wildcard left hand side. +*/
typedef struct _KeyPair
{
 char *name;                    /*+ The key name of the entry. +*/
 union
 {
  int   integer;                /*+ An integer value. +*/
  char *string;                 /*+ A string value. +*/
 }
 value;
}
KeyPair;

/*+ The last of a KeyPair list. +*/
static KeyPair KeyPairEnd={NULL};


/* StartUp section */

/*+ The name of the configuration file. +*/
char *ConfigFile=SPOOL_DIR "/wwwoffle.conf";

/*+ The port number to use for the HTTP proxy port. +*/
int HTTP_Port=8080;

/*+ The port number to use for the wwwoffle port. +*/
int WWWOFFLE_Port=8081;

/*+ The spool directory. +*/
char *SpoolDir=SPOOL_DIR;

/*+ Whether to use the syslog facility or not. +*/
int UseSyslog=0;

/*+ The password required for demon configuration. +*/
char *PassWord=NULL;

/*+ The entries in the StartUp section. +*/
static Entry startup_entries[]={{"http-port"    ,(void*)&HTTP_Port    ,Fixed,PortNumber},
                                {"wwwoffle-port",(void*)&WWWOFFLE_Port,Fixed,PortNumber},
                                {"spool-dir"    ,(void*)&SpoolDir     ,Fixed,DirName},
                                {"use-syslog"   ,(void*)&UseSyslog    ,Fixed,Boolean},
                                {"password"     ,(void*)&PassWord     ,Fixed,String},
                                {NULL           ,NULL                 ,   -1,     -1}};

/*+ The StartUp section. +*/
static Section startup_section={"StartUp",startup_entries};


/* Options Section */

/*+ The option to also fetch images. +*/
int FetchImages=0;

/*+ The number of days to display in the index of the latest pages. +*/
int IndexLatestDays=7;

/*+ The option of a tag that can be added to the bottom of the spooled pages with the date and a refresh button. +*/
int AddInfoRefresh=0;

/*+ The entries in the Options section. +*/
static Entry options_entries[]={{"fetch-images"     ,(void*)&FetchImages    ,Fixed,Boolean},
                                {"index-latest-days",(void*)&IndexLatestDays,Fixed,AgeDays},
                                {"add-info-refresh" ,(void*)&AddInfoRefresh ,Fixed,Boolean},
                                {NULL               ,NULL                   ,   -1,     -1}};

/*+ The Options section. +*/
static Section options_section={"Options",options_entries};


/* LocalHost section */

/*+ The list of localhost hostnames. +*/
static KeyPair **LocalHost=NULL;

/*+ The entries in the LocalHost section. +*/
static Entry localhost_entries[]={{""  ,(void*)&LocalHost,Host,None},
                                  {NULL,NULL             ,  -1,  -1}};

/*+ The LocalHost section. +*/
static Section localhost_section={"LocalHost",localhost_entries};


/* AllowedConnect section */

/*+ The list of allowed hostnames. +*/
static KeyPair **AllowedConnect=NULL;

/*+ The entries in the AllowedConnect section. +*/
static Entry allowedconnect_entries[]={{""  ,(void*)&AllowedConnect,Host,None},
                                       {NULL,NULL                  ,  -1,  -1}};

/*+ The AllowedConnect section. +*/
static Section allowedconnect_section={"AllowedConnect",allowedconnect_entries};


/* DontCache section */

/*+ The list of non-cached hostnames. +*/
static KeyPair **DontCache=NULL;

/*+ The entries in the DontCache section. +*/
static Entry dontcache_entries[]={{""  ,(void*)&DontCache,Host,None},
                                  {NULL,NULL             ,  -1,  -1}};

/*+ The DontCache section. +*/
static Section dontcache_section={"DontCache",dontcache_entries};


/* Proxy section */

/*+ The default proxy server to use. +*/
static char *DefaultProxy=NULL;

/*+ The list of hostnames and proxies. +*/
static KeyPair **Proxies=NULL;

/*+ The entries in the Proxy section. +*/
static Entry proxy_entries[]={{"default",(void*)&DefaultProxy,Fixed,HostAndPort},
                              {""       ,(void*)&Proxies     ,Host ,HostAndPort},
                              {NULL     ,NULL                ,   -1,         -1}};

/*+ The Proxy section. +*/
static Section proxy_section={"Proxy",proxy_entries};


/* Purge section */

/*+ If true then use modification time instead of access time. +*/
int PurgeUseMTime=0;

/*+ The default age for purging files. +*/
static int DefaultPurgeAge=DEF_PURGE_AGE;

/*+ The list of hostnames and purge ages. +*/
static KeyPair **PurgeAges=NULL;

/*+ The entries in the Purge section. +*/
static Entry purge_entries[]={{"use-mtime",(void*)&PurgeUseMTime  ,Fixed,Boolean},
                              {"default"  ,(void*)&DefaultPurgeAge,Fixed,AgeDays},
                              {""         ,(void*)&PurgeAges      ,Host ,AgeDays},
                              {NULL       ,NULL                   ,   -1,     -1}};

/*+ The Purge section. +*/
static Section purge_section={"Purge",purge_entries};


/* All sections */

/*+ The list of sections (NULL terminated). +*/
static Section *sections[]={&startup_section,
                            &options_section,
                            &localhost_section,
                            &allowedconnect_section,
                            &dontcache_section,
                            &proxy_section,
                            &purge_section,
                            NULL};


/* Local functions */
static int ReadFile(FILE *config);
static int ReadSection(FILE *config,Section *section);
static int ParseEntry(Entry *entry,char *left,char *right);
static int ParseValue(char *text,ConfigType type,void *value,int rhs);

static void SaveOldValues(void);
static void RestoreOldValues(void);
static void RemoveOldValues(void);
static void FreeKeyPairList(KeyPair **list,int andvalue);


/*+ Only the first time that the config file is read can the StartUp section be used. +*/
static int first_time=1;

/*+ The error message. +*/
static char *errmsg;


/*++++++++++++++++++++++++++++++++++++++
  Read in the configuration file.

  int fd The file descriptor to write the error message to.
  ++++++++++++++++++++++++++++++++++++++*/

int ReadConfigFile(int fd)
{
 FILE* config;
 int line_no;

 errmsg=NULL;

 config=fopen(ConfigFile,"r");
 if(!config)
   {
    char *msg=(char*)malloc(40+strlen(ConfigFile));
    sprintf(msg,"Cannot open the configuration file '%s'\n",ConfigFile);
    write(fd,msg,strlen(msg));
    free(msg);
    return(1);
   }

 SaveOldValues();

 line_no=ReadFile(config);

 if(errmsg)
    RestoreOldValues();
 else
    RemoveOldValues();

 fclose(config);

 if(errmsg)
   {
    char *msg=(char*)malloc(40+strlen(errmsg)+strlen(ConfigFile));
    sprintf(msg,"Syntax error at line %d in '%s'; %s\n",line_no,ConfigFile,errmsg);
    write(fd,msg,strlen(msg));
    free(msg);
    free(errmsg);
   }

 if(first_time)
    first_time=0;

 if(errmsg)
    return(1);
 else
    return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Get the first specified name of the server host.

  char *GetLocalHost Returns the first named localhost.
  ++++++++++++++++++++++++++++++++++++++*/

char *GetLocalHost(void)
{
 if(LocalHost)
    return((*LocalHost)->name);
 else
    return("localhost");
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is the server (localhost).

  int IsLocalHost Return true if it is the host the server is on.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsLocalHost(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int port=80;
 int isit=0;

 if(colon)
   {
    *colon=0;
    port=atoi(colon+1);
   }

 if(LocalHost)
    for(p=LocalHost;(*p)->name;p++)
       if(!strcmp((*p)->name,host) && port==HTTP_Port)
         {isit=1;break;}

 if(colon)
    *colon=':';

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is allowed to connect.

  int IsAllowedConnect Return true if it is allowed to connect.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsAllowedConnect(char *host)
{
 KeyPair **p;
 int isip=(*host>=0 && *host<=9)?1:0;
 int isit=0;

 if(LocalHost)
    for(p=LocalHost;(*p)->name;p++)
       if(!strcmp((*p)->name,host))
         {isit=1;break;}

 if(AllowedConnect && !isit)
    for(p=AllowedConnect;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {isit=1;break;}

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is to be cached.

  int IsCachedHost Return true if it is to be cached.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsCachedHost(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int isip=(*host>=0 && *host<=9)?1:0;
 int isit=1;

 if(colon)
    *colon=0;

 if(LocalHost)
    for(p=LocalHost;(*p)->name;p++)
       if(!strcmp((*p)->name,host))
         {isit=0;break;}

 if(DontCache && isit)
    for(p=DontCache;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {isit=0;break;}

 if(colon)
    *colon=':';

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Determine the proxy to use for a specified host.

  char *WhichProxy Return a pointer to the proxy.

  char *host The name of the host to be contacted.
  ++++++++++++++++++++++++++++++++++++++*/

char *WhichProxy(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int isip=(*host>=0 && *host<=9)?1:0;
 char *proxy=DefaultProxy;
 int matchlen=0;

 if(colon)
    *colon=0;

 if(Proxies)
    for(p=Proxies;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          strlen((*p)->name)>matchlen &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {
          matchlen=strlen((*p)->name);
          proxy=(*p)->value.string;
         }

 if(colon)
    *colon=':';

 return(proxy);
}


/*++++++++++++++++++++++++++++++++++++++
  Determine the age to use when purging for a specified host.

  int WhatPurgeAge Return the age in days.

  char *host The name of the host to be purged.
  ++++++++++++++++++++++++++++++++++++++*/

int WhatPurgeAge(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int isip=(*host>=0 && *host<=9)?1:0;
 int age=DefaultPurgeAge;
 int matchlen=0;

 if(colon)
    *colon=0;

 if(PurgeAges)
    for(p=PurgeAges;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          strlen((*p)->name)>matchlen &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {
          matchlen=strlen((*p)->name);
          age=(*p)->value.integer;
         }

 if(colon)
    *colon=':';

 return(age);
}


/*++++++++++++++++++++++++++++++++++++++
  Read the data from the file.

  int ReadFile Returns the number of lines read.

  FILE *config The file to read from.
  ++++++++++++++++++++++++++++++++++++++*/

static int ReadFile(FILE *config)
{
 char *line=NULL;
 int line_no=0;

 while((line=fgets_realloc(line,config)))
   {
    int i;
    char *l=line;
    char *r=line+strlen(line)-1;

    line_no++;

    while(*l==' ' || *l=='\t' || *l=='\r' || *l=='\n')
       l++;

    if(!*l || *l=='#')
       continue;

    while(r>l && (*r==' ' || *r=='\t' || *r=='\r' || *r=='\n'))
       *r--=0;

    for(i=0;sections[i];i++)
       if(!strcmp(sections[i]->name,l))
          break;

    if(!sections[i])
      {errmsg=(char*)malloc(32+strlen(l));sprintf(errmsg,"Unrecognised section label '%s'.",l);break;}
    else
       line_no+=ReadSection(config,sections[i]);

    if(errmsg)
       break;
   }

 if(line)
    free(line);

 return(line_no);
}


/*++++++++++++++++++++++++++++++++++++++
  Read all the data from a section.

  int ReadSection Returns the number of lines read.

  FILE *config The file to read from.

  Section *section The section to read.
  ++++++++++++++++++++++++++++++++++++++*/

static int ReadSection(FILE *config,Section *section)
{
 char *line=NULL;
 int line_no=0;
 int open_brace=0;

 while((line=fgets_realloc(line,config)))
   {
    int i;
    char *l=line;
    char *r=line+strlen(line)-1;

    line_no++;

    while(*l==' ' || *l=='\t' || *l=='\r' || *l=='\n')
       l++;

    if(!*l || *l=='#')
       continue;

    while(r>l && (*r==' ' || *r=='\t' || *r=='\r' || *r=='\n'))
       *r--=0;

    if(*l=='{')
      {
       if(*++l)
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Open brace has trailing junk.");break;}
       else
         {
          if(open_brace)
            {errmsg=(char*)malloc(32);strcpy(errmsg,"Open brace repeated.");break;}

          open_brace=1;
         }
      }
    else if(*l=='}')
      {
       if(*++l)
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Close brace has trailing junk.");break;}
       else
         {
          if(!open_brace)
            {errmsg=(char*)malloc(32);strcpy(errmsg,"No open brace seen.");break;}

          break;
         }
      }
    else
      {
       if(!open_brace)
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Open brace missing.");break;}

       if(!first_time && section==*sections)
          continue;

       for(i=0;section->entries[i].name;i++)
          if(!*section->entries[i].name || !strncmp(section->entries[i].name,l,strlen(section->entries[i].name)))
            {
             char *ll,*rr;

             if(*section->entries[i].name)
               {
                ll=l+strlen(section->entries[i].name);

                if(*ll && *ll!='=' && *ll!=' ' && *ll!='\t' && *ll!='\r' && *ll!='\n')
                   continue;
               }
             else
               {
                ll=l;
                while(*ll && *ll!='=' && *ll!=' ' && *ll!='\t' && *ll!='\r' && *ll!='\n')
                   ll++;
               }

             if(section->entries[i].right_type==None)
               {
                *ll=0;
                r=NULL;
               }
             else
               {
                r=strchr(line,'=');
                if(!r)
                  {errmsg=(char*)malloc(32);strcpy(errmsg,"No equal sign for entry.");break;}

                *ll=0;
                if(!*l)
                  {errmsg=(char*)malloc(48);strcpy(errmsg,"Nothing to the left of the equal sign.");break;}

                r++;
                while(*r==' ' || *r=='\t' || *r=='\r' || *r=='\n')
                   r++;

                rr=r;
                while(*rr && *rr!=' ' && *rr!='\t' && *rr!='\r' && *rr!='\n')
                   rr++;

                *rr=0;
               }
             break;
            }

       if(errmsg)
          break;

       if(!section->entries[i].name)
         {errmsg=(char*)malloc(32+strlen(l));sprintf(errmsg,"Unrecognised entry label '%s'.",l);break;}

       ParseEntry(&section->entries[i],l,r);

       if(errmsg)
          break;
      }
   }

 if(line)
    free(line);

 return(line_no);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse an entry from the file.

  int ParseEntry Returns non-zero if there was an error.

  Entry *entries The list of matching entry in the section.

  char *left The string to the left of the equals sign.

  char *right The string to the right of the equals sign.
  ++++++++++++++++++++++++++++++++++++++*/

static int ParseEntry(Entry *entry,char *left,char *right)
{
 if(*entry->name)
   {
    if(ParseValue(right,entry->right_type,entry->pointer,1))
       return(1);
   }
 else
   {
    KeyPair **p,*new=(KeyPair*)malloc(sizeof(KeyPair));
    int i;

    if(ParseValue(left,entry->left_type,(void*)&new->name,0))
      {free(new);return(1);}

    if(entry->right_type==None)
       new->value.string=NULL;
    else
       if(ParseValue(right,entry->right_type,(void*)&new->value,1))
         {free(new);return(1);}

    if(!*(KeyPair***)entry->pointer)
      {
       *(KeyPair***)entry->pointer=(KeyPair**)malloc(8*sizeof(KeyPair*));
       **(KeyPair***)entry->pointer=&KeyPairEnd;
      }

    p=*(KeyPair***)entry->pointer;
    for(i=0;(*p)->name;p++,i++);

    if(i%8==7)
       *(KeyPair***)entry->pointer=(KeyPair**)realloc(*(void**)entry->pointer,(i+9)*sizeof(KeyPair*));

    for(p=*(KeyPair***)entry->pointer;(*p)->name;p++);

    *p=new;
    p++;
    *p=&KeyPairEnd;
   }

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the text and put a value into the location.

  int ParseValue Returns non-zero in case of error.

  char *text The text string to parse.

  ConfigType type The type we are looking for.

  void *value The location to store the value.

  int rhs Set to true if this is the right hand side.
  ++++++++++++++++++++++++++++++++++++++*/

static int ParseValue(char *text,ConfigType type,void *value,int rhs)
{
 switch(type)
   {
   case PortNumber:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a port number, got nothing.");break;}
    if(atoi(text)<=0 || atoi(text)>65535)
      {errmsg=(char*)malloc(32);sprintf(errmsg,"Invalid port number %d.",atoi(text));break;}
    *(int*)value=atoi(text);
    break;

   case AgeDays:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting an age in days, got nothing.");break;}
    *(int*)value=atoi(text);
    break;

   case Boolean:
    if(!*text)
       *(int*)value=0;
    else
       if(!strcasecmp(text,"yes") || !strcasecmp(text,"true") || !strcmp(text,"1"))
          *(int*)value=1;
       else
          *(int*)value=0;
    break;

   case String:
    if(!*text || !strcasecmp(text,"none"))
       *(char**)value=NULL;
    else
      {
       *(char**)value=(char*)malloc(strlen(text)+1);
       strcpy(*(char**)value,text);
      }
    break;

   case DirName:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a directory name, got nothing.");break;}
    if(*text!='/')
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting an absolute directory name '%s'.",text);break;}
    *(char**)value=(char*)malloc(strlen(text)+1);
    strcpy(*(char**)value,text);
    break;

   case Host:
    if(rhs && (!*text || !strcasecmp(text,"none")))
       *(char**)value=NULL;
    else
      {
       char *colon=strchr(text,':');
       if(!*text)
         {errmsg=(char*)malloc(40);strcpy(errmsg,"Expecting a hostname, got nothing.");break;}
       if(colon)
         {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Not expecting a port number, got '%s'.",text);break;}
       *(char**)value=(char*)malloc(strlen(text)+1);
       strcpy(*(char**)value,text);
      }
    break;

   case HostAndPort:
    if(rhs && (!*text || !strcasecmp(text,"none")))
       *(char**)value=NULL;
    else
      {
       char *colon=strchr(text,':');
       if(!*text)
         {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a hostname (and port), got nothing.");break;}
       if(*text==':')
         {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a hostname before the ':', got '%s'.",text);break;}
       if(colon && (atoi(colon+1)<=0 || atoi(colon+1)>65535))
         {errmsg=(char*)malloc(32);sprintf(errmsg,"Invalid port number %d.",atoi(colon+1));break;}
       *(char**)value=(char*)malloc(strlen(text)+1);
       strcpy(*(char**)value,text);
      }
    break;

   case Fixed:
   case None:
    ;
   }

 if(errmsg)
    return(1);
 else
    return(0);
}


/*+ The old value of the option to also fetch images. +*/
static int old_FetchImages=0;

/*+ The old value of the number of days to display in the index of the latest pages. +*/
static int old_IndexLatestDays=0;

/*+ A old value of the option for a tag that can be added to the bottom of the spooled pages with the date and a refresh button. +*/
static int old_AddInfoRefresh=0;

/*+ The old values of the list of localhost hostnames. +*/
static KeyPair **old_LocalHost=NULL;

/*+ The old values of the list of allowed hostnames. +*/
static KeyPair **old_AllowedConnect=NULL;

/*+ The old value of the list of non-cached hostnames. +*/
static KeyPair **old_DontCache=NULL;

/*+ The old value of the default proxy server to use. +*/
static char *old_DefaultProxy=NULL;

/*+ The old value of the list of hostnames and proxies. +*/
static KeyPair **old_Proxies=NULL;

/*+ The old value of the flag that if true then use modification time instead of access time. +*/
static int old_PurgeUseMTime=0;

/*+ The old value of the default age for purging files. +*/
static int old_DefaultPurgeAge=0;

/*+ The old value of the list of hostnames and purge ages. +*/
static KeyPair **old_PurgeAges=NULL;


/*++++++++++++++++++++++++++++++++++++++
  Save the old values in case the re-read of the file fails.
  ++++++++++++++++++++++++++++++++++++++*/

static void SaveOldValues(void)
{
 old_FetchImages=FetchImages;
 old_IndexLatestDays=IndexLatestDays;
 old_AddInfoRefresh=AddInfoRefresh;

 old_LocalHost=LocalHost;
 LocalHost=NULL;

 old_AllowedConnect=AllowedConnect;
 AllowedConnect=NULL;

 old_DontCache=DontCache;
 DontCache=NULL;

 old_DefaultProxy=DefaultProxy;
 DefaultProxy=NULL;
 old_Proxies=Proxies;
 Proxies=NULL;

 old_PurgeUseMTime=PurgeUseMTime;
 old_DefaultPurgeAge=DefaultPurgeAge;
 old_PurgeAges=PurgeAges;
 PurgeAges=NULL;
}


/*++++++++++++++++++++++++++++++++++++++
  Restore the old values if the re-read failed.
  ++++++++++++++++++++++++++++++++++++++*/

static void RestoreOldValues(void)
{
 FetchImages=old_FetchImages;
 IndexLatestDays=old_IndexLatestDays;
 AddInfoRefresh=old_AddInfoRefresh;

 if(LocalHost)
    FreeKeyPairList(LocalHost,0);
 LocalHost=old_LocalHost;

 if(AllowedConnect)
    FreeKeyPairList(AllowedConnect,0);
 AllowedConnect=old_AllowedConnect;

 if(DontCache)
    FreeKeyPairList(DontCache,0);
 DontCache=old_DontCache;

 if(DefaultProxy)
    free(DefaultProxy);
 DefaultProxy=old_DefaultProxy;
 if(Proxies)
    FreeKeyPairList(Proxies,1);
 Proxies=old_Proxies;

 PurgeUseMTime=old_PurgeUseMTime;
 DefaultPurgeAge=old_DefaultPurgeAge;
 if(PurgeAges)
    FreeKeyPairList(PurgeAges,0);
 PurgeAges=old_PurgeAges;
}


/*++++++++++++++++++++++++++++++++++++++
  Remove the old values if the re-read succeeded.
  ++++++++++++++++++++++++++++++++++++++*/

static void RemoveOldValues(void)
{
 if(old_LocalHost)
    FreeKeyPairList(old_LocalHost,0);

 if(old_AllowedConnect)
    FreeKeyPairList(old_AllowedConnect,0);

 if(old_DontCache)
    FreeKeyPairList(old_DontCache,0);

 if(old_DefaultProxy)
    free(old_DefaultProxy);
 if(old_Proxies)
    FreeKeyPairList(old_Proxies,1);

 if(old_PurgeAges)
    FreeKeyPairList(old_PurgeAges,0);
}


/*++++++++++++++++++++++++++++++++++++++
  Free a KeyPair list.

  KeyPair **list The list to free.

  int andvalue If true then free the value field as well.
  ++++++++++++++++++++++++++++++++++++++*/

static void FreeKeyPairList(KeyPair **list,int andvalue)
{
 KeyPair **p;

 for(p=list;(*p)->name;p++)
   {
    free((*p)->name);
    if(andvalue && (*p)->value.string)
       free((*p)->value.string);
    free(*p);
   }
 free(list);
}
