/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffles.c 1.7 1997/01/25 08:13:43 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.0.
  A server to fetch the required pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <signal.h>
#include <unistd.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "sockets.h"
#include "errors.h"


static void usage(void);

/*+ The mode of operation of the server. +*/
typedef enum _Mode
{
 None,                          /*+ Undecided. +*/
 Real,                          /*+ From server host to cache and client. +*/
 RealNoCache,                   /*+ From server host to client. +*/
 RealRefresh,                   /*+ Refresh the page, forced from index. +*/
 Fetch,                         /*+ From server host to cache. +*/
 Spool,                         /*+ From cache to client. +*/
 SpoolGet,                      /*+ Not in cache so record request in outgoing. +*/
 SpoolRefresh,                  /*+ Refresh the page, forced from index. +*/
 Index                          /*+ An index. +*/
}
Mode;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int   client     =-1  , outgoing     =-1  ,spool=-1,server=-1;
 FILE *client_file=NULL,*outgoing_file=NULL;

 char *buffer,*url;
 int status;

 Mode mode=None;

 char *host,*path,*args,*server_host=NULL;
 int port=80;

 /* Parse the command line options */

 if(argc!=3 && argc!=4)
    usage();

 if(!strcmp(argv[1],"-real"))
    mode=Real;
 else if(!strcmp(argv[1],"-fetch"))
    mode=Fetch;
 else if(!strcmp(argv[1],"-spool"))
    mode=Spool;
 else
    exit(1);

 /* Initialise things. */

 InitErrorHandler("wwwoffles");

 signal(SIGPIPE,SIG_IGN);

 /* Set up the client file. */

 client=atoi(argv[2]);

 if(client<=0)
    PrintMessage(Fatal,"Cannot use client file descriptor %d.",client);

 client_file=fdopen(client,"r");
 if(!client_file)
    PrintMessage(Fatal,"Cannot use fdopen on the client file descriptor.");

 /* Set up the outgoing file. */

 if(mode==Fetch)
   {
    outgoing=OpenOutgoingSpoolFile(1);

    if(outgoing==-1)
       PrintMessage(Fatal,"Cannot open the spooled outgoing request to read.");
    else if(outgoing==-2)
      {PrintMessage(Inform,"No more outgoing requests.");return(3);}

    outgoing_file=fdopen(outgoing,"r");
    if(!outgoing_file)
       PrintMessage(Fatal,"Cannot use fdopen on the outgoing file descriptor.");
   }

 /* Get the URL from the request. */

 if(mode==Real || mode==Spool)
    buffer=ParseRequest(client_file,&url);
 else /* mode==Fetch */
    buffer=ParseRequest(outgoing_file,&url);

 if(!url)
   {
    ServerErrorMessage(client,buffer?"Cannot parse the HTTP request":"The HTTP request was empty");

    PrintMessage(Fatal,"Could not parse HTTP request (%s).",buffer?"Parse error":"Empty request");
   }

 PrintMessage(Inform,"URL = %s",url);

 SplitURL(url,&host,&path,&args);

 /* Change the mode based on the URL as required. */

 if(!*host && (mode==Real || mode==Spool))
   {
    if(!strncmp(url,"/outgoing/",10))
      {
       char *copy,*u;

       copy=(char*)malloc(strlen(buffer)+1);
       u=strstr(buffer,url); *u=0;
       sprintf(copy,"%shttp://%s%s",buffer,&url[10],&u[strlen(url)]);
       free(buffer);
       buffer=copy;

       copy=(char*)malloc(strlen(url)+1);
       sprintf(copy,"http://%s",&url[10]);
       free(url);
       url=copy;

       SplitURL(url,&host,&path,&args);

       if(mode==Real)
         {
          mode=RealRefresh;
          OpenWebpageSpoolFile(-1,host,path,args);
         }
       if(mode==Spool)
         {
          spool=OpenWebpageSpoolFile(1,host,path,args);
          if(spool!=-1)
            {mode=SpoolRefresh;close(spool);}
          else
             mode=SpoolGet;
         }
      }
    else
       mode=Index;
   }
 else if(!strncmp(host,"localhost",9) && (mode==Real || mode==Spool))
   {
    mode=RealNoCache;
    argc=3; /* no proxy */
   }
 else if((!*host || !strncmp(host,"localhost",9)) && mode==Fetch)
   {
    PrintMessage(Warning,"A request to fetch a page from localhost is ignored.");
    return(0);
   }
 else if(mode==Real || mode==Spool)
   {
    spool=OpenWebpageSpoolFile(1,host,path,args);

    if(mode==Spool && spool==-1)
       mode=SpoolGet;
    if(mode==Real && spool!=-1)
       mode=Spool;
    if(spool!=-1)
       close(spool);
   }

 /* Set up the outgoing file. */

 if(mode==SpoolGet || mode==SpoolRefresh)
   {
    outgoing=OpenOutgoingSpoolFile(0);

    if(outgoing==-1)
      {
       if(mode==SpoolGet)
          ServerErrorMessage(client,"Cannot open the outgoing request to write");

       PrintMessage(Fatal,"Cannot open the outgoing request to write.");
      }
   }

 /* Set up the file descriptor for the server host. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *colon;

    if(argc==4)
      {
       server_host=(char*)malloc(strlen(argv[3])+1);
       strcpy(server_host,argv[3]);
      }
    else
      {
       server_host=(char*)malloc(strlen(host)+1);
       strcpy(server_host,host);
       MakeRequestNonProxy(url,path,args,&buffer);
      }

    if((colon=strchr(server_host,':')))
      {
       *colon++=0;
       if(*colon)
          port=atoi(colon);
      }

    server=OpenClientSocket(server_host,port);

    if(server==-1)
      {
       if(mode==Fetch)
         {
          spool=OpenWebpageSpoolFile(0,host,path,args);
          if(spool!=-1)
             ServerFetchFailed(spool,url,server_host,port,1);
         }
       if(mode==Real || mode==RealNoCache)
          ServerFetchFailed(client,url,server_host,port,1);

       PrintMessage(Fatal,"Cannot open a socket to the server %s port %d.",server_host,port);
      }
   }

 /* Read request and process it. */

 if(mode==Real || mode==RealNoCache)
   {
    write(server,buffer,strlen(buffer));
   }
 else if(mode==Fetch)
   {
    write(server,buffer,strlen(buffer));

    sprintf(buffer,"Fetching %s\n",url);
    write(client,buffer,strlen(buffer));
   }
 else if(mode==SpoolGet || mode==SpoolRefresh)
   {
    write(outgoing,buffer,strlen(buffer));
   }

 if(buffer)
   {free(buffer); buffer=NULL;}

 /* Parse the reply */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    buffer=ParseReply(server,&status);

    if(status==304 && mode==Fetch)
      {
       PrintMessage(Inform,"Server page is not newer than the one in cache; not fetching.");
       return(0);
      }
    if(status==304 && mode==Real)
       mode=Spool;
   }

 /* Set up the file descriptor for the spool file. */

 if(mode==Real || mode==Fetch || mode==SpoolGet)
   {
    spool=OpenWebpageSpoolFile(0,host,path,args);

    if(spool==-1)
      {
       if(mode==Real || mode==SpoolGet)
          ServerErrorMessage(client,"Cannot open the spooled web page to write");

       PrintMessage(Fatal,"Cannot open the spooled web page to write.");
      }
   }
 else if(mode==Spool)
   {
    spool=OpenWebpageSpoolFile(1,host,path,args);
   }

 /* Read reply and process it. */

 if(!buffer)
    buffer=(char*)malloc(257);
 else
    if(strlen(buffer)<256)
       buffer=(char*)realloc(buffer,257);

 if(mode==Real || mode==RealNoCache)
   {
    int n=strlen(buffer);

    do
      {
       if(mode==Real)
          write(spool,buffer,n);
       write(client,buffer,n);
      }
    while((n=ReadOrTimeout(server,buffer,256))>0);

    if(n<0 && mode==Real)
      {
       lseek(spool,0,SEEK_SET);
       ftruncate(spool,0);
       ServerFetchFailed(spool,url,server_host,port,0);
       PrintMessage(Fatal,"Timed out while reading from server.");
      }
   }
 else if(mode==Fetch)
   {
    int n=strlen(buffer);

    do
      {
       write(spool,buffer,n);
      }
    while((n=ReadOrTimeout(server,buffer,256))>0);

    if(n<0)
      {
       lseek(spool,0,SEEK_SET);
       ftruncate(spool,0);
       ServerFetchFailed(spool,url,server_host,port,0);
       PrintMessage(Fatal,"Timed out while reading from server.");
      }
   }
 else if(mode==Spool)
   {
    char *head="HTTP/1.0 503 WWWOFFLE Fetch Failed\r\n";
    int n=read(spool,buffer,256);

    if(n==0 || !strncmp(buffer,head,strlen(head)))
       OpenWebpageSpoolFile(-1,host,path,args);

    if(n)
       do
         {
          write(client,buffer,n);
         }
       while((n=read(spool,buffer,256))>0);
   }
 else if(mode==SpoolGet)
   {
    WillGet(client,url,0);
    WillGet(spool,url,1);
   }
 else if(mode==RealRefresh || mode==SpoolRefresh)
   {
    Refresh(client,url);
   }
 else /* mode==Index */
   {
    GenerateIndex(client,path,args);
   }

 /* Close down and exit. */

 if(client>0)
    fclose(client_file);

 if(server>0)
    CloseSocket(server);

 if(outgoing>0)
   {
    if(mode==Fetch)
       fclose(outgoing_file);
    else
       close(outgoing);
   }

 if(spool>0)
    close(spool);

 if(buffer)
    free(buffer);

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(void)
{
 fputs("\n"
       "WWWOFFLES - World Wide Web Offline Explorer (Server) - Version 1.0\n"
       "\n",stderr);

 fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
       "                             [http://www.gedanken.demon.co.uk/]\n"
       "\n",stderr);

 fputs("*** Don't run this program, use wwwoffle or wwwoffled instead. ***\n"
       "\n",stderr);

 exit(1);
}
