/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffled.c 1.4 1997/01/22 20:22:33 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.0.
  A demon program to maintain the database and spawn the servers.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>

#include <sys/wait.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#include "wwwoffle.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);
static void sigchild(int signum);
static void sigexit(int signum);


/*+ The proxy server that we use. +*/
char *proxy=NULL;

/*+ The server sockets that we listen on +*/
int http_fd=-1,                 /*+ for the HTTP connections. +*/
    wwwoffle_fd=-1;             /*+ for the WWWOFFLE connections. +*/

/*+ Set to 1 when the demon is to shutdown. +*/
static int closedown=0;

/*+ The online / offline status. +*/
int online=0;

/*+ The current number of active servers. +*/
int n_servers=0;

/*+ The wwwoffle client when fetching. +*/
int fetching=0;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int port1=DEF_HTTP_PORT,
     port2=DEF_WWWOFFLE_PORT;

 char *spool=DEF_SPOOL;

 int i;
 int err;
 struct stat buf;

 /* Parse the command line options */

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-proxy"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffled: The '-proxy' argument requires a hostname.\n"); exit(1);}

       proxy=argv[i];
       continue;
      }

    if(!strcmp(argv[i],"-ports"))
      {
       if(i>=(argc-2))
         {fprintf(stderr,"wwwoffled: The '-ports' argument requires two port numbers.\n"); exit(1);}

       port1=atoi(argv[++i]);
       if(port1<=0 || port1>=65536)
         {fprintf(stderr,"wwwoffled: The port1 number '%s' is invalid.\n",argv[i]); exit(1);}

       port2=atoi(argv[++i]);
       if(port2<=0 || port2>=65536)
         {fprintf(stderr,"wwwoffled: The port2 number '%s' is invalid.\n",argv[i]); exit(1);}

       if(port1==port2)
         {fprintf(stderr,"wwwoffled: The two port numbers are the same '%d'.\n",port1); exit(1);}

       continue;
      }

    if(!strcmp(argv[i],"-spool"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffled: The '-spool' argument requires a spool directory.\n"); exit(1);}

       spool=argv[i];
       continue;
      }

    fprintf(stderr,"wwwoffled: Unknown option '%s'.\n",argv[i]); exit(1);
   }

 /* Initialise things. */

 InitErrorHandler("wwwoffled");

 PrintMessage(Inform,"WWWOFFLE Demon Version 1.0 starting");

 http_fd=OpenServerSocket(port1);
 if(http_fd==-1)
    PrintMessage(Fatal,"Cannot create HTTP server.");

 wwwoffle_fd=OpenServerSocket(port2);
 if(wwwoffle_fd==-1)
    PrintMessage(Fatal,"Cannot create WWWOFFLE server.");

 if(stat(spool,&buf))
   {
    err=mkdir(spool,0755);
    if(err==-1 && errno!=EEXIST)
       PrintMessage(Fatal,"Cannot create spool directory %s [%!s].\n",spool);
   }

 if(!stat(spool,&buf) && !S_ISDIR(buf.st_mode))
    PrintMessage(Fatal,"The spool directory %s is not a directory.\n",spool);

 err=chdir(spool);
 if(err==-1)
    PrintMessage(Fatal,"Cannot change to spool directory %s [%!s].\n",spool);

 err=mkdir("outgoing",0755);
 if(err==-1 && errno!=EEXIST)
    PrintMessage(Fatal,"Cannot create spool 'outgoing' directory [%!s].\n");

 sigchild(0);
 sigexit(0);

 /* Loop around waiting for connections. */

 do
   {
    fd_set readfd;
    int nfds;
    int retval;

    if(http_fd>wwwoffle_fd)
       nfds=http_fd+1;
    else
       nfds=wwwoffle_fd+1;

    FD_ZERO(&readfd);

    FD_SET(wwwoffle_fd,&readfd);

    if(n_servers<MAX_SERVERS)
       FD_SET(http_fd,&readfd);

    retval=select(nfds,&readfd,NULL,NULL,NULL);

    if(retval!=-1)
      {
       if(FD_ISSET(wwwoffle_fd,&readfd))
         {
          char *host;
          int port,client;

          client=AcceptConnect(wwwoffle_fd,&host,&port);
          if(client>=0)
            {
             FILE *file=fdopen(client,"r");

             if(!file)
               {
                PrintMessage(Warning,"Unable to fdopen the wwwoffle control socket [%!s].");

                CloseSocket(client);
               }
             else
               {
                PrintMessage(Inform,"WWWOFFLE connection from %s.",host);

                CommandConnect(file);

                if(fetching!=client)
                   fclose(file);
               }
            }
         }

       if(FD_ISSET(http_fd,&readfd))
         {
          char *host;
          int port,client;

          client=AcceptConnect(http_fd,&host,&port);
          if(client>=0)
            {
             PrintMessage(Inform,"HTTP Proxy connection from %s.",host);

             ForkServer(client,1);

             CloseSocket(client);
            }
         }
      }
   }
 while(!closedown);

 /* Close down and exit. */

 CloseSocket(http_fd);
 CloseSocket(wwwoffle_fd);

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fputs("\n"
       "WWWOFFLED - World Wide Web Offline Explorer (Demon) - Version 1.0\n"
       "\n",stderr);

 if(verbose)
    fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
          "                             [http://www.gedanken.demon.co.uk/]\n"
          "\n",stderr);

 fputs("Usage: wwwoffled [-h] [-proxy host[:port]] [-ports port1 port2] [-spool dir]\n"
       "\n",stderr);

 if(verbose)
    fprintf(stderr,
            "   -h                  : Display this help.\n"
            "\n"
            "   -proxy host[:port]  : The host name and port number of the proxy to use.\n"
            "                         (Default no proxy).\n"
            "   -ports port1 port2  : The port numbers to listen to;\n"
            "                         first for HTTP and second for wwwoffle\n"
            "                         (Defaults %d and %d).\n"
            "   -spool dir          : The directory name to spool to.\n"
            "                         (Default %s).\n"
            "\n",DEF_HTTP_PORT,DEF_WWWOFFLE_PORT,DEF_SPOOL);

 exit(0);
}


/*++++++++++++++++++++++++++++++++++++++
  The signal handler for the child exiting.

  int signum The signal number.
  ++++++++++++++++++++++++++++++++++++++*/

static void sigchild(int signum)
{
 void (*sig)();
 int pid;
 int status;

 if(signum)
    while((pid=waitpid(-1,&status,WNOHANG))>0)
      {
       int exitval=0;

       if(WIFEXITED(status))
         {
          exitval=WEXITSTATUS(status);

          PrintMessage(Inform,"Child wwwoffles exited with status %d (pid=%d).",exitval,pid);
         }
       else if(WIFSIGNALED(status))
         {
          int signal=WTERMSIG(status);

          PrintMessage(Inform,"Child wwwoffles terminated by signal %d (pid=%d).",signal,pid);
         }
       else
          PrintMessage(Inform,"Child wwwoffles finished (pid=%d).",pid);

       if(fetching)
         {
          if(exitval==3)
            {
             char *msg="WWWOFFLE No more to fetch.\n";
             write(fetching,msg,strlen(msg));
             close(fetching);
             fetching=0;
            }
          else
             ForkServer(fetching,0);
         }

       n_servers--;
      }

#if defined(SIGCHLD)
 sig=signal(SIGCHLD,sigchild);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGGHLD handler.\n");
#else
 sig=signal(SIGCLD,sigchild);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGGLD handler.\n");
#endif
}


/*++++++++++++++++++++++++++++++++++++++
  The signal handler for the signals to tell us to exit.

  int signum The signal number.
  ++++++++++++++++++++++++++++++++++++++*/

static void sigexit(int signum)
{
 void (*sig)();

 if(signum)
   {
    closedown=1;

    PrintMessage(Inform,"Exit signalled.");
   }

 sig=signal(SIGINT,sigexit);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGINT handler.\n");

 sig=signal(SIGQUIT,sigexit);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGQUIT handler.\n");

 sig=signal(SIGTERM,sigexit);
 if(sig==SIG_ERR)
    PrintMessage(Warning,"Cannot add SIGTERM handler.\n");
}
