/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffle.c 1.4 1997/01/22 20:18:31 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.0.
  A user level program to interact with the server.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the reqeusted pages. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Get                            /*+ Tell the server to get pages. +*/
}
Action;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int www_npages=0;
 char **www_pages=NULL;

 int purge_age=DEF_AGE;
 int purge_nhosts=0;
 char **purge_hosts=NULL;

 Action action=None;

 int port=-1;

 int i;

 /* Parse the command line options */

 if(argc==1)
    usage(0);

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-port"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-port' argument requires a port number.\n"); exit(1);}

       port=atoi(argv[i]);
       if(port<=0 || port>=65536)
         {fprintf(stderr,"wwwoffle: The port number '%s' is invalid.\n",argv[i]); exit(1);}
       break;
      }
   }

 if(i==argc)
    i=1;
 else if(i==(argc-1))
   {i=1;argc-=2;}
 else
    i++;

 if(!strcmp(argv[i],"-on") || !strcmp(argv[i],"-online"))
   {
    if(argc!=2)
      {fprintf(stderr,"wwwoffle: The '-online' option takes no arguments.\n"); exit(1);}
    action=Online;
   }
 else if(!strcmp(argv[i],"-off") || !strcmp(argv[i],"-offline"))
   {
    if(argc!=2)
      {fprintf(stderr,"wwwoffle: The '-offline' option takes no arguments.\n"); exit(1);}
    action=Offline;
   }
 else if(!strcmp(argv[i],"-fetch"))
   {
    if(argc!=2)
      {fprintf(stderr,"wwwoffle: The '-fetch' option takes no arguments.\n"); exit(1);}
    action=Fetch;
   }
 else if(!strcmp(argv[i],"-purge"))
   {
    for(i++;i<argc;i++)
      {
       int age=atoi(argv[i]);
       char ages[10];

       sprintf(ages,"%d",age);

       if(!strcmp(argv[i],ages))
         {
          purge_age=age;

          if(purge_age<0)
            {fprintf(stderr,"wwwoffle: The age for purging (%d) must be positive.\n",age); exit(1);}
         }
       else
         {
          if(!purge_hosts)
             purge_hosts=(char**)malloc(8*sizeof(char*));
          else if(!(purge_nhosts%8))
             purge_hosts=realloc(purge_hosts,(8+purge_nhosts)*sizeof(char*));

          purge_hosts[purge_nhosts++]=argv[i];
         }
      }

    if(purge_age==0 && !purge_hosts)
      {fprintf(stderr,"wwwoffle: Cannot purge all pages, age=0 requires hostnames.\n"); exit(1);}

    action=Purge;
   }
 else if(argv[i][0]=='-')
   {fprintf(stderr,"wwwoffle: Unknown option '%s'.\n",argv[1]); exit(1);}
 else
   {
    for(;i<argc;i++)
      {
       if(!www_pages)
          www_pages=(char**)malloc(8*sizeof(char*));
       else if(!(www_npages%8))
          www_pages=realloc(www_pages,(8+www_npages)*sizeof(char*));

       www_pages[www_npages++]=argv[i];
      }

    action=Get;
   }

 /* Initialise things. */

 InitErrorHandler("wwwoffle");

 /* The connections to the WWWOFFLE server. */

 if(action!=Get)
   {
    int socket;
    FILE *server_r,*server_w;
    char *buffer=NULL;

    if(port==-1)
       port=DEF_WWWOFFLE_PORT;

    socket=OpenClientSocket("localhost",port);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to server.");

    server_r=fdopen(socket,"r");
    server_w=fdopen(socket,"w");

    /* Send the message. */

    if(action==Online)
       fprintf(server_w,"WWWOFFLE ONLINE\r\n");
    else if(action==Offline)
       fprintf(server_w,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       fprintf(server_w,"WWWOFFLE FETCH\r\n");
    else if(action==Purge)
      {
       fprintf(server_w,"WWWOFFLE PURGE AGE %d\r\n",purge_age);

       if(purge_hosts)
          for(i=0;i<purge_nhosts;i++)
             fprintf(server_w,"WWWOFFLE PURGE HOST %s\r\n",purge_hosts[i]);

       free(purge_hosts);
      }

    fprintf(server_w,"\r\n");
    fflush(server_w);

    while((buffer=fgets_realloc(buffer,server_r)))
       fputs(buffer,stdout);

    fclose(server_w);
    fclose(server_r);
   }

 /* The connections to the http proxy. */

 else /* action==Get */
   {
    int socket;
    char *buffer=(char*)malloc(257);

    if(port==-1)
       port=DEF_HTTP_PORT;

    for(i=0;i<www_npages;i++)
      {
       char *request=(char*)malloc(strlen(www_pages[i])+48);

       socket=OpenClientSocket("localhost",port);

       if(socket==-1)
          PrintMessage(Fatal,"Cannot open connection to server.");

       if(!strncmp(www_pages[i],"http:",5))
         {
          www_pages[i]+=5;
          if(*www_pages[i]=='/') www_pages[i]++;
          if(*www_pages[i]=='/') www_pages[i]++;
         }

       printf("Getting http://%s\n",www_pages[i]);

       sprintf(request,"GET /outgoing/%s HTTP/1.0\r\nAccept: */*\r\n\r\n",www_pages[i]);
       write(socket,request,strlen(request));

       while(read(socket,buffer,256)>0);

       close(socket);
       free(request);
      }

    free(buffer);
    free(www_pages);
   }

 /* exit. */

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fputs("\n"
       "WWWOFFLE - World Wide Web Offline Explorer - Version 1.0\n"
       "\n",stderr);

 if(verbose)
    fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
          "                             [http://www.gedanken.demon.co.uk/]\n"
          "\n",stderr);

 fputs("Usage: wwwoffle -h\n"
       "       wwwoffle -online [-port <port>]\n"
       "       wwwoffle -offline [-port <port>]\n"
       "       wwwoffle -fetch [-port <port>]\n"
       "       wwwoffle -purge [<host> ... ] [<age>] [-port <port>]\n"
       "       wwwoffle <url> [-port <port>]\n"
       "\n",stderr);

 if(verbose)
    fprintf(stderr,
            "wwwoffle -h            : Display this help.\n"
            "\n"
            "wwwoffle -online       : Indicate to the server that the network is active.\n"
            "                         (Proxy requests will be fetched from remote hosts.)\n"
            "\n"
            "wwwoffle -offline      : Indicate to the server that the network is inactive.\n"
            "                         (Proxy requests will be fetched from cache or recorded.)\n"
            "\n"
            "wwwoffle -fetch        : Force the server to fetch the pages that are recorded.\n"
            "\n"
            "wwwoffle -purge        : Force the server to purge pages from the cache.\n"
            "         [<host> ... ]   A list of hosts to purge pages from.\n"
            "         [<age>]         The age of the pages to purge (0 = all pages).\n"
            "\n"
            "wwwoffle <url>         : Fetch the specified URL (or record for fetching).\n"
            "\n"
            "     -port <port.      : The port number to talk to the demon on.\n"
            "                         (Defaults %d for server, %d for http proxy).\n"
            "\n",DEF_WWWOFFLE_PORT,DEF_HTTP_PORT);

 exit(0);
}
