/***************************************
  $Header: /home/amb/wwwoffle/RCS/purge.c 1.1 1997/01/11 13:09:34 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 0.9.
  Purge old files from the cache.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "errors.h"


static int PurgeFiles(char *dirname);

/*+ The time of the oldest file to keep. +*/
static time_t old=0;


/*++++++++++++++++++++++++++++++++++++++
  Purge files from the cache that meet the criteria.

  int age The maximum age of files to keep (in days).

  int nhosts The number of host name that have been specified.

  char **hosts The list of host names that have to be purged.

  int client the file descriptor of the wwwoffle client.
  ++++++++++++++++++++++++++++++++++++++*/

void PurgeCache(int age,int nhosts,char **hosts,int client)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 time_t now=0;

 /* Determine the age of old files. */

 now=time(NULL);

 if(age>0)
    old=now-age*3600*24;
 else
    old=now+3600;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory [%!s]; purge failed.");return;}

 ent=readdir(dir);
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory [%!s]; purge failed.");closedir(dir);return;}

 ent=readdir(dir);  /* skip .  */
 ent=readdir(dir);  /* skip .. */

 /* Search through all of the sub directories. */

 while(ent)
   {
    char *name=ent->d_name;
    struct stat buf;

    if(stat(name,&buf))
       PrintMessage(Warning,"Cannot stat file '%s' [%!s]; race condition?",name);
    else
       if(S_ISDIR(buf.st_mode))
         {
          int match=0;

          if(!nhosts)
             match=1;
          else
             for(i=0;i<nhosts;i++)
                if(!strcmp(name,hosts[i]))
                  {match=1; *hosts[i]=0;}

          if(match)
            {
             char *msg=(char*)malloc(sizeof(name)+16);

             sprintf(msg,"Purging %s\n",name);
             write(client,msg,strlen(msg));
             free(msg);

             if(!PurgeFiles(name) && strcmp(name,"outgoing"))
                if(rmdir(name))
                   PrintMessage(Warning,"Cannot delete 'empty' directory '%s' [%!s].",name);
            }
         }

    ent=readdir(dir);
   }

 closedir(dir);

 for(i=0;i<nhosts;i++)
    if(*hosts[i])
      {
       char *msg=(char*)malloc(sizeof(hosts[i])+24);
       sprintf(msg,"No such directory %s\n",hosts[i]);
       write(client,msg,strlen(msg));
       free(msg);
      }
}


/*++++++++++++++++++++++++++++++++++++++
  Delete the file in the current directory that are older than the specified age.

  int PurgeFiles Returns the number of files that are left.

  char *dirname The name of the directory to purge.
  ++++++++++++++++++++++++++++++++++++++*/

static int PurgeFiles(char *dirname)
{
 int files_left=0;
 DIR *dir;
 struct dirent* ent;

 /* Open the spool subdirectory. */

 if(chdir(dirname))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not purged.",dirname);return(1);}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not purged.",dirname);chdir("..");return(1);}

 ent=readdir(dir);
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not purged.",dirname);closedir(dir);chdir("..");return(1);}

 ent=readdir(dir);  /* skip .  */
 ent=readdir(dir);  /* skip .. */

 /* Check all of the files for age, and delete as needed. */

 while(ent)
   {
    char *name=ent->d_name;
    struct stat buf;

    if(stat(name,&buf))
       PrintMessage(Warning,"Cannot stat file '%s/%s' [%!s]; race condition?",dirname,name);
    else
      {
       time_t t=buf.st_mtime;

       if(t>old)
          files_left++;
       else
          if(unlink(name))
             PrintMessage(Warning,"Cannot unlink file '%s/%s' [%!s].",dirname,name);
      }

    ent=readdir(dir);
   }

 chdir("..");

 return(files_left);
}
