/* UTF8-UCS4-String/lib/xp/xml_expat.cpp
 * 
 * Copyright (C) 2002 Francis James Franklin <fjf@alinameridon.com>
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <cstdio>

#include <expat.h>

#include "utf8string.h"
#include "utf8stringtree.h"

extern "C"
{
  static void s_StartElement (void * context, const char * name, const char ** atts);
  static void s_EndElement (void * context, const char * name);
  static void s_CharacterData (void * context, const char * s, int len);
  static void s_ProcessingInstruction (void * context, const char * target, const char * data);
  static void s_Comment (void * context, const char * data);
  static void s_StartCdataSection (void * context);
  static void s_EndCdataSection (void * context);
  static void s_Default (void * context, const char * s, int len);
  //  static void s_StartNamespaceDecl (void * context, const char * prefix, const char * uri);
  //  static void s_EndNamespaceDecl (void * context, const char * prefix);
  static void s_UnparsedEntityDecl (void * context,
				    const char * entityName,
				    const char * base,
				    const char * systemId,
				    const char * publicId,
				    const char * notationName);
  static void s_NotationDecl (void * context,
			      const char * notationName,
			      const char * base,
			      const char * systemId,
			      const char * publicId);
  static int s_NotStandalone (void * context);
}

extern bool xml_expat_parse (UTF8StringTree & tree, const char * buffer, int length, UTF8String & error)
{
  bool success = true;

  char errbuf[64];

  XML_Parser p = XML_ParserCreate (0);
  if (p == 0)
    {
      success = false;
      error = UTF8String("xml_expat_parse: unable to create expat XML parser!");
    }
  else
    {
      void * context = reinterpret_cast<void *>(&tree);

      XML_SetUserData (p, context);

      XML_SetParamEntityParsing (p, XML_PARAM_ENTITY_PARSING_NEVER);

      XML_SetElementHandler (p, s_StartElement, s_EndElement);
      XML_SetCharacterDataHandler (p, s_CharacterData);
      XML_SetProcessingInstructionHandler (p, s_ProcessingInstruction);
      XML_SetCommentHandler (p, s_Comment);
      XML_SetCdataSectionHandler (p, s_StartCdataSection, s_EndCdataSection);
      XML_SetDefaultHandler (p, s_Default);
      //      XML_SetNamespaceDeclHandler (p, s_StartNamespaceDecl, s_EndNamespaceDecl);
      XML_SetUnparsedEntityDeclHandler (p, s_UnparsedEntityDecl);
      XML_SetNotationDeclHandler (p, s_NotationDecl);
      XML_SetNotStandaloneHandler (p, s_NotStandalone);

      if (XML_Parse (p, buffer, length, 1))
	{
	  error = UTF8String("xml_expat_parse: okay!");
	}
      else
	{
	  success = false;

	  error = UTF8String(XML_ErrorString(XML_GetErrorCode(p)));

	  sprintf (errbuf, " [byte %ld @ line %d col. %d]",
		   XML_GetCurrentByteIndex (p),
		   XML_GetCurrentLineNumber (p),
		   XML_GetCurrentColumnNumber (p));

	  error += UTF8String(errbuf);
	}
      XML_ParserFree (p);
    }
  return success;
}

static void s_StartElement (void * context, const char * name, const char ** atts)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->StartElement (name, atts);
}

static void s_EndElement (void * context, const char * name)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->EndElement (name);
}

static void s_CharacterData (void * context, const char * s, int len)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->CharacterData (s, len);
}

static void s_ProcessingInstruction (void * context, const char * target, const char * data)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->ProcessingInstruction (target, data);
}

static void s_Comment (void * context, const char * data)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->Comment (data);
}

static void s_StartCdataSection (void * context)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->StartCdataSection ();
}

static void s_EndCdataSection (void * context)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->EndCdataSection ();
}

static void s_Default (void * context, const char * s, int len)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->Default (s, len);
}

/*
static void s_StartNamespaceDecl (void * context, const char * prefix, const char * uri)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->StartNamespaceDecl (prefix, uri);
}
static void s_EndNamespaceDecl (void * context, const char * prefix)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->EndNamespaceDecl (prefix);
}
*/

static void s_UnparsedEntityDecl (void * context,
				  const char * entityName,
				  const char * base,
				  const char * systemId,
				  const char * publicId,
				  const char * notationName)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->UnparsedEntityDecl (entityName, base, systemId, publicId, notationName);
}

static void s_NotationDecl (void * context,
			    const char * notationName,
			    const char * base,
			    const char * systemId,
			    const char * publicId)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) tree->NotationDecl (notationName, base, systemId, publicId);
}

static int s_NotStandalone (void * context)
{
  UTF8StringTree * tree = (UTF8StringTree *) context;

  if (tree) return tree->NotStandalone ();

  return 0;
}
