/* UTF8-UCS4-String/lib/xp/utf8stringmap.cpp
 * 
 * Copyright (C) 2002 Francis James Franklin <fjf@alinameridon.com>
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <cstdlib>
#include <cstring>

#include <new>
#include <stdexcept>

#include "utf8stringpair.h"
#include "utf8stringmap.h"

UTF8StringMap::UTF8StringMapValue::UTF8StringMapValue () :
  m_count(1),
  m_pair_max(0),
  m_pair_count(0),
  m_pair(0)
{
  // 
}

UTF8StringMap::UTF8StringMapValue::UTF8StringMapValue (const UTF8StringMapValue & rhs) :
  m_count(1),
  m_pair_max(0),
  m_pair_count(0),
  m_pair(0)
{
  unsigned long count = rhs.m_pair_count;

  if (!grow (count)) return; // throws std::bad_alloc

  bool success = true;
  for (unsigned long i = 0; i < count; i++)
    {
      UTF8StringPair * new_pair = 0;
      try
	{
	  new_pair = new UTF8StringPair(*(rhs.m_pair[i]));
	}
      catch (...)
	{
	  new_pair = 0;
	}
      if (new_pair == 0)
	{
	  success = false;
	  break;
	}
      m_pair[i] = new_pair;
      m_pair_count++;
    }
  if (!success)
    {
      clear ();
      throw std::bad_alloc();
      return;
    }
}

UTF8StringMap::UTF8StringMapValue::~UTF8StringMapValue ()
{
  clear ();
}

void UTF8StringMap::UTF8StringMapValue::clear ()
{
  for (unsigned long i = 0; i < m_pair_count; i++)
    {
      delete m_pair[i];
    }
  if (m_pair) free (m_pair);

  m_pair_max = 0;
  m_pair_count = 0;
  m_pair = 0;
}

bool UTF8StringMap::UTF8StringMapValue::ins (const UTF8String & key, const UTF8String & value, unsigned long & index)
{
  lookup (key, index);

  if (index < m_pair_count)
    if (m_pair[index]->name () == key)
      if (m_pair[index]->value () != value)
	{
	  m_pair[index]->value (value);
	  return true;
	}

  if (!grow ()) return false; // throws std::bad_alloc

  UTF8StringPair * new_pair = 0;
  try
    {
      new_pair = new UTF8StringPair(key,value);
    }
  catch (...)
    {
      new_pair = 0;
    }
  if (new_pair == 0)
    {
      return false;
    }

  if (index < m_pair_count)
    {
      unsigned long bytes = (m_pair_count - index) * sizeof (UTF8StringPair *);

      memmove (m_pair + index + 1, m_pair + index, bytes);
    }
  m_pair[index] = new_pair;
  m_pair_count++;

  return true;
}

void UTF8StringMap::UTF8StringMapValue::del (const UTF8String & key, unsigned long & index)
{
  if (m_pair_count == 0) return;

  if (lookup (key, index) == 0) return;

  UTF8StringPair * old_pair = m_pair[index];

  if (index < (m_pair_count - 1))
    {
      unsigned long bytes = (m_pair_count - 1 - index) * sizeof (UTF8StringPair *);

      memmove (m_pair + index, m_pair + index + 1, bytes);
    }
  m_pair_count--;

  delete old_pair;
}

void UTF8StringMap::UTF8StringMapValue::del (const char * key_utf8str, unsigned long & index)
{
  if (m_pair_count == 0) return;

  if (lookup (key_utf8str, index) == 0) return;

  UTF8StringPair * old_pair = m_pair[index];

  if (index < (m_pair_count - 1))
    {
      unsigned long bytes = (m_pair_count - 1 - index) * sizeof (UTF8StringPair *);

      memmove (m_pair + index, m_pair + index + 1, bytes);
    }
  m_pair_count--;

  delete old_pair;
}

const UTF8String * UTF8StringMap::UTF8StringMapValue::lookup (const UTF8String & key, unsigned long & index) const
{
  /* correct place may be cached in index, so check these cases first
   */
  if (index < m_pair_count) // equality
    {
      if (m_pair[index]->name () == key) return &(m_pair[index]->value ());
    }
  if ((0 < index) && (index + 1 < m_pair_count)) // bounded
    {
      if ((m_pair[index]->name () < key) && (key < m_pair[index+1]->name ())) return 0;
    }

  /* k, so index was wrong; check trivial cases
   */
  if (m_pair_count == 0) // empty map
    {
      index = 0;
      return 0;
    }
  if (key < m_pair[0]->name ()) // add at front
    {
      index = 0;
      return 0;
    }
  if (m_pair[m_pair_count-1]->name () < key) // add at end
    {
      index = m_pair_count;
      return 0;
    }
  if (m_pair[m_pair_count-1]->name () == key) // at end
    {
      index = m_pair_count - 1;
      return &(m_pair[m_pair_count-1]->value ());
    }

  /* oh well, need to search :-(
   */
  unsigned long upper = m_pair_count - 1;
  unsigned long lower = 0;

  while (true)
    {
      if (lower + 1 == upper) break;

      index = (lower + upper) / 2;

      if (key < m_pair[index]->name ())
	{
	  upper = index;
	}
      else
	{
	  lower = index;
	}
    }
  if (m_pair[lower]->name () == key)
    {
      index = lower;
      return &(m_pair[lower]->value ());
    }
  index = upper;
  return 0;
}

const UTF8String * UTF8StringMap::UTF8StringMapValue::lookup (const char * key_utf8str, unsigned long & index) const
{
  if (key_utf8str == 0)
    {
      index = 0; // well, not really, but...
      return 0;
    }
  const char * start = key_utf8str;
  const char * end = key_utf8str + strlen (key_utf8str);

  /* correct place may be cached in index, so check these cases first
   */
  if (index < m_pair_count) // equality
    {
      if (m_pair[index]->name().compare (start, end) == 0) return &(m_pair[index]->value ());
    }
  if ((0 < index) && (index + 1 < m_pair_count)) // bounded
    {
      if ((m_pair[index]->name().compare (start, end) < 0) && (m_pair[index+1]->name().compare (start, end) > 0)) return 0;
    }

  /* k, so index was wrong; check trivial cases
   */
  if (m_pair_count == 0) // empty map
    {
      index = 0;
      return 0;
    }
  if (m_pair[0]->name().compare (start, end) > 0) // add at front
    {
      index = 0;
      return 0;
    }
  if (m_pair[m_pair_count-1]->name().compare (start, end) < 0) // add at end
    {
      index = m_pair_count;
      return 0;
    }
  if (m_pair[m_pair_count-1]->name().compare (start, end) == 0) // at end
    {
      index = m_pair_count - 1;
      return &(m_pair[m_pair_count-1]->value ());
    }

  /* oh well, need to search :-(
   */
  unsigned long upper = m_pair_count - 1;
  unsigned long lower = 0;

  while (true)
    {
      if (lower + 1 == upper) break;

      index = (lower + upper) / 2;

      if (m_pair[index]->name().compare (start, end) > 0)
	{
	  upper = index;
	}
      else
	{
	  lower = index;
	}
    }
  if (m_pair[lower]->name().compare (start, end) == 0)
    {
      index = lower;
      return &(m_pair[lower]->value ());
    }
  index = upper;
  return 0;
}

bool UTF8StringMap::UTF8StringMapValue::grow (unsigned long size)
{
  if (m_pair_count + size <= m_pair_max) return true;

  unsigned long increase = (size < 8) ? 8 : size;
  unsigned long bytes = (m_pair_max + increase) * sizeof (UTF8StringPair *);

  if (m_pair == 0)
    {
      m_pair = (UTF8StringPair **) malloc (bytes);
      if (m_pair == 0)
	{
	  throw std::bad_alloc();
	  return false;
	}
      m_pair_max += increase;
    }
  else
    {
      UTF8StringPair ** more = (UTF8StringPair **) realloc (m_pair, bytes);
      if (more == 0)
	{
	  throw std::bad_alloc();
	  return false;
	}
      m_pair = more;
      m_pair_max += increase;
    }
  return true;
}

UTF8StringMap::UTF8StringMap () :
  m_value(new UTF8StringMapValue)
{
  if (m_value == 0) throw std::bad_alloc();
}

UTF8StringMap::UTF8StringMap (const UTF8StringMap & rhs) :
  m_value(rhs.m_value)
{
  m_value->ref ();
}

UTF8StringMap::~UTF8StringMap ()
{
  if (!m_value->unref ()) delete m_value;
}

UTF8StringMap & UTF8StringMap::operator= (const UTF8StringMap & rhs)
{
  if (m_value == rhs.m_value) return *this;

  if (!m_value->unref ()) delete m_value;
  m_value = rhs.m_value;
  m_value->ref ();

  return *this;
}

void UTF8StringMap::clear ()
{
  if (m_value->count () > 1)
    {
      m_value->unref ();
      m_value = new UTF8StringMapValue;
      if (m_value == 0)
	{
	  throw std::bad_alloc();
	  return;
	}
    }
  else m_value->clear ();
}

bool UTF8StringMap::ins (const UTF8String & key, const UTF8String & value)
{
  unsigned long index = 0;

  const UTF8String * old_value = 0;

  if (old_value = m_value->lookup (key, index))
    if (*old_value == value)
      return true;

  if (m_value->count () > 1)
    {
      m_value->unref ();
      m_value = new UTF8StringMapValue(*m_value);
      if (m_value == 0)
	{
	  throw std::bad_alloc();
	  return false;
	}
    }
  return m_value->ins (key, value, index);
}

void UTF8StringMap::del (const UTF8String & key)
{
  unsigned long index = 0;
  if (m_value->lookup (key, index) == 0) return;

  if (m_value->count () > 1)
    {
      m_value->unref ();
      m_value = new UTF8StringMapValue(*m_value);
      if (m_value == 0)
	{
	  throw std::bad_alloc();
	  return;
	}
    }
  m_value->del (key, index);
}

void UTF8StringMap::del (const char * key_utf8str)
{
  unsigned long index = 0;
  if (m_value->lookup (key_utf8str, index) == 0) return;

  if (m_value->count () > 1)
    {
      m_value->unref ();
      m_value = new UTF8StringMapValue(*m_value);
      if (m_value == 0)
	{
	  throw std::bad_alloc();
	  return;
	}
    }
  m_value->del (key_utf8str, index);
}
