/* UTF8-UCS4-String/lib/xp/include/utf8stringtree.h
 * 
 * Copyright (C) 2002 Francis James Franklin <fjf@alinameridon.com>
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef UTF8_UCS4_STRING__UTF8STRINGTREE_H
#define UTF8_UCS4_STRING__UTF8STRINGTREE_H

#ifdef UTF8_UCS4_STRING_SOURCE
#include "utf8stringmap.h"
#else
#include <UTF8-UCS4-String/utf8stringmap.h>
#endif

class UTF8StringTree
{
public:
  static void xmlns_query (const UTF8StringMap & map, const UTF8String & name, UTF8String & prefix, UTF8String & uri);

  static void style_parse (UTF8StringMap & map, const UTF8String & style);

  enum NodeType
  {
    nt_element,
    nt_text,
    nt_cdata,
    nt_pi,
    nt_comment,
    nt_default
  };
  class Node
    {
    public:
      virtual ~Node () { }
      virtual NodeType type () const = 0;
    };
  class ElementNode : public Node
    {
    private:
      UTF8String m_ns_default;
      UTF8String m_prefix;
      UTF8String m_uri;
      UTF8String m_name;

      Node ** m_node;

      unsigned long m_node_count;
      unsigned long m_node_max;

      UTF8StringMap m_xmlns;
      UTF8StringMap m_style;
      UTF8StringMap m_attrs;

    public:
      ElementNode (const UTF8String & ns_default, // default namespace uri
		   const UTF8String & prefix,     // namespace prefix of element
		   const UTF8String & uri,        // namespace uri of element
		   const char * name,             // element name
		   const char * const * atts,     // element attributes
		   const UTF8StringMap & xmlns,   // namespace prefix->uri map
		   const UTF8StringMap & style);  // CSS style map

      virtual ~ElementNode ();

      NodeType type () const;

      const UTF8String & ns_default () const { return m_ns_default; }
      const UTF8String & prefix () const     { return m_prefix; }
      const UTF8String & uri ()  const       { return m_uri; }
      const UTF8String & name () const       { return m_name; }

      /* ElementNode is responsible for node; do not delete it elsewhere!
       */
      bool append (Node * node);

      const Node * operator[] (unsigned long n) const
	{
	  return (m_node && (n < m_node_count)) ? m_node[n] : 0;
	}
      unsigned long count () const { return m_node_count; }

      const UTF8StringMap & xmlns () const { return m_xmlns; }
      const UTF8StringMap & style () const { return m_style; }
      const UTF8StringMap & attrs () const { return m_attrs; }
    };
  class TextNode : public Node
    {
    private:
      UTF8String m_text;

    public:
      TextNode ();

      virtual ~TextNode ();

      virtual NodeType type () const;

      const UTF8String & text () const { return m_text; }

      void clear ();
      void append (const UTF8String & text);
    };
  class CDATANode : public TextNode
    {
    public:
      CDATANode ();
      ~CDATANode ();
      NodeType type () const;
    };
  class PINode : public TextNode
    {
    private:
      UTF8String m_target;

    public:
      PINode (const char * target);

      ~PINode ();

      NodeType type () const;

      const UTF8String & target () const { return m_target; }
    };
  class CommentNode : public TextNode
    {
    public:
      CommentNode ();
      ~CommentNode ();
      NodeType type () const;
    };
  class DefaultNode : public TextNode
    {
    public:
      DefaultNode ();
      ~DefaultNode ();
      NodeType type () const;
    };
  class ElementFactory
    {
    public:
      class XMLNS
	{
	public:
	  virtual ~XMLNS () { }

	  virtual ElementNode * createElement (const UTF8String & ns_default, // default namespace uri
					       const UTF8String & prefix,     // namespace prefix of element
					       const UTF8String & uri,        // namespace uri of element
					       const char * name,             // element name
					       const char * const * atts,     // element attributes
					       const UTF8StringMap & xmlns,   // namespace prefix->uri map
					       const UTF8StringMap & style    // CSS style map
					       ) const = 0;
	};

      ElementFactory ();

      ~ElementFactory ();

      static ElementFactory * instance (); // make your own if you want

      bool registerNamespace (const UTF8String & uri, const XMLNS * delegate);

      ElementNode * createElement (const UTF8String & ns_default, const UTF8String & prefix, const UTF8String & uri,
				   const char * name, const char * const * atts,
				   const UTF8StringMap & xmlns, const UTF8StringMap & style) const;
    private:
      struct URI_Map
      {
	UTF8String *  uri;
	const XMLNS * delegate;
      };

      URI_Map *     m_map;

      unsigned long m_map_count;
      unsigned long m_map_max;
    };
  class ElementStack
    {
    private:
      ElementNode ** m_stack;

      unsigned long  m_stack_size;
      unsigned long  m_stack_max;

    public:
      ElementStack ();

      ~ElementStack ();

      void clear ();

      bool push (ElementNode * node);

      ElementNode * pop ();
      ElementNode * top ();

      const ElementNode * operator[] (unsigned long n) const
	{
	  return (m_stack && (n < m_stack_size)) ? m_stack[n] : 0;
	}
      unsigned long size () const { return m_stack_size; }
    };

 private:
  char *           m_buffer;

  unsigned long    m_buffer_length;
  unsigned long    m_buffer_max;

  bool             m_stop;

  ElementFactory * m_factory;

  Node **          m_node;

  unsigned long    m_node_count;
  unsigned long    m_node_max;

  TextNode *       m_active;

  ElementNode *    m_tree;
  ElementNode *    m_current;

  ElementStack     m_stack;

  UTF8String       m_uri;
  UTF8String       m_error;

  UTF8StringMap    m_xmlns;
  UTF8StringMap    m_style;

 public:
  UTF8StringTree (UTF8String * uri = 0, ElementFactory * factory = 0);

  ~UTF8StringTree ();

  void clear ();

  const UTF8StringMap & xmlns () const { return m_xmlns; }
  const UTF8StringMap & style () const { return m_style; }

  bool import (const char * buffer, int length, UTF8String & error);

 private:
  /* UTF8StringTree is responsible for node; do not delete it elsewhere!
   */
  bool _append (Node * node);

 public:
  const Node * operator[] (unsigned long n) const
    {
      return (m_node && (n < m_node_count)) ? m_node[n] : 0;
    }
  unsigned long count () const { return m_node_count; }

  const ElementNode * tree ()    const { return m_tree; }
  const ElementNode * current () const { return m_current; }

  const ElementStack & stack () const { return m_stack; }

  const UTF8String & uri () const { return m_uri; }

  /* =========================================================================================
   * SAX callbacks
   */
  void StartElement (const char * name, const char * const * atts);
  void EndElement (const char * name);

  void CharacterData (const char * s, int len);
 private:
  bool _flushCharacterData ();
  bool _flushData ();
  bool _appendData (const char * s, int len);

 public:
  void ProcessingInstruction (const char * target, const char * data);

  void Comment (const char * data);

  void StartCdataSection ();
  void EndCdataSection ();

  void Default (const char * s, int len);
 private:
  bool _flushDefaultData ();

 public:
  /*
   * void StartNamespaceDecl (const char * prefix, const char * uri);
   * void EndNamespaceDecl (const char * prefix);
   */

  void UnparsedEntityDecl (const char * entityName,
			   const char * base,
			   const char * systemId,
			   const char * publicId,
			   const char * notationName);

  void NotationDecl (const char * notationName,
		     const char * base,
		     const char * systemId,
		     const char * publicId);

  int NotStandalone ();
};

#endif /* ! UTF8_UCS4_STRING__UTF8STRINGTREE_H */
