/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* UTF8-UCS4-String/lib/cocoa/UUS_Window.mm
 * 
 * Copyright (C) 2002 Francis James Franklin <fjf@alinameridon.com>
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "UUS_Window.h"

@interface UUS_WindowDelegate : NSWindowController
{
	UUS_Window *	m_window;
}
- (id)initWithWindow:(NSWindow *)window withUUSWindow:(UUS_Window *)uus;
- (void)windowDidResize:(NSNotification *)aNotification;
- (BOOL)windowShouldClose:(id)sender;
@end

@implementation UUS_WindowDelegate

- (id)initWithWindow:(NSWindow *)window withUUSWindow:(UUS_Window *)uus
{
	[super initWithWindow:window];
	m_window = uus;
	return self;
}

- (void)windowDidResize:(NSNotification *)aNotification
{
	m_window->recalibrate ();
}

- (BOOL)windowShouldClose:(id)sender
{
	return m_window->close () ? YES : NO;
}

@end

UUS_Window::UUS_Window (WindowStyle ws) :
	m_ws(ws),
	m_styleMask(NSBorderlessWindowMask),
	m_backingType(NSBackingStoreBuffered),
	m_controller(nil),
	m_window(nil),
	m_view(nil),
	m_opacity(1)
{
	switch (m_ws)
		{
		ws_Raw:
			m_styleMask = NSBorderlessWindowMask;
			break;
		default:
		ws_Panel:
		ws_Normal:
			m_styleMask = NSTitledWindowMask | NSClosableWindowMask | NSMiniaturizableWindowMask | NSResizableWindowMask;
			break;
		}

	NSRect frame = [[NSScreen mainScreen] frame];

	int width  = static_cast<int>(frame.size.width);
	int height = static_cast<int>(frame.size.height);

	m_frame.origin.x = static_cast<float>(width  >> 3);
	m_frame.origin.y = static_cast<float>(height >> 3);

	width  -= width  >> 2;
	height -= height >> 2;

	m_frame.size.width  = static_cast<float>(width);
	m_frame.size.height = static_cast<float>(height);
}

UUS_Window::~UUS_Window ()
{
	if (m_controller)
		{
			if (m_window) [m_controller close];
			[m_controller release];
		}
}

bool UUS_Window::create (bool closable, bool miniaturizable, bool resizable)
{
	if (m_styleMask & NSTitledWindowMask)
		{
			m_styleMask = NSTitledWindowMask;
			if (closable)
				m_styleMask |= NSClosableWindowMask;
			if (miniaturizable)
				m_styleMask |= NSMiniaturizableWindowMask;
			if (resizable)
				m_styleMask |= NSResizableWindowMask;
		}
	return create ();
}

bool UUS_Window::create ()
{
	if (m_window) return true;

	if (m_ws == ws_Panel)
		m_window = (NSWindow *) [NSPanel alloc];
	else
		m_window = [NSWindow alloc];

	if (m_window == 0) return false;
	[m_window initWithContentRect:m_frame styleMask:m_styleMask backing:m_backingType defer:YES];

	if (m_controller)
		{
			[m_controller release];
			m_controller = nil;
		}
	UUS_WindowDelegate * delegate = [UUS_WindowDelegate alloc];
	if (delegate == 0)
		{
			[m_window release];
			m_window = 0;
			return false;
		}
	m_controller = (NSWindowController *) [delegate initWithWindow:m_window withUUSWindow:this];

	[m_window setDelegate:m_controller];
	[m_window setAlphaValue:m_opacity];

	switch (m_ws)
		{
		ws_Raw:
		ws_Panel:
			[m_window setHidesOnDeactivate:YES];
			[m_window setExcludedFromWindowsMenu:YES];
			[m_window setLevel:NSFloatingWindowLevel];
			break;
		default:
		ws_Normal:
			break;
		}

	m_view = [m_window contentView];

	[m_controller showWindow:m_controller];

	// [m_window setFrame:m_frame display:YES];
}

void UUS_Window::destroy ()
{
	if (m_window == nil) return;

	[m_window performClose:m_controller];
}

void UUS_Window::setAlpha (float opacity)
{
	m_opacity = (opacity > 1) ? 1 : ((opacity < 0) ? 0 : opacity);
}

void UUS_Window::setSize (unsigned int width, unsigned int height)
{
	m_frame.size.width  = static_cast<float>(width);
	m_frame.size.height = static_cast<float>(height);

	// TODO: redo...
}

void UUS_Window::setMinSize (unsigned int width, unsigned int height)
{
	if (m_window == nil) return;

	NSSize size;
	size.width  = static_cast<float>(width);
	size.height = static_cast<float>(height);
	[m_window setMinSize:size];
}

bool UUS_Window::replaceMainView (NSView * view)
{
	if (m_window == nil) return false;
	if (m_view   == nil) return false;
	if (  view   == nil) return false;

	[m_window setContentView:view];

	m_view = [m_window contentView];

	m_frame = [m_view frame];

	return true;
}

void UUS_Window::notifyResized ()
{
	// 
}

void UUS_Window::recalibrate ()
{
	m_frame = [m_view frame];
	notifyResized ();
}

bool UUS_Window::closeRequest ()
{
	return true;
}

bool UUS_Window::close ()
{
	bool can_close = closeRequest ();

	if (can_close)
		{
			m_window = 0;
			m_view = 0;
		}
	return can_close;
}
