/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* UTF8-UCS4-String/lib/cocoa/UUS_TreeWindow.mm
 * 
 * Copyright (C) 2002 Francis James Franklin <fjf@alinameridon.com>
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "utf8stringtree.h"

#include "UUS_Window.h"
#include "UUS_TreeWindow.h"
#include "UUS_Table.h"

enum ColumnName
{
	cn_attr_prefix,
	cn_attr_uri,
	cn_attr_name,
	cn_attr_value
};

@interface AttrsDataSource : NSObject
{
	int									m_count;
	const UTF8StringTree::ElementNode *	m_node;
}
-(id)initWithElement:(const UTF8StringTree::ElementNode *)element;
-(id)setElement:(const UTF8StringTree::ElementNode *)element;
-(int)numberOfRowsInTableView:(NSTableView *)aTableView;
-(id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex;
-(BOOL)tableView:(NSTableView *)aTableView shouldEditTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex;
-(id)retain;
-(oneway void)release;
@end

class AttrsTable : public UUS_Table
{
public:
	AttrsTable (const UTF8String & table_name);

	~AttrsTable ();

	void clear (); // disassociate current node

	bool remap (const UTF8StringTree::ElementNode * node);
private:
	bool _map ();

public:
	bool attach (NSView * parent, int x, int y, unsigned int width, unsigned int height, unsigned int mask);

private:
	const UTF8StringTree::ElementNode *	m_node;
	AttrsDataSource *					m_dsrc;
};

class TreeWindow : public UUS_Window, UUS_Table::TableClick
{
private:
	TreeWindow (UUS_Window::WindowStyle ws, const UTF8String & outline_name,
				const UTF8String & attrs_name, const UTF8String & xmlns_name, const UTF8String & style_name);

public:
	~TreeWindow ();

	static TreeWindow * treeWindow (UUS_Window::WindowStyle ws); // use this for construction

	bool setTree (const UTF8StringTree * tree);

	bool create ();
	void destroy ();

	bool closeRequest ();

	/* call-back: user has clicked in a table...
	 */
	void tableClick (const UUS_Table & table);

private:
	const UTF8StringTree *	m_tree;

	NSSplitView *			m_split_view;
	NSView *				m_info_view;
	NSText *				m_text_view;

	UUS_TreeTable *			m_outline_table;
	AttrsTable *			m_attrs_table;
	UUS_MapTable *			m_xmlns_table;
	UUS_MapTable *			m_style_table;

	const UTF8String		m_outline_name;
	const UTF8String		m_attrs_name;
	const UTF8String		m_xmlns_name;
	const UTF8String		m_style_name;
	const UTF8String		m_xmlns_left;
	const UTF8String		m_xmlns_right;
	const UTF8String		m_style_left;
	const UTF8String		m_style_right;

	const UTF8StringMap		m_empty;
};

@implementation AttrsDataSource

-(id)initWithElement:(const UTF8StringTree::ElementNode *)element
{
	[super init];
	m_count = 1;
	m_node = element;
	return self;
}

-(id)setElement:(const UTF8StringTree::ElementNode *)element
{
	m_node = element;
}

-(int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	if (m_node == 0) return 0;
	return static_cast<int>(m_node->attrs().size ());
}

-(id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	static UTF8String prefix("prefix");
	static UTF8String uri("uri");

	if (m_node == 0) return nil; // hmm...
	if (aTableColumn == nil) return nil; // hmm...

	NSNumber * number = (NSNumber *) [aTableColumn identifier];
	if (number == nil) return nil; // hmm...

	UTF8StringMap map = m_node->attrs ();
	const UTF8StringPair * pair = map[rowIndex];

	switch ((ColumnName) [number intValue])
		{
		case cn_attr_prefix:
		case cn_attr_uri:
			{
				UTF8String prefix(m_node->prefix ());
				UTF8String uri(m_node->uri ());

				UTF8StringTree::xmlns_query (m_node->xmlns (), pair->name (), prefix, uri);

				if ((ColumnName) [number intValue] == cn_attr_prefix)
					return [NSString stringWithString:(prefix.ns_str ())]; // ??
				else
					return [NSString stringWithString:(uri.ns_str ())];
			}
			break;
		case cn_attr_name:
			return const_cast<UTF8String &>(pair->name ()).ns_str ();
			break;
		default:
		case cn_attr_value:
			return const_cast<UTF8String &>(pair->value ()).ns_str ();
			break;
		}
}

-(BOOL)tableView:(NSTableView *)aTableView shouldEditTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	return NO;
}

-(id)retain
{
	if (m_count > 0) m_count++;
	return [super retain];
}

-(oneway void)release
{
	if (m_count == 0)
		{
			// 
		}
	[super release];
}

@end

AttrsTable::AttrsTable (const UTF8String & table_name) :
	UUS_Table(table_name,false),
	m_node(0),
	m_dsrc(nil)
{
	// 
}

AttrsTable::~AttrsTable ()
{
	if (m_dsrc) [m_dsrc release];
}

void AttrsTable::clear () // disassociate current node
{
	remap (0);
}

bool AttrsTable::remap (const UTF8StringTree::ElementNode * node)
{
	if (m_node == node) return true;

	m_node = node;
	if (m_dsrc)
		{
			[m_dsrc setElement:node];
			reload ();
		}
	return true;
}

bool AttrsTable::_map ()
{
	if (m_dsrc) return true;

	bool success = true;

	m_dsrc = [AttrsDataSource alloc];
	if (m_dsrc)
		{
			[m_dsrc initWithElement:m_node];

			UTF8String attr_prefix("XML Namespace Prefix");
			UTF8String attr_uri("XML Namespace URI");
			UTF8String attr_name("Attribute Name");
			UTF8String attr_value("Attribute Value");

			column ([NSNumber numberWithInt:((int) cn_attr_prefix)], attr_prefix.ns_str ());
			column ([NSNumber numberWithInt:((int) cn_attr_uri   )], attr_uri.ns_str ());
			column ([NSNumber numberWithInt:((int) cn_attr_name  )], attr_name.ns_str ());
			column ([NSNumber numberWithInt:((int) cn_attr_value )], attr_value.ns_str ());

			set_data_src (m_dsrc);
			set_delegate (m_dsrc); // since our delegate is our data source
		}
	else success = false;

	return success;
}

bool AttrsTable::attach (NSView * parent, int x, int y, unsigned int width, unsigned int height, unsigned int mask)
{
	if (!UUS_Table::attach (parent, x, y, width, height, mask)) return false;
	return _map ();
}

@implementation UUS_TreeWindow

+(UUS_TreeWindow *)windowWithTree:(UUS_Tree *)tree
{
	TreeWindow * window = TreeWindow::treeWindow (UUS_Window::ws_Normal);
	if (window == 0) return nil;

	void * w = reinterpret_cast<void *>(window);

	UUS_TreeWindow * tw = [UUS_TreeWindow alloc];
	if (tw == nil)
		{
			delete window;
			return nil;
		}
	[tw initWithWindow:w];
	[tw setTree:tree];

	return tw;
}

+(UUS_TreeWindow *)panelWithTree:(UUS_Tree *)tree
{
	TreeWindow * window = TreeWindow::treeWindow(UUS_Window::ws_Panel);
	if (window == 0) return nil;

	void * w = reinterpret_cast<void *>(window);

	UUS_TreeWindow * tw = [UUS_TreeWindow alloc];
	if (tw == nil)
		{
			delete window;
			return nil;
		}
	[tw initWithPanel:w];
	[tw setTree:tree];

	return tw;
}

-(id)initWithWindow:(void *)window
{
	[super init];

	m_count = 1;
	m_window = window;
	m_tree = nil;
	m_isPanel = NO;

	return self;
}

-(id)initWithPanel:(void *)window
{
	[super init];

	m_count = 1;
	m_window = window;
	m_tree = nil;
	m_isPanel = YES;

	return self;
}

-(void)setTree:(UUS_Tree *)tree
{
	TreeWindow * window = reinterpret_cast<TreeWindow *>(m_window);

	if (m_tree)
		{
			if (window) window->setTree (0);
			[m_tree release];
		}
	m_tree = tree;
	if (m_tree)
		{
			[m_tree retain];
			if (window) window->setTree (reinterpret_cast<const UTF8StringTree *>([m_tree tree]));
		}
}

-(UUS_Tree *)tree
{
	return m_tree;
}

-(BOOL)show
{
	if (m_window == 0) return NO;

	TreeWindow * window = reinterpret_cast<TreeWindow *>(m_window);
	if (!window->create ()) return NO;

	// TODO

	return YES;
}

-(void)hide
{
	if (m_window == 0) return;

	TreeWindow * window = reinterpret_cast<TreeWindow *>(m_window);
	window->destroy ();
}

-(id)retain
{
	if (m_count > 0) m_count++;
	return [super retain];
}

-(oneway void)release
{
	if (--m_count == 0)
		{
			TreeWindow * window = reinterpret_cast<TreeWindow *>(m_window);
			delete window;
			m_window = 0;
			if (m_tree) [m_tree release];
			m_tree = nil;
		}
	[super release];
}

@end

TreeWindow * TreeWindow::treeWindow (UUS_Window::WindowStyle ws)
{
	TreeWindow * tw = 0;
	try
		{
			UTF8String outline_name("Outline Table");
			UTF8String attrs_name("Attributes Table");
			UTF8String xmlns_name("XMLNS Table");
			UTF8String style_name("Style Table");

			tw = new TreeWindow(ws,outline_name,attrs_name,xmlns_name,style_name);
		}
	catch (...)
		{
			tw = 0;
		}
	return tw;
}

TreeWindow::TreeWindow (UUS_Window::WindowStyle ws, const UTF8String & outline_name,
						const UTF8String & attrs_name, const UTF8String & xmlns_name, const UTF8String & style_name) :
	UUS_Window(ws),
	m_tree(0),
	m_split_view(nil),
	m_info_view(nil),
	m_text_view(nil),
	m_outline_table(new UUS_TreeTable(outline_name)),
	m_attrs_table(new AttrsTable(attrs_name)),
	m_xmlns_table(new UUS_MapTable(xmlns_name)),
	m_style_table(new UUS_MapTable(style_name)),
	m_outline_name(outline_name),
	m_attrs_name(attrs_name),
	m_xmlns_name(xmlns_name),
	m_style_name(style_name),
	m_xmlns_left("XML Namespace Prefix"),
	m_xmlns_right("XML Namespace URI"),
	m_style_left("CSS Style Name"),
	m_style_right("CSS Style Value")
{
	if (m_outline_table) m_outline_table->setClickBack (this);
}

TreeWindow::~TreeWindow ()
{
	if (m_outline_table) delete m_outline_table;

	if (m_attrs_table) delete m_attrs_table;
	if (m_xmlns_table) delete m_xmlns_table;
	if (m_style_table) delete m_style_table;
}

bool TreeWindow::setTree (const UTF8StringTree * tree)
{
	if (m_tree && (tree == 0))
		{
			// TODO ??
		}
	m_tree = tree;

	if (m_outline_table) m_outline_table->setTree (tree);

	if (m_tree == 0) return true;

	// TODO ??
}

@interface MySplitView : NSSplitView
{
	NSText *		m_text;
	NSScrollView *	m_scroll_view;
}
-(id)initWithFrame:(NSRect)frame;
-(void)setText:(NSText *)text scrollView:(NSScrollView *)scroll_view;
-(void)splitViewDidResizeSubviews:(NSNotification *)aNotification;
-(void)drawDividerInRect:(NSRect)aRect;
-(float)dividerThickness;
@end

@implementation MySplitView

-(id)initWithFrame:(NSRect)frame
{
	[super initWithFrame:frame];
	m_text = nil;
	m_scroll_view = nil;
	return self;
}

-(void)setText:(NSText *)text scrollView:(NSScrollView *)scroll_view
{
	m_text = text;
	m_scroll_view = scroll_view;
}

-(void)splitViewDidResizeSubviews:(NSNotification *)aNotification
{
	if ((m_text == nil) || (m_scroll_view == nil)) return;

	[m_text retain];

	[m_scroll_view setDocumentView:nil];

	NSRect frame;
	frame.origin.x = 0;
	frame.origin.y = 0;
	frame.size = [m_scroll_view contentSize];

	[m_text setFrame:frame];
	[m_text setNeedsDisplay:YES];

	[m_scroll_view setDocumentView:m_text];
}

-(void)drawDividerInRect:(NSRect)aRect
{
	[[NSColor knobColor] set];
	[NSBezierPath fillRect:aRect];
}

-(float)dividerThickness
{
	return static_cast<float>(3);
}

@end

bool TreeWindow::create ()
{
	if (m_split_view) return true;

	setSize (600, 400);

	NSRect frame;
	frame.origin.x = 0;
	frame.origin.y = 0;
	frame.size.width  = 600;
	frame.size.height = 400;

	MySplitView * split_view = [MySplitView alloc];
	if (split_view)
		{
			[split_view initWithFrame:frame];
			[split_view setVertical:YES];
			[split_view setIsPaneSplitter:YES];
		}
	else return false;

	/* create outline view here so that it sits in left pane
	 */
	if (!m_outline_table->attach (split_view, 0, 0, 200, 400, NSViewWidthSizable|NSViewHeightSizable))
		{
			[split_view release];
			return false;
		}

	frame.origin.x = 200;
	frame.size.width  = 400;

	NSView * base_view = [NSView alloc];
	if (base_view)
		{
			[base_view initWithFrame:frame];
			[split_view addSubview:base_view];
		}
	else
		{
			[split_view release];
			return false;
		}

	frame.origin.x = 0;
	frame.origin.y = 300;
	frame.size.width  = 400;
	frame.size.height = 100;

	NSView * info_view = [NSView alloc];
	if (info_view)
		{
			[info_view initWithFrame:frame];
			[info_view setAutoresizingMask:NSViewWidthSizable];
			[base_view addSubview:info_view];
		}
	else
		{
			[split_view release];
			return false;
		}

	frame.origin.x = 0;
	frame.origin.y = 0;
	frame.size.width  = 400;
	frame.size.height = 300;

	MySplitView * tri_view = [MySplitView alloc];
	if (tri_view)
		{
			[tri_view initWithFrame:frame];
			[tri_view setAutoresizingMask:(NSViewWidthSizable|NSViewHeightSizable)];
			[tri_view setVertical:NO];
			[tri_view setIsPaneSplitter:YES];
			[base_view addSubview:tri_view];
		}
	else
		{
			[split_view release];
			return false;
		}

	frame.origin.x = 0;
	frame.origin.y = 260;
	frame.size.width  = 400;
	frame.size.height = 40;

	NSScrollView * scroll_view = [NSScrollView alloc];
	if (scroll_view)
		{
			[scroll_view initWithFrame:frame];
			[scroll_view setHasHorizontalScroller:NO];
			[scroll_view setHasVerticalScroller:YES];
			[tri_view addSubview:scroll_view];
		}
	else
		{
			[split_view release];
			return false;
		}

	frame.origin.x = 0;
	frame.origin.y = 0;
	frame.size = [scroll_view contentSize];

	NSText * text_view = [NSText alloc];
	if (text_view)
		{
			[text_view initWithFrame:frame];
			[scroll_view setDocumentView:text_view];
		}
	else
		{
			[split_view release];
			return false;
		}

	[tri_view setText:text_view scrollView:scroll_view];
	[tri_view setDelegate:tri_view];

	/* create attributes table view here so that it sits in top pane
	 */
	if (!m_attrs_table->attach (tri_view, 0, 130, 400, 130, NSViewWidthSizable|NSViewHeightSizable))
		{
			[split_view release];
			return false;
		}

	frame.origin.x = 0;
	frame.origin.y = 0;
	frame.size.width  = 400;
	frame.size.height = 130;

	MySplitView * duo_view = [MySplitView alloc];
	if (duo_view)
		{
			[duo_view initWithFrame:frame];
			[duo_view setVertical:YES];
			[duo_view setIsPaneSplitter:YES];
			[tri_view addSubview:duo_view];
		}
	else
		{
			[split_view release];
			return false;
		}

	/* create other scrolling table views, XMLNS on the left, style on the right
	 */
	if (!m_xmlns_table->attach (duo_view, 0, 0, 200, 150, NSViewWidthSizable|NSViewHeightSizable))
		{
			[split_view release];
			return false;
		}
	if (!m_style_table->attach (duo_view, 150, 0, 200, 150, NSViewWidthSizable|NSViewHeightSizable))
		{
			[split_view release];
			return false;
		}

	if (m_tree)
		{
			m_xmlns_table->remap (m_tree->xmlns (), m_xmlns_left, m_xmlns_right);
			m_style_table->remap (m_tree->style (), m_style_left, m_style_right);
		}
	else
		{
			m_xmlns_table->remap (m_empty, m_xmlns_left, m_xmlns_right);
			m_style_table->remap (m_empty, m_style_left, m_style_right);
		}
	
	if (UUS_Window::create ())
		{
			setMinSize (200, 200);
			if (replaceMainView (split_view))
				{
					m_split_view   = split_view;
					m_info_view    = info_view;
					m_text_view    = text_view;
					return true;
				}
		}
	[split_view release];
	return false;
}

void TreeWindow::destroy ()
{
	UUS_Window::destroy ();
}

bool TreeWindow::closeRequest ()
{
	m_split_view   = nil;
	m_info_view    = nil;

	return true;
}

void TreeWindow::tableClick (const UUS_Table & table)
{
	if ((table.name () == m_outline_name) && m_tree)
		{
			const UTF8StringTree::Node * node = m_outline_table->m_node;

			if (m_split_view && m_text_view)
				{
					[m_text_view setString:@""];
				}

			if (node == 0)
				{
					m_attrs_table->remap (0);
					m_xmlns_table->remap (m_tree->xmlns (), m_xmlns_left, m_xmlns_right);
					m_style_table->remap (m_tree->style (), m_style_left, m_style_right);
				}
			else if (node->type () != UTF8StringTree::nt_element)
				{
					m_attrs_table->remap (0);
					m_xmlns_table->remap (m_empty, m_xmlns_left, m_xmlns_right);
					m_style_table->remap (m_empty, m_style_left, m_style_right);

					const UTF8StringTree::TextNode * text = dynamic_cast<const UTF8StringTree::TextNode *>(node);
					if (m_split_view && m_text_view)
						{
							[m_text_view setString:(const_cast<UTF8String &>(text->text ()).ns_str ())];
						}
				}
			else
				{
					const UTF8StringTree::ElementNode * element = dynamic_cast<const UTF8StringTree::ElementNode *>(node);

					m_attrs_table->remap (element);
					m_xmlns_table->remap (element->xmlns (), m_xmlns_left, m_xmlns_right);
					m_style_table->remap (element->style (), m_style_left, m_style_right);
				}
		}
}
