/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* UTF8-UCS4-String/lib/cocoa/UUS_Table.mm
 * 
 * Copyright (C) 2002 Francis James Franklin <fjf@alinameridon.com>
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "utf8stringtree.h"

#include "UUS_Table.h"

/* ==== UUS_Table ====
 */
UUS_Table::UUS_Table (const UTF8String & table_name, bool is_outline) :
	m_target(0),
	m_table_name(table_name),
	m_is_outline(is_outline),
	m_scroll_view(nil),
	m_table_view(nil),
	m_header_view(nil),
	m_data_src(nil),
	m_delegate(nil)
{
	// 
}

UUS_Table::~UUS_Table ()
{
	// 
	if (m_scroll_view)
		{
			NSView * parent = [m_scroll_view superview];
			if (parent)
				{
					[parent setNeedsDisplayInRect:[m_scroll_view frame]];
					[m_scroll_view removeFromSuperview];
				}
			else [m_scroll_view release];
		}
	if (m_data_src) [m_data_src release];
	if (m_delegate) [m_delegate release];
}

bool UUS_Table::attach (NSView * parent, int x, int y, unsigned int width, unsigned int height, unsigned int mask)
{
	detach ();

	if (parent == nil) return false;
	if (height < 20) return false;

	NSRect frame;
	frame.origin.x = static_cast<float>(x);
	frame.origin.y = static_cast<float>(y);
	frame.size.width  = static_cast<float>(width);
	frame.size.height = static_cast<float>(height);

	if (m_scroll_view == nil)
		{
			m_scroll_view = [NSScrollView alloc];
			if (m_scroll_view == nil)
				{
					m_table_view = nil;
					return false;
				}
			[m_scroll_view initWithFrame:frame];

			frame.origin.x = static_cast<float>(0);
			frame.origin.y = static_cast<float>(17);
			frame.size.width  = static_cast<float>(width);
			frame.size.height = static_cast<float>(height - 17);

			if (m_is_outline)
				m_table_view = [NSOutlineView alloc];
			else
				m_table_view = [NSTableView alloc];
			if (m_table_view == nil)
				{
					[m_scroll_view release];
					return false;
				}
			[m_table_view initWithFrame:frame];

			[m_scroll_view setDocumentView:m_table_view];

			[m_scroll_view setHasHorizontalScroller:YES];
			[m_scroll_view setHasVerticalScroller:YES];

			if (m_data_src) [m_table_view setDataSource:m_data_src];

			if (m_delegate) [m_table_view setDelegate:m_delegate];

			frame.origin.x = static_cast<float>(0);
			frame.origin.y = static_cast<float>(0);
			frame.size.width  = static_cast<float>(width);
			frame.size.height = static_cast<float>(17);

			m_header_view = [NSTableHeaderView alloc];
			if (m_header_view == nil)
				{
					[m_scroll_view release];
					m_table_view = nil;
					return false;
				}
			[m_header_view initWithFrame:frame];
			[m_scroll_view addSubview:m_header_view];

			[m_table_view setHeaderView:m_header_view];
		}
	else [m_scroll_view setFrame:frame];

	[m_scroll_view setAutoresizingMask:mask];

	[parent addSubview:m_scroll_view];
	return true;
}

void UUS_Table::detach ()
{
	if (m_scroll_view == nil) return;

	NSView * parent = [m_scroll_view superview];

	if (parent == nil) return;

	[parent setNeedsDisplayInRect:[m_scroll_view frame]];

	[m_scroll_view retain];
	[m_scroll_view removeFromSuperview];
}

bool UUS_Table::column (id identifier, NSString * header, bool is_outline)
{
	if (m_table_view == nil) return false;

	bool success = true;

	NSTableColumn * tcol = [NSTableColumn alloc];
	if (tcol)
		{
			[tcol initWithIdentifier:identifier];
			[[tcol headerCell] setStringValue:header];
			[m_table_view addTableColumn:tcol];

			if (is_outline && m_is_outline)
				{
					NSOutlineView * view = (NSOutlineView *) m_table_view;
					[view setOutlineTableColumn:tcol];
				}
		}
	else success = false;

	return success;
}

/* remove all columns
 */
void UUS_Table::clear ()
{
	// TODO
}

void UUS_Table::reload ()
{
	if (m_table_view) [m_table_view reloadData];
}

/* data_src will be retained and later released
 * set_data_src (nil) will release the current data source, if any, and empty the table
 */
void UUS_Table::set_data_src (id data_src)
{
	if (m_data_src) [m_data_src release];

	m_data_src = data_src;

	if (m_data_src)
		{
			[m_data_src retain];
			if (m_table_view) [m_table_view setDataSource:m_data_src];
		}
}

/* delegate will be retained and later released
 * set_delegate (nil) will release the current delegate, if any
 */
void UUS_Table::set_delegate (id delegate)
{
	if (m_delegate) [m_delegate release];

	m_delegate = delegate;

	if (m_delegate)
		{
			[m_delegate retain];
			if (m_table_view) [m_table_view setDelegate:delegate];
		}
}

int UUS_Table::selectedRow () // -1 if none
{
	return (m_table_view == nil) ? -1 : [m_table_view selectedRow];
}

id UUS_Table::selectedItem () // nil if none
{
	if (!m_is_outline) return nil;

	NSOutlineView * view = (NSOutlineView *) m_table_view;
	int row = selectedRow ();
	return (row >= 0) ? [view itemAtRow:row] : nil;
}

/* ==== UUS_MapTableDataSource ====
 */
@implementation UUS_MapTableDataSource

-(id)initWithMap:(UTF8StringMap *)map
{
	[super init];
	m_count = 1;
	m_map = map;
	return self;
}

-(int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	if (m_map == 0) return 5;

	return static_cast<int>(m_map->size ());
}

-(id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	if (m_map == 0) return @"fake value 1";
	if (aTableColumn == nil) return @"fake value 2";
	if ([aTableColumn identifier] == nil) return @"fake value 3";

	const UTF8StringPair * pair = (*m_map)[rowIndex];
	if (pair == 0) return @"fake value 4";

	NSNumber * number = (NSNumber *) [aTableColumn identifier];

	if ([number intValue] == 1)
		return const_cast<UTF8String &>(pair->name ()).ns_str ();
	else
		return const_cast<UTF8String &>(pair->value ()).ns_str ();
}

-(BOOL)tableView:(NSTableView *)aTableView shouldEditTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	return NO;
}

-(id)retain
{
	if (m_count > 0) m_count++;
	return [super retain];
}

-(oneway void)release
{
	if (m_count == 0)
		{
			// 
		}
	[super release];
}

@end

/* ==== UUS_MapTable ====
 */
UUS_MapTable::UUS_MapTable (const UTF8String & table_name) :
	UUS_Table(table_name,false),
	m_mapped(false),
	m_dsrc(nil)
{
	// 
}

UUS_MapTable::UUS_MapTable (const UTF8String & table_name, const UTF8StringMap & map,
							const UTF8String & left, const UTF8String & right) :
	UUS_Table(table_name,false),
	m_mapped(true),
	m_dsrc(nil),
	m_map(map),
	m_left(left),
	m_right(right)
{
	// 
}

UUS_MapTable::~UUS_MapTable ()
{
	if (m_dsrc) [m_dsrc release];
}

/* disassociate current map
 */
void UUS_MapTable::clear ()
{
	if (m_dsrc)
		{
			[m_dsrc release];
			m_dsrc = nil;
		}
	set_data_src (nil);
	set_delegate (nil); // since our delegate is our data source
	m_mapped = false;
	UUS_Table::clear ();
}

bool UUS_MapTable::remap (const UTF8StringMap & map, const UTF8String & left, const UTF8String & right)
{
	if ((m_left == left) && (m_right == right))
		{
			m_map = map;
			m_mapped = true;
			reload ();
			return true;
		}
	else
		{
			if (m_mapped) clear ();

			m_map = map;
			m_left = left;
			m_right = right;

			m_mapped = true;
			return _map ();
		}
}

bool UUS_MapTable::_map ()
{
	if (m_dsrc) return true;

	NSNumber * l = [NSNumber numberWithInt:1];
	NSNumber * r = [NSNumber numberWithInt:2];

	bool success = true;

	m_dsrc = [UUS_MapTableDataSource alloc];
	if (m_dsrc)
		{
			[m_dsrc initWithMap:(&m_map)];

			column (l,  m_left.ns_str ());
			column (r, m_right.ns_str ());

			set_data_src (m_dsrc);
			set_delegate (m_dsrc); // since our delegate is our data source
		}
	else success = false;

	return success;
}

bool UUS_MapTable::attach (NSView * parent, int x, int y, unsigned int width, unsigned int height, unsigned int mask)
{
	if (!UUS_Table::attach (parent, x, y, width, height, mask)) return false;
	return m_mapped ? _map () : true;
}

/* ==== OutlineItem ====
 */
@interface OutlineItem : NSObject
{
	const UTF8StringTree *				m_tree;
	const UTF8StringTree::Node *	m_node;
}
-(id)initWithTree:(const UTF8StringTree *)tree withNode:(const UTF8StringTree::Node *)node;
-(const UTF8StringTree *)tree;
-(const UTF8StringTree::Node *)node;
-(int)count;
-(id)typeValue;
-(id)stringValue;
@end

@implementation OutlineItem

-(id)initWithTree:(const UTF8StringTree *)tree withNode:(const UTF8StringTree::Node *)node
{
	[super init];
	m_tree = tree;
	m_node = node;
	return self;
}

-(const UTF8StringTree *)tree
{
	return m_tree;
}

-(const UTF8StringTree::Node *)node
{
	return m_node;
}

-(int)count
{
	if (m_node == 0) return 0;

	if (m_node->type () == UTF8StringTree::nt_element)
		{
			const UTF8StringTree::ElementNode * element = dynamic_cast<const UTF8StringTree::ElementNode *>(m_node);
			return static_cast<int>(element->count ());
		}
	else return 0;
}

-(id)typeValue
{
	if (m_node == 0) return @"Bad Node!";

	switch (m_node->type ())
		{
		case UTF8StringTree::nt_element:
			return @"element";
		case UTF8StringTree::nt_text:
			return @"text";
		case UTF8StringTree::nt_cdata:
			return @"cdata";
		case UTF8StringTree::nt_pi:
			return @"pi";
		case UTF8StringTree::nt_comment:
			return @"comment";
		case UTF8StringTree::nt_default:
			return @"default";
		default:
			return @"unknown";
		}
}

-(id)stringValue
{
	if (m_node == 0) return @"Bad Node!";

	switch (m_node->type ())
		{
		case UTF8StringTree::nt_element:
			{
				const UTF8StringTree::ElementNode * element = dynamic_cast<const UTF8StringTree::ElementNode *>(m_node);
				return const_cast<UTF8String &>(element->name ()).ns_str ();
			}
		case UTF8StringTree::nt_pi:
			{
				const UTF8StringTree::PINode * pi = dynamic_cast<const UTF8StringTree::PINode *>(m_node);
				return const_cast<UTF8String &>(pi->target ()).ns_str ();
			}
		case UTF8StringTree::nt_text:
		case UTF8StringTree::nt_cdata:
		case UTF8StringTree::nt_comment:
		case UTF8StringTree::nt_default:
		default:
			return @"";
		}
}

@end

/* ==== UUS_TreeTableDataSource ====
 */
@implementation UUS_TreeTableDataSource

-(id)initWithTree:(const UTF8StringTree *)tree withTable:(UUS_TreeTable *)table
{
	[super init];
	m_count = 0;
	m_tree = tree;
	m_table = table;
	return self;
}

-(id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)item
{
	if (m_tree == 0) return nil;

	OutlineItem * o_item = nil;

	if (item == nil)
		{
			o_item = [OutlineItem alloc];
			if (o_item) [o_item initWithTree:m_tree withNode:((*m_tree)[index])];
		}
	else
		{
			const UTF8StringTree::Node * node = [((OutlineItem *) item) node];
			if (node->type () == UTF8StringTree::nt_element)
				{
					const UTF8StringTree::ElementNode * element = dynamic_cast<const UTF8StringTree::ElementNode *>(node);

					o_item = [OutlineItem alloc];
					if (o_item) [o_item initWithTree:m_tree withNode:((*element)[index])];
				}
		}
	return o_item;
}

-(BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item
{
	if (item == nil) return m_tree ? YES : NO;

	OutlineItem * o_item = (OutlineItem *) item;
	return [o_item count] ? YES : NO;
}

-(int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item
{
	if (m_tree == 0) return 0;

	if (item == nil) return static_cast<int>(m_tree->count ());

	OutlineItem * o_item = (OutlineItem *) item;
	return [o_item count];
}

-(id)outlineView:(NSOutlineView *)outlineView objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
	if (item == nil) return nil;
	OutlineItem * o_item = (OutlineItem *) item;

	NSNumber * number = (NSNumber *) [tableColumn identifier];

	if ([number intValue] == 1)
		return [o_item typeValue];
	else
		return [o_item stringValue];
}

-(BOOL)outlineView:(NSOutlineView *)outlineView shouldEditTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
	return NO;
}

-(void)outlineViewSelectionDidChange:(NSNotification *)notification
{
	OutlineItem * o_item = (OutlineItem *) m_table->selectedItem ();
	if (o_item)
		{
			m_table->m_node = [o_item node];
		}
	else m_table->m_node = 0;

	m_table->click ();
}

-(id)retain
{
	if (m_count > 0) m_count++;
	return [super retain];
}

-(oneway void)release
{
	if (m_count == 0)
		{
			// 
		}
	[super release];
}

@end

/* ==== UUS_TreeTable ====
 */
UUS_TreeTable::UUS_TreeTable (const UTF8String & table_name) :
	UUS_Table(table_name,true),
	m_dsrc(nil),
	m_tree(0),
	m_node(0),
	m_left("XML Tree Node Type"),
	m_right("Node Name/Target/Data")
{
	// 
}

UUS_TreeTable::~UUS_TreeTable ()
{
	if (m_dsrc) [m_dsrc release];
}

void UUS_TreeTable::clear ()
{
	setTree (0);
}

bool UUS_TreeTable::setTree (const UTF8StringTree * tree)
{
	if (m_dsrc)
		{
			[m_dsrc release];
			m_dsrc = nil;
		}
	set_data_src (nil);
	set_delegate (nil); // since our delegate is our data source

	UUS_Table::clear ();

	m_tree = tree;

	return exists () ? _tree () : true;
}

bool UUS_TreeTable::_tree ()
{
	if (m_dsrc) return true;

	NSNumber * l = [NSNumber numberWithInt:1];
	NSNumber * r = [NSNumber numberWithInt:2];

	bool success = true;

	m_dsrc = [UUS_TreeTableDataSource alloc];
	if (m_dsrc)
		{
			[m_dsrc initWithTree:m_tree withTable:this];

			column (l,  m_left.ns_str (), true);
			column (r, m_right.ns_str ());

			set_data_src (m_dsrc);
			set_delegate (m_dsrc); // since our delegate is our data source
		}
	else success = false;

	return success;
}

bool UUS_TreeTable::attach (NSView * parent, int x, int y, unsigned int width, unsigned int height, unsigned int mask)
{
	if (!UUS_Table::attach (parent, x, y, width, height, mask)) return false;
	return _tree ();
}
