
/*
** (c) 2004 Herbert Poetzl
**
** V0.01	flag ioctls
**
*/

#define VERSION "V0.01"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/utsname.h>
 
#ifdef O_LARGEFILE
#define OPEN_FLAGS (O_RDONLY|O_NONBLOCK|O_LARGEFILE)
#else
#define OPEN_FLAGS (O_RDONLY|O_NONBLOCK)
#endif


static	char	err_flag = 0;

static	char	opt_flag = 0;

static	char *	cmd_name = NULL;
static	char *	name = NULL;

static	int 	num_flag = 0;


/* proc ioctls */

#define FIOC_GETXFLG    _IOR('x', 5, long)
#define FIOC_SETXFLG    _IOW('x', 6, long)
#define FIOC_GETXFLG64    _IOR('x', 5, long long)
#define FIOC_SETXFLG64    _IOW('x', 6, long long)

static bool use_64 = false;

int getxflg(int fd, int *flag)
{
	int r, c;

	r = ioctl (fd, FIOC_GETXFLG, &c);
	if (r == -1){
		use_64 = true;
		r = ioctl (fd, FIOC_GETXFLG64, &c);
	}
	if (flag)
	    *flag = c;
	return r;
}

int setxflg(int fd, int flag)
{
	int r, c;

	c = (int) flag;
	r = ioctl (fd, use_64 ? FIOC_SETXFLG64 : FIOC_SETXFLG, &c);
	return r;
}


int do_getxflg(int fd, int *flag)
{
    	int r;
	
	if ((r=getxflg(fd, flag))) {
	    switch(errno) {
	    case EPERM:
 		fprintf(stderr,
		    "%s: insufficient permission.\n",
		    cmd_name);
    	    	exit(3);
		break;
	    case ENOTTY:
 		fprintf(stderr,
		    "%s: ioctl not supported on %s\n",
		    cmd_name, name);
		break;
	    default:
 		fprintf(stderr,
		    "%s: unknown error on getxflg.\n",
		    cmd_name);
 		exit(5);
	    }
	}
    	return r;
}

int do_setxflg(int fd, int flag)
{
    	int r;
	
	if ((r=setxflg(fd, num_flag))) {
	    switch(errno) {
	    case EROFS:
 		fprintf(stderr,
		    "%s: read only filesystem.\n",
		    cmd_name);
 		exit(4);
	    case EPERM:
 		fprintf(stderr,
		    "%s: insufficient permission.\n",
		    cmd_name);
 		exit(3);
	    case ENOTTY:
 		fprintf(stderr,
		    "%s: ioctl not supported on %s\n",
		    cmd_name, name);
		break;
	    default:
 		fprintf(stderr,
		    "%s: unknown error on setxflg.\n",
		    cmd_name);
 		exit(5);
	    }	
    	}
    	return r;
}
	

#define OPTIONS "+hdDeEf:"

int	main(int argc, char *argv[])
{
	extern int optind;
	extern char *optarg;
	char c, errflg = 0;
	int r;

	{
		struct utsname uts;
		if (uname(&uts)!=-1){
			//printf ("sysname :%s:\n",uts.sysname);
			//printf ("release :%s:\n",uts.release);
			if (strcmp(uts.release,"2.6.")< 0){
				fprintf (stderr,"vproc: Do nothing on old kernel\n");
				exit (0);
			}
		}else{
			// Can't tell the version of the OS, better do nothing
			fprintf (stderr,"Can't tell the version of the OS. Weird\n");
			exit (-1);
		}
	}

	
	cmd_name = argv[0];
	while ((c = getopt(argc, argv, OPTIONS)) != (char)EOF) {
		switch (c) {
		case 'h':
			fprintf(stderr,
				"This is %s " VERSION "\n"
				"options are:\n"
				"-h        print this help message\n"
				"-d        hide in context > 1\n"
				"-D        hide in any context\n"
				"-e        show everywhere\n"
				"-E        show only in context 1\n"
				"-f <num>  flag value in decimal\n"
				"--        end of options\n"
				,cmd_name);
			exit(0);
			break;
		case 'd':
			num_flag = 3;
			opt_flag = 1;
			break;
		case 'D':
			num_flag = 1;
			opt_flag = 1;
			break;
		case 'e':
			num_flag = 0;
			opt_flag = 1;
			break;
		case 'E':
			num_flag = 2;
			opt_flag = 1;
			break;
		case 'f':
			num_flag = atol(optarg);
			opt_flag = 1;
			break;
		case '?':
		default:
			errflg++;
			break;
		}
	}        
	if (errflg) {
		fprintf(stderr, 
			"Usage: %s -[" OPTIONS "]\n"
			"%s -h for help.\n",
			cmd_name, cmd_name);
		exit(2);
	}

	for (; optind < argc; optind++) {
		struct stat64 buf;

		name = argv[optind];
		if (stat64(name, &buf) != 0) {
			fprintf(stderr,"%s: unable to stat %s\n",
			cmd_name, name);
			continue;
		} 
	    
		int fd = open(name, OPEN_FLAGS);
		if (fd < 0) {
			fprintf(stderr,"%s: unable to access %s\n",
				cmd_name, name);
			continue;
		}

		int flag;
		r = do_getxflg(fd, &flag);
		if (!r && opt_flag) r = do_setxflg(fd, num_flag);
		close(fd);
		if (r) continue;
	    
		if (!opt_flag) {
			static char *where[] = {
				"visible everywhere",
				"visible on the host only",
				"visible in context 1 only",
				"hidden in user contexts"
			};
	    	fprintf(stdout,"%s is %s.\n",name, where[flag & 0x3]);
		}
		/* output changes */
	}
	return err_flag?1:0;
}

