/*
	2001-11-15 

	Guillaum Dallaire vserver@guillaum.org
 	
	Distributed under the Gnu Public License, see the License file
	in this package.
*/
/*
	vserver-stat help you to see all the active context currently in the kernel
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <syscall.h>

#define PROC_DIR_NAME "/proc"
#define CTX_DIR_NAME "/var/run/vservers/"
#define CTX_NAME_MAX_LEN 50

#ifndef __NR_new_s_context
	#define __NR_new_s_context	226
#endif

_syscall3(int, new_s_context, int, newctx, int, remove_cap, int, flags);

struct ctx_list
{
	int ctx;
	int process_count;
	int VmSize_total;
	int VmRSS_total;
	long stime_total, utime_total;
	char name[CTX_NAME_MAX_LEN];
	struct ctx_list *next;
} *my_ctx_list;

struct process_info
{
	long VmSize;
	long VmRSS;
	long stime, utime;
	long cstime, cutime;	// dead children time
	int s_context;
};

char *process_name;

void usage()
{
	fprintf(stderr, "%s version %s\n", process_name, VERSION);
	fprintf(stderr, "(no argument needed)\n\n");
	fprintf(stderr, "Show informations about all the active context.\n\n");
	fprintf(stderr, "	CTX#		Context number\n");
	fprintf(stderr, "			#0 = root context\n");
	fprintf(stderr, "			#1 = monitoring context\n");
	fprintf(stderr, "	PROC QTY	Quantity of processes in each context\n");
	fprintf(stderr, "	VSZ		Number of pages of virtual memory\n");
	fprintf(stderr, "	RSS		Resident set size\n");
	fprintf(stderr, "	utime		User-mode CPU time accumulated\n");
	fprintf(stderr, "	ctime		Kernel-mode CPU time accumulated\n");
	fprintf(stderr, "\n");

}

// insert a new record to the list
struct ctx_list *insert_ctx(int ctx, struct ctx_list *next)
{
	struct ctx_list *new;

	new = (struct ctx_list *)malloc(sizeof(struct ctx_list));
	new->ctx = ctx;
	new->process_count = 0;
	new->VmSize_total = 0;
	new->VmRSS_total = 0;
	new->next = next;
	new->name[0] = '\0';

	return new;
}

// find the ctx record with the ctx number
struct ctx_list *find_ctx(struct ctx_list *list, int ctx)
{
	// very simple search engine...
	while(list != NULL)
	{
		// find
		if (list->ctx == ctx)
		{
			return list;
		}
		list = list->next;
	}
	return NULL;
}

// compute the process info into the list
void add_ctx(struct ctx_list *list, struct process_info *process)
{
	list->process_count ++;
	list->VmSize_total += process->VmSize;
	list->VmRSS_total += process->VmRSS;
	list->utime_total += process->utime + process->cutime;
	list->stime_total += process->stime + process->cstime;
}

// increment the count number in the ctx record using ctx number
void count_ctx(struct ctx_list *list, struct process_info *process)
{
	struct ctx_list *prev = list;

	if (process == NULL) return;

	// search
	while(list != NULL)
	{
		// find
		if (list->ctx == process->s_context)
		{
			add_ctx(list, process);
			return;
		}
		// insert between
		if (list->ctx > process->s_context)
		{
			prev->next = insert_ctx(process->s_context, list);
			add_ctx(prev->next, process);
			return;
		}
		// ++
		prev = list;
		list = list->next;
	}
	// add at the end
	prev->next = insert_ctx(process->s_context, NULL);
	add_ctx(prev->next, process);
}

// free mem
void free_ctx(struct ctx_list *list)
{
	struct ctx_list *prev;

	for(;list != NULL; list = prev)
	{
		prev = list->next;		
		free(list);
	}
}

// show the ctx_list with name from /var/run/servers/*.ctx
void show_ctx(struct ctx_list *list)
{
	// fill the ctx_list using the /var/run/servers/*.ctx file(s)
	int bind_ctx_name(struct ctx_list *list)
	{
		// fetch the context number in /var/run/vservers/'filename'
		int fetch_ctx_number(char *filename)
		{
			int fd;
			int ctx;
			char buf[25];

			// open file
			if ((fd = open(filename, O_RDONLY, 0)) == -1)
				return -1;
			// put the file in a small buffer
			if (read(fd, buf, sizeof(buf)) < 1)
				return -1;

			close(fd);

			sscanf(buf, "S_CONTEXT=%d", &ctx);
			return ctx;
		}

		/* begin bind_ctx_name */

		DIR *ctx_dir;
		struct dirent *dir_entry;
		char *p;
		char ctx_name[CTX_NAME_MAX_LEN];
		struct ctx_list *ctx;
		int ctx_number;

		// open the /var/run/vservers directory
		if ((ctx_dir = opendir(CTX_DIR_NAME)) == NULL)
		{
			fprintf(stderr, "%s: in openning %s: %s\n", process_name, CTX_DIR_NAME, strerror(errno));
			return -1;
		}
	
		chdir(CTX_DIR_NAME);
		while ((dir_entry = readdir(ctx_dir)) != NULL)
		{
			strncpy(ctx_name, dir_entry->d_name, sizeof(ctx_name));
			p = strstr(ctx_name, ".ctx");
			if (p != NULL) // make sure that it is a .ctx file..
			{
				*p = '\0'; // remove the .ctx in the file name
				if ((ctx_number = fetch_ctx_number(dir_entry->d_name)) > 1)
				{
					if ((ctx = find_ctx(list, ctx_number)) != NULL)
						strncpy(ctx->name, ctx_name, CTX_NAME_MAX_LEN);
				}
			}
			// else fprintf(stderr, "invalid file %s in %s\n", dir_entry->d_name, CTX_DIR_NAME);
		}
		closedir(ctx_dir);	
		return 0;
	}

	char *convert_time(long t, char *str)
	{
		unsigned hh, mm, ss, ms;

		ms = t % 100;
		t /= 100;

		ss = t%60;
		t /= 60;
		mm = t%60;
		t /= 60;
		hh = t%24;
 		t /= 24;

		/* FIXME: add day output */

		if (hh > 0)
	  		snprintf(str, 25, "%02uh%02um%02u", hh, mm, ss);
		else
		{
	  		snprintf(str, 25, "%02um%02u.%02u", mm, ss, ms);
		}
		return str;
	}
	
	/* begin show_ctx */
	char utime[25], stime[25];

	// now we have all the active context, fetch the name
	// from /var/run/vservers/*.ctx
	bind_ctx_name(list);

	printf("CTX#   PROC QTY       VSZ        RSS    utime    stime NAME\n");
	while(list != NULL)
	{
		if (list->ctx == 1)
			strncpy(list->name, "monitoring server", CTX_NAME_MAX_LEN);

		printf("%-7d%8d%10u %10u %s %s %s\n", list->ctx, list->process_count, 
			list->VmSize_total / 1024, list->VmRSS_total, 
			convert_time(list->utime_total, utime), convert_time(list->stime_total, stime), list->name);
		list = list->next;
	}
}

// open the process's status file to get the ctx number, and other stat
struct process_info *get_process_info(char *pid)
{
	int fd;
	char buffer[1024];
	char *p;
	static struct process_info process;

	// open the proc/#/status file
	snprintf(buffer, sizeof(buffer),  "/proc/%s/status", pid);
	if ((fd = open(buffer, O_RDONLY, 0)) == -1)
		return NULL;
	// put the file in a buffer
	if (read(fd, buffer, sizeof(buffer)) < 1)
		return NULL;

	close(fd);

	// find the s_context entry
	if ((p = strstr(buffer, "s_context:")) == NULL)
		return NULL;

	sscanf(p, "s_context: %d", &process.s_context);

	// open the /proc/#/stat file
	snprintf(buffer, sizeof(buffer),  "/proc/%s/stat", pid);
	if ((fd = open(buffer, O_RDONLY, 0)) == -1)
		return NULL;
	// put the file in a buffer
	if (read(fd, buffer, sizeof(buffer)) < 1)
		return NULL;

	close(fd);

	p = strchr(buffer, ')');		// go after the PID (process_name)
	sscanf(p + 2,
		"%*s "
		"%*s %*s %*s %*s %*s "
		"%*s %*s %*s %*s %*s %lu %lu "
		"%ld %ld %*s %*s %*s %*s "
		"%*s %lu "
		"%ld ", &process.utime, &process.stime,
			&process.cutime, &process.cstime,
 			&process.VmSize, &process.VmRSS);

	return &process;
}

int main(int argc, char **argv)
{
	DIR *proc_dir;
	struct dirent *dir_entry;
	pid_t my_pid;

	// for error msg
	process_name = argv[0];

	if (argc > 1)
	{
		usage();
		return 0;
	}

	// do not include own stat
	my_pid = getpid();

	// try to switch in context 1
	if (new_s_context(1, 0, 0) < 0)
	{
		fprintf(stderr, "%s: unable to switch in context security #1\n", process_name);
		return -1;
	}

	// create the fist...
	my_ctx_list = insert_ctx(0, NULL);
	// init with the default name for the context 0
	strncpy(my_ctx_list->name, "root server", CTX_NAME_MAX_LEN);

	// open the /proc dir
	if ((proc_dir = opendir(PROC_DIR_NAME)) == NULL)
	{
		fprintf(stderr, "%s: %s\n", process_name, strerror(errno));
		return -1;
	}
	
	chdir(PROC_DIR_NAME);
	while ((dir_entry = readdir(proc_dir)) != NULL)
	{
		// select only process file
		if (!isdigit(*dir_entry->d_name))
			continue;

		if (atoi(dir_entry->d_name) != my_pid)
			count_ctx(my_ctx_list, get_process_info(dir_entry->d_name));
		
	}
	closedir(proc_dir);

	// output the ctx_list	
	show_ctx(my_ctx_list);

	// free the ctx_list
	free_ctx(my_ctx_list);

	return 0;
}
