/*
	This utility is used to build a new vserver using a reference vserver.
	It uses hard link whenever possible instead of duplicating files.
	Once done, it sets the immutable bits.
*/
#include <stdio.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>

#include <string>
#include <vector>
#include <list>
#include <set>
#include <algorithm>
#include <iostream.h>
#include <pfstream.h>
#include <linux/ext2_fs.h>
#include "vutil.h"

// Patch to help compile this utility on unpatched kernel source
#ifndef EXT2_IMMUTABLE_FILE_FL
	#define EXT2_IMMUTABLE_FILE_FL	0x00000010
	#define EXT2_IMMUTABLE_LINK_FL	0x00008000
#endif


using namespace std;

static bool testmode = false;
static int debug = 0;
static int  ext2flags = EXT2_IMMUTABLE_FILE_FL | EXT2_IMMUTABLE_LINK_FL;
static struct {
	int nblink;
	int nbcopy;
	long size_copy;
	int nbdir;
	int nbsymlink;
	int nbspc;
} stats;


static void usage()
{
	cerr <<
		"vbuild version " << VERSION <<
		"\n\n"
		"vbuild [ options ] reference-server new-vservers\n"
		"\n"
		"--test: Show what will be done, do not do it.\n"
		"--debug: Prints some debugging messages.\n"
		"--noflags: Do not put any immutable flags on the file\n"
		"--immutable: Set the immutable_file bit on the files.\n"
		"--immutable-mayunlink: Sets the immutable_link flag on files.\n"
		"--stats: Produce statistics on the number of file linked\n"
		"         copied and so on.\n"
		"\n"
		"By default, the immutable_file and	immutable_link flags are\n"
		"set on the files. So if you want no immutable flags, you must\n"
		"use --noflags. If you want a single flag, you must use\n"
		"--noflags first, then the --immutable or --immutable-mayunlink\n"
		"flag.\n"
		;
}

class PACKAGE{
public:
	string name;
	string version;	// version + release
	PACKAGE(string &_name, string &_version)
		: name (_name), version(_version)
	{
	}
	PACKAGE(const char *_name, const char *_version)
		: name (_name), version(_version)
	{
	}
	PACKAGE(const string &line)
	{
		*this = line;
	}
	PACKAGE & operator = (const string &_line)
	{
		string line (_line);
		string::iterator pos = find (line.begin(),line.end(),'=');
		if (pos != line.end()){
			name = string(line.begin(),pos);
			version = string(pos + 1,line.end());
		}
		return *this;
	}
	bool operator == (const PACKAGE &v) const
	{
		return name == v.name && version == v.version;
	}
	// Load the file member of the package, but exclude configuration file
	void loadfiles(const string &ref, set<string> &files)
	{
		if (debug > 2) cout << "Loading files for package " << name << endl;
		string cmd = "| /usr/sbin/chroot " + ref + " /bin/rpm -ql --dump "
			+ name + "-" + version;
		ipfstream oo (cmd.c_str());
		while (1){
			char tmp[1000];
			int mode=-1,type=-1;
			oo.getline (tmp,sizeof(tmp)-1);
			if (tmp[0] == '\0') break;
			char *start = tmp;
			for (int i=0; i<8; i++){
				char *pt = start;
				while (*pt > ' ') pt++;
				if (*pt == ' ') *pt++ = '\0';
				if (i == 4){
					sscanf(start,"%o",&mode);
				}else if (i==7){
					type = atoi(start);
				}
				start = pt;
					
			}						
			if (type == 0 && S_ISREG(mode)){
				files.insert (tmp);
			}else if (debug > 10){
				cout << "Package " << name << " exclude " << tmp << endl;
			}
		}
		if (debug > 2) cout << "Done\n";
	}
	#if 0
	bool locate(const string &path)
	{
		return find (files.begin(),files.end(),path) != files.end();
	}
	#endif
};

/*
	Load the list of all packages in a vserver
*/
static void vunify_loadallpkg (string &refserver, list<PACKAGE> &packages)
{
	string cmd = "| /usr/sbin/chroot " + refserver + " /bin/rpm -qa"
		+ " --queryformat \"%{name}=%{version}-%{release}\\n\"";
	// cout << "command " << cmd << endl;
	ipfstream oo (cmd.c_str());
	copy (istream_iterator<string>(oo),istream_iterator<string>()
		,inserter(packages,packages.begin()));
}

/*
	Set the immutable flag on a file
*/
static int setext2flag (const char *fname, bool set)
{
	int ret = -1;
	if (testmode){
		ret = 0;
	}else{
		int fd = open (fname,O_RDONLY);
		if (fd == -1){
			cerr << "Can't open file " << fname 
				<< " (" << strerror(errno) << ")\n";
		}else{
			int flags = set ? ext2flags : 0;
			ret = ioctl (fd,EXT2_IOC_SETFLAGS,&flags);
			close (fd);
			if (ret == -1){
				cerr << "Can't " << (set ? "set" : "unset")
					<< " immutable flag on file "
					<< fname
					<< " (" << strerror(errno) << ")\n";
			}
		}
	}
	return ret;
}

static int vbuild_mkdir (const char *path, mode_t mode)
{
	int ret = -1;
	if (testmode){
		printf ("mkdir %s; chmod %o %s\n",path,mode,path);
		ret = 0;
	}else{
		ret = mkdir (path,mode);
	}
	return ret;
}

static int vbuild_mknod(const char *path, mode_t mode, dev_t dev)
{
	int ret = -1;
	if (testmode){
		printf ("mknod %s %o %02x:%02x\n",path,mode,major(dev),minor(dev));
		ret = 0;
	}else{
		ret = mknod (path,mode,dev);
	}
	return ret;
}
static int vbuild_symlink(const char *src, const char *dst)
{
	int ret = -1;
	if (testmode){
		printf ("ln -s %s %s\n",src,dst);
		ret = 0;
	}else{
		ret = symlink (src,dst);
	}
	return ret;
}

static int vbuild_link(const char *src, const char *dst)
{
	int ret = -1;
	if (testmode){
		printf ("ln %s %s\n",src,dst);
		ret = 0;
	}else{
		ret = link (src,dst);
	}
	return ret;
}

static int vbuild_chown(const char *path, uid_t uid, gid_t gid)
{
	int ret = -1;
	if (testmode){
		printf ("chown %d.%d %s\n",uid,gid,path);
		ret = 0;
	}else{
		ret = chown (path,uid,gid);
	}
	return ret;
}

static int vbuild_file_copy(
	const char *src,
	const char *dst,
	struct stat &st)
{
	int ret = -1;
	if (testmode){
		printf ("cp -a %s %s\n",src,dst);
		ret = 0;
	}else{
		ret = file_copy (src,dst,st);
	}
	return ret;
}

static int vbuild_copy (
	string refserv,
	string newserv,
	string logical_dir,
	set<string> &files)
{
	int ret = -1;
	if (debug > 0) printf ("Copying directory %s\n",logical_dir.c_str());
	DIR *dir = opendir (refserv.c_str());
	if (dir == NULL){
		fprintf (stderr,"Can't open directory %s (%s)\n",refserv.c_str()
			,strerror(errno));
	}else{
		struct dirent *ent;
		ret = 0;
		while ((ent=readdir(dir))!=NULL){
			if (strcmp(ent->d_name,".")==0 || strcmp(ent->d_name,"..")==0){
				continue;
			}
			string file = refserv + "/" + ent->d_name;
			struct stat st;
			if (lstat(file.c_str(),&st) == -1){
				fprintf (stderr,"Can't lstat file %s (%s)\n"
					,file.c_str(),strerror(errno));
				ret = -1;
			}else{
				string newfile = newserv + "/" + ent->d_name;
				if (S_ISDIR(st.st_mode)){
					if (vbuild_mkdir (newfile.c_str(),st.st_mode)==-1){
						fprintf (stderr,"Can't mkdir %s (%s)\n"
							,newfile.c_str(),strerror(errno));
						ret = -1;
					}else{
						stats.nbdir++;
						if (vbuild_chown(newfile.c_str(),st.st_uid,st.st_gid)==-1){
							fprintf (stderr,"Can't chown %s (%s)\n"
								,newfile.c_str(),strerror(errno));
							ret = -1;
						}
						ret |= vbuild_copy (file,newfile
							,logical_dir + "/" + ent->d_name,files);
					}
				}else if (S_ISLNK(st.st_mode)){
					char path[PATH_MAX];
					int len = readlink(file.c_str(),path,sizeof(path)-1);
					if (len < 0){
						fprintf (stderr,"Can't readlink %s (%s)\n"
							,file.c_str(),strerror(errno));
						ret = -1;
					}else{
						path[len] = '\0';
						stats.nbsymlink++;
						if (vbuild_symlink (path,newfile.c_str())==-1){
							fprintf (stderr,"Can't symlink %s to %s (%s)\n",
								newfile.c_str(),path,strerror(errno));
						}
					}
				}else if (S_ISBLK(st.st_mode)
					|| S_ISCHR(st.st_mode)
					|| S_ISFIFO(st.st_mode)){
					stats.nbspc++;
					if (vbuild_mknod (newfile.c_str(),st.st_mode,st.st_rdev)==-1){
						fprintf (stderr,"Can't mknod %s (%s)\n"
							,newfile.c_str(),strerror(errno));
						ret = -1;
					}
				}else if (S_ISSOCK(st.st_mode)){
					// Do nothing
				}else{
					// Ok, this is a file. We either copy it or do a link
					string logical_file = logical_dir + "/" + ent->d_name;
					if (files.find (logical_file)==files.end()){
						if (debug > 1) printf ("Copying file %s\n",file.c_str());
						if (vbuild_file_copy (file.c_str(),newfile.c_str(),st)==-1){
							fprintf (stderr,"Can't copy %s to %s (%s)\n",
								file.c_str(),newfile.c_str(),strerror(errno));
							ret = -1;
						}else{
							stats.size_copy += st.st_size;
							stats.nbcopy++;
						}
					}else{
						if (debug > 2) printf ("Linking file %s\n",file.c_str());
						setext2flag (file.c_str(),false);
						stats.nblink++;
						if (vbuild_link (file.c_str(),newfile.c_str())==-1){
							fprintf (stderr,"Can't link %s to %s (%s)\n",
								file.c_str(),newfile.c_str(),strerror(errno));
							ret = -1;
						}
						setext2flag (file.c_str(),true);
					}
				}
			}
		}
		closedir(dir);
	}
	return ret;
}

int main (int argc, char *argv[])
{
	int ret = -1;
	bool statistics = false;
	int i;
	for (i=1; i<argc; i++){
		const char *arg = argv[i];
		//const char *opt = argv[i+1];
		if (strcmp(arg,"--test")==0){
			testmode = true;
		}else if (strcmp(arg,"--debug")==0){
			debug++;
		}else if (strcmp(arg,"--stats")==0){
			statistics = true;
		}else if (strcmp(arg,"--noflags")==0){
			ext2flags = 0;
		}else if (strcmp(arg,"--immutable")==0){
			ext2flags |= EXT2_IMMUTABLE_FILE_FL;
		}else if (strcmp(arg,"--immutable-mayunlink")==0){
			ext2flags |= EXT2_IMMUTABLE_LINK_FL;
		}else{
			break;
		}
	}
	if (i!=argc-2){
		usage();
	}else{
		string refserv = argv[i++];
		string newserv = argv[i];
		list<PACKAGE> packages;
		// Load the files which are not configuration files from
		// the packages
		vunify_loadallpkg (refserv,packages);
		set<string> files;
		for (list<PACKAGE>::iterator it=packages.begin(); it!=packages.end(); it++){
			(*it).loadfiles(refserv,files);
		}
		// Now, we do a recursive copy of refserv into newserv
		umask (0);
		mkdir (newserv.c_str(),0755);
		stats.nbdir = stats.nblink = stats.nbcopy = stats.nbsymlink = 0;
		stats.nbspc = 0;
		stats.size_copy = 0;
		ret = vbuild_copy (refserv,newserv,"",files);
		if (statistics){
			printf ("Directory created: %d\n",stats.nbdir);
			printf ("Files copied     : %d\n",stats.nbcopy);
			printf ("Bytes copied     : %ld\n",stats.size_copy);
			printf ("Files linked     : %d\n",stats.nblink);
			printf ("Files symlinked  : %d\n",stats.nbsymlink);
			printf ("Special files    : %d\n",stats.nbspc);
		}
	}
	return ret;
}


