#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <syscall.h>
#include <linux/capability.h>
#include <asm/unistd.h>

#ifndef __NR_new_s_context
	#define __NR_new_s_context	222
#endif

_syscall3(int, new_s_context, int, newctx, int, remove_cap, int, flags);

static void usage()
{
	fprintf (stderr,"chcontext version %s\n",VERSION);
	fprintf (stderr
		,"chcontext [ options ] command arguments ...\n"
		 "\n"
		 "chcontext allocate a new security context and executes\n"
		 "a command in that context.\n"
		 "By default, a new/unused context is allocated\n"
		 "\n"

		 "--cap CAP_NAME\n"
		 "\tRemove a capability from the command. This option may be\n"
		 "\trepeated several time.\n"
		 "\tSee /usr/include/linux/capability.h\n"

		 "--ctx num\n"
		 "\tSelect the context. On root in context 0 is allowed to\n"
		 "\tselect a specific context.\n"
		 "\tContext number 1 is special. It can see all processes\n"
		 "\tin any contexts, but can't kill them though.\n"

		 "--disconnect\n"
		 "\tStart the command in background and make the process\n"
		 "\ta child of process 1.\n"

		 "--domainname new_domainname\n"
		 "\tSet the domainname (NIS) in the new security context.\n"

		 "--hostname new_hostname\n"
		 "\tSet the hostname in the new security context\n"
		 "\tThis is need because if you create a less privileged\n"
		 "\tsecurity context, it may be unable to change its hostname\n"

		 "--lock\n"
		 "\tLock in the new context. The new process is trapped\n"
		 "\tand can't use chcontext anymore.\n"

		 "--secure\n"
		 "\tRemove all the capabilities to make a virtual server trustable\n"

		 "--silent\n"
		 "\tDo not print the allocated context number.\n"
		 "\n"
		 "Information about context is found in /proc/self/status\n");
}


int main (int argc, char *argv[])
{
	int ret = -1;
	int i;
	int ctx = -1;
	int disconnect = 0;
	int silent = 0;
	int flags = 0;
	unsigned remove_cap = 0;
	unsigned long secure = (1<<CAP_LINUX_IMMUTABLE)
		|(1<<CAP_NET_BROADCAST)
		|(1<<CAP_NET_ADMIN)
		|(1<<CAP_NET_RAW)
		|(1<<CAP_IPC_LOCK)
		|(1<<CAP_IPC_OWNER)
		|(1<<CAP_SYS_MODULE)
		|(1<<CAP_SYS_RAWIO)
		|(1<<CAP_SYS_PACCT)
		|(1<<CAP_SYS_ADMIN)
		|(1<<CAP_SYS_BOOT)
		|(1<<CAP_SYS_NICE)
		|(1<<CAP_SYS_RESOURCE)
		|(1<<CAP_SYS_TIME)
		|(1<<CAP_MKNOD);
	const char *hostname=NULL, *domainname=NULL;

	for (i=1; i<argc; i++){
		const char *arg = argv[i];
		const char *opt = argv[i+1];
		if (strcmp(arg,"--ctx")==0){
			ctx = atoi(opt);
			i++;
		}else if (strcmp(arg,"--disconnect")==0){
			disconnect = 1;
		}else if (strcmp(arg,"--silent")==0){
			silent = 1;
		}else if (strcmp(arg,"--lock")==0){
			flags |= 1;
		}else if (strcmp(arg,"--cap")==0){
			static struct {
				const char *option;
				int bit;
			}tbcap[]={
				{"CAP_LINUX_IMMUTABLE",CAP_LINUX_IMMUTABLE},
				{"CAP_NET_BIND_SERVICE",CAP_NET_BIND_SERVICE},
				{"CAP_NET_BROADCAST",CAP_NET_BROADCAST},
				{"CAP_NET_ADMIN",	CAP_NET_ADMIN},
				{"CAP_NET_RAW",	CAP_NET_RAW},
				{"CAP_IPC_LOCK",	CAP_IPC_LOCK},
				{"CAP_IPC_OWNER",	CAP_IPC_OWNER},
				{"CAP_SYS_MODULE",CAP_SYS_MODULE},
				{"CAP_SYS_RAWIO",	CAP_SYS_RAWIO},
				{"CAP_SYS_PACCT",	CAP_SYS_PACCT},
				{"CAP_SYS_ADMIN",	CAP_SYS_ADMIN},
				{"CAP_SYS_BOOT",	CAP_SYS_BOOT},
				{"CAP_SYS_NICE",	CAP_SYS_NICE},
				{"CAP_SYS_RESOURCE",CAP_SYS_RESOURCE},
				{"CAP_SYS_TIME",	CAP_SYS_TIME},
				{"CAP_MKNOD",		CAP_MKNOD},
				{NULL,0}
			};
			int j;
			for (j=0; tbcap[j].option != NULL; j++){
				if (strcasecmp(tbcap[j].option,opt)==0){
					remove_cap |= (1<<tbcap[j].bit);
					break;
				}
			}
			if (tbcap[j].option == NULL){
				fprintf (stderr,"Unknown capability %s\n",opt);
			}
			i++;
		}else if (strcmp(arg,"--secure")==0){
			remove_cap |= secure;
		}else if (strcmp(arg,"--hostname")==0){
			hostname = opt;
			i++;
		}else if (strcmp(arg,"--domainname")==0){
			domainname = opt;
			i++;
		}else{
			break;
		}
	}
	if (i == argc){
		usage();
	}else if (argv[i][0] == '-'){
		usage();
	}else{
		int newctx = new_s_context(ctx,0,flags);
		if (newctx != -1){
			if (hostname != NULL){
				sethostname (hostname,strlen(hostname));
				if (!silent){
					printf ("Host name is now %s\n",hostname);
				}
			}
			if (domainname != NULL){
				setdomainname (domainname,strlen(domainname));
				if (!silent){
					printf ("Domain name is now %s\n",domainname);
				}
			}
			if (remove_cap != 0) new_s_context (-2,remove_cap,0);
			if (!silent) printf ("New security context is %d\n",newctx);
			if (disconnect == 0 || fork()==0){
				execvp (argv[i],argv+i);
			}
		}else{
			perror ("Can't set the new security context\n");
		}
	}
	return ret;
}

