#pragma implementation
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include <utime.h>
#include "vutil.h"
#include <sys/ioctl.h>
#include <linux/ext2_fs.h>

bool testmode;
int debug;

int file_copy (const char *src, const char *dst, struct stat &st)
{
	int ret = -1;
	FILE *fin = fopen (src,"r");
	if (fin == NULL){
		fprintf (stderr,"Can't open file %s (%s)\n",src,strerror(errno));
	}else{
		FILE *fout = fopen (dst,"w");
		if (fout == NULL){
			fprintf (stderr,"Can't open file %s (%s)\n",dst,strerror(errno));
		}else{
			char buf[8192];
			int len;
			while ((len=fread(buf,1,sizeof(buf),fin))>0){
				fwrite (buf,1,len,fout);
			}
			fflush (fout);
			ret = 0;
			if (fchown (fileno(fout),st.st_uid,st.st_gid)==-1){
				fprintf (stderr,"Can't chown file %s (%s)\n"
					,dst,strerror(errno));
				ret = -1;
			}else if (fchmod (fileno(fout),st.st_mode)==-1){
				fprintf (stderr,"Can't chmod file %s (%s)\n"
					,dst,strerror(errno));
				ret = -1;
			}
			fclose(fout);
			struct utimbuf timbuf;
			timbuf.modtime = st.st_mtime;
			timbuf.actime = st.st_atime;
			if (utime (dst,&timbuf)==-1){
				fprintf (stderr,"Can't set time stamp on file %s (%s)\n"
					,dst,strerror(errno));
			}
		}
		fclose (fin);
	}
	return ret;
}

/*
	Set the immutable flag on a file
*/
int setext2flag (const char *fname, bool set, int ext2flags)
{
	int ret = -1;
	if (testmode){
		ret = 0;
	}else{
		int fd = open (fname,O_RDONLY);
		if (fd == -1){
			fprintf (stderr,"Can't open file %s (%s)\n",fname 
				,strerror(errno));
		}else{
			int flags = set ? ext2flags : 0;
			ret = ioctl (fd,EXT2_IOC_SETFLAGS,&flags);
			close (fd);
			if (ret == -1){
				fprintf (stderr,"Can't %s immutable flag on file %s (^s)\n"
					,(set ? "set" : "unset")
					,fname
					,strerror(errno));
			}
		}
	}
	return ret;
}

int vbuild_mkdir (const char *path, mode_t mode)
{
	int ret = -1;
	if (testmode){
		printf ("mkdir %s; chmod %o %s\n",path,mode,path);
		ret = 0;
	}else{
		ret = mkdir (path,mode);
	}
	return ret;
}

int vbuild_mknod(const char *path, mode_t mode, dev_t dev)
{
	int ret = -1;
	if (testmode){
		printf ("mknod %s %o %02x:%02x\n",path,mode,major(dev),minor(dev));
		ret = 0;
	}else{
		ret = mknod (path,mode,dev);
	}
	return ret;
}
int vbuild_symlink(const char *src, const char *dst)
{
	int ret = -1;
	if (testmode){
		printf ("ln -s %s %s\n",src,dst);
		ret = 0;
	}else{
		ret = symlink (src,dst);
	}
	return ret;
}

int vbuild_link(const char *src, const char *dst)
{
	int ret = -1;
	if (testmode){
		printf ("ln %s %s\n",src,dst);
		ret = 0;
	}else{
		ret = link (src,dst);
	}
	return ret;
}

int vbuild_unlink(const char *path)
{
	int ret = -1;
	if (testmode){
		printf ("unlink %s\n",path);
		ret = 0;
	}else{
		ret = unlink (path);
	}
	return ret;
}

int vbuild_chown(const char *path, uid_t uid, gid_t gid)
{
	int ret = -1;
	if (testmode){
		printf ("chown %d.%d %s\n",uid,gid,path);
		ret = 0;
	}else{
		ret = chown (path,uid,gid);
	}
	return ret;
}

int vbuild_file_copy(
	const char *src,
	const char *dst,
	struct stat &st)
{
	int ret = -1;
	if (testmode){
		printf ("cp -a %s %s\n",src,dst);
		ret = 0;
	}else{
		ret = file_copy (src,dst,st);
	}
	return ret;
}

/*
	Load the list of all packages in a vserver
*/
void vutil_loadallpkg (string &refserver, list<PACKAGE> &packages)
{
	string cmd = "/usr/sbin/chroot " + refserver + " /bin/rpm -qa"
		+ " --queryformat \"%{name}=%{version}-%{release}\\n\"";
	// cout << "command " << cmd << endl;
	#if 0
		cmd = "| " + cmd;
		ipfstream oo (cmd.c_str());
		copy (istream_iterator<string>(oo),istream_iterator<string>()
			,inserter(packages,packages.begin()));
		int status;
		wait (&status);
	#else
		FILE *fin = popen (cmd.c_str(),"r");
		if (fin == NULL){
			fprintf (stderr,"Can't execute %s (%s)\n",cmd.c_str()
				,strerror(errno));
		}else{
			char line[1000];
			while (fgets(line,sizeof(line)-1,fin)!=NULL){
				int last = strlen(line)-1;
				if (last >= 0 && line[last] == '\n') line[last] = '\0';
				packages.push_back (PACKAGE(line));
			}
			pclose (fin);
		}
	#endif
}

int vutil_lstat (string path, struct stat &st)
{
	int ret = 0;
	if (lstat(path.c_str(),&st) == -1){
		fprintf (stderr,"Can't lstat file %s (%s)\n"
			,path.c_str(),strerror(errno));
		ret = -1;
	}
	return ret;
}

