/*
	Prints the specs of a network device in shell like form

	ADDR=
	NETMASK=
	BCAST=
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <net/if.h>

static void usage()
{
	fprintf (stderr,"ifspec version %s\n",VERSION);
	fprintf (stderr
		,"ifspec network-device [ ipaddr netmask broadcast ]\n"
		 "prints device specification in a shell usable way\n");
	exit (-1);
}

static int ifconfig_ioctl(
	int fd,
	const char *ifname,
	int cmd,
	struct ifreq *ifr)
{
	strcpy(ifr->ifr_name, ifname);
	return ioctl(fd, cmd,ifr);
}



/*
	Fetch the IP number of an interface from the kernel.
	Assume the device is already available in the kernel
	Return -1 if any error.
*/
int ifconfig_print (
	const char *ifname,
	const char *addrstr,
	const char *maskstr,
	const char *bcaststr)
{
	int ret = -1;
	int skfd = socket(AF_INET, SOCK_DGRAM, 0);
	if (skfd != -1){
		struct ifreq ifr;
		if (addrstr != NULL && addrstr[0] != '\0'){
			printf ("ADDR=%s\n",addrstr);
		}else if (ifconfig_ioctl(skfd,ifname,SIOCGIFADDR, &ifr) >= 0){
			struct sockaddr_in *sin = (struct sockaddr_in*)&ifr.ifr_addr;
			unsigned long addr = ntohl(sin->sin_addr.s_addr);
			printf ("ADDR=%lu.%lu.%lu.%lu\n"
				,(addr>>24)&0xff
				,(addr>>16)&0xff
				,(addr>>8)&0xff
				,addr&0xff);
			ret = 0;
		}
		if (maskstr != NULL && maskstr[0] != '\0'){
			printf ("NETMASK=%s\n",maskstr);
		}else		if (ifconfig_ioctl(skfd,ifname,SIOCGIFNETMASK, &ifr) >= 0){
			struct sockaddr_in *sin = (struct sockaddr_in*)&ifr.ifr_addr;
			unsigned long addr = ntohl(sin->sin_addr.s_addr);
			printf ("NETMASK=%lu.%lu.%lu.%lu\n"
				,(addr>>24)&0xff
				,(addr>>16)&0xff
				,(addr>>8)&0xff
				,addr&0xff);
			ret = 0;
		}
		if (bcaststr != NULL && bcaststr[0] != '\0'){
			printf ("BCAST=%s\n",bcaststr);
		}else if (ifconfig_ioctl(skfd,ifname,SIOCGIFBRDADDR, &ifr) >= 0){
			struct sockaddr_in *sin = (struct sockaddr_in*)&ifr.ifr_addr;
			unsigned long addr = ntohl(sin->sin_addr.s_addr);
			printf ("BCAST=%lu.%lu.%lu.%lu\n"
				,(addr>>24)&0xff
				,(addr>>16)&0xff
				,(addr>>8)&0xff
				,addr&0xff);
			ret = 0;
		}
		close (skfd);
	}
	return ret;
}


int main (int argc, char *argv[])
{
	int ret = -1;
	if (argc < 2){
		usage();
	}else{
		const char *addrstr = argc >= 3 ? argv[2] : NULL;
		const char *maskstr = argc >= 4 ? argv[3] : NULL;
		const char *bcaststr = argc >= 5 ? argv[4] : NULL;
		ret = ifconfig_print (argv[1],addrstr,maskstr,bcaststr);
	}
	return ret;
}



