/*
 *	@(#)Viewer.java 1.16 01/01/11 07:24:46
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.universe;


import java.awt.BorderLayout;
import java.awt.event.*;
import java.awt.Frame;
import java.awt.Panel;
import java.awt.GraphicsConfiguration;
import java.net.URL;
import javax.media.j3d.*;
import javax.vecmath.*;
import com.sun.j3d.audioengines.javasound.*;

/**
 * The Viewer class holds all the information that describes the physical
 * and virtual "presence" in the Java 3D universe.  The Viewer object
 * consists of:
 * <UL>
 * <LI>Physical Objects</LI>
 *  <UL>
 *   <LI>Canavs3D - used to render with.</LI>
 *   <LI>PhysicalEnvironment - holds characteristics of the hardware platform
 *    being used to render on.</LI>
 *   <LI>PhysicalBody -  holds the physical characteristics and personal
 *    preferences of the person who will be viewing the Java 3D universe.</LI>
 *  </UL>
 * <LI>Virtual Objects</LI>
 *  <UL>
 *   <LI>View - the Java 3D View object.</LI>
 *   <LI>ViewerAvatar - the geometry that is used by Java 3D to represent the
 *    person viewing the Java 3D universe.</LI>
 *  </UL>
 * </UL>
 *
 * @see Canvas3D
 * @see PhysicalEnvironment
 * @see PhysicalBody
 * @see View
 * @see ViewerAvatar
 */
public class Viewer {

    protected        Canvas3D            canvas;
    protected        View                view;
    protected        ViewerAvatar	 avatar = null;
    protected static PhysicalBody        physicalBody = null;
    protected static PhysicalEnvironment physicalEnvironment = null;

    // If no Canvas3D object is provided during construction
    // the Viewer object will create one and place it within
    // a Panel within a Frame.
    protected        Frame               j3dFrame = null;
    protected        Panel               j3dPanel = null;

    // Reference to the ViewingPlatform object
    ViewingPlatform      viewingPlatform = null;

    /**
     * Creates a default viewer object. The default values are used to create
     * the PhysicalBody, PhysicalEnvironment.  A single RGB, double buffered
     * and depth buffered Canvas3D object is created.  The View is created
     * with a front clip distance of 0.1f and a back clip distance of 10.0f.
     */
    public Viewer() {
        // Call main constructor with default values.
        this(null, null);
    }

    /**
     * Creates a default viewer object. The default values are used to create
     * the PhysicalBody, PhysicalEnvironment.  The View is created
     * with a front clip distance of 0.1f and a back clip distance of 10.0f.
     *
     * @param userCanvas The Canavs3D object to render to.  If this object is
     *  null a single RGB, double buffered and depth buffered Canvas3D
     *  object is created.
     */
    public Viewer(Canvas3D userCanvas) {
        // Call main constructor.
        this(userCanvas, null);
    }

    /**
     * Creates a default viewer object. The default values are used to create
     * the PhysicalEnvironment.  A single RGB, double buffered
     * and depth buffered Canvas3D object is created.  The View is created
     * with a front clip distance of 0.1f and a back clip distance of 10.0f.
     *
     * @param userConfig The URL of the user configuration file used to
     *  initialize the PhysicalBody object.  If null, the default PhysicalBody
     *  values are used.
     */
    public Viewer(URL userConfig) {
        // Call main constructor.
        this(null, userConfig);
    }

    /**
     * Creates a default viewer object. The default values are used to create
     * the PhysicalEnvironment.  The View is created
     * with a front clip distance of 0.1f and a back clip distance of 10.0f.
     *
     * @param userCanvas The Canavs3D object to render to.  If this object is
     *  null a single RGB, double buffered and depth buffered Canvas3D
     *  object is created.
     * @param userConfig The URL of the user configuration file used to
     *  initialize the PhysicalBody object.  If null, the default PhysicalBody
     *  values are used.
     */
    public Viewer(Canvas3D userCanvas, URL userConfig) {

        // Only one PhysicalBody per Universe.
        if (physicalBody == null) {
            physicalBody = new PhysicalBody();
            if (userConfig != null) {
                // TODO:open config file and set PhysicalBody attr's here
            }
        }

        // Only one PhysicalEnvironment per Universe.
        if (physicalEnvironment == null) {
            physicalEnvironment = new PhysicalEnvironment();
            // TODO:open system config file and set PhysicalEnvironment here
        }

        // Create Canav3D object if none was passed in.
        if (userCanvas == null) {
            // Canvas3D should be array, determined by GraphicsEnvironment
            // and config files
	    GraphicsConfiguration config =
		SimpleUniverse.getPreferredConfiguration();
	    canvas = new Canvas3D(config);
	    
            // If no canvas was provided need to also create the Frame
            // and Panel for the Canvas3D.
            j3dFrame = new Frame();
            j3dFrame.setLayout(new BorderLayout());
            // TODO: want config option to allow for full screen.
            j3dFrame.setSize(256, 256);
	    
            // Put the Canvas3D into a Panel.
            j3dPanel = new Panel();
            j3dPanel.setLayout(new BorderLayout());
            j3dPanel.add("Center", canvas);
	    
            j3dFrame.add("Center", j3dPanel);
            j3dFrame.show();

            // Add windowListener to detect window close event.
            j3dFrame.addWindowListener(new WindowAdapter()
            {
                public void windowClosing(WindowEvent winEvent)
                {
                    System.exit(0);
                }
            });
        }
        else
            canvas = userCanvas;

        // Create a View and attach the Canvas3D and the physical
        // body and environment to the view.
        view = new View();

        view.addCanvas3D(canvas);
        view.setPhysicalBody(physicalBody);
        view.setPhysicalEnvironment(physicalEnvironment);
    }

    /**
     * Returns the View object associated with the Viewer object.
     *
     * @return The View object of this Viewer.
     */
    public View getView() {
        return view;
    }

    /**
     * Set the ViewingPlatform object used by this Viewer.
     *
     * @param platform The ViewingPlatform object to set for this
     *  Viewer object.  Use null to unset the current value and
     *  not assign assign a new ViewingPlatform object.
     */
    public void setViewingPlatform(ViewingPlatform platform) {
	if (viewingPlatform != null) {
	    viewingPlatform.removeViewer(this);
	}

        viewingPlatform = platform;

        if (platform != null) {
            view.attachViewPlatform(platform.getViewPlatform());
            platform.addViewer(this);

            if (avatar != null)
                viewingPlatform.setAvatar(this, avatar);
        }
        else
            view.attachViewPlatform(null);
    }
    /**
     * Get the ViewingPlatform object used by this Viewer.
     *
     * @return The ViewingPlatform object used by this
     *  Viewer object. 
     */
    public ViewingPlatform getViewingPlatform() {
	return viewingPlatform;
    }

    /**
     * Sets the geometry to be associated with the viewer's avatar.  The
     * avatar is the geometry used to represent the viewer in the virtual
     * world.
     *
     * @param avatar The geometry to associate with this Viewer object.
     *  Passing in null will cause any geometry associated with the Viewer
     *  to be removed from the scen graph.
     */
    public void setAvatar(ViewerAvatar avatar) {
        // Just return if trying to set the same ViewerAvatar object.
        if (this.avatar == avatar)
            return;

        this.avatar = avatar;
        if (viewingPlatform != null)
            viewingPlatform.setAvatar(this, this.avatar);
    }

    /**
     * Gets the geometry associated with the viewer's avatar.  The
     * avatar is the geometry used to represent the viewer in the virtual
     * world.
     *
     * @return The root of the scene graph that is used to represent the
     *  viewer's avatar.
     */
    public ViewerAvatar getAvatar() {
        return avatar;
    }

    /**
     * Returns the PhysicalBody object associated with the Viewer object.
     *
     * @return A reference to the PhysicalBody object.
     */
    public PhysicalBody getPhysicalBody() {
        return physicalBody;
    }

    /**
     * Returns the PhysicalEnvironment object associated with the Viewer
     * object.
     *
     * @return A reference to the PhysicalEnvironment object.
     */
    public PhysicalEnvironment getPhysicalEnvironment() {
        return physicalEnvironment;
    }

    /**
     * Returns the canvas associated with this Viewer object.
     */
    public Canvas3D getCanvases() {
        // TODO:should return array
        return canvas;
    }

    /**
     * Returns the Frame created by this Viewer object.  If a Viewer
     * is constructed without a Canvas3D object then the Viewer object
     * will create a Canva3D object, a Panel to place the Canvas3D
     * object into and a Frame to place the Panel in.
     *
     * @return The Frame created by this object.  If a Frame was
     *  not created by the Viewer object, null is returned.
     */
    public Frame getFrame() {
        return j3dFrame;
    }

    /**
     * Returns the Panel created by this Viewer object.  If a Viewer
     * is constructed without a Canvas3D object then the Viewer object
     * will create a Canva3D object, a Panel to place the Canvas3D
     * object into and a Frame to place the Panel in.
     *
     * @return The Panel created by this object.  If a Panel was
     *  not created by the Viewer object, null is returned.
     */
    public Panel getPanel() {
        return j3dPanel;
    }

    /**
     * Used to create and initialize the AudioDevice, used for sound rendering.
     *
     * @param BranchGroup The BranchGroup to attach to this Universe's Locale.
     * @return reference to created AudioDevice, or null if error occurs.
     */
    public AudioDevice createAudioDevice() {
        // TODO:open system config file and construct proper AudioDevice here
        //      For now, JavaSoundMixer is assumed to be the desired AudioDevice
        //      implementation.
        if (physicalEnvironment != null) {
            // TODO: what about others? Uninitialize?
            JavaSoundMixer javaSoundMixer =
              new JavaSoundMixer(physicalEnvironment);
            javaSoundMixer.initialize();
            return javaSoundMixer; // return reference to created AudioDevice
        }
        else
            return null;
    }

}
