/*
 *	@(#)SimpleUniverse.java 1.18 01/01/11 07:24:45
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.universe;

import java.awt.GraphicsEnvironment;
import java.awt.GraphicsConfiguration;
import java.net.URL;

import javax.media.j3d.*;
import javax.vecmath.*;

/**
 * This class sets up a minimal user environment to quickly and easily
 * get a Java 3D program up and running.  This utility class creates
 * all the necessary objects on the "view" side of the scene graph.
 * Specifically, this class creates a locale, a single ViewingPlatform, 
 * and a Viewer object (both with their default values).
 * Many basic Java 3D applications
 * will find that SimpleUniverse provides all necessary functionality
 * needed by their applications. More sophisticated applications
 * may find that they need more control in order to get extra functionality
 * and will not be able to use this class.
 * 
 * @see Viewer
 * @see ViewingPlatform
 */
public class SimpleUniverse extends VirtualUniverse {

    /**
     * Locale reference needed to create the "view" portion
     * of the scene graph.
     */
    protected Locale          locale;

    /**
     * Viewer reference needed to create the "view" portion
     * of the scene graph.
     */
    protected Viewer          viewer;

    /**
     * Creates a locale, a single ViewingPlatform, and
     * and a Viewer object (both with their default values).
     *
     * @see Locale
     * @see Viewer
     * @see ViewingPlatform
     */
    public SimpleUniverse() {
        // call main constructor with default values.
        this(null, 1, null, null);
    }

    /**
     * Creates a locale, a single ViewingPlatform, and a Viewer object
     * (with default values).  The ViewingPlatform is created with the
     * specified number of TransformGroups.
     *
     * @param numTransforms The number of transforms to be in the
     * MultiTransformGroup object.
     *
     * @see Locale
     * @see Viewer
     * @see ViewingPlatform
     *
     * @since Java 3D 1.2.1
     */
    public SimpleUniverse(int numTransforms) {
	// call main constructor with default values except numTransforms
	this(null, numTransforms, null, null);
    }

    /**
     * Creates a locale, a single ViewingPlatform (with default values), and
     * and a Viewer object.  The Viewer object uses default values for
     * everything but the canvas.
     *
     * @param canvas The canvas to associate with the Viewer object.  Passing
     *  in null will cause this parameter to be ignored and a canvas to be
     *  created by the utility.
     *
     * @see Locale
     * @see Viewer
     * @see ViewingPlatform
     */
    public SimpleUniverse(Canvas3D canvas) {
        // call main constructor with default values for everything but
        // the canvas parameter.
        this(null, 1, canvas, null);
    }

    /**
     * Creates a locale, a single ViewingPlatform, and a Viewer object
     * The Viewer object uses default values for everything but the canvas.
     * The ViewingPlatform is created with the specified number of
     * TransformGroups.
     *
     * @param canvas The canvas to associate with the Viewer object.  Passing
     * in null will cause this parameter to be ignored and a canvas to be
     * created by the utility.
     * @param numTransforms The number of transforms to be in the
     * MultiTransformGroup object.
     *
     * @see Locale
     * @see Viewer
     * @see ViewingPlatform
     * @see MultiTransformGroup
     *
     * @since Java 3D 1.2.1
     */
    public SimpleUniverse(Canvas3D canvas, int numTransforms) {
	// call main constructor with default values except canvas
	// and numTransforms
	this(null, numTransforms, canvas, null);
    }

    /**
     * Creates the "view" side of the scene graph.  The passed in parameters
     * override the default values where appropriate.
     *
     * @param origin The origin used to set the origin of the Locale object.
     *  If this object is null, then 0.0 is used.
     * @param numTransforms The number of transforms to be in the
     *  MultiTransformGroup object.
     * @param canvas The canvas to draw into.  If this is null, it is
     *  ignored and a canvas will be created by the utility.
     * @param userConfig The URL to the user's configuration file, used
     *  by the Viewer object.  Passing in null causes the default values
     *  to be used.
     *
     * @see Locale
     * @see Viewer
     * @see ViewingPlatform
     * @see MultiTransformGroup
     */
    public SimpleUniverse(HiResCoord origin, int numTransforms,
      Canvas3D canvas, URL userConfig) {
	ViewingPlatform vwp;

        // See if the HiResCoord was given
        if (origin == null)
            locale = new Locale(this);
        else
            locale = new Locale(this, origin);

        // Create the ViewingPlatform and Viewer objects, passing
        // down the appropriate parameters.
	vwp = new ViewingPlatform(numTransforms);
        viewer = new Viewer(canvas, userConfig);
        viewer.setViewingPlatform(vwp);

        // Add the ViewingPlatform to the locale - the scene
        // graph is now "live".
        locale.addBranchGraph(vwp);
    }


    /**
     * Creates the "view" side of the scene graph.  The passed in parameters
     * override the default values where appropriate.
     *
     * @param viewingPlatform The viewingPlatform to use to create
     *  the "view" side of the scene graph.
     * @param viewer The viewer object to use to create
     *  the "view" side of the scene graph.
     */
    public SimpleUniverse(ViewingPlatform viewingPlatform, Viewer viewer) {
        locale = new Locale(this);

        // Assign object references.
        this.viewer = viewer;

        // Add the ViewingPlatform to the Viewer object.
        this.viewer.setViewingPlatform(viewingPlatform);

        // Add the ViewingPlatform to the locale - the scene
        // graph is now "live".
        locale.addBranchGraph(viewingPlatform);
    }

    /**
     * Returns the Locale object associated with this scene graph.
     *
     * @return The Locale object used in the construction of this scene
     *  graph.
     */
    public Locale getLocale() {
        return locale;
    }

    /**
     * Returns the Viewer object associated with this scene graph.  
     * SimpleUniverse creates a single Viewer object for use in the
     * scene graph.
     * 
     * @return The Viewer object associated with this scene graph.
     */
    public Viewer getViewer() {
        return viewer;
    }

    /**
     * Returns the ViewingPlatform object associated with this scene graph.
     *
     * @return The ViewingPlatform object of this scene graph.
     */
    public ViewingPlatform getViewingPlatform() {
        return viewer.getViewingPlatform();
    }

    /**
     * Returns the Canvas3D object associated with this Java 3D Universe.
     *
     * @return A reference to the Canvas3D object associated with the
     *  Viewer object.  This method is equivalent to calling getCanvas(0).
     *
     * @see Viewer
     */
    public Canvas3D getCanvas() {
        return getCanvas(0);
    }

    /**
     * Returns the Canvas3D object at the specified index associated with
     * this Java 3D Universe.
     *
     * @param canvasNum The index of the Canvas3D object to retrieve.
     *  If there is no Canvas3D object for the given index, null is returned.
     *
     * @return A reference to the Canvas3D object associated with the
     */
    public Canvas3D getCanvas(int canvasNum) {
        return viewer.getCanvases();
    }

    /**
     * Used to add Nodes to the geometry side (as opposed to the view side)
     * of the scene graph.  This is a short cut to getting the Locale object
     * and calling that object's addBranchGraph() method.
     *
     * @param BranchGroup The BranchGroup to attach to this Universe's Locale.
     */
    public void addBranchGraph(BranchGroup bg) {
        locale.addBranchGraph(bg);
    }

    /**
     * Finds the preferred <code>GraphicsConfiguration</code> object
     * for the system.  This object can then be used to create the
     * Canvas3D objet for this system.
     *
     * @return The best <code>GraphicsConfiguration</code> object for
     *  the system.
     */
    public static GraphicsConfiguration getPreferredConfiguration() {
        GraphicsConfigTemplate3D template = new GraphicsConfigTemplate3D();
        String stereo;

        // Check if the user has set the Java 3D stereo option.
        // Getting the system properties causes appletviewer to fail with a
        //  security exception without a try/catch.

        stereo = (String) java.security.AccessController.doPrivileged(
           new java.security.PrivilegedAction() {
           public Object run() {
               return System.getProperty("j3d.stereo");
           }
        });

        // update template based on properties.
        if (stereo != null) {
            if (stereo.equals("REQUIRED"))
                template.setStereo(template.REQUIRED);
            else if (stereo.equals("PREFERRED"))
                template.setStereo(template.PREFERRED);
        }

        // Return the GraphicsConfiguration that best fits our needs.
        return GraphicsEnvironment.getLocalGraphicsEnvironment().
                getDefaultScreenDevice().getBestConfiguration(template);
    }
}
