/*
 *	@(#)Sphere.java 1.40 01/01/11 07:23:36
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.geometry;

import com.sun.j3d.utils.geometry.*;
import java.io.*;
import java.util.*;
import javax.media.j3d.*;
import javax.vecmath.*;
import java.math.*;

/**
 * Sphere is a geometry primitive created with a given radius and resolution.
 * It is centered at the origin.
 * <p>
 * When a texture is applied to a Sphere, it is mapped CCW from the back
 * of the sphere.
 */

public class Sphere extends Primitive {

  /**
   * Sphere shape identifier, used by <code>getShape</code>.
   *
   * @see Sphere#getShape
   */
  public static final int BODY = 0;

  static final int MID_REZ_DIV = 15;
  float  radius;
  int    divisions;

  /**
   *   Constructs a Sphere of a given radius. Normals are generated
   *   by default, texture coordinates are not. The resolution defaults to  
   *   15 divisions along sphere's axes. Appearance defaults to white.
   *   @param radius Radius
   */
  public Sphere (float radius) {
    this(radius,  GENERATE_NORMALS, MID_REZ_DIV);
  }

  /**  
   *   Constructs a default Sphere of radius of 1.0.
   *   Resolution defaults to 15 divisions. Appearance defaults to white.
   */
  public Sphere() {
    this(1.0f, GENERATE_NORMALS, MID_REZ_DIV);
  }

  /**  
   *   Constructs a Sphere of a given radius and appearance.
   *   @param radius Radius
   *   @param appearance Appearance
   */

  public Sphere (float radius, Appearance ap) {
    this(radius, GENERATE_NORMALS, MID_REZ_DIV, ap);
  }

  /**  
   *   Constructs a Sphere of a given radius and appearance with
   *   additional parameters specified by the Primitive flags.
   *   @param radius Radius
   *   @param flags 
   *   @param ap appearance
   */
  public Sphere(float radius, int primflags, Appearance ap) {
    this(radius, primflags, MID_REZ_DIV, ap);
  }

  /**  
   *   Constructs a Sphere of a given radius and number of divisions
   *   with additional parameters specified by the Primitive flags.
   *   Appearance defaults to white.
   *   @param radius Radius
   *   @param divisions Divisions
   *   @param primflags Primflags
   */
  public Sphere(float radius, int primflags, int divisions) {
    this(radius, primflags, divisions, null);
  }


  /**
   * Obtains Sphere's shape node that contains the geometry.
   * This allows users to modify the appearance or geometry.
   * @param partId The part to return (must be BODY for Spheres)
   * @return The Shape3D object associated with the partId.  If an
   * invalid partId is passed in, null is returned.
   */
  public Shape3D getShape(int partId) {
    if (partId != BODY) return null;
//     return (Shape3D)((Group)getChild(0)).getChild(BODY);
    return (Shape3D)getChild(BODY);
  }

  /** Obtains Sphere's shape node that contains the geometry.
   */
  public Shape3D getShape() {
//     return (Shape3D)((Group)getChild(0)).getChild(BODY);
      return (Shape3D)getChild(BODY);
  }

  /** Sets appearance of the Sphere.
   */
  public void setAppearance(Appearance ap) {
//     ((Shape3D)((Group)getChild(0)).getChild(BODY)).setAppearance(ap);
      ((Shape3D)getChild(BODY)).setAppearance(ap);
  }

    /**
     * Gets the appearance of the specified part of the sphere.
     *
     * @param partid identifier for a given subpart of the sphere
     *
     * @return The appearance object associated with the partID.  If an
     * invalid partId is passed in, null is returned.
     *
     * @since Java 3D 1.2.1
     */
    public Appearance getAppearance(int partId) {
	if (partId != BODY) return null;
	return getShape(partId).getAppearance();
    }


  /**  
   *   Constructs a customized Sphere of a given radius, 
   *   number of divisions, and appearance, with additional parameters
   *   specified by the Primitive flags.  The resolution is defined in
   *   terms of number of subdivisions along the sphere's axes. More
   *   divisions lead to more finely tesselated objects. 
   *   <p>
   *   If the appearance is null, the sphere defaults to a white appearance.
   */
  public Sphere(float radius, int primflags, int divisions, Appearance ap) {
    super();

    double rho, drho, theta, dtheta;
    double vx, vy, vz;
    double s, t, ds, dt;
    int i, j;
    double sign;

    this.radius = radius;
    this.divisions = divisions;

    /* 
     *     The sphere algorithm evaluates spherical angles along regular
     * units. For each spherical coordinate, (theta, rho), a (x,y,z) is
     * evaluated (along with the normals and texture coordinates).
     * 
     *       The spherical angles theta varies from 0 to 2pi and rho from 0
     * to pi. Sample points depends on the number of divisions.
     */

    flags = primflags;
    
    //Depending on whether normal inward bit is set.
    boolean outside;
    if ((flags & GENERATE_NORMALS_INWARD) != 0) {
      sign = -1.0;
      outside = false;
    }
    else {
	sign = 1.0;
	outside = true;
    }

    // delta theta and delta rho depends on divsions.
    dtheta = 2.0 * Math.PI / (double) divisions;
    drho = Math.PI / (double) divisions;

    t = 0.0;
    ds = 1.0 / divisions;
    dt = 1.0 / divisions;

    GeomBuffer cache = getCachedGeometry(Primitive.SPHERE,
					    radius, 0.0f, 0.0f, 
					    divisions, 0, primflags);
    Shape3D shape;
    if (cache != null) {
      shape = new Shape3D(cache.getComputedGeometry());
      numVerts += cache.getNumVerts();
      numTris += cache.getNumTris();
    } else {
      // Create a geometry buffer with given number points allocated.    
      GeomBuffer gbuf = new GeomBuffer(divisions*(divisions + 1)* 2);

      for (i = divisions; i > 0; i--) {
 	rho = i * drho;
	
	gbuf.begin(GeomBuffer.QUAD_STRIP);
	s = 0.0;
	for (j = divisions; j >= 0; j--) {
	    // Takes care of boundary case.
	    if (j == divisions)	  
		theta = 0.0 - Math.PI/2.0;
	    else 
		theta =((double) j) * dtheta - Math.PI/2.0;
	    
	    // Second quad vertex.
	    vx = Math.cos(theta) * Math.sin((i-1)*drho);
	    vy = sign * Math.cos((i-1)*drho);
	    vz = Math.sin(theta) * Math.sin((i-1)*drho);
		
	    // Send to Buffer
	    gbuf.normal3d( vx*sign, vy*sign, vz*sign );
	    gbuf.texCoord2d(s, t+dt);
	    gbuf.vertex3d( vx*radius, vy*radius, vz*radius );
	    
	    // First quad vertex.
	    // Evaluate spherical coords to get unit sphere positions. 
	    vx = Math.cos(theta) * Math.sin(rho);
	    vy = sign * Math.cos(rho);
	    vz = Math.sin(theta) * Math.sin(rho);
		
	    // Send to buffer.
	    gbuf.normal3d( vx*sign, vy*sign, vz*sign );
	    gbuf.texCoord2d(s, t);
	    gbuf.vertex3d( vx*radius, vy*radius, vz*radius );
	    
	    // increment s
	    s += ds;
	}
	gbuf.end();
	t += dt;
      }
      
      shape = new Shape3D(gbuf.getGeom(flags));
      numVerts = gbuf.getNumVerts();
      numTris = gbuf.getNumTris();
      if ((primflags & Primitive.GEOMETRY_NOT_SHARED) == 0) {
	cacheGeometry(Primitive.SPHERE,
		      radius, 0.0f, 0.0f, 
		      divisions, 0, primflags, gbuf);
      }
    }

    if ((flags & ENABLE_APPEARANCE_MODIFY) != 0) {
	shape.setCapability(Shape3D.ALLOW_APPEARANCE_READ);
	shape.setCapability(Shape3D.ALLOW_APPEARANCE_WRITE);
    }

    if ((flags & ENABLE_GEOMETRY_PICKING) != 0) {
        shape.setCapability(Shape3D.ALLOW_GEOMETRY_READ);
    }
    
    this.addChild(shape);

    if (ap == null) {
      setAppearance();
    }
    else setAppearance(ap);
  }

    /**
     * Used to create a new instance of the node.  This routine is called
     * by <code>cloneTree</code> to duplicate the current node.
     * <code>cloneNode</code> should be overridden by any user subclassed
     * objects.  All subclasses must have their <code>cloneNode</code>
     * method consist of the following lines:
     * <P><blockquote><pre>
     *     public Node cloneNode(boolean forceDuplicate) {
     *         UserSubClass usc = new UserSubClass();
     *         usc.duplicateNode(this, forceDuplicate);
     *         return usc;
     *     }
     * </pre></blockquote>
     * @param forceDuplicate when set to <code>true</code>, causes the
     *  <code>duplicateOnCloneTree</code> flag to be ignored.  When
     *  <code>false</code>, the value of each node's
     *  <code>duplicateOnCloneTree</code> variable determines whether
     *  NodeComponent data is duplicated or copied.
     *
     * @see Node#cloneTree
     * @see Node#duplicateNode
     * @see NodeComponent#setDuplicateOnCloneTree
     */
    public Node cloneNode(boolean forceDuplicate) {
        Sphere s = new Sphere(radius, flags, divisions, getAppearance());
        s.duplicateNode(this, forceDuplicate);

        return s;
    }

    /**
     * Copies all node information from <code>originalNode</code> into
     * the current node.  This method is called from the
     * <code>cloneNode</code> method which is, in turn, called by the
     * <code>cloneTree</code> method.
     * <P>
     * For any <i>NodeComponent</i> objects
     * contained by the object being duplicated, each <i>NodeComponent</i>
     * object's <code>duplicateOnCloneTree</code> value is used to determine
     * whether the <i>NodeComponent</i> should be duplicated in the new node
     * or if just a reference to the current node should be placed in the
     * new node.  This flag can be overridden by setting the
     * <code>forceDuplicate</code> parameter in the <code>cloneTree</code>
     * method to <code>true</code>.
     *
     * @param originalNode the original node to duplicate.
     * @param forceDuplicate when set to <code>true</code>, causes the
     *  <code>duplicateOnCloneTree</code> flag to be ignored.  When
     *  <code>false</code>, the value of each node's
     *  <code>duplicateOnCloneTree</code> variable determines whether
     *  NodeComponent data is duplicated or copied.
     *
     * @see Node#cloneTree
     * @see Node#cloneNode
     * @see NodeComponent#setDuplicateOnCloneTree
     */
    public void duplicateNode(Node originalNode, boolean forceDuplicate) {
        super.duplicateNode(originalNode, forceDuplicate);
    }

    /**
     * Returns the radius of the sphere
     *
     * @since Java 3D 1.2.1
     */
    public float getRadius() {
	return radius;
    }

    /**
     * Returns the number of divisions
     *
     * @since Java 3D 1.2.1
     */
    public int getDivisions() {
	return divisions;
    }

}
