/*
 *	@(#)CompressionStream.java 1.20 01/01/11 07:23:15
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.compression ;

import com.sun.j3d.utils.geometry.* ;
import javax.media.j3d.* ;
import javax.vecmath.* ;
import java.util.* ;

/**
 * This class is used as input to a geometry compressor.  It collects elements
 * such as vertices, normals, colors, mesh references, and quantization
 * parameters in an ordered stream.  This stream is then traversed during
 * the compression process and used to build the compressed output buffer.
 *
 * @see GeometryCompressor
 */
public class CompressionStream {
    private static final boolean benchmark = false ;

    // Mesh buffer normal substitution is unavailable in Level I.
    private static final boolean noMeshNormalSubstitution = true ;

    /**
     * This flag indicates that a vertex starts a new triangle or line strip.
     */
    static final int RESTART = 1 ;

    /**
     * This flag indicates that the next triangle in the strip is defined by
     * replacing the middle vertex of the previous triangle in the strip.
     * Equivalent to REPLACE_OLDEST for line strips.
     */
    static final int REPLACE_MIDDLE = 2 ;

    /**
     * This flag indicates that the next triangle in the strip is defined by
     * replacing the oldest vertex of the previous triangle in the strip.
     * Equivalent to REPLACE_MIDDLE for line strips.
     */
    static final int REPLACE_OLDEST = 3 ;

    /**
     * This flag indicates that a vertex is to be pushed into the mesh buffer.
     */
    static final int MESH_PUSH = 1 ;

    /**
     * This flag indicates that a vertex does not use the mesh buffer.
     */
    static final int NO_MESH_PUSH = 0 ;

    /**
     * Type of this stream, either CompressedGeometryHeader.POINT_BUFFER,
     * CompressedGeometryHeader.LINE_BUFFER, or
     * CompressedGeometryHeader.TRIANGLE_BUFFER
     */
    int streamType ;

    /**
     * A mask indicating which components are present in each vertex, as
     * defined by GeometryArray.
     */
    int vertexFormat ;

    /**
     * Boolean indicating RGB colors are bundled with the vertices.
     */
    boolean vertexColor3 ;

    /**
     * Boolean indicating RGBA colors are bundled with the vertices.
     */
    boolean vertexColor4 ;

    /**
     * Boolean indicating normals are bundled with the vertices.
     */
    boolean vertexNormals ;

    /**
     * Axes-aligned box enclosing all vertices in model coordinates.
     */
    Point3d mcBounds[] = new Point3d[2] ;

    /**
     * Axes-aligned box enclosing all vertices in normalized coordinates.
     */
    Point3d ncBounds[] = new Point3d[2] ;

    /**
     * Axes-aligned box enclosing all vertices in quantized coordinates.
     */
    Point3i qcBounds[] = new Point3i[2] ;

    /**
     * Center for normalizing positions to the unit cube.
     */
    double center[] = new double[3] ;

    /**
     * Maximum position range along the 3 axes.
     */
    double positionRangeMaximum ;

    /**
     * Scale for normalizing positions to the unit cube.
     */
    double scale ;

    /**
     * Current position component (X, Y, and Z) quantization value.  This can
     * range from 1 to 16 bits and has a default of 16.<p>
     *
     * At 1 bit of quantization it is not possible to express positive
     * absolute or delta positions.
     */
    int positionQuant ;

    /**
     * Current color component (R, G, B, A) quantization value.  This can
     * range from 2 to 16 bits and has a default of 9.<p>
     *
     * A color component is represented with a signed fixed-point value in
     * order to be able express negative deltas; the default of 9 bits
     * corresponds to the 8-bit color component range of the graphics hardware
     * commonly available.  Colors must be non-negative, so the lower limit of
     * quantization is 2 bits.
     */
    int colorQuant ;

    /**
     * Current normal component (U and V) quantization value.  This can range
     * from 0 to 6 bits and has a default of 6.<p>
     *
     * At 0 bits of quantization normals are represented only as 6 bit
     * sextant/octant pairs and 14 specially encoded normals (the 6 axis
     * normals and the 8 octant midpoint normals); since U and V can only be 0
     * at the minimum quantization, the totally number of unique normals is 
     * 12 + 14 = 26.
     */
    int normalQuant ;

    /**
     * Flag indicating position quantization change.
     */
    boolean positionQuantChanged ;

    /**
     * Flag indicating color quantization change.
     */
    boolean colorQuantChanged ;

    /**
     * Flag indicating normal quantization change.
     */
    boolean normalQuantChanged ;

    /**
     * Last quantized position.
     */
    int lastPosition[] = new int[3] ;

    /**
     * Last quantized color.
     */
    int lastColor[] = new int[4] ;

    /**
     * Last quantized normal's sextant.
     */
    int lastSextant ;

    /**
     * Last quantized normal's octant.
     */
    int lastOctant ;

    /**
     * Last quantized normal's U encoding parameter.
     */
    int lastU ;

    /**
     * Last quantized normal's V encoding parameter.
     */
    int lastV ;

    /**
     * Flag indicating last normal used a special encoding.
     */
    boolean lastSpecialNormal ;

    /**
     * Flag indicating the first position in this stream.
     */
    boolean firstPosition ;

    /**
     * Flag indicating the first color in this stream.
     */
    boolean firstColor ;

    /**
     * Flag indicating the first normal in this stream.
     */
    boolean firstNormal ;

    /**
     * The total number of bytes used to create the uncompressed geometric
     * elements in this stream, useful for performance analysis.  This
     * excludes mesh buffer references.
     */
    int byteCount ;

    /**
     * The number of vertices created for this stream, excluding mesh buffer
     * references.
     */
    int vertexCount ;

    /**
     * The number of mesh buffer references created for this stream.
     */
    int meshReferenceCount ;

    /**
     * Mesh buffer mirror used for computing deltas during quantization pass
     * and a limited meshing algorithm for unstripped data.
     */
    MeshBuffer meshBuffer = new MeshBuffer() ;


    // Collection which holds the elements of this stream.
    private Collection stream ;

    // True if preceding stream elements were colors or normals.  Used to flag
    // color and normal mesh buffer substitution when computing deltas during
    // quantization pass.
    private boolean lastElementColor = false ;
    private boolean lastLastElementColor = false ;
    private boolean lastElementNormal = false ;
    private boolean lastLastElementNormal = false ;

    // Some convenient temporary holding variables.
    private Point3f p3f = new Point3f() ;
    private Color3f c3f = new Color3f() ;
    private Color4f c4f = new Color4f() ;
    private Vector3f n3f = new Vector3f() ;


    // Private constructor for common initializations.
    private CompressionStream() {
	this.stream = new LinkedList() ;

	byteCount = 0 ;
	vertexCount = 0 ;
	meshReferenceCount = 0 ;

	mcBounds[0] = new Point3d(Double.POSITIVE_INFINITY,
				  Double.POSITIVE_INFINITY,
				  Double.POSITIVE_INFINITY) ;
	mcBounds[1] = new Point3d(Double.NEGATIVE_INFINITY,
				  Double.NEGATIVE_INFINITY,
				  Double.NEGATIVE_INFINITY) ;

	qcBounds[0] = new Point3i(Integer.MAX_VALUE,
				  Integer.MAX_VALUE,
				  Integer.MAX_VALUE) ;
	qcBounds[1] = new Point3i(Integer.MIN_VALUE,
				  Integer.MIN_VALUE,
				  Integer.MIN_VALUE) ;

	/* normalized bounds computed from quantized bounds */
	ncBounds[0] = new Point3d() ;
	ncBounds[1] = new Point3d() ;
    }

    /**
     * Creates a new CompressionStream for the specified geometry type and
     * vertex format.<p>
     *
     * @param streamType type of data in this stream, either
     * CompressedGeometryHeader.POINT_BUFFER,
     * CompressedGeometryHeader.LINE_BUFFER, or
     * CompressedGeometryHeader.TRIANGLE_BUFFER
     *
     * @param vertexFormat a mask indicating which components are
     * present in each vertex, as used by GeometryArray
     *
     * @see GeometryCompressor
     * @see GeometryArray
     */
    CompressionStream(int streamType, int vertexFormat) {
	this() ;
	setStreamType(streamType) ;
	setVertexFormat(vertexFormat) ;
    }

    private int getConnectionType(GeometryArray ga) {
	if (ga instanceof TriangleStripArray ||
	    ga instanceof IndexedTriangleStripArray ||
	    ga instanceof TriangleFanArray ||
	    ga instanceof IndexedTriangleFanArray ||
	    ga instanceof TriangleArray ||
	    ga instanceof IndexedTriangleArray ||
	    ga instanceof QuadArray ||
	    ga instanceof IndexedQuadArray)

	    return CompressedGeometryHeader.TRIANGLE_BUFFER ;

	else if (ga instanceof LineArray ||
		 ga instanceof IndexedLineArray ||
		 ga instanceof LineStripArray ||
		 ga instanceof IndexedLineStripArray)

	    return CompressedGeometryHeader.LINE_BUFFER ;

	else
	    return CompressedGeometryHeader.POINT_BUFFER ;
    }

    private void setStreamType(int streamType) {
	this.streamType = streamType ;
    }

    private void setVertexFormat(int vertexFormat) {
	this.vertexFormat = vertexFormat ;
	vertexColor3 = vertexColor4 = vertexNormals = false ;

	if ((vertexFormat & GeometryArray.NORMALS) == GeometryArray.NORMALS)
	    vertexNormals = true ;

	if ((vertexFormat & GeometryArray.COLOR_3) == GeometryArray.COLOR_3)
	    if ((vertexFormat & GeometryArray.COLOR_4) == GeometryArray.COLOR_4)
		vertexColor4 = true ;
	    else
		vertexColor3 = true ;
    }

    /**
     * Iterates across all compression stream elements and applies
     * quantization parameters, encoding consecutive vertices as delta values
     * whenever possible.  Each geometric element is mapped to a HuffmanNode
     * object containing its resulting bit length, right shift (trailing 0
     * count), and absolute or relative status.<p>
     * 
     * Positions are normalized to span a unit cube via an offset and a
     * uniform scale factor that maps the midpoint of the object extents along
     * each dimension to the origin, and the longest dimension of the object to
     * the open interval (-1.0 .. +1.0).  The geometric endpoints along that
     * dimension are both one quantum away from unity; for example, at a
     * position quantization of 6 bits, an object would be normalized so that
     * its most negative dimension is at (-1 + 1/64) and the most positive is
     * at (1 - 1/64).<p>
     * 
     * Normals are assumed to be of unit length.  Color components are clamped
     * to the [0..1) range, where the right endpoint is one quantum less
     * than 1.0.<p>
     *
     * @param huffmanTable Table which will map geometric compression stream
     * elements to HuffmanNode objects describing each element's data
     * representation.  This table can then be processed with Huffman's
     * algorithm to optimize the bit length of descriptor tags according to
     * the number of geometric elements mapped to each tag.
     */
    void quantize(HuffmanTable huffmanTable) {
	// Set up default initial quantization parameters.  The position and
	// color parameters specify the number of bits for each X, Y, Z, R, G,
	// B, or A component.  The normal quantization parameter specifies the
	// number of bits for each U and V component.
	positionQuant = 16 ;
	colorQuant = 9 ;
	normalQuant = 6 ;

	// Compute position center and scaling for normalization to the unit
	// cube.  This is a volume bounded by the open intervals (-1..1) on
	// each axis.
	center[0] = (mcBounds[1].x + mcBounds[0].x) / 2.0 ;
	center[1] = (mcBounds[1].y + mcBounds[0].y) / 2.0 ;
	center[2] = (mcBounds[1].z + mcBounds[0].z) / 2.0 ;

	double xRange = mcBounds[1].x - mcBounds[0].x ;
	double yRange = mcBounds[1].y - mcBounds[0].y ;
	double zRange = mcBounds[1].z - mcBounds[0].z ;

	if (xRange > yRange)
	    positionRangeMaximum = xRange ;
	else
	    positionRangeMaximum = yRange ;

	if (zRange > positionRangeMaximum)
	    positionRangeMaximum = zRange ;

	// Adjust the range of the unit cube to match the default
	// quantization.
	//
	// This scale factor along with the center values computed above will
	// produce 16-bit integer representations of the floating point
	// position coordinates ranging symmetrically about 0 from -32767 to
	// +32767.  -32768 is not used and the normalized floating point
	// position coordinates of -1.0 as well as +1.0 will not be
	// represented.
	//
	// Applications which wish to seamlessly stitch together compressed
	// objects will need to be aware that the range of normalized
	// positions will be one quantum away from the [-1..1] endpoints of
	// the unit cube and should adjust scale factors accordingly.
	scale = (2.0 / positionRangeMaximum) * (32767.0 / 32768.0) ;

	// Flag quantization change.
	positionQuantChanged = colorQuantChanged = normalQuantChanged = true ;

	// Flag first position, color, and normal.
	firstPosition = firstColor = firstNormal = true ;

	// Apply quantization.
	Iterator i = stream.iterator() ;
	while (i.hasNext()) {
	    Object o = i.next() ;

	    if (o instanceof CompressionStreamElement) {
		((CompressionStreamElement)o).quantize(this, huffmanTable) ;

		// Keep track of whether last two elements were colors or
		// normals for mesh buffer component substitution semantics.
		lastLastElementColor = lastElementColor ;
		lastLastElementNormal = lastElementNormal ;
		lastElementColor = lastElementNormal = false ;

		if (o instanceof CompressionStreamColor)
		    lastElementColor = true ;
		else if (o instanceof CompressionStreamNormal)
		    lastElementNormal = true ;
	    }
	}

	// Compute the bounds in normalized coordinates.
	ncBounds[0].x = (double)qcBounds[0].x / 32768.0 ;
	ncBounds[0].y = (double)qcBounds[0].y / 32768.0 ;
	ncBounds[0].z = (double)qcBounds[0].z / 32768.0 ;

	ncBounds[1].x = (double)qcBounds[1].x / 32768.0 ;
	ncBounds[1].y = (double)qcBounds[1].y / 32768.0 ;
	ncBounds[1].z = (double)qcBounds[1].z / 32768.0 ;
    }

    /**
     * Iterates across all compression stream elements and builds the
     * compressed geometry command stream output.<p>
     *
     * @param huffmanTable Table which maps geometric elements in this stream
     * to tags describing the encoding parameters (length, shift, and
     * absolute/relative status) to be used for their representations in the
     * compressed output.  All tags must be 6 bits or less in length, and the
     * sum of the number of bits in the tag plus the number of bits in the
     * data it describes must be at least 6 bits in length.
     *
     * @param outputBuffer CommandStream to use for collecting the compressed
     * bits.
     */
    void outputCommands(HuffmanTable huffmanTable, CommandStream outputBuffer) {
	//
	// The first command output is setState to indicate what data is
	// bundled with each vertex.  Although the semantics of geometry
	// decompression allow setState to appear anywhere in the stream, this
	// cannot be handled by the current Java 3D software decompressor,
	// which internally decompresses an entire compressed buffer into a
	// single retained object sharing a single consistent vertex format.
	// This limitation may be removed in subsequent releases of Java 3D.
	//
	int bnv = (vertexNormals? 1 : 0) ;
	int bcv = ((vertexColor3 || vertexColor4)? 1 : 0) ;
	int cap = (vertexColor4? 1 : 0) ;

	int command = CommandStream.SET_STATE | bnv ;
	long data = (bcv << 2) | (cap << 1) ;

	// Output the setState command.
	outputBuffer.addCommand(command, 8, data, 3) ;

	// Output the Huffman table commands.
	huffmanTable.outputCommands(outputBuffer) ;

	// Output each compression stream element's data.
	Iterator i = stream.iterator() ;
	while (i.hasNext()) {
	    Object o = i.next() ;
	    if (o instanceof CompressionStreamElement)
		((CompressionStreamElement)o).outputCommand(huffmanTable,
							    outputBuffer) ;
	}

	// Finish the header-forwarding interleave and long-word align.
	outputBuffer.end() ;
    }

    /**
     * Retrieve the total size of the uncompressed geometric data in bytes,
     * excluding mesh buffer references.
     * @return uncompressed byte count
     */
    int getByteCount() {
	return byteCount ;
    }

    /**
     * Retrieve the the number of vertices created for this stream, excluding
     * mesh buffer references.
     * @return vertex count
     */
    int getVertexCount() {
	return vertexCount ;
    }

    /**
     * Retrieve the number of mesh buffer references created for this stream. 
     * @return mesh buffer reference count
     */
    int getMeshReferenceCount() {
	return meshReferenceCount ;
    }

    /**
     * Stream element that sets position quantization during quantize pass.
     */
    class PositionQuant extends CompressionStreamElement {
	int value ;

	PositionQuant(int value) {
	    this.value = value ;
	}

	void quantize(CompressionStream s, HuffmanTable t) {
	    positionQuant = value ;
	    positionQuantChanged = true ;

	    // Adjust range of unit cube scaling to match quantization.
	    scale = (2.0 / positionRangeMaximum) *
		(((double)((1 << (value-1)) - 1))/((double)(1 << (value-1)))) ;
	}

	public String toString() {
	    return "positionQuant: " + value ;
	}
    }

    /**
     * Stream element that sets normal quantization during quantize pass.
     */
    class NormalQuant extends CompressionStreamElement {
	int value ;

	NormalQuant(int value) {
	    this.value = value ;
	}

	void quantize(CompressionStream s, HuffmanTable t) {
	    normalQuant = value ;
	    normalQuantChanged = true ;
	}

	public String toString() {
	    return "normalQuant: " + value ;
	}
    }

    /**
     * Stream element that sets color quantization during quantize pass.
     */
    class ColorQuant extends CompressionStreamElement {
	int value ;

	ColorQuant(int value) {
	    this.value = value ;
	}

	void quantize(CompressionStream s, HuffmanTable t) {
	    colorQuant = value ;
	    colorQuantChanged = true ;
	}

	public String toString() {
	    return "colorQuant: " + value ;
	}
    }

    /**
     * Stream element that references the mesh buffer.
     */
    class MeshReference extends CompressionStreamElement {
	int stripFlag, meshIndex ;

	MeshReference(int stripFlag, int meshIndex) {
	    this.stripFlag = stripFlag ;
	    this.meshIndex = meshIndex ;
	    meshReferenceCount++ ;
	}

	void quantize(CompressionStream s, HuffmanTable t) {
	    // Retrieve the vertex from the mesh buffer mirror and set up the
	    // data needed for the next stream element to compute its deltas.
	    CompressionStreamVertex v = meshBuffer.getVertex(meshIndex) ;
	    lastPosition[0] = v.xAbsolute ;
	    lastPosition[1] = v.yAbsolute ;
	    lastPosition[2] = v.zAbsolute ;

	    // Set up last color data if it exists and previous elements
	    // don't override it.
	    if (v.color != null && !lastElementColor &&
		!(lastElementNormal && lastLastElementColor)) {
		lastColor[0] = v.color.rAbsolute ;
		lastColor[1] = v.color.gAbsolute ;
		lastColor[2] = v.color.bAbsolute ;
		lastColor[3] = v.color.aAbsolute ;
	    }

	    // Set up last normal data if it exists and previous element
	    // doesn't override it.
	    if (v.normal != null && !lastElementNormal &&
		!(lastElementColor && lastLastElementNormal)) {
		lastSextant = v.normal.sextant ;
		lastOctant = v.normal.octant ;
		lastU = v.normal.uAbsolute ;
		lastV = v.normal.vAbsolute ;
		lastSpecialNormal = v.normal.specialNormal ;
	    }
	}

	void outputCommand(HuffmanTable t, CommandStream outputBuffer) {
	    int command = CommandStream.MESH_B_R ;
	    long data = stripFlag & 0x1 ;

	    command |= (((meshIndex & 0xf) << 1) | (stripFlag >> 1)) ;
	    outputBuffer.addCommand(command, 8, data, 1) ;
	}

	public String toString() {
	    return
		"meshReference: stripFlag " + stripFlag +
		" meshIndex " + meshIndex ;
	}
    }


    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param stripFlag vertex replacement flag, either RESTART,
     * REPLACE_OLDEST, or REPLACE_MIDDLE
     */
    void addVertex(Point3f pos, int stripFlag) {
	stream.add(new CompressionStreamVertex(this, pos,
					       (Vector3f)null, (Color3f)null,
					       stripFlag, NO_MESH_PUSH)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param stripFlag vertex replacement flag, either RESTART,
     * REPLACE_OLDEST, or REPLACE_MIDDLE
     */
    void addVertex(Point3f pos, Vector3f norm, int stripFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, norm, (Color3f)null, stripFlag, NO_MESH_PUSH)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART,
     * REPLACE_OLDEST, or REPLACE_MIDDLE
     */
    void addVertex(Point3f pos, Color3f color, int stripFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, (Vector3f)null, color, stripFlag, NO_MESH_PUSH)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART,
     * REPLACE_OLDEST, or REPLACE_MIDDLE
     */
    void addVertex(Point3f pos, Color4f color, int stripFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, (Vector3f)null, color, stripFlag, NO_MESH_PUSH)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART,
     * REPLACE_OLDEST, or REPLACE_MIDDLE
     */
    void addVertex(Point3f pos, Vector3f norm, Color3f color,
			  int stripFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, norm, color, stripFlag, NO_MESH_PUSH)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART,
     * REPLACE_OLDEST, or REPLACE_MIDDLE
     */
    void addVertex(Point3f pos, Vector3f norm, Color4f color,
			  int stripFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, norm, color, stripFlag, NO_MESH_PUSH)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, int stripFlag, int meshFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, (Vector3f)null, (Color3f)null, stripFlag, meshFlag)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, Vector3f norm,
			  int stripFlag, int meshFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, norm, (Color3f)null, stripFlag, meshFlag)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, Color3f color,
			  int stripFlag, int meshFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, (Vector3f)null, color, stripFlag, meshFlag)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, Color4f color,
			  int stripFlag, int meshFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, (Vector3f)null, color, stripFlag, meshFlag)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, Vector3f norm, Color3f color,
			  int stripFlag, int meshFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, norm, color, stripFlag, meshFlag)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param color color data
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, Vector3f norm, Color4f color,
			  int stripFlag, int meshFlag) {
	stream.add(new CompressionStreamVertex
	    (this, pos, norm, color, stripFlag, meshFlag)) ;
    }

    /**
     * Copy vertex data and add it to the end of this stream.
     * @param pos position data
     * @param norm normal data
     * @param color color data, either Color3f or Color4f, determined by
     * current vertex format
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshFlag if MESH_PUSH the vertex is pushed into the mesh buffer
     */
    void addVertex(Point3f pos, Vector3f norm,
		   Object color, int stripFlag, int meshFlag) {

	if (vertexColor3) 
	    stream.add(new CompressionStreamVertex
		       (this, pos, norm, (Color3f)color, stripFlag, meshFlag)) ;
	else
	    stream.add(new CompressionStreamVertex
		       (this, pos, norm, (Color4f)color, stripFlag, meshFlag)) ;
    }

    /**
     * Add a mesh buffer reference to this stream.
     * @param stripFlag vertex replacement flag, either RESTART, REPLACE_OLDEST,
     * or REPLACE_MIDDLE
     * @param meshIndex index of vertex to retrieve from the mesh buffer
     */
    void addMeshReference(int stripFlag, int meshIndex) {
	stream.add(new MeshReference(stripFlag, meshIndex)) ;
    }

    /**
     * Copy the given color to the end of this stream and use it as a global
     * state change that applies to all subsequent vertices.
     */
    void addColor(Color3f c3f) {
	stream.add(new CompressionStreamColor(this, c3f)) ;
    }

    /**
     * Copy the given color to the end of this stream and use it as a global
     * state change that applies to all subsequent vertices.
     */
    void addColor(Color4f c4f) {
	stream.add(new CompressionStreamColor(this, c4f)) ;
    }

    /**
     * Copy the given normal to the end of this stream and use it as a global
     * state change that applies to all subsequent vertices.
     */
    void addNormal(Vector3f n) {
	stream.add(new CompressionStreamNormal(this, n)) ;
    }

    /**
     * Add a new position quantization value to the end of this stream that
     * will apply to all subsequent vertex positions.
     *
     * @param value number of bits to quantize each position's X, Y,
     * and Z components, ranging from 1 to 16 with a default of 16
     */
    void addPositionQuantization(int value) {
	stream.add(new PositionQuant(value)) ;
    }

    /**
     * Add a new color quantization value to the end of this stream that will
     * apply to all subsequent colors.
     *
     * @param value number of bits to quantize each color's R, G, B, and
     * alpha components, ranging from 2 to 16 with a default of 9
     */
    void addColorQuantization(int value) {
	stream.add(new ColorQuant(value)) ;
    }

    /**
     * Add a new normal quantization value to the end of this stream that will
     * apply to all subsequent normals.  This value specifies the number of
     * bits for each normal's U and V components.
     *
     * @param value number of bits for quantizing U and V, ranging from 0 to
     * 6 with a default of 6
     */
    void addNormalQuantization(int value) {
	stream.add(new NormalQuant(value)) ;
    }

    /**
     * Convert a GeometryArray to compression stream elements and add them to
     * this stream.
     *
     * @param ga GeometryArray to convert
     * @exception IllegalArgumentException if GeometryArray has a
     * dimensionality or vertex format inconsistent with the CompressionStream
     */
    void addGeometryArray(GeometryArray ga) {
	if (streamType != getConnectionType(ga))
	    throw new IllegalArgumentException
		("GeometryArray has an inconsistent dimensionality") ;
	else if (vertexFormat != ga.getVertexFormat())
	    throw new IllegalArgumentException
		("GeometryArray has an inconsistent vertex format") ;

	// Get index arrays if needed.
	int indexCount = 0 ;
	int colorIndices[] = null ;
	int normalIndices[] = null ;
	int positionIndices[] = null ;
	boolean indexedGeometry = false ;

	if (ga instanceof IndexedGeometryArray) {
	    indexedGeometry = true ;
	    IndexedGeometryArray iga = (IndexedGeometryArray)ga ;
	    indexCount = iga.getIndexCount() ;

	    positionIndices = new int[indexCount] ;
	    iga.getCoordinateIndices(0, positionIndices) ;

	    if (vertexNormals) {
		normalIndices = new int[indexCount] ;
		iga.getNormalIndices(0, normalIndices) ;
	    }

	    if (vertexColor3 || vertexColor4) {
		colorIndices = new int[indexCount] ;
		iga.getColorIndices(0, colorIndices) ;
	    }
	}

	// Check topology.
	int stripCount = 0 ;
	int stripCounts[] = null ;
	int replaceCode = RESTART ;
	int vertexCount = ga.getVertexCount() ;

	if (ga instanceof TriangleStripArray ||
	    ga instanceof IndexedTriangleStripArray ||
	    ga instanceof LineStripArray ||
	    ga instanceof IndexedLineStripArray)
	    replaceCode = REPLACE_OLDEST ;

	else if (ga instanceof TriangleFanArray ||
		 ga instanceof IndexedTriangleFanArray)
	    replaceCode = REPLACE_MIDDLE ;

	if (replaceCode == RESTART) {
	    // Geometry isn't explicitly stripped.
	    // Convert it to a canonical strip representation.
	    int verticesPerPrimitive ;

	    if (ga instanceof QuadArray ||
		ga instanceof IndexedQuadArray) {
		verticesPerPrimitive = 4 ;
		replaceCode = REPLACE_MIDDLE ;
	    }

	    else if (ga instanceof TriangleArray ||
		     ga instanceof IndexedTriangleArray)
		verticesPerPrimitive = 3 ;

	    else if (ga instanceof LineArray ||
		     ga instanceof IndexedLineArray)
		verticesPerPrimitive = 2 ;

	    else
		verticesPerPrimitive = 1 ;

	    if (indexedGeometry)
		stripCount = indexCount/verticesPerPrimitive ;
	    else
		stripCount = vertexCount/verticesPerPrimitive ;

	    stripCounts = new int[stripCount] ;
	    Arrays.fill(stripCounts, verticesPerPrimitive) ;

	} else {
	    // Get strip data.
	    if (indexedGeometry) {
		IndexedGeometryStripArray igsa ;
		igsa = (IndexedGeometryStripArray)ga ;

		stripCount = igsa.getNumStrips() ;
		stripCounts = new int[stripCount] ;
		igsa.getStripIndexCounts(stripCounts) ;

	    } else {
		GeometryStripArray gsa ;
		gsa = (GeometryStripArray)ga ;

		stripCount = gsa.getNumStrips() ;
		stripCounts = new int[stripCount] ;
		gsa.getStripVertexCounts(stripCounts) ;
	    }
	}

	// Get vertex data.
	Vector3f normals[] = null ;
	Color3f colors3[] = null ;
	Color4f colors4[] = null ;

	Point3f positions[] = new Point3f[vertexCount] ;
	for (int i = 0 ; i < vertexCount ; i++)
	    positions[i] = new Point3f() ;

	ga.getCoordinates(0, positions) ;

	if (vertexNormals) {
	    normals = new Vector3f[vertexCount] ;
	    for (int i = 0 ; i < vertexCount ; i++)
		normals[i] = new Vector3f() ;

	    ga.getNormals(0, normals) ;
	}
	if (vertexColor3) {
	    colors3 = new Color3f[vertexCount] ;
	    for (int i = 0 ; i < vertexCount ; i++)
		colors3[i] = new Color3f() ;

	    ga.getColors(0, colors3) ;
	}
	else if (vertexColor4) {
	    colors4 = new Color4f[vertexCount] ;
	    for (int i = 0 ; i < vertexCount ; i++)
		colors4[i] = new Color4f() ;

	    ga.getColors(0, colors4) ;
	}

	// Build the compression stream for this shape's geometry.
	//
	// Always push vertices into the mesh buffer unless they match ones
	// already there; if they do, generate mesh buffer references instead.
	// This reduces the number of vertices when non-stripped abutting
	// facets are added to the stream.
	//
	// Note: Level II geometry compression semantics allow the mesh buffer
	// normals to be substituted with the value of an immediately
	// preceding SetNormal command, but this is unavailable in Level I.
	int v = 0 ;
	int stripFlag ;
	if (indexedGeometry) {
	    for (int i = 0 ; i < stripCount ; i++) {
		stripFlag = RESTART ;

		for (int j = 0 ; j < stripCounts[i] ; j++) {
		    int pi = positionIndices[v] ;
		    int r = meshBuffer.getMeshReference(pi) ;

		    if ((r == meshBuffer.NOT_FOUND) ||
			(vertexNormals && noMeshNormalSubstitution &&
			 (normalIndices[v] != meshBuffer.getNormalIndex(r)))) {

			int ni = vertexNormals? normalIndices[v] : -1 ;
			int ci = (vertexColor3 || vertexColor4)?
			         colorIndices[v] : -1 ;

			Point3f p = positions[pi] ;
			Vector3f n = vertexNormals? normals[ni] : null ;
			Object c = vertexColor3? (Object)colors3[ci] :
			           vertexColor4? (Object)colors4[ci] : null ;

			addVertex(p, n, c, stripFlag, MESH_PUSH) ;
			meshBuffer.push(pi, ci, ni) ;

		    } else {
			if (vertexNormals && !noMeshNormalSubstitution &&
			    normalIndices[v] != meshBuffer.getNormalIndex(r))
			    addNormal(normals[normalIndices[v]]) ;

			if (vertexColor3 &&
			    colorIndices[v] != meshBuffer.getColorIndex(r))
			    addColor(colors3[colorIndices[v]]) ;
			
			else if (vertexColor4 &&
				 colorIndices[v] != meshBuffer.getColorIndex(r))
			    addColor(colors4[colorIndices[v]]) ;

			addMeshReference(stripFlag, r) ;
		    }
		    stripFlag = replaceCode ;
		    v++ ;
		}
	    }
	} else {
	    for (int i = 0 ; i < stripCount ; i++) {
		stripFlag = RESTART ;

		for (int j = 0 ; j < stripCounts[i] ; j++) {
		    Point3f p = positions[v] ;
		    int r = meshBuffer.getMeshReference(p) ;

		    if ((r == meshBuffer.NOT_FOUND) ||
			(vertexNormals && noMeshNormalSubstitution &&
			 (! normals[v].equals(meshBuffer.getNormal(r))))) {

			Vector3f n = vertexNormals? normals[v] : null ;
			Object c = vertexColor3? (Object)colors3[v] :
			           vertexColor4? (Object)colors4[v] : null ;

			addVertex(p, n, c, stripFlag, MESH_PUSH) ;
			meshBuffer.push(p, c, n) ;

		    } else {
			if (vertexNormals && !noMeshNormalSubstitution &&
			    (! normals[v].equals(meshBuffer.getNormal(r))))
			    addNormal(normals[v]) ;

			if (vertexColor3 &&
			    (! colors3[v].equals(meshBuffer.getColor3(r))))
			    addColor(colors3[v]) ;

			else if (vertexColor4 &&
				 (! colors4[v].equals(meshBuffer.getColor4(r))))
			    addColor(colors4[v]) ;

			addMeshReference(stripFlag, r) ;
		    }
		    stripFlag = replaceCode ;
		    v++ ;
		}
	    }
	}
    }

    /**
     * Print the stream to standard output.
     */
    void print() {
	System.out.println("\nstream has " + stream.size() + " entries") ;
	System.out.println("uncompressed size " + byteCount + " bytes") ;
	System.out.println("upper position bound: " + mcBounds[1].toString()) ;
	System.out.println("lower position bound: " + mcBounds[0].toString()) ;
	System.out.println("X, Y, Z centers (" +
			   ((float)center[0]) + " " +
			   ((float)center[1]) + " " +
			   ((float)center[2]) + ")\n" +
			   "scale " + ((float)scale) + "\n") ;

	Iterator i = stream.iterator() ;
	while (i.hasNext()) {
	    System.out.println(i.next().toString() + "\n") ;
	}
    }


    ////////////////////////////////////////////////////////////////////////////
    //									      //
    // The following constructors are currently the only public members of    //
    // this class.  All other members are subject to revision.                //
    //									      //
    ////////////////////////////////////////////////////////////////////////////

    /**
     * Creates a CompressionStream from an array of Shape3D scene graph
     * objects.  These Shape3D objects may only consist of a GeometryArray
     * component and an optional Appearance component.  The resulting stream
     * may be used as input to the GeometryCompressor methods.<p>
     *
     * Each Shape3D in the array must be of the same dimensionality (point,
     * line, or surface) and have the same vertex format as the others.
     * Texture coordinates are ignored.<p>
     *
     * If a color is specified in the material attributes for a Shape3D then
     * that color is added to the CompressionStream as the current global
     * color.  Subsequent colors as well as any colors bundled with vertices
     * will override it.  Only the material diffuse colors are used; all other
     * appearance attributes are ignored.<p>
     *
     * @param positionQuant
     * number of bits to quantize each position's X, Y,
     * and Z components, ranging from 1 to 16
     *
     * @param colorQuant
     * number of bits to quantize each color's R, G, B, and
     * alpha components, ranging from 2 to 16
     *
     * @param normalQuant
     * number of bits for quantizing each normal's U and V components, ranging
     * from 0 to 6
     *
     * @param shapes
     * an array of Shape3D scene graph objects containing
     * GeometryArray objects, all with the same vertex format and
     * dimensionality
     *
     * @exception IllegalArgumentException if any Shape3D has an inconsistent
     * dimensionality or vertex format, or if any Shape3D contains a geometry
     * component that is not a GeometryArray
     *
     * @see Shape3D
     * @see GeometryArray
     * @see GeometryCompressor
     */
    public CompressionStream(int positionQuant, int colorQuant,
			     int normalQuant, Shape3D shapes[]) {
	this() ;

	if ((shapes == null) || (shapes[0] == null))
	    throw new IllegalArgumentException("null Shape3D") ;

	long startTime = 0 ;
	if (benchmark) startTime = System.currentTimeMillis() ;

	Geometry g = shapes[0].getGeometry() ;
	if (! (g instanceof GeometryArray))
	    throw new IllegalArgumentException
		("Shape3D at index 0 is not a GeometryArray") ;

	GeometryArray ga = (GeometryArray)g ;
	int vertexFormat = ga.getVertexFormat() ;
	int streamType = getConnectionType(ga) ;

	setStreamType(streamType) ;
	setVertexFormat(vertexFormat) ;

	// Add global quantization parameters to the start of the stream.
	addPositionQuantization(positionQuant) ;
	addColorQuantization(colorQuant) ;
	addNormalQuantization(normalQuant) ;

	// Loop through all shapes.
	for (int s = 0 ; s < shapes.length ; s++) {
	    g = shapes[s].getGeometry() ;
	    if (! (g instanceof GeometryArray))
		throw new IllegalArgumentException
		    ("Shape3D at index " + s + " is not a GeometryArray") ;

	    // Check for material color and add it to the stream if it exists.
	    Appearance a = shapes[s].getAppearance() ;
	    if (a != null) {
		Material m = a.getMaterial() ;
		if (m != null) {
		    m.getDiffuseColor(c3f) ;
		    if (vertexColor4) {
			c4f.set(c3f.x, c3f.y, c3f.z, 1.0f) ;
			addColor(c4f) ;
		    } else
			addColor(c3f) ;
		}
	    }

	    // Add the geometry array to the stream.
	    addGeometryArray((GeometryArray)g) ;
	}

	if (benchmark) {
	    long t = System.currentTimeMillis() - startTime ;
	    System.out.println
		("\nCompressionStream:\n" + shapes.length + " shapes in " +
		 (t / 1000f) + " sec") ;
	}
    }

    /**
     * Creates a CompressionStream from an array of Shape3D scene graph
     * objects.  These Shape3D objects may only consist of a GeometryArray
     * component and an optional Appearance component.  The resulting stream
     * may be used as input to the GeometryCompressor methods.<p>
     *
     * Each Shape3D in the array must be of the same dimensionality (point,
     * line, or surface) and have the same vertex format as the others.
     * Texture coordinates are ignored.<p>
     *
     * If a color is specified in the material attributes for a Shape3D then
     * that color is added to the CompressionStream as the current global
     * color.  Subsequent colors as well as any colors bundled with vertices
     * will override it.  Only the material diffuse colors are used; all other
     * appearance attributes are ignored.<p>
     *
     * Defaults of 16, 9, and 6 bits are used as the quantization values for
     * positions, colors, and normals respectively.  These are the maximum
     * resolution values defined for positions and normals; the default of 9
     * for color is the equivalent of the 8 bits of RGBA component resolution
     * commonly available in graphics frame buffers.<p>
     *
     * @param shapes
     * an array of Shape3D scene graph objects containing
     * GeometryArray objects, all with the same vertex format and
     * dimensionality.
     *
     * @exception IllegalArgumentException if any Shape3D has an inconsistent
     * dimensionality or vertex format, or if any Shape3D contains a geometry
     * component that is not a GeometryArray
     *
     * @see Shape3D
     * @see GeometryArray
     * @see GeometryCompressor
     */
    public CompressionStream(Shape3D shapes[]) {
	this(16, 9, 6, shapes) ;
    }

    /**
     * Creates a CompressionStream from an array of GeometryInfo objects.  The
     * resulting stream may be used as input to the GeometryCompressor
     * methods.<p>
     *
     * Each GeometryInfo in the array must be of the same dimensionality
     * (point, line, or surface) and have the same vertex format as the
     * others.  Texture coordinates are ignored.<p>
     *
     * @param positionQuant
     * number of bits to quantize each position's X, Y,
     * and Z components, ranging from 1 to 16
     *
     * @param colorQuant
     * number of bits to quantize each color's R, G, B, and
     * alpha components, ranging from 2 to 16
     *
     * @param normalQuant
     * number of bits for quantizing each normal's U and V components, ranging
     * from 0 to 6
     *
     * @param geometry
     * an array of GeometryInfo objects, all with the same
     * vertex format and dimensionality
     *
     * @exception IllegalArgumentException if any GeometryInfo object has an
     * inconsistent dimensionality or vertex format
     *
     * @see GeometryInfo
     * @see GeometryCompressor
     */
    public CompressionStream(int positionQuant, int colorQuant,
			     int normalQuant, GeometryInfo geometry[]) {
	this() ;

	if ((geometry == null) || (geometry[0] == null))
	    throw new IllegalArgumentException("null GeometryInfo") ;

	GeometryArray ga = geometry[0].getGeometryArray() ;
	int vertexFormat = ga.getVertexFormat() ;
	int streamType = getConnectionType(ga) ;

	setStreamType(streamType) ;
	setVertexFormat(vertexFormat) ;

	// Add global quantization parameters to the start of the stream.
	addPositionQuantization(positionQuant) ;
	addColorQuantization(colorQuant) ;
	addNormalQuantization(normalQuant) ;

	// Loop through all GeometryInfo objects and add them to the stream.
	for (int i = 0 ; i < geometry.length ; i++) {
	    addGeometryArray(geometry[i].getGeometryArray()) ;
	}
    }

    /**
     * Creates a CompressionStream from an array of GeometryInfo objects.  The
     * resulting stream may be used as input to the GeometryCompressor
     * methods.<p>
     *
     * Each GeometryInfo in the array must be of the same dimensionality
     * (point, line, or surface) and have the same vertex format as the
     * others.  Texture coordinates are ignored.<p>
     *
     * Defaults of 16, 9, and 6 bits are used as the quantization values for
     * positions, colors, and normals respectively.  These are the maximum
     * resolution values defined for positions and normals; the default of 9
     * for color is the equivalent of the 8 bits of RGBA component resolution
     * commonly available in graphics frame buffers.<p>
     *
     * @param geometry
     * an array of GeometryInfo objects, all with the same
     * vertex format and dimensionality
     *
     * @exception IllegalArgumentException if any GeometryInfo object has an
     * inconsistent dimensionality or vertex format
     *
     * @see GeometryInfo
     * @see GeometryCompressor
     */
    public CompressionStream(GeometryInfo geometry[]) {
	this(16, 9, 6, geometry) ;
    }
}
