/*
 *	@(#)TCBKeyFrame.java 1.13 01/01/11 07:21:58
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.behaviors.interpolators;

import javax.media.j3d.*;
import java.util.*;
import javax.vecmath.*;
import com.sun.j3d.internal.J3dUtilsI18N;

/**
 * This class represents a Key Frame that can be used for Kochanek-Bartels
 * (TCB) spline interpolation.
 *
 * @since Java3D 1.1
 */

public class TCBKeyFrame {
  
    // Position, Rotation and Scale
    public Point3f position; 
    public Quat4f  quat;
    public Point3f scale; 

    // Tension, Continuity & Bias
    public float  tension;
    public float  continuity;
    public float  bias;

    // Sample Time 
    public float  knot;

    // Interpolation type (linear = 0 -> spline interpolation)
    public int linear; 

    // default constructor 
    TCBKeyFrame () {
        tension = continuity = bias = 0.0f;
    }

    public TCBKeyFrame (TCBKeyFrame kf) {
        this(kf.knot, kf.linear, kf.position, kf.quat, kf.scale,
                     kf.tension, kf.continuity, kf.bias);
        
    }

    /*
     * Creates a key frame using the given inputs. 
     *
     * @param k knot value for this key frame 
     * @param l the linear flag (0 - Spline Interp, 1, Linear Interp
     * @param pos the position at the key frame
     * @param q the rotations at the key frame
     * @param s the scales at the key frame
     * @param t tension (-1.0 < t < 1.0)
     * @param c continuity (-1.0 < c < 1.0)
     * @param b bias (-1.0 < b < 1.0)
     */
    public TCBKeyFrame (float k, int l, Point3f pos, Quat4f q, Point3f s, 
                        float t, float c, float b) {

        knot     = k; 
        linear   = l;
        position = new Point3f(pos);
        quat     = new Quat4f(q);
        scale    = new Point3f(s);

        // Check for valid tension continuity and bias values
        if (t < -1.0f || t > 1.0f) {
            throw new IllegalArgumentException(J3dUtilsI18N.getString("TCBKeyFrame0"));
        }

        if (b < -1.0f || b > 1.0f) {
            throw new IllegalArgumentException(J3dUtilsI18N.getString("TCBKeyFrame1"));
        }

        if (c < -1.0f || c > 1.0f) {
            throw new IllegalArgumentException(J3dUtilsI18N.getString("TCBKeyFrame2"));
        }

        // copy valid tension, continuity and bias values
        tension = t;
        continuity = c;
        bias = b;
    }

    /*
     * Prints information comtained in this key frame 
     * @param tag string tag for identifying debug message
     */
    public void debugPrint (String tag) {
        System.out.println ("\n" + tag); 
        System.out.println (" knot = " + knot); 
        System.out.println (" linear = " + linear); 
        System.out.println (" position(x,y,z) = " + position.x + " " 
                               + position.y + " " + position.z);
              
        System.out.println (" tension = " + tension); 
        System.out.println (" continuity = " + continuity); 
        System.out.println (" bias = " + bias); 
    }
}

