/*
 *	@(#)KBSplinePathInterpolator.java 1.5 01/01/11 07:22:01
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.behaviors.interpolators;

import javax.media.j3d.*;
import java.util.*;
import javax.vecmath.*;
import com.sun.j3d.internal.J3dUtilsI18N;

/**
 * KBSplinePathInterpolator behavior.  This class defines the base class for
 * all Kochanek-Bartels (also known as TCB or Tension-Continuity-Bias) 
 * Spline Path Interpolators.  
 *
 * @since Java3D 1.2
 */

public abstract class KBSplinePathInterpolator extends Interpolator {
  
    private   int             keysLength;
    protected KBKeyFrame[]    keyFrames;

    /**
     * This value is the distance between knots 
     * value which can be used in further calculations by the subclass.
     */
    protected float currentU;

    /**
     * 
     */
    protected int lowerKnot;
    protected int upperKnot;

    // non-public, default constructor used by cloneNode
    KBSplinePathInterpolator() {
    }
  
    /**
     * Constructs a new KBSplinePathInterpolator object that interpolates
     * between keyframes. It takes at least two key frames. The first key 
     * frame's knot must have a value of 0.0 and the last knot must have a
     * value of 1.0.  An intermediate key frame with index k must have a 
     * knot value strictly greater than the knot value of a key frame with 
     * index less than k. Once this constructor has all the valid key frames
     * it creates its own list of key fames that duplicates the first key frame
     * at the beginning of the list and the last key frame at the end of the
     * list. 
     * @param alpha the alpha object for this interpolator
     * @param keys an array of KBKeyFrame. Requires at least two key frames.
     */
    public KBSplinePathInterpolator(Alpha alpha, KBKeyFrame keys[]) {
	super(alpha);

        // Make sure that we have at least two key frames
        keysLength = keys.length;
        if (keysLength < 2) {
	    throw new IllegalArgumentException(J3dUtilsI18N.getString("KBSplinePathInterpolator0"));

        }

        // Make sure that the first key frame's knot is equal to 0.0 
	if (keys[0].knot < -0.0001 || keys[0].knot > 0.0001) {
	    throw new IllegalArgumentException(J3dUtilsI18N.getString("KBSplinePathInterpolator1"));
	}
	
        // Make sure that the last key frames knot is equal to 1.0 
	if (keys[keysLength-1].knot -1.0 < -0.0001 || keys[keysLength-1].knot -1.0 > 0.0001) {
	    throw new IllegalArgumentException(J3dUtilsI18N.getString("KBSplinePathInterpolator2"));
	}

        // Make sure that all the knots are in sequence 
	for (int i = 0; i < keysLength; i++)  {
	    if (i>0 && keys[i].knot < keys[i-1].knot) {
		throw new IllegalArgumentException(J3dUtilsI18N.getString("KBSplinePathInterpolator3"));
	    }
	}

        // Make space for a leading and trailing key frame in addition to 
        // the keys passed in
        keyFrames = new KBKeyFrame[keysLength+2];
        keyFrames[0] = new KBKeyFrame();
        keyFrames[0] = keys[0];
        for (int i = 1; i < keysLength+1; i++) {
           keyFrames[i] = keys[i-1]; 
        }
        keyFrames[keysLength+1] = new KBKeyFrame();
        keyFrames[keysLength+1] = keys[keysLength-1]; 

        // Make key frame length reflect the 2 added key frames
        keysLength += 2;
    }
  
    /**
     * This method retrieves the length of the key frame array.
     * @return the number of key frames 
     */
    public int getArrayLength(){
	return keysLength;
    }
  
    /**
     * This method retrieves the key frame at the specified index.
     * @param index the index of the key frame requested
     * @return the key frame at the associated index
     */
    public KBKeyFrame getKeyFrame (int index) {

        // We add 1 to index because we have added a leading keyFrame
	return this.keyFrames[index+1];
    }

    /**
     * This method computes the bounding knot indices and interpolation value
     * "CurrentU" given the current value of alpha and the knots[] array.
     */
    protected void computePathInterpolation() {
	float value = (this.getAlpha()).value();

        // skip knots till we find the two we fall between  
        int i = 1;
	int len = keysLength - 2;
        while ((value > keyFrames[i].knot) && (i < len)) {
          i++;
        }
	
        if (i == 1) {
            currentU = 0f;
            lowerKnot = 1;
            upperKnot = 2;
        } else {
           currentU = (value - keyFrames[i-1].knot)/
                      (keyFrames[i].knot - keyFrames[i-1].knot);
           lowerKnot = i-1;
           upperKnot = i;
        }
    }

    public void duplicateNode(Node originalNode, boolean forceDuplicate) {
        super.duplicateNode(originalNode, forceDuplicate);
        KBSplinePathInterpolator originalSpline = 
                                  (KBSplinePathInterpolator) originalNode;
        setAlpha(originalSpline.getAlpha());
        keysLength = originalSpline.keysLength;
        keyFrames = new KBKeyFrame[keysLength];
        System.arraycopy(originalSpline.keyFrames, 0,
                         keyFrames, 0, keysLength);
    }

}
