/*
 *	@(#)KBRotPosScaleSplinePathInterpolator.java 1.5 01/01/11 07:22:00
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.utils.behaviors.interpolators;

import javax.media.j3d.*;
import java.util.*;
import javax.vecmath.*;


/**
 * KBRotPosScaleSplinePathInterpolator; A rotation and position path 
 * interpolation behavior node using Kochanek-Bartels cubic splines
 * (also known as TCB or Tension-Continuity-Bias Splines).
 *
 * @since Java3D 1.2
 */

/**
 * KBRotPosScaleSplinePathInterpolator behavior.  This class defines a 
 * behavior that varies the rotational, translational, and scale components 
 * of its target TransformGroup by using the Kochanek-Bartels cubic spline 
 * interpolation to interpolate among a series of key frames
 * (using the value generated by the specified Alpha object).  The
 * interpolated position, orientation, and scale are used to generate
 * a transform in the local coordinate system of this interpolator.  
 */

public class KBRotPosScaleSplinePathInterpolator 
                                        extends KBSplinePathInterpolator {

    TransformGroup target;
    Transform3D    axis        = new Transform3D();
    Transform3D    axisInverse = new Transform3D();
    Transform3D    rotation    = new Transform3D();
    Transform3D    temp        = new Transform3D();
    Matrix4d       pitchMat    = new Matrix4d();   // pitch matrix
    Matrix4d       bankMat     = new Matrix4d();   // bank matrix
    Matrix4d       tMat        = new Matrix4d();   // transformation matrix
    Matrix4d       sMat        = new Matrix4d();   // scale matrix
    //Quat4f         iQuat       = new Quat4f();     // interpolated quaternion
    Vector3f       iPos        = new Vector3f();   // interpolated position 
    Point3f        iScale      = new Point3f();    // interpolated scale 
    float          iHeading, iPitch, iBank;        // interpolated heading,
                                                   // pitch and bank

    KBCubicSplineCurve   cubicSplineCurve = new KBCubicSplineCurve();
    KBCubicSplineSegment cubicSplineSegments[]; 
    int                  numSegments;
    int                  currentSegmentIndex;
    KBCubicSplineSegment currentSegment;

    // non-public, default constructor used by cloneNode
    KBRotPosScaleSplinePathInterpolator() {
    }

    /**
      * Constructs a new KBRotPosScaleSplinePathInterpolator object that 
      * varies the rotation, translation, and scale of the target 
      * TransformGroup's transform.  At least 2 key frames are required for 
      * this interpolator. The first key
      * frame's knot must have a value of 0.0 and the last knot must have a
      * value of 1.0.  An intermediate key frame with index k must have a
      * knot value strictly greater than the knot value of a key frame with
      * index less than k.
      * @param alpha the alpha object for this interpolator
      * @param target the TransformGroup node affected by this interpolator
      * @param axisOfRotPosScale the transform that specifies the local
      * coordinate system in which this interpolator operates.
      * @param keys an array of key frames that defien the motion path 
      */
    public KBRotPosScaleSplinePathInterpolator(Alpha alpha,
				       TransformGroup target,
				       Transform3D axisOfRotPosScale,
				       KBKeyFrame keys[]) {
	super(alpha, keys);

	this.target = target;
	axis.set(axisOfRotPosScale);
	axisInverse.invert(axis);

        // Create a spline curve using the derived key frames
        cubicSplineCurve = new KBCubicSplineCurve(this.keyFrames); 
        numSegments = cubicSplineCurve.numSegments;

    }

    /**
      * This method sets the axis of rotation, translation, and scale
      * for this interpolator, RotPosScale, which defines the local
      * coordinate system in which this interpolator operates.
      * @param axisOfRotPosScale the interpolators axis of RotPosScale 
      */ 
    public void setAxisOfRotPosScale(Transform3D axisOfRotPosScale) {
        this.axis.set(axisOfRotPosScale);
        this.axisInverse.invert(this.axis);
    }
       
    /**
      * This method retrieves this interpolator's axis of rotation, translation,
      * and scale.
      * @return the interpolator's axis of RotPosScale
      */ 
    public Transform3D getAxisOfRotPosScale() {
        return new Transform3D(this.axis);
    }

    /**
      * This method sets the target TransformGroup for this interpolator.
      * @param target the target TransformGroup reference
      */
    public void setTarget(TransformGroup target) {
	this.target = target;
    }

    /**
      * This method retrieves this interpolator's target TransformGroup
      * reference.
      * @return the interpolator's target TransformGroup reference
      */
    public TransformGroup getTarget() {
	return target;
    }

    /*
     * The KBRotPosScaleSplinePathInterpolator's initialize routine uses 
     * the default initialization routine.
     */

    /**
     * This method is invoked by the behavior scheduler every frame.  It maps
     * the alpha value that corresponds to the current time into translation,
     * rotation, and scale values, computes a transform based on these values,
     * and updates the specified TransformGroup node with this new transform.
     * @param criteria enumeration of criteria that have triggered this wakeup
     */
    public void processStimulus(Enumeration criteria) {
	/* Handle stimulus */

	if (this.getAlpha() != null) {

            // compute the current value of u from alpha and the 
            // determine lower and upper knot points
	    computePathInterpolation();

            // Determine the segment within which we will be interpolating
            currentSegmentIndex = this.lowerKnot - 1;
    
            // if we are at the start of the curve 
	    if (currentSegmentIndex == 0 && currentU == 0f) {

                iHeading = keyFrames[1].heading;
                iPitch = keyFrames[1].pitch;
                iBank = keyFrames[1].bank;
                iPos.set(keyFrames[1].position);
	        iScale.set(keyFrames[1].scale);

            // if we are at the end of the curve 
	    } else if (currentSegmentIndex == (numSegments-1) && 
                       currentU == 1.0) {

                iHeading = keyFrames[upperKnot].heading;
                iPitch = keyFrames[upperKnot].pitch;
                iBank = keyFrames[upperKnot].bank;
                iPos.set(keyFrames[upperKnot].position);
	        iScale.set(keyFrames[upperKnot].scale);

            // if we are somewhere in between the curve
            } else {

                // Get a reference to the current spline segment i.e. the
                // one bounded by lowerKnot and upperKnot 
                currentSegment 
                         = cubicSplineCurve.getSegment(currentSegmentIndex);

                // interpolate quaternions 
                iHeading = currentSegment.getInterpolatedHeading (currentU);
                iPitch = currentSegment.getInterpolatedPitch (currentU);
                iBank = currentSegment.getInterpolatedBank (currentU);

                // interpolate position
                currentSegment.getInterpolatedPositionVector(currentU,iPos); 

                // interpolate position
                currentSegment.getInterpolatedScale(currentU,iScale);

	    }

            // Generate a transformation matrix in tMat using interpolated 
            // heading, pitch and bank
            pitchMat.setIdentity();
            pitchMat.rotX(-iPitch);
            bankMat.setIdentity();
            bankMat.rotZ(iBank);
	    tMat.setIdentity();
            tMat.rotY(-iHeading);
            tMat.mul(pitchMat);
            tMat.mul(bankMat);

            // TODO: Vijay - Handle Non-Uniform scale
            // Currently this interpolator does not handle non uniform scale 
            // We cheat by just taking the x scale component

            // Scale the transformation matrix
	    sMat.set((double)iScale.x);
	    tMat.mul(sMat);

	    // Set the translation components.
	    tMat.m03 = iPos.x;
	    tMat.m13 = iPos.y;
	    tMat.m23 = iPos.z;
	    rotation.set(tMat);
    
	    // construct a Transform3D from:  axis * rotation * axisInverse 
	    temp.mul(axis, rotation);
	    temp.mul(temp, axisInverse);
	    target.setTransform(temp); 
	}
        
	wakeupOn(defaultWakeupCriterion);

    }

        // Needed so cloneTree will operate correctly
    public Node cloneNode(boolean forceDuplicate) {
        KBRotPosScaleSplinePathInterpolator spline =
          new KBRotPosScaleSplinePathInterpolator();

        spline.duplicateNode(this, forceDuplicate);
        return spline;
    }
    
    public void duplicateNode(Node originalNode, boolean forceDuplicate) {
        super.duplicateNode(originalNode, forceDuplicate);
	
	KBRotPosScaleSplinePathInterpolator interpolator =
	    (KBRotPosScaleSplinePathInterpolator)originalNode;
        setAxisOfRotPosScale(interpolator.axis);
        target = interpolator.target;
        cubicSplineCurve = new KBCubicSplineCurve(interpolator.keyFrames); 
        numSegments = cubicSplineCurve.numSegments;
    }

    public void updateNodeReferences(NodeReferenceTable refTable) {
        TransformGroup tg =
            (TransformGroup)refTable.getNewObjectReference(target);
        setTarget(tg);
    }

}


