/*
 *	@(#)ObjectFileMaterials.java 1.10 01/01/11 07:26:50
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.j3d.loaders.objectfile;

import javax.media.j3d.Appearance;
import javax.media.j3d.Material;
import javax.media.j3d.Shape3D;
import javax.vecmath.Color3f;
import com.sun.j3d.loaders.ParsingErrorException;
import com.sun.j3d.loaders.IncorrectFormatException;
import java.io.FileNotFoundException;
import java.io.StringReader;
import java.io.Reader;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.BufferedInputStream;
import java.io.InputStreamReader;
import java.util.Hashtable;
import com.sun.j3d.loaders.objectfile.DefaultMaterials;
import java.net.URL;
import java.awt.Toolkit;
import java.awt.Image;
import java.awt.image.BufferedImage;
import javax.media.j3d.Texture2D;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import java.awt.image.DataBufferInt;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.TexCoordGeneration;
import java.security.AccessController;
import java.security.PrivilegedAction;
import javax.media.j3d.GeometryArray;
import com.sun.j3d.utils.image.TextureLoader;
import javax.media.j3d.TransparencyAttributes;


class ObjectFileMaterials implements ImageObserver
{
  // DEBUG
  // 1 = Name of materials
  // 16 = Tokens
  private static final int DEBUG = 0;

  private String curName = null;
  private ObjectFileMaterial cur = null;

  private Hashtable materials;			// key=String name of material
						// value=ObjectFileMaterial

  private String basePath;
  private boolean fromUrl;


  private class ObjectFileMaterial {

    public Color3f Ka;
    public Color3f Kd;
    public Color3f Ks;
    public int illum;
    public float Ns;
    public Texture2D t;
    public boolean transparent;

    public void ObjectFileMaterial()
    {
      Ka = null;
      Kd = null;
      Ks = null;
      illum = -1;
      Ns = -1.0f;
      t = null;
    } // End of ObjectFileMaterial
  }



  void assignMaterial(String matName, Shape3D shape)
  {
    ObjectFileMaterial p = null;

    if ((DEBUG & 1) != 0) System.out.println("Color " + matName);
    
    Material m = new Material();
    p = (ObjectFileMaterial)materials.get(matName);
    Appearance a = new Appearance();

    if (p != null) {
      // Set ambient & diffuse color
      if (p.Ka != null) m.setAmbientColor(p.Ka);
      if (p.Kd != null) m.setDiffuseColor(p.Kd);

      // Set specular color
      if ((p.Ks != null) && (p.illum != 1)) m.setSpecularColor(p.Ks);
      else if (p.illum == 1) m.setSpecularColor(0.0f, 0.0f, 0.0f);

      if (p.illum >= 1) m.setLightingEnable(true);
      else if (p.illum == 0) m.setLightingEnable(false);

      if (p.Ns != -1.0f) m.setShininess(p.Ns);

      if (p.t != null) {
	a.setTexture(p.t);
	// Create Texture Coordinates if not already present
	if ((((GeometryArray)shape.getGeometry()).getVertexFormat() &
	     GeometryArray.TEXTURE_COORDINATE_2) == 0) {
	  TexCoordGeneration tcg = new TexCoordGeneration();
	  a.setTexCoordGeneration(tcg);
	}
      }

      if (p.transparent) 
	a.setTransparencyAttributes(
	  new TransparencyAttributes(TransparencyAttributes.NICEST, 0.0f));
    }
    a.setMaterial(m);
    if ((DEBUG & 1) != 0) System.out.println(m);
    shape.setAppearance(a);
  } // End of assignMaterial



  private void readName(ObjectFileParser st)
  {
    if (!st.getToken()) return;

    if (st.ttype == ObjectFileParser.TT_WORD) {

      if (curName != null) materials.put(curName, cur);
      curName = new String(st.sval);
      cur = new ObjectFileMaterial();
    }
    st.skipToNextLine();
  } // End of readName



  private void readAmbient(ObjectFileParser st)
  {
    Color3f p = new Color3f();

    if (st.getNumber()) {
      p.x = (float)st.nval;
      if (st.getNumber()) {
        p.y = (float)st.nval;
        if (st.getNumber()) {
          p.z = (float)st.nval;
        } else return;
      } else return;
    } else return;

    cur.Ka = p;

    st.skipToNextLine();
  } // End of readAmbient



  private void readDiffuse(ObjectFileParser st)
  {
    Color3f p = new Color3f();

    if (st.getNumber()) {
      p.x = (float)st.nval;
      if (st.getNumber()) {
        p.y = (float)st.nval;
        if (st.getNumber()) {
          p.z = (float)st.nval;
        } else return;
      } else return;
    } else return;

    cur.Kd = p;

    st.skipToNextLine();
  } // End of readDiffuse



  private void readSpecular(ObjectFileParser st)
  {
    Color3f p = new Color3f();

    if (st.getNumber()) {
      p.x = (float)st.nval;
      if (st.getNumber()) {
        p.y = (float)st.nval;
        if (st.getNumber()) {
          p.z = (float)st.nval;
        } else return;
      } else return;
    } else return;

    cur.Ks = p;

    st.skipToNextLine();
  } // End of readSpecular



  private void readIllum(ObjectFileParser st)
  {
    float f;

    if (st.getNumber()) {
      cur.illum = (int)st.nval;
    }

    st.skipToNextLine();
  } // End of readSpecular



  private void readShininess(ObjectFileParser st)
  {
    float f;

    if (st.getNumber()) {
      cur.Ns = (float)st.nval;
      if (cur.Ns < 1.0f) cur.Ns = 1.0f;
      else if (cur.Ns > 128.0f) cur.Ns = 128.0f;
    }

    st.skipToNextLine();
  } // End of readSpecular



  private void scaleAndLoad(BufferedImage bi, boolean luminance, boolean alpha)
  {
    String s = null;
    if (luminance && alpha) s = new String("LUM8_ALPHA8");
    else if (luminance) s = new String("LUMINANCE");
    else if (alpha) s = new String("RGBA");
    else s = new String("RGB");

    TextureLoader tl = new TextureLoader(bi, s, TextureLoader.GENERATE_MIPMAP);
    cur.t = (Texture2D)tl.getTexture();
    cur.t.setMinFilter(Texture2D.MULTI_LEVEL_LINEAR);
  } // end of scaleAndLoad



  public void readMapKd(ObjectFileParser st)
  {
    // Filenames are case sensitive
    st.lowerCaseMode(false);

    if (!st.getToken()) return;

    st.lowerCaseMode(true);

    if (st.ttype == ObjectFileParser.TT_WORD) {
      // Convert filename to lower case for extension comparisons
      String suffix =
	st.sval.substring(st.sval.lastIndexOf('.') + 1).toLowerCase();

      if ((suffix.equals("jpg")) || (suffix.equals("gif")) ||
	  (suffix.equals("jpeg")) || (suffix.equals("jpe")) ||
	  (suffix.equals("xbm"))) {

	// Use Java toolkit to load jpeg file
	final Image im[] = new Image[1];
	final Toolkit tk = Toolkit.getDefaultToolkit();
	final String fn = new String(basePath + st.sval);
	AccessController.doPrivileged(
	  new PrivilegedAction() {
	    public Object run() {
	      im[0] = tk.getImage(fn);
	      return null;
	    }
	  }
	);

        // Tell it to start loading
        tk.prepareImage(im[0], -1, -1, this);

        // Wait until it's fully loaded
	boolean doneLoading = false;
	int flags = 0;
        try {
	  do {
	    flags = tk.checkImage(im[0], -1, -1, this);
	    doneLoading = (flags & (ALLBITS | ERROR | ABORT)) != 0;
	    Thread.sleep(10);
	  } while (!doneLoading);
	}
	catch (InterruptedException e) {
	  // If Thread.sleep() is interrupted, the texture won't be used
	}

	boolean loadError = (flags & (ERROR | ABORT)) != 0;
	if (doneLoading && !loadError) {

	  int w = im[0].getWidth(this);
	  int h = im[0].getHeight(this);

          // Need to get the data from an Image into a BufferedImage
	  BufferedImage bi = new BufferedImage(w, h,
	    BufferedImage.TYPE_INT_ARGB);
	  int intPixels[] =
	    ((DataBufferInt)bi.getRaster().getDataBuffer()).getData();
	  PixelGrabber pg =
	    new PixelGrabber(im[0], 0, 0, w, h, intPixels, 0, w);
	  try {
	    pg.grabPixels();
	    scaleAndLoad(bi, false, false);
	  }
	  catch (InterruptedException e) {
	    // if grabPixels gets interrupted, the texture map won't get used
	  }
	}
      } else if ((suffix.equals("int")) || (suffix.equals("inta")) ||
		 (suffix.equals("rgb")) || (suffix.equals("rgba")) ||
		 (suffix.equals("bw")) || (suffix.equals("sgi"))) {
	try {
	  RgbFile f = new RgbFile(basePath + st.sval);
	  BufferedImage bi = f.getImage();
	  cur.transparent = suffix.equals("inta") || suffix.equals("rgba");
	  if (bi != null) {
	    scaleAndLoad(bi, suffix.equals("int") || suffix.equals("inta"),
	      cur.transparent);
	  }
	}
	catch (FileNotFoundException e) {
	  // Texture won't get loaded if file can't be found
	}
      }
    }
    st.skipToNextLine();
  } // End of readMapKd



  private void readFile(ObjectFileParser st)
  {
    int t;
    if (!st.getToken()) return;
    while (st.ttype != ObjectFileParser.TT_EOF) {

      // Print out one token for each line
      if ((DEBUG & 16) != 0) {
        System.out.print("Token ");
        if (st.ttype == ObjectFileParser.TT_EOL) System.out.println("EOL");
        else if (st.ttype == ObjectFileParser.TT_WORD)
          System.out.println(st.sval);
        else System.out.println((char)st.ttype);
      }

      if (st.ttype == ObjectFileParser.TT_WORD) {
	if (st.sval.equals("newmtl")) {
	  readName(st);
	} else if (st.sval.equals("ka")) {
	  readAmbient(st);
	} else if (st.sval.equals("kd")) {
	  readDiffuse(st);
	} else if (st.sval.equals("ks")) {
	  readSpecular(st);
	} else if (st.sval.equals("illum")) {
	  readIllum(st);
	} else if (st.sval.equals("d")) {
	  if (!st.skipToNextLine()) return;
	} else if (st.sval.equals("ns")) {
	  readShininess(st);
	} else if (st.sval.equals("tf")) {
	  if (!st.skipToNextLine()) return;
	} else if (st.sval.equals("sharpness")) {
	  if (!st.skipToNextLine()) return;
	} else if (st.sval.equals("map_kd")) {
	  readMapKd(st);
	} else if (st.sval.equals("map_ka")) {
	  if (!st.skipToNextLine()) return;
	} else if (st.sval.equals("map_ks")) {
	  if (!st.skipToNextLine()) return;
	} else if (st.sval.equals("map_ns")) {
	  if (!st.skipToNextLine()) return;
	} else if (st.sval.equals("bump")) {
	  if (!st.skipToNextLine()) return;
	}
      }

      if (!st.skipToNextLine()) return;

      // Get next token
      if (!st.getToken()) return;
    }
    if (curName != null) materials.put(curName, cur);
  } // End of readFile



  void readMaterialFile(boolean fromUrl, String basePath, String fileName)
  {
    Reader reader;

    this.basePath = basePath;
    this.fromUrl = fromUrl;

    try {
      if (fromUrl) {
	reader = (Reader)(new InputStreamReader(
	  new BufferedInputStream(
	    (new URL(basePath + fileName).openStream()))));
      } else {
	reader = new BufferedReader(new FileReader(basePath + fileName));
      }
    }
    catch (Exception e) {
      // couldn't find it - ignore mtllib
      return;
    }
    if ((DEBUG & 1) != 0)
      System.out.println("Material file: " + basePath + fileName);

    ObjectFileParser st = new ObjectFileParser(reader);
    readFile(st);
  }  // End of readMaterialFile



  ObjectFileMaterials()
  {
    Reader reader = new StringReader((new DefaultMaterials()).materials);

    ObjectFileParser st = new ObjectFileParser(reader);
    materials = new Hashtable(50);
    readFile(st);
  } // End of ObjectFileMaterials



  /**
   * Implement the ImageObserver interface.  Needed to load jpeg and gif
   * files using the Toolkit.
   */
  public boolean imageUpdate(Image img, int flags, int x, int y, int w, int h)
  {
    return (flags & (ALLBITS | ABORT)) == 0;
  } // End of imageUpdate

} // End of class ObjectFileMaterials

// End of file ObjectFileMaterials.java
