/*
 *	@(#)JavaSoundMixer.java 1.60 01/02/13 08:58:37
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

/*
 * Audio device driver using Java Sound Mixer Engine
 */

package com.sun.j3d.audioengines.javasound;

import java.net.URL;
import java.io.InputStream;
import javax.vecmath.*;
import javax.media.j3d.*;
import com.sun.j3d.audioengines.*;
import java.util.ArrayList;

/**
 * The JavaSoundMixer Class defines an audio output device that accesses
 * JavaSound JavaSoundMixer functionality Hae stream data.
 */

public class JavaSoundMixer extends AudioEngine3D
{
    /**
     ** Debug print mechanism for Sound nodes
     **/ 
    static final
    boolean  debugFlag = false;

    static final // 'static final' so internal error prints will not be compiled
    boolean internalErrors = false;
 
    void debugPrint(String message) {
        if (debugFlag)
            System.out.println(message);
    }
 
    void debugPrintln(String message) {
        if (debugFlag)
            System.out.println(message);
    }

    // Determines method to call for added or setting sound into ArrayList
    static final int ADD_TO_LIST = 1;
    static final int SET_INTO_LIST = 2;

    /**
     * Reverb Parameters
     *
     * dirty flag checked and cleared by render()
     */
    static int REFLECTION_COEFF_CHANGED =  1;
    static int REVERB_DELAY_CHANGED     =  2;
    static int REVERB_ORDER_CHANGED     =  4; 

    int      reverbDirty = 0xFFFF;   
    int      lastReverbSpeed = 0; 
    boolean  reverbOn = false;
    int      reverbType = 1;  // Reverb type 1 equals NONE in JavaSound engine

    /*
     * Construct a new JavaSoundMixer with the specified P.E.
     * @param physicalEnvironment the physical environment object where we
     * want access to this device.
     */  
    public JavaSoundMixer(PhysicalEnvironment physicalEnvironment ) {
        super(physicalEnvironment);
    }

    /*
     * OVERRIDEN methods from AudioEngine
     */
    /**
     * Query total number of channels available for sound rendering
     * for this audio device.
     * @returns number of maximum voices play simultaneously on JavaSound Mixer.
     */  
    public int getTotalChannels() {
        // totalChannels = com.sun.j3d.audio.J3DHaeStream.getTotalChannels();
        // return(totalChannels);
        return(com.sun.j3d.audio.J3DHaeStream.getTotalChannels());
    }
 
     /* 
      * NEW interfaces to mixer/engine specific methods 
      */
    /**
     * Code to initialize the device
     * @return flag: true is initialized sucessfully, false if error
     */
    public boolean initialize() {
        if (com.sun.j3d.audio.J3DHaeStream.initialize()) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer: J3DHaeStream.initialize returned true");
            return true;
        }
        else {
            if (debugFlag)
                debugPrintln("JavaSoundMixer: J3DHaeStream.initialize returned false");
            return false;
        }
    }

    /**
     * Code to close the device
     * @return flag: true is closed sucessfully, false if error
     */
    public boolean close() {
        if (com.sun.j3d.audio.J3DHaeStream.close()) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer: J3DHaeStream.close returned true");
            return true;
        }
        else {
            if (debugFlag)
                debugPrintln("JavaSoundMixer: J3DHaeStream.close returned false");
            return false;
        }
    }

    // Initialize the AudioContainer then create the sample data on audio device
    // Return index
    int loadSound(MediaContainer soundData) {
        int   index = JSSample.NULL_SAMPLE;
        int   error;
        int   dataType = com.sun.j3d.audio.JavaSoundParams.UNSUPPORTED_DATA_TYPE;
        // %%% TODO MediaContiner may not have capability flags set for query
        //     if need be change all calls to initAudioContainer param 
        String path = soundData.getURLString();
        URL url = soundData.getURLObject();
        InputStream inputStream = soundData.getInputStream();
        boolean cacheFlag = soundData.getCacheEnable();
        if (url != null) {
            index = com.sun.j3d.audio.J3DHaeStream.initAudioContainer(
                            url, cacheFlag);
        }
        else if (path != null)  {
            // generate url from string, and pass url to driver
            if (debugFlag) {
                debugPrint("AudioEngine3D.loadSound with path = " + path);
            }
            try  {
                url = new URL(path);
            }
            catch (Exception e) {
                // %%% TODO throw an exception from this package
                //     How to access javax.media.j3d.J3dI18N.getString from this package
                // throw new SoundException(javax.media.j3d.J3dI18N.getString("JavaSoundMixer0"));
                return JSSample.NULL_SAMPLE;
            }
            index = com.sun.j3d.audio.J3DHaeStream.initAudioContainer(
                            url, cacheFlag);
        }
        else if (inputStream != null) {
            index = com.sun.j3d.audio.J3DHaeStream.initAudioContainer(
                            inputStream, cacheFlag);
        }
        
        if (index == JSSample.NULL_SAMPLE) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer.loadSound loadSound failed");
            return JSSample.NULL_SAMPLE;
        }

        if (index < 0) {  // return value -1 denotes problem initializing ACIS
            if (debugFlag)
                debugPrintln("JavaSoundMixer loadSound " +
                           "initAudioContainer failed");
            return  JSSample.NULL_SAMPLE;
        }
        dataType = com.sun.j3d.audio.J3DHaeStream.getDataType(index);
        if (dataType == com.sun.j3d.audio.JavaSoundParams.UNSUPPORTED_DATA_TYPE) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer loadSound " +
                           "encountered unknown data type");
            return  JSSample.NULL_SAMPLE;
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if (debugFlag) {
                debugPrintln("JavaSoundMixer.loadSound dataType = BUFFERED");
                debugPrintln("               loadSound call Clip.loadSample");
            }
            if (inputStream != null)
                error = com.sun.j3d.audio.J3DHaeClip.loadSample(inputStream, index);
            else 
                error = com.sun.j3d.audio.J3DHaeClip.loadSample(url, index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            if (debugFlag) {
                debugPrintln("JavaSoundMixer.loadSound dataType = STREAMING");
                debugPrintln("               loadSound call Stream.loadSample");
            }
            if (inputStream != null)
                error = com.sun.j3d.audio.J3DHaeStream.loadSample(inputStream, index);
            else 
                error = com.sun.j3d.audio.J3DHaeStream.loadSample(url, index);
        }
        else  {  // dataType = XXXX_MIDI_DATA
            if (debugFlag) {
                debugPrintln("JavaSoundMixer.loadSound dataType = MIDI");
                debugPrintln("               loadSound call Midi.loadSample");
            }
            if (inputStream != null)
                error = -1;  // should not be possible, internal error...
            else 
                error = com.sun.j3d.audio.J3DHaeMidi.loadSample(url, index);
        }  
        if (error < 0) {
            if (debugFlag) {
                debugPrintln("               loadSample returns error");
                debugPrintln("               so loadSound returns NULL_SAMPLE");
            }
            return  JSSample.NULL_SAMPLE;
        }
        else {
            if (debugFlag) {
                debugPrintln("               loadSound returns error");
            }
            return index;
        }
    }


    /*
     * OVERRIDEN methods from AudioEngine3D
     */

    /**
     * Code to load sound data into a channel of device channel
     * Load sound as one or mores sample into the Java Sound Mixer:
     *   a) as either a STREAM or CLIP based on whether cached is enabled
     *   b) positional and directional sounds use THREE samples per
     *      sound so index return for the first sample associated with
     *      prepared sound is incremented by 2.
     * @return index to the (first or only) sample.
     */
    public int prepareSound(int soundType, MediaContainer soundData) {
        int   index = JSSample.NULL_SAMPLE;
        if (soundData == null) {
            index = this.loadSound(null);
            return index;
        }

        JSSample sample = null;
        JSPositionalSample posSample = null;
        JSDirectionalSample dirSample = null;
        int  methodType  = ADD_TO_LIST;
        index = this.loadSound(soundData);
        int   dataType;
        dataType = com.sun.j3d.audio.J3DHaeStream.getDataType(index);
        if (dataType==com.sun.j3d.audio.JavaSoundParams.UNSUPPORTED_DATA_TYPE) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer.prepareSound get dataType failed");
            return JSSample.NULL_SAMPLE;
        }

        /*
         * Is array list capacity extended automatically when add used?
         */
        synchronized(samples) {
          int samplesSize = samples.size();
          if (index >= samplesSize) {
            samples.ensureCapacity(index+1);
            methodType = ADD_TO_LIST;
            if (debugFlag)
                debugPrintln("samples increased capacity to " + (index+1) +
                           ", method type ADD"); 
            // put null samples into this sparce list
            for (int i=samplesSize; i<index; i++) {
                if (debugFlag)
                    debugPrintln("add null reference to sample, list");
                samples.add(i, null);
            }
          }
          else {
            methodType = SET_INTO_LIST;
            if (debugFlag)
                debugPrintln("samples increased capacity to " + (index+1) +
                           ", method type SET");
          }

          if (soundType == AudioDevice3D.CONE_SOUND) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer.prepareSound soundType = CONE");
            dirSample = new JSDirectionalSample();
            if (methodType == SET_INTO_LIST)
                 samples.set(index, dirSample);
            else
                 samples.add(index, dirSample);
            sample = dirSample;
            posSample = dirSample;

          }
          else if (soundType == AudioDevice3D.POINT_SOUND) {
            if (debugFlag)
                debugPrintln("JavaSoundMixer.prepareSound soundType = POINT");
            posSample = new JSPositionalSample();
            if (methodType == SET_INTO_LIST)
                 samples.set(index, posSample);
            else
                 samples.add(index, posSample);
            sample = posSample;
          }
          else {  // soundType == AudioDevice3D.BACKGROUND_SOUND
            if (debugFlag)
                debugPrintln("JavaSoundMixer.prepareSound soundType = BACKGROUND");
            sample = new JSSample();
            if (methodType == SET_INTO_LIST)
                 samples.set(index, sample);
            else
                 samples.add(index, sample);
          }

          /*
           * Point and Cone Sounds (Positional and Directional Sample) 
           * use TWO samples per sound in this implementation.
           * Load this same sound as another sample in another channel
           * for stereo processing later.
           */
          // %%% TODO for now only Point and Cone MIDI files are processed
          //     as Background sound
          if ( (posSample != null) &&
             (dataType!=com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA &&
              dataType!=com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA) )
          {
            int secondIndex = JSSample.NULL_SAMPLE;
            int reverbIndex = JSSample.NULL_SAMPLE;
            int errorFlag;
            int errorFlag2;
            secondIndex = this.loadSound(soundData);
            reverbIndex = this.loadSound(soundData);
            if (secondIndex == JSSample.NULL_SAMPLE  ||
                reverbIndex == JSSample.NULL_SAMPLE) {
                /**
                 * Error getting a second or third sample so clear out the 
                 * first (and second) and return an error
                 */
                if (debugFlag)
                    debugPrintln(
                       "JavaSoundMixer.prepareSound add'l loadSounds failed");
                clearSound(index);
                if (secondIndex != JSSample.NULL_SAMPLE)
                    clearSound(secondIndex);
                samples.set(index, null);
                return JSSample.NULL_SAMPLE;
            }

            if (debugFlag) {
                debugPrintln("load SECOND and REVERB INDICES = " + 
                            secondIndex + ", " + reverbIndex);
                debugPrintln("samples size = " +samples.size());
            }

            int highestIndex = -1;
            if (secondIndex < reverbIndex)
                highestIndex = reverbIndex;
            else  // secondIndex > reverbIndex
                highestIndex = secondIndex;
            int oldSamplesSize = samples.size();
            if (highestIndex >= oldSamplesSize) {
                if (debugFlag)
                    debugPrintln("samples list increased capacity to " +
                           (highestIndex+1));
                samples.ensureCapacity(highestIndex+1);
            }

            if (secondIndex >= oldSamplesSize) {
                methodType = ADD_TO_LIST;
                if (debugFlag)
                    debugPrintln(" ADD second sample, index " + secondIndex +
                               ", to list with size " + samples.size());
                samples.add(secondIndex, sample);
            }
            else {
                methodType = SET_INTO_LIST;
                if (debugFlag)
                    debugPrintln(" SET second sample, index " + secondIndex +
                               ", into list with size " + samples.size());
                samples.set(secondIndex, sample);
            }
            posSample.setSecondIndex(secondIndex);

            if (reverbIndex >= oldSamplesSize) {
                methodType = ADD_TO_LIST;
                if (debugFlag)
                    debugPrintln(" ADD reverb sample, index " + reverbIndex +
                               ", to list with size " + samples.size());
                samples.add(reverbIndex, sample);
            }
            else {
                methodType = SET_INTO_LIST;
                if (debugFlag)
                    debugPrintln(" SET reverb sample, index " + reverbIndex +
                               ", into list with size " + samples.size());
                samples.set(reverbIndex, sample);
            }
            posSample.setReverbIndex(reverbIndex);
          }

        } // synchronized samples ArrayList

        /*
         * Since no error occurred while loading, save all the characteristics
         * for the sound in the sample.
         */
        sample.setDirtyFlags(0xFFFF);
        sample.setSoundType(soundType);
        sample.setSoundData(soundData);

        int rate = com.sun.j3d.audio.J3DHaeStream.getSampleRate(index);
        sample.setSampleRate(rate);
        
        sample.setDataType(dataType);
        if (debugFlag)  {
            debugPrint("               prepareSound dataType = "+dataType);
            debugPrintln("JavaSoundMixer.prepareSound returned "+index);
        }
        return index;
    }

    /**
     * Clears the fields associated with sample data for this sound.
     * %%% TODO call J3DHaeXXXX clear vectors method
     */  
    public void clearSound(int index) {
        JSSample sample = null;
        if ( (sample = (JSSample)getSample(index)) == null)
            return;
        sample.clear();
        synchronized(samples) {
            samples.set(index, null);
        }
    }

    /**
     * Save a reference to the local to virtual world coordinate space
     */
    public void  setVworldXfrm(int index, Transform3D trans) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: setVworldXfrm for index " + index);
        super.setVworldXfrm(index, trans);
        if (debugFlag) {
            double[] matrix = new double[16];
            trans.get(matrix);
            debugPrintln("JavaSoundMixer     column-major transform ");
            debugPrintln("JavaSoundMixer         " + matrix[0]+", "+matrix[1]+
                         ", "+matrix[2]+", "+matrix[3]);
            debugPrintln("JavaSoundMixer         " + matrix[4]+", "+matrix[5]+
                         ", "+matrix[6]+", "+matrix[7]);
            debugPrintln("JavaSoundMixer         " + matrix[8]+", "+matrix[9]+
                         ", "+matrix[10]+", "+matrix[11]);
            debugPrintln("JavaSoundMixer         " + matrix[12]+", "+matrix[13]+
                         ", "+matrix[14]+", "+matrix[15]);
        }
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;
        int   soundType = sample.getSoundType();

        if (soundType == AudioDevice3D.CONE_SOUND) {
            JSDirectionalSample dirSample = null;
            if ((dirSample = (JSDirectionalSample)getSample(index)) == null)
                return;
            dirSample.setXformedDirection();
            dirSample.setXformedPosition();
            // flag that VirtualWorld transform set
            dirSample.setVWrldXfrmFlag(true);
        }
        else if (soundType == AudioDevice3D.POINT_SOUND) {
            JSPositionalSample posSample = null;
            if ((posSample = (JSPositionalSample)getSample(index)) == null)
                return;
            posSample.setXformedPosition();
            // flag that VirtualWorld transform set
            posSample.setVWrldXfrmFlag(true);
        }
        return;
    }

    public void   setPosition(int index, Point3d position) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: setPosition for index " + index);
        super.setPosition(index, position);
        JSPositionalSample posSample = null;
        if ((posSample = (JSPositionalSample)getSample(index)) == null)
            return;
        int   soundType = posSample.getSoundType();
        if ( (soundType == AudioDevice3D.POINT_SOUND) ||
             (soundType == AudioDevice3D.CONE_SOUND) ) {
            posSample.setXformedPosition();
        }
        return;
    }
 
    public void setDirection(int index, Vector3d direction) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: setDirection for index " + index);
        super.setDirection(index, direction);
        JSDirectionalSample dirSample = null;
        if ((dirSample = (JSDirectionalSample)getSample(index)) == null)
            return;
        int   soundType = dirSample.getSoundType();
        if (soundType == AudioDevice3D.CONE_SOUND) {
            dirSample.setXformedDirection();
        }
        return;
    }
 
    public void setReflectionCoefficient(float coefficient) {
        super.setReflectionCoefficient(coefficient);
        reverbDirty |= REFLECTION_COEFF_CHANGED;
        return;
    }

    public void setReverbDelay(float reverbDelay) {
        super.setReverbDelay(reverbDelay);
        reverbDirty |= REVERB_DELAY_CHANGED;
        return;
    }

    public void setReverbOrder(int reverbOrder) {
        super.setReverbOrder(reverbOrder);
        reverbDirty |=  REVERB_ORDER_CHANGED;
        return;
    }

    /*
     * %%% QUESTION if this is used, for now, exclusively, to start a Background
     *      or any single sampled Sounds, why are there if-else cases to handle
     *      Point and Cone sounds??
     * 
     *
     * %%% TODO For now background sounds are not reverberated
     */
    public int   startSample(int index) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: STARTSample for index " + index);

        int returnValue = -1;
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return JSSample.NULL_SAMPLE;

        AuralParameters attribs = getAuralParameters();
        int   soundType = sample.getSoundType();
        boolean muted = sample.getMuteFlag();
        if (muted) {
            if (debugFlag)
                debugPrintln("                            MUTEd start");
            sample.setLoopCount(0);
            sample.leftGain = 0.0f;
            sample.rightGain = 0.0f; 
            sample.leftDelay = 0;
            sample.rightDelay = 0;
            if (soundType != AudioDevice3D.BACKGROUND_SOUND)
                setFilter(index, false, Sound.NO_FILTER);
        }
        else {
            // %%% KLUDGE 
            // %%% Setting AuralParameters before you start sound doesn't work
            sample.render(sample.getDirtyFlags(), getView(), attribs);
            scaleSampleRate(index, sample.rateRatio);
            // filtering
            if (soundType != AudioDevice3D.BACKGROUND_SOUND)
                setFilter(index, sample.getFilterFlag(), sample.getFilterFreq());
        }

        int   dataType = sample.getDataType();
        int   loopCount = sample.getLoopCount();
        float leftGain = sample.leftGain;
        float rightGain = sample.rightGain;
        int   leftDelay = (int)(sample.leftDelay * attribs.getRolloff());
        int   rightDelay = (int)(sample.rightDelay * attribs.getRolloff());
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            if (soundType == AudioDevice3D.BACKGROUND_SOUND) {
                returnValue = com.sun.j3d.audio.J3DHaeStream.startSample(index,
                    loopCount, leftGain);
                if (debugFlag)
                    debugPrintln("JavaSoundMixer                    " +
                      "start stream backgroundSound with gain " + leftGain);
            }
            else { // soundType is POINT_SOUND or CONE_SOUND
                // start up main left and right channels for spatial rendered sound
                returnValue = com.sun.j3d.audio.J3DHaeStream.startSamples(index,
                       ((JSPositionalSample)sample).getSecondIndex(),
                       loopCount, leftGain, rightGain, leftDelay, rightDelay);
                //
                // start up reverb channel w/out delay even if reverb not on now
                // 
                float reverbGain = 0.0f;
                if (!muted && reverbOn) {
                    reverbGain = sample.getGain() *
                           attribs.getReflectionCoefficient();
                }
                int reverbRtrnVal = com.sun.j3d.audio.J3DHaeStream.startSample(
                       ((JSPositionalSample)sample).getReverbIndex(), loopCount,
			       reverbGain);
                if (debugFlag)
                    debugPrintln("JavaSoundMixer                    " +
                      "start stream positionalSound with gain "+ leftGain +
                      ", " + rightGain);
            }
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if (soundType == AudioDevice3D.BACKGROUND_SOUND) {
                returnValue = com.sun.j3d.audio.J3DHaeClip.startSample(index,
                    loopCount, leftGain );
                if (debugFlag)
                    debugPrintln("JavaSoundMixer                    " +
                      "start buffer backgroundSound with gain " + leftGain);
            }
            else { // soundType is POINT_SOUND or CONE_SOUND
                // start up main left and right channels for spatial rendered sound
                returnValue = com.sun.j3d.audio.J3DHaeClip.startSamples(index,
                        ((JSPositionalSample)sample).getSecondIndex(),
                        loopCount, leftGain, rightGain, leftDelay, rightDelay);
                // 
                // start up reverb channel w/out delay even if reverb not on now
                //  
                float reverbGain = 0.0f; 
                if (!muted && reverbOn) { 
                    reverbGain = sample.getGain() * 
                           attribs.getReflectionCoefficient(); 
                }
                int reverbRtrnVal = com.sun.j3d.audio.J3DHaeClip.startSample(
                        ((JSPositionalSample)sample).getReverbIndex(),
                        loopCount, reverbGain);

                if (debugFlag)
                    debugPrintln("JavaSoundMixer                    " +
                      "start stream positionalSound with gain " + leftGain
                      + ", " + rightGain);
            }
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            if (soundType == AudioDevice3D.BACKGROUND_SOUND) {
                returnValue = com.sun.j3d.audio.J3DHaeMidi.startSample(index,
                    loopCount, leftGain);
                if (debugFlag)
                    debugPrintln("JavaSoundMixer                    " +
                      "start Midi backgroundSound with gain " + leftGain);
            }
            else { // soundType is POINT_SOUND or CONE_SOUND
                // start up main left and right channels for spatial rendered sound
                returnValue = com.sun.j3d.audio.J3DHaeMidi.startSamples(index,
                       ((JSPositionalSample)sample).getSecondIndex(),
                       loopCount, leftGain, rightGain, leftDelay, rightDelay);
                /****
                // %%% TODO positional MIDI sounds not supported.
                //     The above startSamples really just start on sample
                //     Don't bother with reverb channel for now.

                //
                // start up reverb channel w/out delay even if reverb not on now
                // 
                float reverbGain = 0.0f;
                if (!muted && reverbOn) {
                    reverbGain = sample.getGain() *
                           attribs.getReflectionCoefficient();
                }
                int reverbRtrnVal = com.sun.j3d.audio.J3DHaeMidi.startSample(
                       ((JSPositionalSample)sample).getReverbIndex(), loopCount,
			       reverbGain);
                *******/
                if (debugFlag)
                    debugPrintln("JavaSoundMixer                    " +
                      "start Midi positionalSound with gain "+ leftGain +
                      ", " + rightGain);
            }
        }
        else {
            if (internalErrors)
                debugPrintln( 
                    "JavaSoundMixer: Internal Error startSample dataType " +
                    dataType + " invalid");
            return JSSample.NULL_SAMPLE;
        }

        if (returnValue < 0) {
            if (internalErrors)
                debugPrintln( 
                    "JavaSoundMixer: Internal Error startSample for index " +
                    index + " failed");
            return JSSample.NULL_SAMPLE;
        }
        else {
            if (debugFlag)
                debugPrintln("                startSample worked, " +
                       "returning " + returnValue);
            // %%% KLUDGE Set AuralParameters AFTER sound started 
            if (!muted) {
                if (reverbDirty > 0) {
                    if (debugFlag) {
                        debugPrintln("startSample: reverb settings are:");
                        debugPrintln("    coeff = "+
                              attribs.getReflectionCoefficient() +
                              ", delay = " + attribs.getReverbDelay() +
                              ", order = " + attribs.getReverbOrder());
                    }
                    float delayTime = attribs.getReverbDelay() * attribs.getRolloff();
                    calcReverb(index, attribs.getReflectionCoefficient(),
                          delayTime,  attribs.getReverbOrder());
                }
                // %%% KLUDGE it apprears that reverb has to be reset in
                //            JavaSound engine when sound re-started??
                // force reset of reverb parameters when sound is started
                setReverb(index);
            }
            return returnValue;
        }
    }
    
    public int   stopSample(int index) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: STOPSample for index " + index);
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return -1;

        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();

        int returnValue = 0;
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) || 
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeStream.stopSamples(index, 
                         ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeStream.stopSample(
                         ((JSPositionalSample)sample).getReverbIndex());
            }
            else
                returnValue = com.sun.j3d.audio.J3DHaeStream.stopSample(index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) || 
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeClip.stopSamples(index, 
                         ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeClip.stopSample(
                         ((JSPositionalSample)sample).getReverbIndex());
            }
            else
                returnValue = com.sun.j3d.audio.J3DHaeClip.stopSample(index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {

            /*****
            // %%% TODO positional sounds NOT supported yet
            if ( (soundType == AudioDevice3D.CONE_SOUND) || 
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeMidi.stopSamples(index, 
                         ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeMidi.stopSample(
                         ((JSPositionalSample)sample).getReverbIndex());
            }
            else
            *****/
                returnValue = com.sun.j3d.audio.J3DHaeMidi.stopSample(index);
        }
        else {
            if (internalErrors)
                debugPrintln( "JavaSoundMixer: Internal Error stopSample dataType " +
                  dataType + " invalid");
            return -1;
        }

        if (returnValue < 0) {
            if (internalErrors)
                debugPrintln( "JavaSoundMixer: Internal Error stopSample(s) for index " +
                    index + " failed");
            return -1;
        }
        else {
            // set fields in sample to reset for future start
            sample.stop();
            if (debugFlag)
                debugPrintln("JavaSoundMixer: stopSample for index " +
                       index + " worked, returning " + returnValue);
            return 0;
        }
    }

    public void pauseSample(int index) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: PAUSESample for index " + index);
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();
        int returnValue = 0;
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeStream.pauseSamples(index,
                         ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeStream.pauseSample(
                         ((JSPositionalSample)sample).getReverbIndex());
            }
            else
                returnValue = com.sun.j3d.audio.J3DHaeStream.pauseSample(index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeClip.pauseSamples(index,
                         ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeClip.pauseSample(
                         ((JSPositionalSample)sample).getReverbIndex());
            }
            else
                returnValue = com.sun.j3d.audio.J3DHaeClip.pauseSample(index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            /*******
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeMidi.pauseSamples(index,
                         ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeMidi.pauseSample(
                         ((JSPositionalSample)sample).getReverbIndex());
            }
            else
            *****/
                returnValue = com.sun.j3d.audio.J3DHaeMidi.pauseSample(index);
        }
        else {
            if (internalErrors)
                debugPrintln( 
                  "JavaSoundMixer: Internal Error pauseSample dataType " +
                  dataType + " invalid");
        }
        if (returnValue < 0) {
            if (internalErrors)
                debugPrintln( "JavaSoundMixer: Internal Error pauseSample " +
                    "for index " + index + " failed");
        }
        return;
    }

    public void unpauseSample(int index) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: UNPAUSESample for index " + index);
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();
        int returnValue = 0;
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeStream.unpauseSamples(index,
                      ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeStream.unpauseSample(
                      ((JSPositionalSample)sample).getReverbIndex());
            }
            else
                returnValue = com.sun.j3d.audio.J3DHaeStream.unpauseSample(index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeClip.unpauseSamples(index,
                      ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeClip.unpauseSample(
                      ((JSPositionalSample)sample).getReverbIndex());
            }
            else
                returnValue = com.sun.j3d.audio.J3DHaeClip.unpauseSample(index);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            /*********
            // %%% TODO positional Midi sounds
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                returnValue = com.sun.j3d.audio.J3DHaeMidi.unpauseSamples(index,
                      ((JSPositionalSample)sample).getSecondIndex());
                returnValue = com.sun.j3d.audio.J3DHaeMidi.unpauseSample(
                      ((JSPositionalSample)sample).getReverbIndex());
            }
            else
            *********/
                returnValue = com.sun.j3d.audio.J3DHaeMidi.unpauseSample(index);
        }
        else {
            if (internalErrors)
                debugPrintln( 
                      "JavaSoundMixer: Internal Error unpauseSample dataType " +
                      dataType + " invalid");
        }
        if (returnValue < 0) { 
            if (internalErrors)
                debugPrintln( "JavaSoundMixer: Internal Error unpauseSample " +                    "for index " + index + " failed");
 
        }
        return;
    }

    public void updateSample(int index) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: UPDATESample for index " + index);
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        int   loopCount = sample.getLoopCount();
        float leftGain;
        float rightGain;
        int   leftDelay;
        int   rightDelay;
        int   soundType = sample.getSoundType();
        boolean muted = sample.getMuteFlag();

        if (muted) {
            leftGain = sample.leftGain;
            rightGain = sample.rightGain;
            if (leftGain == 0.0f && rightGain == 0.0f) {
                if (debugFlag)
                    debugPrintln("                           " +
                        "MUTE flag true but gains already 0");
            }
            else {
                if (debugFlag)
                    debugPrintln("                           " +
                        "MUTE during update");
                sample.leftGain = sample.rightGain = 0.0f;
                setStereoGain(index, 0.0f, 0.0f);
            }
            if (soundType != AudioDevice3D.BACKGROUND_SOUND)
                setFilter(index, false, Sound.NO_FILTER);
        }
        else {
            // If reverb parameters changed resend to audio device
            AuralParameters attribs = getAuralParameters();
            if (reverbDirty > 0) {
                if (debugFlag) {
                    debugPrintln("updateSample: reverb settings are:");
                    debugPrintln("    coeff = " + attribs.getReflectionCoefficient() +
                        ", delay = " + attribs.getReverbDelay() +
                        ", order = " + attribs.getReverbOrder());
                }
                float delayTime = attribs.getReverbDelay() * attribs.getRolloff();
                calcReverb(index, attribs.getReflectionCoefficient(),
                          delayTime,  attribs.getReverbOrder());
            }
            // %%% TODO Only re-set reverb if values different
            //     For now force reset to ensure that reverb is currently correct
            setReverb(index);  // ensure that reverb is currently correct
            sample.render(sample.getDirtyFlags(), getView(), attribs);

            // filtering
            if (soundType != AudioDevice3D.BACKGROUND_SOUND)
                setFilter(index, sample.getFilterFlag(), sample.getFilterFreq());
             
            leftGain = sample.leftGain;
            rightGain = sample.rightGain;
            leftDelay = (int)(sample.leftDelay * attribs.getRolloff());
            rightDelay = (int)(sample.rightDelay * attribs.getRolloff());
            scaleSampleRate(index, sample.rateRatio);

            if (debugFlag) {
                debugPrintln("                           " +
                        "StereoGain during update " + leftGain +
                        ", " + rightGain);
                debugPrintln("                           " +
                        "StereoDelay during update " + leftDelay +
                        ", " + rightDelay);
            }
            setStereoGain(index, leftGain, rightGain);
            setStereoDelay(index, leftDelay, rightDelay);
        }
        return;
    }

    public void   muteSample(int index) {
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        if (debugFlag)
            debugPrintln("JavaSoundMixer: muteSample");
        sample.setMuteFlag(true);
        return;
    }

    public void   unmuteSample(int index) {
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        if (debugFlag)
            debugPrintln("JavaSoundMixer: unmuteSample");
        sample.setMuteFlag(false);

        // since while mute the reverb type and state was not updated...
        // Reverb has to be recalculated when sound is unmuted .
        reverbDirty = 0xFFFF;  // force an update of reverb params
        sample.setDirtyFlags(0xFFFF); // heavy weight forcing of gain/delay update

        // %%% TODO force an update of ALL parameters that could have changed
        //     while muting disabled...

        return;
    }

    public long  getSampleDuration(int index) {
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return -1L;
        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();
        long duration;
        if (debugFlag) {
             debugPrintln("JavaSoundMixer: getSampleDuration for index "+index);
             debugPrintln("                getSampleDuration dataType "+ dataType);
        }
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA)
            duration = com.sun.j3d.audio.J3DHaeStream.getSampleDuration(index);

        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA)
            duration = com.sun.j3d.audio.J3DHaeClip.getSampleDuration(index);
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            duration = com.sun.j3d.audio.J3DHaeMidi.getSampleDuration(index);
        }
        else {
            if (internalErrors)
                debugPrintln( ": Internal Error getSampleDuration dataType " + 
                      dataType + " invalid");
            return -1L;
        }
        if (debugFlag)
             debugPrintln("                return duration " + duration);
        return duration;
        
    }

    public int   getNumberOfChannelsUsed(int index) {
        /*
         * Calls same method with different signature containing the
         * sample's mute flag passed as the 2nd parameter.
         */ 
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return 0;
        else 
            return getNumberOfChannelsUsed(index, sample.getMuteFlag());
    }

    public int   getNumberOfChannelsUsed(int index, boolean muted) {
        /*
         * The JavaSoundMixer implementation uses THREE channels to render
         * the stereo image of each Point and Cone Sounds:
         *   Two for rendering the right and left portions of the rendered
         *   spatialized sound image - panned hard right or left respectively.
         * This implementation uses one channel to render Background sounds
         * whether the sample is mono or stereo.
         * %%% TODO When muted is implemented, that flag should be check
         *     so that zero is returned.
         */ 
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return 0;

        int   soundType = sample.getSoundType();
        int   dataType = sample.getDataType();

        // %%% TODO for now positional Midi sound used only 1 sample
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||
            dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA)
            return 1;

        if (soundType == BACKGROUND_SOUND)
            return 1;
        else  // for Point and Cone sounds
            return 3;
    }

    public long  getStartTime(int index) {
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return 0L;

        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();

        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA)
            return (long)(com.sun.j3d.audio.J3DHaeStream.getStartTime(index));
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA)
            return (long)(com.sun.j3d.audio.J3DHaeClip.getStartTime(index));
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA)
            return (long)(com.sun.j3d.audio.J3DHaeMidi.getStartTime(index));
        else {
            if (internalErrors)
                debugPrintln( 
                  "JavaSoundMixer: Internal Error getStartTime dataType " +
                  dataType + " invalid");
            return 0L;
        }
    }

    /*
     * Methods called during rendering
     */
    void setStereoGain(int index, float leftGain, float rightGain) {
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        AuralParameters attribs = getAuralParameters();
        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();
        boolean muted = sample.getMuteFlag();

        if (debugFlag)
            debugPrintln("setStereoGain for index "+index+" " + leftGain +
                        ", " + rightGain);
        /* 
         * start up reverb channel w/out delay even if reverb not on now
         */ 
        // %%% TODO For now sum left & rightGains for reverb gain
        float reverbGain = 0.0f; 
        if (!muted && reverbOn) { 
            reverbGain = sample.getGain() *
                         attribs.getReflectionCoefficient(); 
        }
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            com.sun.j3d.audio.J3DHaeStream.setSampleGain(index, leftGain);
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                com.sun.j3d.audio.J3DHaeStream.setSampleGain(
                      ((JSPositionalSample)sample).getSecondIndex(), rightGain);
                com.sun.j3d.audio.J3DHaeStream.setSampleGain(
                        ((JSPositionalSample)sample).getReverbIndex(),
                        reverbGain);
            }
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            com.sun.j3d.audio.J3DHaeClip.setSampleGain(index, leftGain);
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                com.sun.j3d.audio.J3DHaeClip.setSampleGain(
                        ((JSPositionalSample)sample).getSecondIndex(), rightGain);
                com.sun.j3d.audio.J3DHaeClip.setSampleGain(
                        ((JSPositionalSample)sample).getReverbIndex(), 
                        reverbGain);
            }
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            // Stereo samples not used for Midi Song playback
            com.sun.j3d.audio.J3DHaeMidi.setSampleGain(index,
                    (leftGain+rightGain) );
            // %%% JavaSound does not support Midi song panning yet
            /****
            // -1.0 far left, 0.0 center, 1.0 far right
            position = (leftGain - rightGain) / (leftGain + rightGain);
            com.sun.j3d.audio.J3DHaeMidi.setSamplePan(index, position);
            *****/
            /*****
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                com.sun.j3d.audio.J3DHaeMidi.setSampleGain(
                      ((JSPositionalSample)sample).getSecondIndex(), rightGain);
                com.sun.j3d.audio.J3DHaeMidi.setSampleGain(
                        ((JSPositionalSample)sample).getReverbIndex(),
                        reverbGain);
            }
            *****/
        }
        else {
            if (internalErrors)
                debugPrintln( "JavaSoundMixer: Internal Error setSampleGain dataType " +
                      dataType + " invalid");
            return;
        }
    }

    void  setStereoDelay(int index, int leftDelay, int rightDelay) {
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();
        if (debugFlag)
            debugPrintln("setStereoDelay for index "+index+" " + leftDelay +
                        ", " + rightDelay);
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                com.sun.j3d.audio.J3DHaeStream.setSampleDelay(
                         index, leftDelay);
                com.sun.j3d.audio.J3DHaeStream.setSampleDelay(
                         ((JSPositionalSample)sample).getSecondIndex(), rightDelay);
            }
            else 
               com.sun.j3d.audio.J3DHaeStream.setSampleDelay(index, 0);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                com.sun.j3d.audio.J3DHaeClip.setSampleDelay(
                         index, leftDelay);
                com.sun.j3d.audio.J3DHaeClip.setSampleDelay(
                         ((JSPositionalSample)sample).getSecondIndex(), rightDelay);
            }
            else
                com.sun.j3d.audio.J3DHaeClip.setSampleDelay(index, 0);
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            /************
            if ( (soundType == AudioDevice3D.CONE_SOUND) ||
                 (soundType == AudioDevice3D.POINT_SOUND) ) {
                com.sun.j3d.audio.J3DHaeMidi.setSampleDelay(
                         index, leftDelay);
                com.sun.j3d.audio.J3DHaeMidi.setSampleDelay(
                         ((JSPositionalSample)sample).getSecondIndex(), rightDelay);
            }
            else 
            ********/
               com.sun.j3d.audio.J3DHaeMidi.setSampleDelay(index, 0);
        }
        else {
            if (internalErrors)
                debugPrintln( "JavaSoundMixer: Internal Error setSampleDelay dataType " +
                  dataType + " invalid");
            return;
        }
    }

    void  calcReverb(int index, float reflection, float delay, int order) {
        /*
         * pick the JavaSound reverb type that matches the input parameters
         * as best as possible.
         *
	 *  Hae Reverb Types     Size      Persistance        Delay
	 *  ================  -----------  ------------    -----------
         *  1 None (dry)
	 *  2 "Closet"         very small  very short <.5   fast smooth
         *  3 "Garage"         med. large    medium 1.0       medium
         *  4 "Acoustic Lab"   med. small     short .5       med. fast
         *  5 "Cavern"           large       long >2.0       very slow
	 *  6 "Dungeon"          medium    med. long 1.5     med. slow
         * 
         *
         * Order is NOT controllable, nor does it have a natural parallel.
         * For this reason Order and Reflection are tied together as to
         * affect 'Decay Speed'.  This speed paired with the size of the 
         * space implied by the Delay parameter determine the JavaSound
         * Reverb type that is set:
         *
         *                      |  Short:               Long:
         *                Speed |    Coeff <= 0.9         Coeff > 0.9
         *  Size                |    Order <= 8           Order > 8
         * ---------------------------------------------------------------
         *  small (<100ms)      |    2 "Closet"           4 "Acoustic Lab"
         *  medium  (<500ms)    |    3 "Garage"           6 "Dungeon"
         *  large   (>500ms)    |    6 "Dungeon"          5 "Cavern"
         *
         * Size (delay) parameter takes precidence.
         * Last changed Reflection Coeff or Order is what is used.
         *   If both Coeff and Order changed, coeff takes precidence.
         *
         * %%% FEATURE of JavaSound
         *     Dungeon sound like it is flipping the pan value of the sound
         *     and applying reverb to the inverted pan position - like the
         *     sound is bouncing off the opposite wall of the Dungeon.
         */
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        int   dataType = sample.getDataType();
        int   soundType = sample.getSoundType();

        /*
         * Remember Coeff change is choosen over Order change if BOTH made
         * otherwise the last one changed take precidence.
         */
        // %%% TODO this last reverb "speed" (paramaeter type set) should
        // %%%      be set in AuralParameters class...
        if ((reverbDirty & REFLECTION_COEFF_CHANGED) > 0)
            lastReverbSpeed = REFLECTION_COEFF_CHANGED;
        else if ((reverbDirty & REVERB_ORDER_CHANGED) > 0)
            lastReverbSpeed = REVERB_ORDER_CHANGED;
        // otherwise no change to either, so use last

        if (reflection <= 0.0 || delay <= 0.0 || order <= 0) 
            reverbOn = false; 
        else  {
            reverbOn = true;
            if (delay < 100.0f)  {
                // "small" reverberant space
                if ((lastReverbSpeed & REVERB_ORDER_CHANGED) > 0) {
                    if (order <= 8)
                        reverbType = 2;
                    else
                        reverbType = 4;
                }
                else { // REFLECTION_COEFF_CHANGED or no change
                    if (reflection <= 0.9f)
                        reverbType = 2;
                    else
                        reverbType = 4;
                }
            }
            else if (delay < 500.0f)  {
                // "medium" reverberant space
                if ((lastReverbSpeed & REVERB_ORDER_CHANGED) > 0) {
                    if (order <= 8)
                        reverbType = 3;
                    else
                        reverbType = 6;
                }
                else {  // REFLECTION_COEFF_CHANGED of no change
                    if (reflection <= 0.9f)
                        reverbType = 3;
                    else
                        reverbType = 6;
                }
            }
            else  { // delay >= 500.0f
                // "large" reverberant space
                if ((lastReverbSpeed & REVERB_ORDER_CHANGED) > 0) {
                    if (order <= 8)
                        reverbType = 6;
                    else
                        reverbType = 5;
                }
                else { // REFLECTION_COEFF_CHANGED or no change
                    if (reflection <= 0.9f)
                        reverbType = 6;
                    else
                        reverbType = 5;
                }
            }
        }

        if (debugFlag)
            debugPrintln("JavaSoundMixer: setReverb index " + 
                index + ", type = " + reverbType + ", flag = " + reverbOn);

        reverbDirty = 0;   // clear the attribute reverb dirty flags
    }

    void  setReverb(int index) {
        /*
         * Only third sample of multisample sounds has reverb parameters set.
         *
         * For now, only positional and directional sounds are reverberated.
         */
       
        JSSample sample = null;
        if ((sample = (JSSample)getSample(index)) == null)
            return;

        int      soundType = sample.getSoundType();
        int      dataType = sample.getDataType();
        
        // %%% QUESTION Should reverb be applied to background sounds?
        if ( (soundType == AudioDevice3D.CONE_SOUND) ||
             (soundType == AudioDevice3D.POINT_SOUND) ) {
            if (debugFlag)
                debugPrintln("setReverb called with type, on = " +
                      reverbType + ", " + reverbOn);
            JSPositionalSample posSample = null; 
            if ((posSample = (JSPositionalSample)getSample(index)) == null) 
                return;

            int reverbIndex = posSample.getReverbIndex();
            if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA)
                com.sun.j3d.audio.J3DHaeStream.setReverb(reverbIndex, 
                         reverbType, reverbOn);
            else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA)
                com.sun.j3d.audio.J3DHaeClip.setReverb(reverbIndex,
                         reverbType, reverbOn);
            else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                     dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA)
                com.sun.j3d.audio.J3DHaeMidi.setReverb(reverbIndex, 
                         reverbType, reverbOn);
            else {
                if (internalErrors)
                    debugPrintln( "JavaSoundMixer: Internal Error setReverb " +
                          "dataType " + dataType + " invalid");
            }
        }
    }

    void  scaleSampleRate(int index, float scaleFactor) {
        if (debugFlag)
            debugPrintln("JavaSoundMixer: scaleSampleRate index " + 
                index + ", scale factor = " + scaleFactor);
        JSSample sample = null; 
        if ((sample = (JSSample)getSample(index)) == null) 
            return;
        int   dataType = sample.getDataType();
        if (debugFlag)
             debugPrintln(" scaleSampleRate.dataType = " + dataType +
                               "using sample " + sample + " from samples[" +
                                 index +"]");
        int   soundType = sample.getSoundType();

        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA) {
            com.sun.j3d.audio.J3DHaeStream.scaleSampleRate(index, scaleFactor);
            if (soundType != AudioDevice3D.BACKGROUND_SOUND)  {
                com.sun.j3d.audio.J3DHaeStream.scaleSampleRate(
                    ((JSPositionalSample)sample).getSecondIndex(), scaleFactor);
                com.sun.j3d.audio.J3DHaeStream.scaleSampleRate(
                    ((JSPositionalSample)sample).getReverbIndex(), scaleFactor);
            }
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            com.sun.j3d.audio.J3DHaeClip.scaleSampleRate(index, scaleFactor);
            if (soundType != AudioDevice3D.BACKGROUND_SOUND) {
                com.sun.j3d.audio.J3DHaeClip.scaleSampleRate(
                    ((JSPositionalSample)sample).getSecondIndex(), scaleFactor);
                com.sun.j3d.audio.J3DHaeClip.scaleSampleRate(
                    ((JSPositionalSample)sample).getReverbIndex(), scaleFactor);
            }
        }
        else if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||             
                 dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            com.sun.j3d.audio.J3DHaeMidi.scaleSampleRate(index, scaleFactor);
            /******
            if (soundType != AudioDevice3D.BACKGROUND_SOUND)  {
                com.sun.j3d.audio.J3DHaeMidi.scaleSampleRate(
                    ((JSPositionalSample)sample).getSecondIndex(), scaleFactor);
                com.sun.j3d.audio.J3DHaeMidi.scaleSampleRate(
                    ((JSPositionalSample)sample).getReverbIndex(), scaleFactor);
            }
            ******/
        }
        else {
            if (internalErrors)
                debugPrintln( 
                  "JavaSoundMixer: Internal Error scaleSampleRate dataType " +
                  dataType + " invalid");
        }
    }

// %%% TEMPORARY Override of method due to bug in Java Sound
    public void   setLoop(int index, int count) {
        JSSample sample = null; 
        if ((sample = (JSSample)getSample(index)) == null) 
            return;
        int dataType = sample.getDataType();
 
        // %%% WORKAROUND
        //     Bug in Java Sound engine hangs when INFINITE_LOOP count
        //     for Audio Wave data.  Leave count unchanged for Midi data.
        if (dataType==com.sun.j3d.audio.JavaSoundParams.STREAMING_AUDIO_DATA ||
            dataType==com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            if (count == Sound.INFINITE_LOOPS) {
                // LoopCount of 'loop Infinitely' forced to largest positive int
                count = 0x7FFFFFF;
            }
        }
        super.setLoop(index, count);
        return;
    }

    // Perform device specific filtering
    // Assumes that this is called for positional and directional sounds
    // not background sounds, so there are at lease two samples assigned
    // per sound.
    // %%% TODO - remove assumption from method
    void setFilter(int index, boolean filterFlag, float filterFreq) {
        JSPositionalSample posSample = null;
        if ((posSample = (JSPositionalSample)getSample(index)) == null) 
            return;
        int dataType = posSample.getDataType();

        // Filtering can NOT be performed on MIDI Songs
        if (dataType == com.sun.j3d.audio.JavaSoundParams.STREAMING_MIDI_DATA ||
            dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_MIDI_DATA) {
            return;
        }

        int secondIndex = posSample.getSecondIndex();
        if (dataType == com.sun.j3d.audio.JavaSoundParams.BUFFERED_AUDIO_DATA) {
            com.sun.j3d.audio.J3DHaeClip.setFiltering(index, filterFlag,filterFreq);
            com.sun.j3d.audio.J3DHaeClip.setFiltering(secondIndex, 
                                                    filterFlag, filterFreq);
        }
        else { // dataType == JavaSoundParams.STREAMING_AUDIO_DATA
            com.sun.j3d.audio.J3DHaeStream.setFiltering(index, filterFlag,filterFreq);
            com.sun.j3d.audio.J3DHaeStream.setFiltering(secondIndex, 
                                                    filterFlag, filterFreq);
        }
        // %%% QUESTION should reverb channel be filtered???

        if (debugFlag) {
            debugPrintln("JavaSoundMixer:setFilter " +
                         "of non-backgroundSound by (" +
                         filterFlag + ", " + filterFreq + ")");
        }
    }

}
