/*
 *	@(#)Sample.java 1.17 01/01/11 07:25:35
 *
 * Copyright (c) 1996-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

/*
 * Sample object
 */

package com.sun.j3d.audioengines;

import javax.media.j3d.*;
import javax.vecmath.*;

/**
 * The Sample Class defines the data and methods associated with a sound
 * sample played thru the AudioDevice.
 * This contains all the data fields for non-spatialized and spatialized
 * (positional and directional) sound samples.
 */

public class Sample
{

    /**
     ** Debug print mechanism for Sound nodes
     **/ 
    static final
    protected boolean  debugFlag = false;

    static final // 'static final' so internal error prints will not be compiled
    protected boolean internalErrors = false;

    protected void debugPrint(String message) {
        if (debugFlag)
            System.out.println(message);
    }

    protected void debugPrintln(String message) {
        if (debugFlag)
            System.out.println(message);
    }

   /**
    * Null Sound identifier denotes sound is not created or initialized
    */
    protected static final int NULL_SAMPLE = -1;

    /**
     *  sound data associated with sound source
     */  
    protected MediaContainer  soundData = null;

    /**
     *  sound data associated with sound source
     */  
    protected int soundType = -1;

    /**
     *  Overall Scale Factor applied to sound.
     */  
    protected float gain = 1.0f;  // Valid values are >= 0.0.

    /**
     *  Number of times sound is looped/repeated during play
     */ 
    protected int   loopCount = 0;  //  Range from 0 to POSITIVE_INFINITY(-1)

    protected long    duration = 0;
    protected int     numberOfChannels = 0;
    protected boolean mute = false;  // denotes if sample is muted
                          // (playing with zero gain)

    /*
     *
     * Fields associated with positional sound samples
     *
     */
    /*
     * Local to Vworld transform
     */
    protected Transform3D vworldXfrm = new Transform3D();
    protected boolean     vwXfrmFlag = false;

    /*
     * Origin of Sound source in Listener's space.
     */
    protected Point3f     position = new Point3f(0.0f, 0.0f, 0.0f);

    /*
     * Pairs of distances and gain scale factors that define piecewise linear
     * gain attenuation between each pair.
     */  
    protected double[]  attenuationDistance = null; 
    protected float[]   attenuationGain = null;;

    /**
     * dirty flags denoting what has changed since last rendering
     */  
    protected int dirtyFlags = 0xFFFF;

    /*
     *
     * Direction sample fields
     *
     */
    /**
     * The Cone Sound's direction vector.  This is the cone axis.
     */
    protected Vector3f	direction = new Vector3f(0.0f, 0.0f, 1.0f);

    /**
     * Pairs of distances and gain scale factors that define piecewise linear
     * gain BACK attenuation between each pair.
     * These are used for defining elliptical attenuation regions.
     */  
    protected double[]     backAttenuationDistance = null;
    protected float[]     backAttenuationGain = null;

    /**
     * Directional Sound's gain can be attenuated based on the listener's
     * location off-angle from the source source direction.
     * This can be set by three parameters:
     *     angular distance in radians
     *     gain scale factor
     *     filtering (currently the only filtering supported is lowpass)
     */
    protected double[]	angularDistance = {0.0, (Math.PI * 0.5)};
    protected float[]	angularGain     = {1.0f, 0.0f};

    /**
     *   Distance Filter
     *   Each sound source is attenuated by a filter based on it's distance
     *   from the listener.
     *   For now the only supported filterType will be LOW_PASS frequency 
     *   cutoff.
     *   At some time full FIR filtering will be supported.
     */ 
    protected static final int  NO_FILTERING  = -1;
    protected static final int  LOW_PASS      =  1;

    protected int         angularFilterType      = NO_FILTERING;
    protected float[]     angularFilterCutoff = {Sound.NO_FILTER, Sound.NO_FILTER};

    /*
     * Construct a new audio device Sample object
     */  
    public Sample() {
        if (debugFlag)
            debugPrintln("Sample constructor");
    }
    
    public long  getDuration() {
        return 0;
    }

    public long  getStartTime() {
        return 0;
    }

    public int    getNumberOfChannelsUsed() {
        return 0;
    }

    public void  setDirtyFlags(int flags) {
        dirtyFlags = flags;
    }

    public int   getDirtyFlags() {
        return dirtyFlags;
    }

    public void  setSoundType(int type) {
        soundType = type;
    }

    public int   getSoundType() {
        return soundType;
    }

    public void  setSoundData(MediaContainer ref) {
        soundData = ref;
    }

    public MediaContainer getSoundData() {
        return soundData;
    }

    public void  setMuteFlag(boolean flag) {
        mute = flag;
    }

    public boolean getMuteFlag() {
        return mute;
    }

    public void  setVWrldXfrmFlag(boolean flag) {
        // this flag is ONLY true if the VirtualWorld Transform is ever set
        vwXfrmFlag = flag;
    }

    public boolean getVWrldXfrmFlag() {
        return vwXfrmFlag;
    }

    public void  setGain(float scaleFactor) {
        gain = scaleFactor;
        return;
    }

    public void   setLoopCount(int count) {
        loopCount = count; 
        return;
    }

    public int   getLoopCount() {
        return loopCount; 
    }

    public void   setPosition(Point3d position) {
        this.position.set(position);
        return;
    }

    public void setDistanceGain(
              double[] frontDistance, float[]  frontAttenuationScaleFactor, 
              double[] backDistance, float[]  backAttenuationScaleFactor) {
        if (frontDistance != null) {
            int size = frontDistance.length;
            attenuationDistance = new double[size];
            attenuationGain = new float[size];
            for (int i=0; i<size; i++) {
                attenuationDistance[i] = frontDistance[i];
                attenuationGain[i] = frontAttenuationScaleFactor[i];
            }
        }
        else {
            attenuationDistance = null;
            attenuationGain = null;
        }
        if (backDistance != null && frontDistance != null) {
            int size = backDistance.length;
            backAttenuationDistance = new double[size];
            backAttenuationGain = new float[size];
            for (int i=0; i<size; i++) {
                backAttenuationDistance[i] = backDistance[i];
                backAttenuationGain[i] = backAttenuationScaleFactor[i];
            }
        }
        else {
            backAttenuationDistance = null;
            backAttenuationGain = null;
        }
        return;
    }

    public void setDirection(Vector3d direction) {
        this.direction.set(direction);
        return;
    }


    public void setAngularAttenuation(int filterType, double[] angle, 
                 float[] attenuationScaleFactor, float[] filterCutoff) {
        if (angle != null) {
            int size = angle.length;
            angularDistance = new double[size];
            angularGain = new float[size];
            if (filterType != NO_FILTERING && filterCutoff != null)
                angularFilterCutoff = new float[size];
            else
                angularFilterCutoff = null;
            for (int i=0; i<size; i++) {
                angularDistance[i] = angle[i];
                angularGain[i] = attenuationScaleFactor[i];
                if (filterType != NO_FILTERING)
                    angularFilterCutoff[i] = filterCutoff[i];
            }
            angularFilterType = filterType;
        }
        else {
            angularDistance = null;
            angularGain = null;
            angularFilterCutoff = null;
            angularFilterType = NO_FILTERING;
        }
    }

    /**
     * Clears/re-initialize fields associated with sample data
     * for this sound,
     * and frees any device specific data associated with this sample.
     */  
    public void clear() {
        if (debugFlag)
            debugPrintln("Sample.clear() entered");
        soundData = (MediaContainer)null;
        soundType = NULL_SAMPLE;
        gain = 1.0f;
        loopCount = 0; 
        duration = 0;
        numberOfChannels = 0;
        vworldXfrm.setIdentity();
        vwXfrmFlag = false;
        position.set(0.0f, 0.0f, 0.0f);
        attenuationDistance = null;
        attenuationGain = null;
        direction.set(0.0f, 0.0f, 1.0f);
        backAttenuationDistance = null;
        backAttenuationGain = null;
        angularDistance[0] = 0.0f;
        angularDistance[1] = (float)(Math.PI) * 0.5f;
        angularGain[0] = 1.0f;
        angularGain[1] = 0.0f;
        angularFilterType = NO_FILTERING;
        angularFilterCutoff[0] = Sound.NO_FILTER;
        angularFilterCutoff[1] = Sound.NO_FILTER;
        if (debugFlag)
            debugPrintln("Sample.clear() entered");
    }

    /*
     * Render
     */
    public void render(int dirtyFlags, View view, AuralParameters attribs) {
        // meant to be overridden
    }

    /*
     * Stop
     */
    public void stop() {
        // meant to be overridden
    }
}
