
//
// Set.java
//

/*
VisAD system for interactive analysis and visualization of numerical
data.  Copyright (C) 1996 - 1998 Bill Hibbard, Curtis Rueden, Tom
Rink and Dave Glowacki.
 
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License in file NOTICE for more details.
 
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

package visad;

//
// TO DO:
//
// finish GriddednDSet using VisAD Enumerated logic.
//
// add MultiGridded for union of grids
// Field evalualtion as mean of non-missing values in each grid
//
// Field and FlatField: options (mode?) for Interpolated and NotInterpolated
//
// Field and FlatField: test for operations on IntegerSet and optimize
//
// change lots of methods to default protection (visible within package only)
//
//

/**
   Set is the abstract superclass of the VisAD hierarchy of sets.<P>
   Sets are subsets of R^n for n>0.  Set objects are immutable.<P>
*/
public abstract class Set extends DataImpl {

  int DomainDimension; // this is a subset of R^DomainDimension
  int Length;          // number of samples
  CoordinateSystem DomainCoordinateSystem;

  Unit[] SetUnits;
  ErrorEstimate[] SetErrors;

  /** caches of Set-s that return false and true to equals();
      useful for GriddedSet and other Set subclasses which have
      expensive equals tests;
      WLH 6 Nov 97 - don't use these. they hinder garbage collection
      and won't accomplish much practical
  private static final int CACHE_LENGTH = 10;
  private Set[] NotEqualsCache;
  private int LastNotEquals;
  private Set[] EqualsCache;
  private int LastEquals;
*/

  /** construct a Set object */
  public Set(MathType type) throws VisADException {
    this(type, null, null, null);
  }

  /** construct a Set object with a non-default CoordinateSystem 
      and non-default Unit-s */
  public Set(MathType type, CoordinateSystem coord_sys) throws VisADException {
    this(type, coord_sys, null, null);
  }

  /** construct a Set object with a non-default CoordinateSystem
      and non-default Unit-s */
  public Set(MathType type, CoordinateSystem coord_sys, Unit[] units,
             ErrorEstimate[] errors) throws VisADException {
    super(adjustType(type));
    Length = 0;
    DomainDimension = getDimension(type);
    RealTupleType DomainType = ((SetType) Type).getDomain();
    CoordinateSystem cs = DomainType.getCoordinateSystem();
    if (coord_sys == null) {
      DomainCoordinateSystem = cs;
    }
    else {
      if (cs == null || !cs.getReference().equals(coord_sys.getReference())) {
        throw new CoordinateSystemException(
          "Set: coord_sys must match Type.DefaultCoordinateSystem");
      }
      DomainCoordinateSystem = coord_sys;
    }
    if (DomainCoordinateSystem != null &&
        !Unit.canConvertArray(DomainCoordinateSystem.getCoordinateSystemUnits(),
                              DomainType.getDefaultUnits())) {
      throw new UnitException("Set: CoordinateSystem Units must be " +
                              "convertable with DomainType default Units");
    }

    if (units == null) {
      SetUnits = (DomainCoordinateSystem == null) ?
                 DomainType.getDefaultUnits() :
                 DomainCoordinateSystem.getCoordinateSystemUnits();
    }
    else {
      if (units.length != DomainDimension) {
        throw new UnitException("Set: units dimension does not match");
      }
      SetUnits = new Unit[DomainDimension];
      Unit[] dunits = DomainType.getDefaultUnits();
      for (int i=0; i<DomainDimension; i++) {
        if (units[i] == null && dunits[i] != null) {
          SetUnits[i] = dunits[i];
        }
        else {
          SetUnits[i] = units[i];
        }
      }
    }
    if(!Unit.canConvertArray(SetUnits, DomainType.getDefaultUnits())) {
      throw new UnitException("Set: Units must be convertable with " +
                              "DomainType default Units");
    }
    if (SetUnits == null) SetUnits = new Unit[DomainDimension];

    SetErrors = new ErrorEstimate[DomainDimension];
    if (errors != null) {
      if (errors.length != DomainDimension) {
        throw new SetException("Set: errors dimension does not match");
      }
      for (int i=0; i<DomainDimension; i++) SetErrors[i] = errors[i];
    }
  }

  /** get DomainDimension (i.e., this is a subset of R^DomainDimension) */
  static int getDimension(MathType type) throws VisADException {
    if (type instanceof SetType) {
      return ((SetType) type).getDomain().getDimension();
    }
    else if (type instanceof RealTupleType) {
      return ((RealTupleType) type).getDimension();
    }
    else if (type instanceof RealType) {
      return 1;
    }
    else {
      throw new TypeException("Set: Type must be SetType or RealTupleType");
    }
  }

  static MathType adjustType(MathType type) throws VisADException {
    if (type instanceof SetType) {
      return type;
    }
    else if (type instanceof RealTupleType) {
      return new SetType(type);
    }
    else if (type instanceof RealType) {
      return new SetType(type);
    }
    else {
      throw new TypeException("Set: Type must be SetType, RealTupleType" +
                              " or RealType");
    }
  }

  public Unit[] getSetUnits() {
    return Unit.copyUnitsArray(SetUnits);
  }

  public ErrorEstimate[] getSetErrors() {
    return ErrorEstimate.copyErrorsArray(SetErrors);
  }

  /** get DomainCoordinateSystem */
  public CoordinateSystem getCoordinateSystem() {
    return DomainCoordinateSystem;
  }

  /** get DomainDimension (i.e., this is a subset of R^DomainDimension) */
  public int getDimension() {
    return DomainDimension;
  }

  /** for non-SimpleSet, ManifoldDimension = DomainDimension */
  public int getManifoldDimension() {
    return DomainDimension;
  }

  /** get the number of samples */
  public int getLength() throws VisADException {
    return Length;
  }

  /** get an int array ennumerating all index values of this set */
  public int[] getWedge() {
    int[] wedge = new int[Length];
    for (int i=0; i<Length; i++) wedge[i] = i;
    return wedge;
  }

  public float[][] getSamples(boolean copy) throws VisADException {
    int n = getLength();
    int[] indices = new int[n];
    // do NOT call getWedge
    for (int i=0; i<n; i++) indices[i] = i;
    return indexToValue(indices);
  }

  //
  // must eventually move indexToValue and valueToIndex to SimpleSet
  // and add logic for UnionSet to Field and FlatField
  //
  /** convert an array of 1-D indices to an array of values in R^DomainDimension */
  public abstract float[][] indexToValue(int[] index) throws VisADException;

  /** convert an array of values in R^DomainDimension to an array of 1-D indices */
  public abstract int[] valueToIndex(float[][] value) throws VisADException;

  public DataShadow computeRanges(ShadowType type, DataShadow shadow)
         throws VisADException {
    return computeRanges(type, shadow, null, false);
  }

  /** this default does not set ranges - it is used by FloatSet and DoubleSet */
  public DataShadow computeRanges(ShadowType type, DataShadow shadow,
                                  double[][] ranges, boolean domain)
         throws VisADException {
    setAnimationSampling(type, shadow, domain);
    return shadow;
  }

  /** domain == true is this is the domain of a Field */
  void setAnimationSampling(ShadowType type, DataShadow shadow, boolean domain)
       throws VisADException {
    // default does nothing
  }

  /** merge 1D sets; used for default animation set */
  Set merge1DSets(Set set) throws VisADException {
    if (DomainDimension != 1 || set.getDimension() != 1 ||
        equals(set)) return this;
    int length = getLength();
    // all indices in this
    int[] indices = getWedge();
    // all values in this
    float[][] values = indexToValue(indices);
    // transform values from this to set
    ErrorEstimate[] errors_out = new ErrorEstimate[1];
    values = CoordinateSystem.transformCoordinates(
                   ((SetType) set.getType()).getDomain(),
                   set.getCoordinateSystem(), set.getSetUnits(),
                   null /* set.getSetErrors() */,
                   ((SetType) Type).getDomain(),
                   DomainCoordinateSystem,
                   SetUnits, null /* SetErrors */, values);
    // set indices for values in this
    int[] test_indices = set.valueToIndex(values);
    // find indices of set not covered by this
    int set_length = set.getLength();
    boolean[] set_indices = new boolean[set_length];
    for (int i=0; i<set_length; i++) set_indices[i] = true;
    for (int i=0; i<length; i++) set_indices[test_indices[i]] = false;
    // now set_indices = true for indices of set not covered by this
    int num_new = 0;
    for (int i=0; i<set_length; i++) if (set_indices[i]) num_new++;
    if (num_new == 0) return this; // all covered, so nothing to do
    // not all covered, so merge values of this with values of set
    // not covered; first get uncovered indices
    int[] new_indices = new int[num_new];
    num_new = 0;
    for (int i=0; i<set_length; i++) {
      if (set_indices[i]) {
        new_indices[num_new] = i;
        num_new++;
      }
    }
    // get uncovered values
    float[][] new_values = set.indexToValue(new_indices);
    // transform values for Units and CoordinateSystem
    new_values = CoordinateSystem.transformCoordinates(
                     ((SetType) Type).getDomain(),
                     DomainCoordinateSystem, SetUnits, null /* errors_out */,
                     ((SetType) set.getType()).getDomain(),
                     set.getCoordinateSystem(), set.getSetUnits(),
                     null /* set.getSetErrors() */, new_values);
    // merge uncovered values with values of this
    float[][] all_values = new float[1][length + num_new];
    for (int i=0; i<length; i++) all_values[0][i] = values[0][i];
    for (int i=0; i<num_new; i++) {
      all_values[0][length + i] = new_values[0][i];
    }
    // sort all_values then construct Gridded1DSet
    // just use ErrorEstimates from this
    QuickSort.sort(all_values[0]);
    return new Gridded1DSet(Type, all_values, all_values[0].length,
                            DomainCoordinateSystem, SetUnits,
                            SetErrors, false);
  }

  public Set makeSpatial(SetType type, float[][] samples) throws VisADException {
    throw new SetException("Set.makeSpatial: not valid for this Set");
  }

  public VisADGeometryArray make1DGeometry(float[][] color_values)
         throws VisADException {
    throw new SetException("Set.make1DGeometry: not valid for this Set");
  }

  public VisADGeometryArray make2DGeometry(float[][] color_values)
         throws VisADException {
    throw new SetException("Set.make2DGeometry: not valid for this Set");
  }

  public VisADGeometryArray make3DGeometry(float[][] color_values)
         throws VisADException {
    throw new SetException("Set.make3DGeometry: not valid for this Set");
  }

  public VisADGeometryArray makePointGeometry(float[][] color_values)
         throws VisADException {
    throw new SetException("Set.makePointGeometry: not valid for this Set");
  }

  /** return basic lines in array[0], fill-ins in array[1]
      and labels in array[2] */
  public VisADGeometryArray[] makeIsoLines(float interval, float low,
                      float hi, float base, float[] fieldValues,
                      float[][] color_values, boolean[] swap)
         throws VisADException {
    throw new SetException("Set.makeIsoLines: not valid for this Set");
  }

  public VisADGeometryArray makeIsoSurface(float isolevel,
         float[] fieldValues, float[][] color_values)
         throws VisADException {
    throw new SetException("Set.makeIsoSurface: not valid for this Set");
  }

  public static double[][] floatToDouble(float[][] value) {
    if (value == null) return null;
    double[][] val = new double[value.length][];
    for (int i=0; i<value.length; i++) {
      if (value[i] == null) {
        val[i] = null;
      }
      else {
        val[i] = new double[value[i].length];
        for (int j=0; j<value[i].length; j++) {
          val[i][j] = value[i][j];
        }
      }
    }
    return val;
  }

  public static float[][] doubleToFloat(double[][] value) {
    if (value == null) return null;
    float[][] val = new float[value.length][];
    for (int i=0; i<value.length; i++) {
      if (value[i] == null) {
        val[i] = null;
      }
      else {
        val[i] = new float[value[i].length];
        for (int j=0; j<value[i].length; j++) {
          val[i][j] = (float) value[i][j];
        }
      }
    }
    return val;
  }

  public void getNeighbors( int[][] neighbors )
              throws VisADException
  {
    throw new UnimplementedException("Set: getNeighbors()");
  }

  public void getNeighbors( int[][] neighbors, float[][] weights )
         throws VisADException
  {
    throw new UnimplementedException("Set: getNeighbors()");
  }

  public int[][] getNeighbors( int dimension )
                 throws VisADException
  {
    throw new UnimplementedException("Set: getNeighbors()");
  }


  /** test set against a cache of Set-s not equal to this */
  public boolean testNotEqualsCache(Set set) {
/* WLH 6 Nov 97
    if (NotEqualsCache == null || set == null) return false;
    for (int i=0; i<CACHE_LENGTH; i++) {
      if (set == NotEqualsCache[i]) return true;
    }
*/
    return false;
  }

  /** add set to a cache of Set-s not equal to this */
  public void addNotEqualsCache(Set set) {
/* WLH 6 Nov 97
    if (NotEqualsCache == null) {
      NotEqualsCache = new Set[CACHE_LENGTH];
      for (int i=0; i<CACHE_LENGTH; i++) NotEqualsCache[i] = null;
      LastNotEquals = 0;
    }
    NotEqualsCache[LastNotEquals] = set;
    LastNotEquals = (LastNotEquals + 1) % CACHE_LENGTH;
*/
  }

  /** test set against a cache of Set-s equal to this */
  public boolean testEqualsCache(Set set) {
/* WLH 6 Nov 97
    if (EqualsCache == null || set == null) return false;
    for (int i=0; i<CACHE_LENGTH; i++) {
      if (set == EqualsCache[i]) return true;
    }
*/
    return false;
  }

  /** add set to a cache of Set-s equal to this */
  public void addEqualsCache(Set set) {
/* WLH 6 Nov 97
    if (EqualsCache == null) {
      EqualsCache = new Set[CACHE_LENGTH];
      for (int i=0; i<CACHE_LENGTH; i++) EqualsCache[i] = null;
      LastEquals = 0;
    }
    EqualsCache[LastEquals] = set;
    LastEquals = (LastEquals + 1) % CACHE_LENGTH;
*/
  }

  /** test equality of SetUnits and DomainCoordinateSystem
      between this and set */
  public boolean equalUnitAndCS(Set set) {
    if (DomainCoordinateSystem == null) {
      if (set.DomainCoordinateSystem != null) return false;
    }
    else {
      if (!DomainCoordinateSystem.equals(set.DomainCoordinateSystem)) {
        return false;
      }
    }
    if (SetUnits != null || set.SetUnits != null) {
      if (SetUnits == null || set.SetUnits == null) return false;
      int n = SetUnits.length;
      if (n != set.SetUnits.length) return false;
      for (int i=0; i<n; i++) {
        if (SetUnits[i] == null && set.SetUnits[i] == null) continue;
        if (SetUnits[i] == null || set.SetUnits[i] == null) return false;
        if (!SetUnits[i].equals(set.SetUnits[i])) return false;
      }
    }
    return true;
  }

  /** test for equality */
  public abstract boolean equals(Object set);

  /** clone this Set */
  public abstract Object clone();

  /** copy this Set, but give it a new MathType;
      this is safe, since constructor checks consistency of
      DomainCoordinateSystem and SetUnits with Type */
  public abstract Object cloneButType(MathType type) throws VisADException;

  public String longString() throws VisADException {
    return longString("");
  }

  public String longString(String pre) throws VisADException {
    throw new TypeException("Set.longString");
  }

  /* run 'java visad.Set' to test the MathType and Set classes */
  public static void main(String args[]) throws VisADException {

    // visad.type.Crumb crumb_type = new visad.type.Crumb(); // this works
    // visad.data.Crumb crumb_data = new visad.data.Crumb(); // this works
    // but this does not work:   Crumb crumb = new Crumb();

    RealType vis_radiance = new RealType("vis_radiance", null, null);
    RealType ir_radiance = new RealType("ir_radiance", null, null);
    RealType count = new RealType("count", null, null);

    RealType[] types = {RealType.Latitude, RealType.Longitude};
    RealTupleType earth_location = new RealTupleType(types);

    RealType[] types2 = {vis_radiance, ir_radiance};
    RealTupleType radiance = new RealTupleType(types2);

    try {
      TupleType radiancexxx = new TupleType(types2);
    }
    catch (TypeException e) {
      // System.out.println(e);
      e.printStackTrace(); // same as e.printStackTrace(System.out);
    }
    try {
      FunctionType image_tuplexxx = new FunctionType(earth_location, radiance);
    }
    catch (TypeException e) {
      System.out.println(e);
    }
    try {
      FunctionType image_visxxx = new FunctionType(earth_location, vis_radiance);
    }
    catch (TypeException e) {
      System.out.println(e);
    }

    FunctionType image_tuple = new FunctionType(earth_location, radiance);
    FunctionType image_vis = new FunctionType(earth_location, vis_radiance);
    FunctionType image_ir = new FunctionType(earth_location, ir_radiance);

    FunctionType ir_histogram = new FunctionType(ir_radiance, count);

    System.out.println(image_tuple);
    System.out.println(ir_histogram);

    Linear2DSet set2d = new Linear2DSet(earth_location,
                                        0.0, 127.0, 128, 0.0, 127.0, 128);
    Linear1DSet set1d = new Linear1DSet(ir_radiance, 0.0, 255.0, 256);
    Integer2DSet iset2d = new Integer2DSet(earth_location, 128, 128);
    Integer1DSet iset1d = new Integer1DSet(ir_radiance, 256);

    FlatField imaget1 = new FlatField(image_tuple, set2d);
    FlatField imagev1 = new FlatField(image_vis, set2d);
    FlatField imager1 = new FlatField(image_ir, set2d);
    FlatField histogram1 = new FlatField(ir_histogram, set1d);

    System.out.println(imaget1);
    System.out.println(histogram1);

    System.out.println(set2d);
    System.out.println(set1d);
    System.out.println(iset2d);
    System.out.println(iset1d);

    if (set1d instanceof IntegerSet) System.out.println(" set1d ");
    if (set2d instanceof IntegerSet) System.out.println(" set2d ");
    if (iset1d instanceof IntegerSet) System.out.println(" iset1d ");
    if (iset2d instanceof IntegerSet) System.out.println(" iset2d ");
    System.out.println("");

    int i = 14;
    short s = 12;
    byte b = 10;
    Real t = new Real(1.0);
    Real x = new Real(12);
    Real y = new Real(12L);
    Real u = new Real(i);
    Real v = new Real(s);
    Real w = new Real(b);

    System.out.println(t);
    System.out.println("" + x + " " + y + " " + u + " " + v + " " + w);
  }

/* Here's the output:

iris 235% java visad.Set
visad.TypeException: TupleType: all components are RealType, must use RealTupleType
        at visad.TupleType.<init>(TupleType.java:47)
        at visad.Set.main(Set.java:155)
FunctionType (Real): (Latitude(degrees), Longitude(degrees)) -> (vis_radiance, ir_radiance)
FunctionType (Real): (ir_radiance) -> count
FlatField  missing

FlatField  missing

Linear2DSet: Length = 16384
  Dimension 1: Length = 128 Range = 0 to 127
  Dimension 2: Length = 128 Range = 0 to 127

Linear1DSet: Length = 256 Range = 0 to 255

Integer2DSet: Length = 16384
  Dimension 1: Length = 128
  Dimension 2: Length = 128

Integer1DSet: Length = 256

 iset1d 
 iset2d 

1
12 12 14 12 10
iris 236% 

*/

}

