
//
// BaseUnit.java
//

/*
 * Copyright 1997, University Corporation for Atmospheric Research
 * See file LICENSE for copying and redistribution conditions.
 *
 * $Id: BaseUnit.java,v 1.4 1998/06/04 20:07:06 steve Exp $
 */

package visad;

import java.util.Vector;
import java.io.Serializable;

/**
 * A class that represents the base units of a system of units.
 * @author Steven R. Emmerson
 *
 * This is part of Steve Emerson's Unit package that has been
 * incorporated into VisAD.
 */
public final class BaseUnit
    extends	Unit
    implements	Serializable
{
    /**
     * Name of the unit (e.g. "meter").
     */
    private final String		unitName;

    /**
     * Quantity of the unit (e.g. "Length").
     */
    private final String		quantityName;

    /**
     * Derived unit associated with base unit (for computational efficiency).
     */
    final DerivedUnit			derivedUnit;

    /**
     * Global database of base units (to prevent multiple base units for the
     * same quantity).
     */
    private static final Vector	baseUnits = new Vector(8);


    
    /**
     * Raise a base unit to a power.
     *
     * @param power	The power to raise this unit by.
     * @return		The unit resulting from raising this unit to 
     *			<code>power</code>.
     * @promise		This unit has not been modified.
     */
    public Unit pow(int power)
    {
	return derivedUnit.pow(power);
    }

    /**
     * Return the name of this unit.
     *
     * @return          The name of this unit (e.g. "meter").
     */
    public String unitName()
    {
	return unitName;
    }

    /**
     * Return the name of the quantity associated with this unit.
     *
     * @return          The name this units quantity (e.g. "Length").
     */
    public String quantityName()
    {
	return quantityName;
    }

    /**
     * Return a string representation of this unit.
     *
     * @return          A string representation of this unit (e.g. "meter").
     */
    public String toString()
    {
	return unitName;
    }

    /**
     * Create a new base unit.
     *
     * @param quantityName	The name of the associated quantity (e.g. 
     *				"Length").
     * @param unitName		The name for the unit (e.g. "meter").
     * @return          	A new base unit or the previously created one
     *				with the same names.
     * @require			The arguments are non-null.  The quantity
     *				name has not been used before or the unit name
     *				is the same as before.
     * @promise			The new quantity and unit has been added to the
     *				database.
     * @exception UnitException	Attempt to redefine the base unit 
     *				associated with <code>quantityName</code>.
     */
    public static synchronized BaseUnit addBaseUnit(String quantityName,
						    String unitName)
	throws UnitException
    {
	BaseUnit	baseUnit = quantityNameToUnit(quantityName);

	if (baseUnit == null)
	    return new BaseUnit(unitName, quantityName);

	if (baseUnit.unitName.equals(unitName))
	    return baseUnit;

	throw new UnitException("Attempt to redefine quantity \"" + 
	    quantityName + "\" base unit from \"" + baseUnit.unitName +
	    "\" to \"" + unitName + "\"");
    }

    /**
     * Find the base unit with the given name.
     *
     * @param unitName		The name of the unit (e.g. "meter").
     * @return          	The existing base unit with the given name
     *				or <code>null</code> if no such units exists.
     * @require			The argument is non-null.
     */
    public static synchronized BaseUnit unitNameToUnit(String unitName)
    {
	for (int i = 0; i < baseUnits.size(); ++i)
	{
	    BaseUnit	baseUnit = (BaseUnit)baseUnits.elementAt(i);

	    if (baseUnit.unitName.equals(unitName))
		return baseUnit;
	}

	return null;
    }

    /**
     * Find the base unit for the given quantity.
     *
     * @param unitName		The name of the quantity (e.g. "Length").
     * @return          	The existing base unit for the given quantity
     *				or <code>null</code> if no such unit exists.
     * @require			The argument is non-null.
     */
    public static synchronized BaseUnit quantityNameToUnit(String quantityName)
    {
	for (int i = 0; i < baseUnits.size(); ++i)
	{
	    BaseUnit	baseUnit = (BaseUnit)baseUnits.elementAt(i);

	    if (baseUnit.quantityName.equals(quantityName))
		return baseUnit;
	}

	return null;
    }


    /**
     * Test this class.
     *
     * @param args		Arguments (ignored).
     * @exception UnitException	A problem occurred.
     */
    public static void main(String[] args)
	throws	UnitException
    {
	BaseUnit	meter = BaseUnit.addBaseUnit("Length", "meter");

	System.out.println("meter=\"" + meter + "\"");
	System.out.println("(Unit)meter=\"" + (Unit)meter + "\"");
	System.out.println("meter^2=\"" + meter.pow(2) + "\"");
	System.out.println("((Unit)meter)^2=\"" + ((Unit)meter).pow(2) + "\"");

	BaseUnit	second = BaseUnit.addBaseUnit("Time", "second");

	System.out.println("meter*second=\"" + meter.multiply(second) + "\"");
	System.out.println("meter/(Unit)second=\"" + 
	    meter.divide((Unit)second) + "\"");

	System.out.println("meter.toThis(5,meter)=" + meter.toThis(5,meter));
	System.out.println("meter.toThat(5,meter)=" + meter.toThat(5,meter));

	System.out.println("meter.toThis(5,(Unit)meter)=" +
	    meter.toThis(5,(Unit)meter));
	System.out.println("meter.toThat(5,(Unit)meter)=" +
	    meter.toThat(5,(Unit)meter));

	double[] values = meter.toThis(new double[] {1,2},meter);

	System.out.println("meter.toThis({1,2},meter)=" + 
	    values[0] + "," + values[1]);

	values = meter.toThat(new double[] {1,2},meter);

	System.out.println("meter.toThat({1,2},meter)=" + 
	    values[0] + "," + values[1]);

	values = meter.toThis(new double[] {1,2},(Unit)meter);

	System.out.println("meter.toThis({1,2},(Unit)meter)=" + 
	    values[0] + "," + values[1]);

	values = meter.toThat(new double[] {1,2},(Unit)meter);

	System.out.println("meter.toThat({1,2},(Unit)meter)=" + 
	    values[0] + "," + values[1]);

	System.out.println("Checking exceptions:");
	try
	{
	    meter.toThis(5,second);
	    System.err.println("ERROR: second -> meter");
	    System.exit(1);
	}
	catch (UnitException e)
	{
	    System.out.println(e.getMessage());
	}
	try
	{
	    meter.toThat(5,second);
	    System.err.println("ERROR: meter -> second");
	    System.exit(1);
	}
	catch (UnitException e)
	{
	    System.out.println(e.getMessage());
	}
	try
	{
	    BaseUnit	foo = BaseUnit.addBaseUnit("Length", "foot");
	    System.err.println("ERROR: \"foot\" added");
	    System.exit(1);
	}
	catch (UnitException e)
	{
	    System.out.println(e.getMessage());
	}
    }

    /**
     * Convert values to this unit from a base unit.
     *
     * @param values	The values to be converted.
     * @param that      The unit of <code>values</code>.
     * @return          The converted values in units of this unit.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThis(double[] values, BaseUnit that)
	throws UnitException
    {
	if (equals(that))
	{
	    double[]	newValues = new double[values.length];

	    for (int i = 0; i < values.length; ++i)
		newValues[i] = values[i];

	    return newValues;
	}

	throw new UnitException("Attempt to convert from unit \"" +
	    that + "\" to unit \"" + this + "\"");
    }

    float[] toThis(float[] values, BaseUnit that)
        throws UnitException
    {
        if (equals(that))
        {
            float[]    newValues = new float[values.length];
 
            for (int i = 0; i < values.length; ++i)
                newValues[i] = values[i];
 
            return newValues;
        }
 
        throw new UnitException("Attempt to convert from unit \"" +
            that + "\" to unit \"" + this + "\"");
    }

    /**
     * Convert values to this unit from a derived unit.
     *
     * @param values	The values to be converted.
     * @param that      The unit of <code>values</code>.
     * @return          The converted values in units of this unit.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThis(double[] values, DerivedUnit that)
	throws UnitException
    {
	return derivedUnit.toThis(values, that);
    }

    float[] toThis(float[] values, DerivedUnit that)
        throws UnitException
    {
        return derivedUnit.toThis(values, that);
    }

    /**
     * Convert values to this unit from a scaled unit.
     *
     * @param values	The values to be converted.
     * @param that      The unit of <code>values</code>.
     * @return          The converted values in units of this unit.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThis(double[] values, ScaledUnit that)
	throws UnitException
    {
	return that.toThat(values, derivedUnit);
    }

    float[] toThis(float[] values, ScaledUnit that)
        throws UnitException
    {
        return that.toThat(values, derivedUnit);
    }

    /**
     * Convert values to this unit from a offset unit.
     *
     * @param values	The values to be converted.
     * @param that      The unit of <code>values</code>.
     * @return          The converted values in units of this unit.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThis(double[] values, OffsetUnit that)
	throws UnitException
    {
	return that.toThat(values, derivedUnit);
    }

    float[] toThis(float[] values, OffsetUnit that)
        throws UnitException
    {
        return that.toThat(values, derivedUnit);
    }

    /**
     * Convert values from this unit to a base unit.
     *
     * @param values	The values to be converted in units of this unit.
     * @param that      The unit to which to convert the values.
     * @return          The converted values.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThat(double[] values, BaseUnit that)
	throws UnitException
    {
	if (equals(that))
	{
	    double[]	newValues = new double[values.length];

	    for (int i = 0; i < values.length; ++i)
		newValues[i] = values[i];

	    return newValues;
	}

	throw new UnitException("Attempt to convert from unit \"" +
	    this + "\" to unit \"" + that + "\"");
    }

    float[] toThat(float[] values, BaseUnit that)
        throws UnitException
    {
        if (equals(that))
        {
            float[]    newValues = new float[values.length];
 
            for (int i = 0; i < values.length; ++i)
                newValues[i] = values[i];
 
            return newValues;
        }
 
        throw new UnitException("Attempt to convert from unit \"" +
            this + "\" to unit \"" + that + "\"");
    }

    /**
     * Convert values from this unit to a derived unit.
     *
     * @param values	The values to be converted in units of this unit.
     * @param that      The unit to which to convert the values.
     * @return          The converted values.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThat(double[] values, DerivedUnit that)
	throws UnitException
    {
	return derivedUnit.toThat(values, that);
    }

    float[] toThat(float[] values, DerivedUnit that)
        throws UnitException
    {
        return derivedUnit.toThat(values, that);
    }

    /**
     * Convert values from this unit to a scaled unit.
     *
     * @param values	The values to be converted in units of this unit.
     * @param that      The unit to which to convert the values.
     * @return          The converted values.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThat(double[] values, ScaledUnit that)
	throws UnitException
    {
	return that.toThis(values, derivedUnit);
    }

    float[] toThat(float[] values, ScaledUnit that)
        throws UnitException
    {
        return that.toThis(values, derivedUnit);
    }

    /**
     * Convert values from this unit to a offset unit.
     *
     * @param values	The values to be converted in units of this unit.
     * @param that      The unit to which to convert the values.
     * @return          The converted values.
     * @require		The units are identical.
     * @promise		Neither unit has been modified.
     * @exception	The units are not convertible.
     */
    double[] toThat(double[] values, OffsetUnit that)
	throws UnitException
    {
	return that.toThis(values, derivedUnit);
    }

    float[] toThat(float[] values, OffsetUnit that)
        throws UnitException
    {
        return that.toThis(values, derivedUnit);
    }

    /**
     * Construct a base unit from the names for the quantity and unit.
     *
     * @param unitName		Name of the unit (e.g. "meter").
     * @param quantityName	Name of the quantity (e.g. "Length").
     */
    private BaseUnit(String unitName, String quantityName)
    {
	this.unitName = unitName;
	this.quantityName = quantityName;
	baseUnits.addElement(this);
	derivedUnit = new DerivedUnit(this);
    }

  /** added by WLH 11 Feb 98 */
  public boolean equals(Unit unit) {
    return (unit instanceof BaseUnit) &&
           unitName.equals(((BaseUnit) unit).unitName) &&
           quantityName.equals(((BaseUnit) unit).quantityName);
  }

    /**
     * Multiply a base unit by another base unit.
     *
     * @param that	The base unit with which to multiply this unit.
     * @return		The product of the two units.
     * @promise		Neither unit has been modified.
     */
    Unit multiply(BaseUnit that)
    {
	return derivedUnit.multiply(that.derivedUnit);
    }

    /**
     * Multiply a base unit by a derived unit.
     *
     * @param that	The derived unit with which to multiply this unit.
     * @return          The product of the two units.
     * @promise		Neither unit has been modified.
     */
    Unit multiply(DerivedUnit that)
    {
	return derivedUnit.multiply(that);
    }

    /**
     * Multiply a base unit by a scaled unit.
     *
     * @param that      The scaled unit with which to multiply this unit.
     * @return          The product of the two units.
     * @promise		Neither unit has been modified.
     */
    Unit multiply(ScaledUnit that)
    {
	return derivedUnit.multiply(that);
    }

    /**
     * Divide a base unit by a base unit.
     *
     * @param that      The base unit to divide into this unit.
     * @return          The quotient of the two units.
     * @promise		Neither unit has been modified.
     */
    Unit divide(BaseUnit that)
    {
	return derivedUnit.divide(that.derivedUnit);
    }

    /**
     * Divide a base unit by a derived unit.
     *
     * @param that      The derived unit to divide into this unit.
     * @return          The quotient of the two units.
     * @promise		Neither unit has been modified.
     */
    Unit divide(DerivedUnit that)
    {
	return derivedUnit.divide(that);
    }

    /**
     * Divide a base unit by a scaled unit.
     *
     * @param that      The scaled unit to divide into this unit.
     * @return          The quotient of the two units.
     * @promise		Neither unit has been modified.
     */
    Unit divide(ScaledUnit that)
    {
	return derivedUnit.divide(that);
    }

/*
   added by Bill Hibbard for VisAD
   so that BaseUnits from different JVM's can be equal
*/
/*
  public boolean equals(Unit unit) {
    if (!(unit instanceof BaseUnit)) return false;
    return quantityName.equals(((BaseUnit) unit).quantityName) &&
           unitName.equals(((BaseUnit) unit).unitName);
  }
*/

}
