/* vi:set ts=8 sts=4 sw=4:
 *
 * VIM - Vi IMproved	by Bram Moolenaar
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */

/*
 * mark.c: functions for setting marks and jumping to them
 */

#include "vim.h"

/*
 * This file contains routines to maintain and manipulate marks.
 */

/*
 * If a named file mark's lnum is non-zero, it is valid.
 * If a named file mark's fnum is non-zero, it is for an existing buffer,
 * otherwise it is from .viminfo and namedfm[n].fname is the file name.
 * There are marks 'A - 'Z (set by user) and '0 to '9 (set when writing
 * viminfo).
 */
#define EXTRA_MARKS 10					/* marks 0-9 */
static xfmark_t namedfm[NMARKS + EXTRA_MARKS];		/* marks with file nr */

static void fname2fnum __ARGS((xfmark_t *fm));
static void fmarks_check_one __ARGS((xfmark_t *fm, char_u *name, buf_t *buf));
static char_u *mark_line __ARGS((pos_t *mp, int lead_len));
static void show_one_mark __ARGS((int, char_u *, pos_t *, char_u *, int current));
static void cleanup_jumplist __ARGS((void));
#ifdef FEAT_VIMINFO
static void write_one_filemark __ARGS((FILE *fp, xfmark_t *fm, int c1, int c2));
#endif

/*
 * Set named mark 'c' at current cursor position.
 * Returns OK on success, FAIL if bad name given.
 */
    int
setmark(c)
    int		c;
{
    int		i;

    /* Check for a special key (may cause islower() to crash). */
    if (c < 0)
	return FAIL;

    if (c == '\'' || c == '`')
    {
	setpcmark();
	/* keep it even when the cursor doesn't move */
	curwin->w_prev_pcmark = curwin->w_pcmark;
	return OK;
    }

#ifndef EBCDIC
    if (c > 'z')	    /* some islower() and isupper() cannot handle
				characters above 127 */
	return FAIL;
#endif
    if (islower(c))
    {
	i = c - 'a';
	curbuf->b_namedm[i] = curwin->w_cursor;
#ifdef FEAT_VIRTUALEDIT
	curbuf->b_namedmc[i] = curwin->w_coladd;
#endif
	return OK;
    }
    if (isupper(c))
    {
	i = c - 'A';
	namedfm[i].fmark.mark = curwin->w_cursor;
#ifdef FEAT_VIRTUALEDIT
	namedfm[i].fmark.coladd = curwin->w_coladd;
#endif
	namedfm[i].fmark.fnum = curbuf->b_fnum;
	vim_free(namedfm[i].fname);
	namedfm[i].fname = NULL;
	return OK;
    }
    return FAIL;
}

/*
 * Set the previous context mark to the current position and add it to the
 * jump list.
 */
    void
setpcmark()
{
    int		i;
#ifdef JUMPLIST_ROTATE
    xfmark_t	tempmark;
#endif
    xfmark_t	*fm;

    /* for :global the mark is set only once */
    if (global_busy)
	return;

    curwin->w_prev_pcmark = curwin->w_pcmark;
    curwin->w_pcmark = curwin->w_cursor;

#ifdef JUMPLIST_ROTATE
    /*
     * If last used entry is not at the top, put it at the top by rotating
     * the stack until it is (the newer entries will be at the bottom).
     * Keep one entry (the last used one) at the top.
     */
    if (curwin->w_jumplistidx < curwin->w_jumplistlen)
	++curwin->w_jumplistidx;
    while (curwin->w_jumplistidx < curwin->w_jumplistlen)
    {
	tempmark = curwin->w_jumplist[curwin->w_jumplistlen - 1];
	for (i = curwin->w_jumplistlen - 1; i > 0; --i)
	    curwin->w_jumplist[i] = curwin->w_jumplist[i - 1];
	curwin->w_jumplist[0] = tempmark;
	++curwin->w_jumplistidx;
    }
#endif

    /* If jumplist is full: remove oldest entry */
    if (++curwin->w_jumplistlen > JUMPLISTSIZE)
    {
	curwin->w_jumplistlen = JUMPLISTSIZE;
	vim_free(curwin->w_jumplist[0].fname);
	for (i = 1; i < JUMPLISTSIZE; ++i)
	    curwin->w_jumplist[i - 1] = curwin->w_jumplist[i];
    }
    curwin->w_jumplistidx = curwin->w_jumplistlen;
    fm = &curwin->w_jumplist[curwin->w_jumplistlen - 1];

    fm->fmark.mark = curwin->w_pcmark;
    fm->fmark.fnum = curbuf->b_fnum;
    fm->fname = NULL;
}

/*
 * To change context, call setpcmark(), then move the current position to
 * where ever, then call checkpcmark().  This ensures that the previous
 * context will only be changed if the cursor moved to a different line.
 * If pcmark was deleted (with "dG") the previous mark is restored.
 */
    void
checkpcmark()
{
    if (curwin->w_prev_pcmark.lnum != 0
	    && (equal(curwin->w_pcmark, curwin->w_cursor)
		|| curwin->w_pcmark.lnum == 0))
    {
	curwin->w_pcmark = curwin->w_prev_pcmark;
	curwin->w_prev_pcmark.lnum = 0;		/* Show it has been checked */
    }
}

/*
 * move "count" positions in the jump list (count may be negative)
 */
    pos_t *
movemark(count)
    int count;
{
    pos_t	*pos;
    xfmark_t	*jmp;

    cleanup_jumplist();

    if (curwin->w_jumplistlen == 0)	    /* nothing to jump to */
	return (pos_t *)NULL;

    for (;;)
    {
	if (curwin->w_jumplistidx + count < 0
		|| curwin->w_jumplistidx + count >= curwin->w_jumplistlen)
	    return (pos_t *)NULL;

	/*
	 * if first CTRL-O or CTRL-I command after a jump, add cursor position
	 * to list.  Careful: If there are duplicates (CTRL-O immidiately after
	 * starting Vim on a file), another entry may have been removed.
	 */
	if (curwin->w_jumplistidx == curwin->w_jumplistlen)
	{
	    setpcmark();
	    --curwin->w_jumplistidx;	/* skip the new entry */
	    if (curwin->w_jumplistidx + count < 0)
		return (pos_t *)NULL;
	}

	curwin->w_jumplistidx += count;

	jmp = curwin->w_jumplist + curwin->w_jumplistidx;
	if (jmp->fmark.fnum == 0)
	    fname2fnum(jmp);
	if (jmp->fmark.fnum != curbuf->b_fnum)
	{
	    /* jump to other file */
	    if (buflist_findnr(jmp->fmark.fnum) == NULL)
	    {					     /* Skip this one .. */
		count += count < 0 ? -1 : 1;
		continue;
	    }
	    if (buflist_getfile(jmp->fmark.fnum, jmp->fmark.mark.lnum,
							    0, FALSE) == FAIL)
		return (pos_t *)NULL;
	    /* Set lnum again, autocommands my have changed it */
	    curwin->w_cursor.lnum = jmp->fmark.mark.lnum;
	    curwin->w_cursor.col = jmp->fmark.mark.col;
#ifdef FEAT_VIRTUALEDIT
	    curwin->w_coladd = jmp->fmark.coladd;
#endif
	    pos = (pos_t *)-1;
	}
	else
	    pos = &(jmp->fmark.mark);
	return pos;
    }
}

#if defined(FEAT_VIRTUALEDIT) || defined(PROTO)
static pos_t *getmark2 __ARGS((int c, int changefile, int fcoladd));

/*
 * Call getmark() and set curwin->w_coladd.
 */
    pos_t *
getmark_coladd(c, changefile)
    int		c;
    int		changefile;		/* allowed to edit another file */
{
    return getmark2(c, changefile, TRUE);
}
#endif

/*
 * Find mark "c".
 * Return pointer to pos_t if found (caller needs to check lnum!)
 *	  NULL if there is no mark called 'c'.
 *	  -1 if mark is in other file (only if changefile is TRUE)
 */
    pos_t *
getmark(c, changefile)
    int		c;
    int		changefile;		/* allowed to edit another file */
{
#ifdef FEAT_VIRTUALEDIT
    return getmark2(c, changefile, FALSE);
}

    static pos_t *
getmark2(c, changefile, fcoladd)
    int		c;
    int		changefile;		/* allowed to edit another file */
    int		fcoladd;		/* advance column?? */
{
    int			coladd = 0;
#endif
    pos_t		*posp;
#ifdef FEAT_VISUAL
    pos_t		*startp, *endp;
#endif
    static pos_t	pos_copy;

    posp = NULL;

    /* Check for special key, can't be a mark name and might cause islower()
     * to crash. */
    if (c < 0)
	return posp;
#ifndef EBCDIC
    if (c > '~')			/* check for islower()/isupper() */
	;
    else
#endif
        if (c == '\'' || c == '`')	/* previous context mark */
    {
	pos_copy = curwin->w_pcmark;	/* need to make a copy because */
	posp = &pos_copy;		/*   w_pcmark may be changed soon */
    }
    else if (c == '"')			/* to pos when leaving buffer */
	posp = &(curbuf->b_last_cursor);
    else if (c == '^')			/* to where Insert mode stopped */
	posp = &(curbuf->b_last_insert);
    else if (c == '.')			/* to where last change was made */
	posp = &(curbuf->b_last_change);
    else if (c == '[')			/* to start of previous operator */
	posp = &(curbuf->b_op_start);
    else if (c == ']')			/* to end of previous operator */
	posp = &(curbuf->b_op_end);
    else if (c == '{' || c == '}')	/* to previous/next paragraph */
    {
	pos_t	pos;
	oparg_t	oa;

	pos = curwin->w_cursor;
	if (findpar(&oa, c == '}' ? FORWARD : BACKWARD, 1L, NUL, FALSE))
	{
	    pos_copy = curwin->w_cursor;
	    posp = &pos_copy;
	}
	curwin->w_cursor = pos;
#ifdef FEAT_VIRTUALEDIT
	curwin->w_coladd = 0;
#endif
    }
    else if (c == '(' || c == ')')	/* to previous/next sentence */
    {
	pos_t	pos;

	pos = curwin->w_cursor;
	if (findsent(c == ')' ? FORWARD : BACKWARD, 1L))
	{
	    pos_copy = curwin->w_cursor;
	    posp = &pos_copy;
	}
	curwin->w_cursor = pos;
#ifdef FEAT_VIRTUALEDIT
	curwin->w_coladd = 0;
#endif
    }
#ifdef FEAT_VISUAL
    else if (c == '<' || c == '>')	/* start/end of visual area */
    {
	startp = &curbuf->b_visual_start;
	endp = &curbuf->b_visual_end;
	if ((c == '<') == lt_coladd(*startp, curbuf->b_visual_start_coladd,
					  *endp, curbuf->b_visual_end_coladd))
	{
	    posp = startp;
#ifdef FEAT_VIRTUALEDIT
	    coladd = curbuf->b_visual_start_coladd;
#endif
	}
	else
	{
	    posp = endp;
#ifdef FEAT_VIRTUALEDIT
	    coladd = curbuf->b_visual_end_coladd;
#endif
	}
	/*
	 * For Visual line mode, set mark at begin or end of line
	 */
	if (curbuf->b_visual_mode == 'V')
	{
	    pos_copy = *posp;
	    posp = &pos_copy;
	    if (c == '<')
		pos_copy.col = 0;
	    else
		pos_copy.col = MAXCOL;
	}
    }
#endif
    else if (islower(c))		/* normal named mark */
    {
	posp = &(curbuf->b_namedm[c - 'a']);
#ifdef FEAT_VIRTUALEDIT
	coladd = curbuf->b_namedmc[c - 'a'];
#endif
    }
    else if (isupper(c) || vim_isdigit(c))	/* named file mark */
    {
	if (vim_isdigit(c))
	    c = c - '0' + NMARKS;
	else
	    c -= 'A';
	posp = &(namedfm[c].fmark.mark);
#ifdef FEAT_VIRTUALEDIT
	coladd = namedfm[c].fmark.coladd;
#endif

	if (namedfm[c].fmark.fnum == 0)
	    fname2fnum(&namedfm[c]);
	if (namedfm[c].fmark.fnum != curbuf->b_fnum)
	{
	    /* mark is in another file */
	    if (namedfm[c].fmark.mark.lnum != 0
				       && changefile && namedfm[c].fmark.fnum)
	    {
		if (buflist_getfile(namedfm[c].fmark.fnum,
				      (linenr_t)1, GETF_SETMARK, FALSE) == OK)
		{
		    /* Set the lnum now, autocommands could have changed it */
		    curwin->w_cursor.lnum = namedfm[c].fmark.mark.lnum;
		    curwin->w_cursor.col = namedfm[c].fmark.mark.col;
#ifdef FEAT_VIRTUALEDIT
		    if (fcoladd)
			curwin->w_coladd = namedfm[c].fmark.coladd;
		    else
			curwin->w_coladd = 0;
#endif
		    return (pos_t *)-1;
		}
	    }
	    posp = &pos_copy;		/* mark exists, but is not valid in
					    current buffer */
	    pos_copy.lnum = 0;
	}
    }

#ifdef FEAT_VIRTUALEDIT
    if (fcoladd)
	curwin->w_coladd = coladd;
    else
	curwin->w_coladd = 0;
#endif

    return posp;
}

/*
 * Search for the next named mark in the current file.
 *
 * Returns pointer to pos_t of the next mark or NULL if no mark is found.
 */
    pos_t *
getnextmark(startpos, dir, beginline)
    pos_t	*startpos;	/* where to start */
    int		dir;	/* direction for search */
    int		beginline;
{
    int		i;
    pos_t	*result = NULL;
    pos_t	pos = *startpos;

    /* When searching backward and leaving the cursor on the first non-blank,
     * position must be in a previous line.
     * When searching forward and leaving the cursor on the first non-blank,
     * position must be in a next line. */
    if (dir == BACKWARD && beginline)
	pos.col = 0;
    else if (dir == FORWARD && beginline)
	pos.col = MAXCOL;

    for (i = 0; i < NMARKS; i++)
    {
	if (curbuf->b_namedm[i].lnum > 0)
	{
	    if (dir == FORWARD)
	    {
		if ((result == NULL || lt(curbuf->b_namedm[i], *result))
			&& lt(pos, curbuf->b_namedm[i]))
		    result = &curbuf->b_namedm[i];
	    }
	    else
	    {
		if ((result == NULL || lt(*result, curbuf->b_namedm[i]))
			&& lt(curbuf->b_namedm[i], pos))
		    result = &curbuf->b_namedm[i];
	    }
	}
    }

    return result;
}

/*
 * For an xtended filemark: set the fnum from the fname.
 * This is used for marks obtained from the .viminfo file.  It's postponed
 * until the mark is used to avoid a long startup delay.
 */
    static void
fname2fnum(fm)
    xfmark_t	*fm;
{
    char_u	*p;

    if (fm->fname != NULL)
    {
	/*
	 * First expand "~/" in the file name to the home directory.
	 * Try to shorten the file name.
	 */
	expand_env(fm->fname, NameBuff, MAXPATHL);
	mch_dirname(IObuff, IOSIZE);
	p = shorten_fname(NameBuff, IObuff);

	/* buflist_new() will call fmarks_check_names() */
	(void)buflist_new(NameBuff, p, (linenr_t)1, FALSE, FALSE);
    }
}

/*
 * Check all file marks for a name that matches the file name in buf.
 * May replace the name with an fnum.
 * Used for marks that come from the .viminfo file.
 */
    void
fmarks_check_names(buf)
    buf_t	*buf;
{
    char_u	*name;
    int		i;
    win_t	*wp;

    if (buf->b_ffname == NULL)
	return;

    name = home_replace_save(buf, buf->b_ffname);
    if (name == NULL)
	return;

    for (i = 0; i < NMARKS + EXTRA_MARKS; ++i)
	fmarks_check_one(&namedfm[i], name, buf);

#ifndef FEAT_WINDOWS
    wp = curwin;
#else
    for (wp = firstwin; wp != NULL; wp = wp->w_next)
#endif
	for (i = 0; i < wp->w_jumplistlen; ++i)
	    fmarks_check_one(&wp->w_jumplist[i], name, buf);

    vim_free(name);
}

    static void
fmarks_check_one(fm, name, buf)
    xfmark_t	*fm;
    char_u	*name;
    buf_t	*buf;
{
    if (fm->fmark.fnum == 0
	    && fm->fname != NULL
	    && fnamecmp(name, fm->fname) == 0)
    {
	fm->fmark.fnum = buf->b_fnum;
	vim_free(fm->fname);
	fm->fname = NULL;
    }
}

/*
 * Check a if a position from a mark is valid.
 * Give and error message and return FAIL if not.
 */
    int
check_mark(pos)
    pos_t    *pos;
{
    if (pos == NULL)
    {
	EMSG(_(e_umark));
	return FAIL;
    }
    if (pos->lnum == 0)
    {
	EMSG(_(e_marknotset));
	return FAIL;
    }
    if (pos->lnum > curbuf->b_ml.ml_line_count)
    {
	EMSG(_(e_markinval));
	return FAIL;
    }
    return OK;
}

/*
 * clrallmarks() - clear all marks in the buffer 'buf'
 *
 * Used mainly when trashing the entire buffer during ":e" type commands
 */
    void
clrallmarks(buf)
    buf_t	*buf;
{
    static int		i = -1;

    if (i == -1)	/* first call ever: initialize */
	for (i = 0; i < NMARKS + 1; i++)
	{
	    namedfm[i].fmark.mark.lnum = 0;
	    namedfm[i].fname = NULL;
	}

    for (i = 0; i < NMARKS; i++)
	buf->b_namedm[i].lnum = 0;
    buf->b_op_start.lnum = 0;		/* start/end op mark cleared */
    buf->b_op_end.lnum = 0;
    buf->b_last_cursor.lnum = 1;	/* '" mark cleared */
    buf->b_last_cursor.col = 0;
    buf->b_last_insert.lnum = 0;	/* '^ mark cleared */
    buf->b_last_change.lnum = 0;	/* '. mark cleared */
}

/*
 * Get name of file from a filemark.
 * When it's in the current buffer, return the text at the mark.
 * Returns an allocated string.
 */
    char_u *
fm_getname(fmark, lead_len)
    fmark_t	*fmark;
    int		lead_len;
{
    if (fmark->fnum == curbuf->b_fnum)		    /* current buffer */
	return mark_line(&(fmark->mark), lead_len);
    return buflist_nr2name(fmark->fnum, FALSE, TRUE);
}

/*
 * Return the line at mark "mp".  Truncate to fit in window.
 * The returned string has been allocated.
 */
    static char_u *
mark_line(mp, lead_len)
    pos_t	*mp;
    int		lead_len;
{
    char_u	*s, *p;
    int		len;

    if (mp->lnum == 0 || mp->lnum > curbuf->b_ml.ml_line_count)
	return vim_strsave((char_u *)"-invalid-");
    s = vim_strnsave(skipwhite(ml_get(mp->lnum)), (int)Columns);
    if (s == NULL)
	return NULL;
    /* Truncate the line to fit it in the window */
    len = 0;
    for (p = s; *p != NUL; ++p)
    {
	len += ptr2cells(p);
	if (len >= Columns - lead_len)
	    break;
#ifdef FEAT_MBYTE
	if (has_mbyte)
	    p += mb_ptr2len_check(p) - 1;
#endif
    }
    *p = NUL;
    return s;
}

/*
 * print the marks
 */
    void
do_marks(eap)
    exarg_t	*eap;
{
    char_u	*arg = eap->arg;
    int		i;
    char_u	*name;

    if (arg != NULL && *arg == NUL)
	arg = NULL;

    show_one_mark('\'', arg, &curwin->w_pcmark, NULL, TRUE);
    for (i = 0; i < NMARKS; ++i)
	show_one_mark(i + 'a', arg, &curbuf->b_namedm[i], NULL, TRUE);
    for (i = 0; i < NMARKS + EXTRA_MARKS; ++i)
    {
	if (namedfm[i].fmark.fnum != 0)
	    name = fm_getname(&namedfm[i].fmark, 15);
	else
	    name = namedfm[i].fname;
	if (name != NULL)
	{
	    show_one_mark(i >= NMARKS ? i - NMARKS + '0' : i + 'A',
		    arg, &namedfm[i].fmark.mark, name,
		    namedfm[i].fmark.fnum == curbuf->b_fnum);
	    if (namedfm[i].fmark.fnum != 0)
		vim_free(name);
	}
    }
    show_one_mark('"', arg, &curbuf->b_last_cursor, NULL, TRUE);
    show_one_mark('[', arg, &curbuf->b_op_start, NULL, TRUE);
    show_one_mark(']', arg, &curbuf->b_op_end, NULL, TRUE);
    show_one_mark('^', arg, &curbuf->b_last_insert, NULL, TRUE);
    show_one_mark('.', arg, &curbuf->b_last_change, NULL, TRUE);
#ifdef FEAT_VISUAL
    show_one_mark('<', arg, &curbuf->b_visual_start, NULL, TRUE);
    show_one_mark('>', arg, &curbuf->b_visual_end, NULL, TRUE);
#endif
    show_one_mark(-1, arg, NULL, NULL, FALSE);
}

    static void
show_one_mark(c, arg, p, name, current)
    int		c;
    char_u	*arg;
    pos_t	*p;
    char_u	*name;
    int		current;	/* in current file */
{
    static int	did_title = FALSE;
    int		mustfree = FALSE;

    if (c == -1)			    /* finish up */
    {
	if (did_title)
	    did_title = FALSE;
	else
	{
	    if (arg == NULL)
		MSG(_("No marks set"));
	    else
		EMSG2(_("No marks matching \"%s\""), arg);
	}
    }
    /* don't output anything if 'q' typed at --more-- prompt */
    else if (!got_int
	    && (arg == NULL || vim_strchr(arg, c) != NULL)
	    && p->lnum != 0)
    {
	if (!did_title)
	{
	    /* Highlight title */
	    MSG_PUTS_TITLE(_("\nmark line  col file/text"));
	    did_title = TRUE;
	}
	msg_putchar('\n');
	if (!got_int)
	{
	    sprintf((char *)IObuff, " %c %6ld %4d ", c, p->lnum, p->col);
	    msg_outtrans(IObuff);
	    if (name == NULL && current)
	    {
		name = mark_line(p, 15);
		mustfree = TRUE;
	    }
	    if (name != NULL)
	    {
		msg_outtrans_attr(name, current ? hl_attr(HLF_D) : 0);
		if (mustfree)
		    vim_free(name);
	    }
	}
	out_flush();		    /* show one line at a time */
    }
}

/*
 * print the jumplist
 */
/*ARGSUSED*/
    void
ex_jumps(eap)
    exarg_t	*eap;
{
    int		i;
    char_u	*name;

    cleanup_jumplist();
    /* Highlight title */
    MSG_PUTS_TITLE(_("\n jump line  col file/text"));
    for (i = 0; i < curwin->w_jumplistlen; ++i)
    {
	if (curwin->w_jumplist[i].fmark.mark.lnum != 0)
	{
	    if (curwin->w_jumplist[i].fmark.fnum == 0)
		fname2fnum(&curwin->w_jumplist[i]);
	    name = fm_getname(&curwin->w_jumplist[i].fmark, 16);
	    if (name == NULL)	    /* file name not available */
		continue;

	    msg_putchar('\n');
	    sprintf((char *)IObuff, "%c %2d %5ld %4d ",
		i == curwin->w_jumplistidx ? '>' : ' ',
		i > curwin->w_jumplistidx ? i - curwin->w_jumplistidx
					  : curwin->w_jumplistidx - i,
		curwin->w_jumplist[i].fmark.mark.lnum,
		curwin->w_jumplist[i].fmark.mark.col);
	    msg_outtrans(IObuff);
	    msg_outtrans_attr(name,
			    curwin->w_jumplist[i].fmark.fnum == curbuf->b_fnum
							? hl_attr(HLF_D) : 0);
	    vim_free(name);
	}
	out_flush();
    }
    if (curwin->w_jumplistidx == curwin->w_jumplistlen)
	MSG_PUTS("\n>");
}

#define one_adjust(add) \
    { \
	lp = add; \
	if (*lp >= line1 && *lp <= line2) \
	{ \
	    if (amount == MAXLNUM) \
		*lp = 0; \
	    else \
		*lp += amount; \
	} \
	else if (amount_after && *lp > line2) \
	    *lp += amount_after; \
    }

/* don't delete the line, just put at first deleted line */
#define one_adjust_nodel(add) \
    { \
	lp = add; \
	if (*lp >= line1 && *lp <= line2) \
	{ \
	    if (amount == MAXLNUM) \
		*lp = line1; \
	    else \
		*lp += amount; \
	} \
	else if (amount_after && *lp > line2) \
	    *lp += amount_after; \
    }

/*
 * Adjust marks between line1 and line2 (inclusive) to move 'amount' lines.
 * Must be called before changed_*(), appended_lines() or deleted_lines().
 * May be called before or after changing the text.
 * When deleting lines line1 to line2, use an 'amount' of MAXLNUM: The marks
 * within this range are made invalid.
 * If 'amount_after' is non-zero adjust marks after line2.
 * Example: Delete lines 34 and 35: mark_adjust(34, 35, MAXLNUM, -2);
 * Example: Insert two lines below 55: mark_adjust(56, MAXLNUM, 2, 0);
 *				   or: mark_adjust(56, 55, MAXLNUM, 2);
 */
    void
mark_adjust(line1, line2, amount, amount_after)
    linenr_t	line1;
    linenr_t	line2;
    long	amount;
    long	amount_after;
{
    int		i;
    int		fnum = curbuf->b_fnum;
    linenr_t	*lp;
    win_t	*win;

    if (line2 < line1 && amount_after == 0L)	    /* nothing to do */
	return;

    /* named marks, lower case and upper case */
    for (i = 0; i < NMARKS; i++)
    {
	one_adjust(&(curbuf->b_namedm[i].lnum));
	if (namedfm[i].fmark.fnum == fnum)
	    one_adjust(&(namedfm[i].fmark.mark.lnum));
    }
    for (i = NMARKS; i < NMARKS + EXTRA_MARKS; i++)
    {
	if (namedfm[i].fmark.fnum == fnum)
	    one_adjust(&(namedfm[i].fmark.mark.lnum));
    }

    /* previous context mark */
    one_adjust(&(curwin->w_pcmark.lnum));

    /* previous pcmark */
    one_adjust(&(curwin->w_prev_pcmark.lnum));

    /* last Insert position */
    one_adjust(&(curbuf->b_last_insert.lnum));

    /* last change position */
    one_adjust(&(curbuf->b_last_change.lnum));

#ifdef FEAT_VISUAL
    /* Visual area */
    one_adjust_nodel(&(curbuf->b_visual_start.lnum));
    one_adjust_nodel(&(curbuf->b_visual_end.lnum));
#endif

#ifdef FEAT_QUICKFIX
    /* quickfix marks */
    qf_mark_adjust(line1, line2, amount, amount_after);
#endif

    /*
     * Adjust items in all windows related to the current buffer.
     */
    for (win = firstwin; win != NULL; win = W_NEXT(win))
    {
	/* Marks in the jumplist.  When deleting lines, this may create
	 * duplicate marks in the jumplist, they will be removed later. */
	for (i = 0; i < win->w_jumplistlen; ++i)
	    if (win->w_jumplist[i].fmark.fnum == fnum)
		one_adjust_nodel(&(win->w_jumplist[i].fmark.mark.lnum));

	if (win->w_buffer == curbuf)
	{
	    /* marks in the tag stack */
	    for (i = 0; i < win->w_tagstacklen; i++)
		if (win->w_tagstack[i].fmark.fnum == fnum)
		    one_adjust_nodel(&(win->w_tagstack[i].fmark.mark.lnum));

	    /* topline and cursor position for windows with the same buffer
	     * other than the current window */
	    if (win != curwin)
	    {
		if (win->w_topline >= line1 && win->w_topline <= line2)
		{
		    if (amount == MAXLNUM)	    /* topline is deleted */
		    {
			if (line1 <= 1)
			    win->w_topline = 1;
			else
			    win->w_topline = line1 - 1;
		    }
		    else		/* keep topline on the same line */
			win->w_topline += amount;
		}
		else if (amount_after && win->w_topline > line2)
		    win->w_topline += amount_after;
		if (win->w_cursor.lnum >= line1 && win->w_cursor.lnum <= line2)
		{
		    if (amount == MAXLNUM) /* line with cursor is deleted */
		    {
			if (line1 <= 1)
			    win->w_cursor.lnum = 1;
			else
			    win->w_cursor.lnum = line1 - 1;
			win->w_cursor.col = 0;
		    }
		    else		/* keep cursor on the same line */
			win->w_cursor.lnum += amount;
		}
		else if (amount_after && win->w_cursor.lnum > line2)
		    win->w_cursor.lnum += amount_after;
	    }

#ifdef FEAT_FOLDING
	    /* adjust folds */
	    foldMarkAdjust(win, line1, line2, amount, amount_after);
#endif
	}
    }
}

/*
 * When deleting lines, this may create duplicate marks in the
 * jumplist. They will be removed here for the current window.
 */
    static void
cleanup_jumplist()
{
    int	    i;
    int	    from, to;

    to = 0;
    for (from = 0; from < curwin->w_jumplistlen; ++from)
    {
	if (curwin->w_jumplistidx == from)
	    curwin->w_jumplistidx = to;
	for (i = from + 1; i < curwin->w_jumplistlen; ++i)
	    if (curwin->w_jumplist[i].fmark.fnum
					== curwin->w_jumplist[from].fmark.fnum
		    && curwin->w_jumplist[from].fmark.fnum != 0
		    && curwin->w_jumplist[i].fmark.mark.lnum
				  == curwin->w_jumplist[from].fmark.mark.lnum)
		break;
	if (i >= curwin->w_jumplistlen)	    /* no duplicate */
	    curwin->w_jumplist[to++] = curwin->w_jumplist[from];
	else
	    vim_free(curwin->w_jumplist[from].fname);
    }
    if (curwin->w_jumplistidx == curwin->w_jumplistlen)
	curwin->w_jumplistidx = to;
    curwin->w_jumplistlen = to;
}

#if defined(FEAT_WINDOWS) || defined(PROTO)
/*
 * Copy the jumplist from window "from" to window "to".
 */
    void
copy_jumplist(from, to)
    win_t	*from;
    win_t	*to;
{
    int		i;

    for (i = 0; i < from->w_jumplistlen; ++i)
    {
	to->w_jumplist[i] = from->w_jumplist[i];
	if (from->w_jumplist[i].fname != NULL)
	    to->w_jumplist[i].fname = vim_strsave(from->w_jumplist[i].fname);
    }
    to->w_jumplistlen = from->w_jumplistlen;
    to->w_jumplistidx = from->w_jumplistidx;
}

/*
 * Free items in the jumplist of window "wp".
 */
    void
free_jumplist(wp)
    win_t	*wp;
{
    int		i;

    for (i = 0; i < wp->w_jumplistlen; ++i)
	vim_free(wp->w_jumplist[i].fname);
}
#endif

    void
set_last_cursor(win)
    win_t	*win;
{
    win->w_buffer->b_last_cursor = win->w_cursor;
}

#if defined(FEAT_VIMINFO) || defined(PROTO)
    int
read_viminfo_filemark(line, fp, force)
    char_u	*line;
    FILE	*fp;
    int		force;
{
    char_u	*str;
    xfmark_t	*fm;
    int		i;

    /* We only get here if line[0] == '\'' or '-'.
     * Illegal mark names are ignored (for future expansion). */
    str = line + 1;
    if (
#ifndef EBCDIC
	    *str <= 127 &&
#endif
	    ((*line == '\'' && (isdigit(*str) || isupper(*str)))
	     || (*line == '-' && *str == '\'')))
    {
	if (*str == '\'')
	{
	    /* If the jumplist isn't full insert fmark as oldest entry */
	    if (curwin->w_jumplistlen == JUMPLISTSIZE)
		fm = NULL;
	    else
	    {
		for (i = curwin->w_jumplistlen; i > 0; --i)
		    curwin->w_jumplist[i] = curwin->w_jumplist[i - 1];
		++curwin->w_jumplistidx;
		++curwin->w_jumplistlen;
		fm = &curwin->w_jumplist[0];
		fm->fmark.mark.lnum = 0;
		fm->fname = NULL;
	    }
	}
	else if (isdigit(*str))
	    fm = &namedfm[*str - '0' + NMARKS];
	else
	    fm = &namedfm[*str - 'A'];
	if (fm != NULL && (fm->fmark.mark.lnum == 0 || force))
	{
	    str = skipwhite(str + 1);
	    fm->fmark.mark.lnum = getdigits(&str);
	    str = skipwhite(str);
	    fm->fmark.mark.col = getdigits(&str);
#ifdef FEAT_VIRTUALEDIT
	    fm->fmark.coladd = 0;
#endif
	    fm->fmark.fnum = 0;
	    str = skipwhite(str);
	    vim_free(fm->fname);
	    fm->fname = viminfo_readstring(str, fp);
	}
    }
    return vim_fgets(line, LSIZE, fp);
}

    void
write_viminfo_filemarks(fp)
    FILE	*fp;
{
    int		i;
    char_u	*name;
    buf_t	*buf;
    xfmark_t	*fm;

    if (get_viminfo_parameter('\'') == 0)
	return;

    fprintf(fp, _("\n# File marks:\n"));

    /*
     * Find a mark that is the same file and position as the cursor.
     * That one, or else the last one is deleted.
     * Move '0 to '1, '1 to '2, etc. until the matching one or '9
     * Set '0 mark to current cursor position.
     */
    if (curbuf->b_ffname != NULL && !removable(curbuf->b_ffname))
    {
	name = buflist_nr2name(curbuf->b_fnum, TRUE, FALSE);
	for (i = NMARKS; i < NMARKS + EXTRA_MARKS - 1; ++i)
	    if (namedfm[i].fmark.mark.lnum == curwin->w_cursor.lnum
		    && (namedfm[i].fname == NULL
			    ? namedfm[i].fmark.fnum == curbuf->b_fnum
			    : (name != NULL
				    && STRCMP(name, namedfm[i].fname) == 0)))
		break;
	vim_free(name);

	vim_free(namedfm[i].fname);
	for ( ; i > NMARKS; --i)
	    namedfm[i] = namedfm[i - 1];
	namedfm[NMARKS].fmark.mark = curwin->w_cursor;
#ifdef FEAT_VIRTUALEDIT
	namedfm[NMARKS].fmark.coladd = curwin->w_coladd;
#endif
	namedfm[NMARKS].fmark.fnum = curbuf->b_fnum;
	namedfm[NMARKS].fname = NULL;
    }

    /* Write the filemarks '0 - '9 and 'A - 'Z */
    for (i = 0; i < NMARKS + EXTRA_MARKS; i++)
	write_one_filemark(fp, &namedfm[i], '\'',
				     i < NMARKS ? i + 'A' : i - NMARKS + '0');

    /* Write the jumplist with -' */
    fprintf(fp, _("\n# Jumplist (newest first):\n"));
    setpcmark();	/* add current cursor position */
    cleanup_jumplist();
    for (fm = &curwin->w_jumplist[curwin->w_jumplistlen - 1];
					   fm >= &curwin->w_jumplist[0]; --fm)
    {
	if (fm->fmark.fnum == 0
		|| ((buf = buflist_findnr(fm->fmark.fnum)) != NULL
		    && !removable(buf->b_ffname)))
	    write_one_filemark(fp, fm, '-', '\'');
    }
}

    static void
write_one_filemark(fp, fm, c1, c2)
    FILE	*fp;
    xfmark_t	*fm;
    int		c1;
    int		c2;
{
    char_u	*name;

    if (fm->fmark.mark.lnum == 0)	/* not set */
	return;

    if (fm->fmark.fnum != 0)		/* there is a buffer */
	name = buflist_nr2name(fm->fmark.fnum, TRUE, FALSE);
    else
	name = fm->fname;		/* use name from .viminfo */
    if (name == NULL)
	return;

    fprintf(fp, "%c%c  %ld  %ld  ", c1, c2, (long)fm->fmark.mark.lnum,
						    (long)fm->fmark.mark.col);
    viminfo_writestring(fp, name);
    if (fm->fmark.fnum != 0)
	vim_free(name);
}

/*
 * Return TRUE if "name" is on removable media (depending on 'viminfo').
 */
    int
removable(name)
    char_u  *name;
{
    char_u  *p;
    char_u  part[51];
    int	    retval = FALSE;

    name = home_replace_save(NULL, name);
    if (name != NULL)
    {
	for (p = p_viminfo; *p; )
	{
	    copy_option_part(&p, part, 51, ", ");
	    if (part[0] == 'r'
			   && STRNICMP(part + 1, name, STRLEN(part + 1)) == 0)
	    {
		retval = TRUE;
		break;
	    }
	}
	vim_free(name);
    }
    return retval;
}

static void write_one_mark __ARGS((FILE *fp_out, int c, pos_t *pos));

/*
 * Write all the named marks for all buffers.
 * Return the number of buffers for which marks have been written.
 */
    int
write_viminfo_marks(fp_out)
    FILE	*fp_out;
{
    int		count;
    buf_t	*buf;
    int		is_mark_set;
    int		i;
#ifdef FEAT_WINDOWS
    win_t	*win;

    /*
     * Set b_last_cursor for the all buffers that have a window.
     */
    for (win = firstwin; win != NULL; win = win->w_next)
	set_last_cursor(win);
#else
	set_last_cursor(curwin);
#endif

    fprintf(fp_out, _("\n# History of marks within files (newest to oldest):\n"));
    count = 0;
    for (buf = firstbuf; buf != NULL; buf = buf->b_next)
    {
	/*
	 * Only write something if buffer has been loaded and at least one
	 * mark is set.
	 */
	if (buf->b_marks_read)
	{
	    if (buf->b_last_cursor.lnum != 0)
		is_mark_set = TRUE;
	    else
	    {
		is_mark_set = FALSE;
		for (i = 0; i < NMARKS; i++)
		    if (buf->b_namedm[i].lnum != 0)
		    {
			is_mark_set = TRUE;
			break;
		    }
	    }
	    if (is_mark_set && buf->b_ffname != NULL
		      && buf->b_ffname[0] != NUL && !removable(buf->b_ffname))
	    {
		home_replace(NULL, buf->b_ffname, IObuff, IOSIZE, TRUE);
		fprintf(fp_out, "\n> ");
		viminfo_writestring(fp_out, IObuff);
		write_one_mark(fp_out, '"', &buf->b_last_cursor);
		write_one_mark(fp_out, '^', &buf->b_last_insert);
		write_one_mark(fp_out, '.', &buf->b_last_change);
		for (i = 0; i < NMARKS; i++)
		    write_one_mark(fp_out, 'a' + i, &buf->b_namedm[i]);
		count++;
	    }
	}
    }

    return count;
}

    static void
write_one_mark(fp_out, c, pos)
    FILE	*fp_out;
    int		c;
    pos_t	*pos;
{
    if (pos->lnum != 0)
	fprintf(fp_out, "\t%c\t%ld\t%d\n", c, (long)pos->lnum, (int)pos->col);
}

/*
 * Handle marks in the viminfo file:
 * fp_out == NULL   read marks for current buffer only
 * fp_out != NULL   copy marks for buffers not in buffer list
 */
    void
copy_viminfo_marks(line, fp_in, fp_out, count, eof)
    char_u	*line;
    FILE	*fp_in;
    FILE	*fp_out;
    int		count;
    int		eof;
{
    buf_t	*buf;
    int		num_marked_files;
    int		load_marks;
    int		copy_marks_out;
    char_u	*str;
    int		i;
    char_u	*p;
    char_u	*name_buf;
    pos_t	pos;

    if ((name_buf = alloc(LSIZE)) == NULL)
	return;
    num_marked_files = get_viminfo_parameter('\'');
    while (!eof && (count < num_marked_files || fp_out == NULL))
    {
	if (line[0] != '>')
	{
	    if (line[0] != '\n' && line[0] != '\r' && line[0] != '#')
	    {
		if (viminfo_error(_("Missing '>'"), line))
		    break;	/* too many errors, return now */
	    }
	    eof = vim_fgets(line, LSIZE, fp_in);
	    continue;		/* Skip this dud line */
	}

	/*
	 * Handle long line and translate escaped characters.
	 * Find file name, set str to start.
	 * Ignore leading and trailing white space.
	 */
	str = skipwhite(line + 1);
	str = viminfo_readstring(str, fp_in);
	if (str == NULL)
	    continue;
	p = str + STRLEN(str);
	while (p != str && (*p == NUL || vim_isspace(*p)))
	    p--;
	if (*p)
	    p++;
	*p = NUL;

	/*
	 * If fp_out == NULL, load marks for current buffer.
	 * If fp_out != NULL, copy marks for buffers not in buflist.
	 */
	load_marks = copy_marks_out = FALSE;
	if (fp_out == NULL)
	{
	    if (curbuf->b_ffname != NULL)
	    {
		home_replace(NULL, curbuf->b_ffname, name_buf, LSIZE, TRUE);
		if (fnamecmp(str, name_buf) == 0)
		    load_marks = TRUE;
	    }
	}
	else /* fp_out != NULL */
	{
	    /* This is slow if there are many buffers!! */
	    for (buf = firstbuf; buf != NULL; buf = buf->b_next)
		if (buf->b_ffname != NULL)
		{
		    home_replace(NULL, buf->b_ffname, name_buf, LSIZE, TRUE);
		    if (fnamecmp(str, name_buf) == 0)
			break;
		}

	    /*
	     * copy marks if the buffer has not been loaded
	     */
	    if (buf == NULL || !buf->b_marks_read)
	    {
		copy_marks_out = TRUE;
		fputs("\n> ", fp_out);
		viminfo_writestring(fp_out, str);
		count++;
	    }
	}
	vim_free(str);

	while (!(eof = vim_fgets(line, LSIZE, fp_in)) && line[0] == TAB)
	{
	    if (load_marks)
	    {
		if (line[1] != NUL)
		    sscanf((char *)line + 2, "%ld %d", &pos.lnum, &pos.col);
		switch (line[1])
		{
		    case '"': curbuf->b_last_cursor = pos; break;
		    case '^': curbuf->b_last_insert = pos; break;
		    case '.': curbuf->b_last_change = pos; break;
		    default:  if ((i = line[1] - 'a') >= 0 && i < NMARKS)
				  curbuf->b_namedm[i] = pos;
		}
	    }
	    else if (copy_marks_out)
		fputs((char *)line, fp_out);
	}
	if (load_marks)
	    break;
    }
    vim_free(name_buf);
}
#endif /* FEAT_VIMINFO */
