/* vi:set ts=8 sts=4 sw=4:
 *
 * VIM - Vi IMproved	by Bram Moolenaar
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */

/*
 * misc2.c: Various functions.
 */

#include "vim.h"

#ifdef HAVE_FCNTL_H
# include <fcntl.h>	    /* for chdir() */
#endif

#if defined(FEAT_VIRTUALEDIT) || defined(PROTO)
static int coladvance2 __ARGS((int addspaces, int finetune, colnr_t wcol));

/*
 * Return TRUE if in the current mode we need to use virtual.
 */
    int
virtual_active()
{
    return (ve_all
#ifdef FEAT_VISUAL
	    || (ve_block && VIsual_active && VIsual_mode == Ctrl_V)
#endif
	    || (ve_insert && (State & INSERT)));
}

/*
 * Get the screen position of character number pos on the current line.
 */
    int
getviscol()
{
    int		x = 0;
    char_u	*line = ml_get_curline();
    char_u	*p;

    for (p = line; (colnr_t)(p - line) < curwin->w_cursor.col; )
	x += lbr_chartabsize_adv(&p, x);

    return x + curwin->w_coladd;
}

/*
 * Go to column "wcol", and add/insert white space as neccessary to get the
 * cursor in that column.
 * The caller must have saved the cursor line for undo!
 */
    int
coladvance_force(wcol)
    colnr_t wcol;
{
    return coladvance2(TRUE, FALSE, wcol);
}
#endif

/*
 * Try to advance the Cursor to the specified column.
 * If virtual editing: fine tune the cursor position.
 *
 * return OK if desired column is reached, FAIL if not
 */
    int
coladvance(wcol)
    colnr_t	wcol;
{
#ifdef FEAT_VIRTUALEDIT
    return coladvance2(FALSE, virtual_active(), wcol);
}

    static int
coladvance2(addspaces, finetune, wcol)
    int		addspaces;	/* change the text to achieve our goal? */
    int		finetune;	/* change char offset for the excact column */
    colnr_t	wcol;		/* column to move to */
{
#endif
    int		idx;
    char_u	*ptr;
    char_u	*line;
    colnr_t	col;
    int		csize;
    int		one_more;

    one_more = (State & INSERT) || restart_edit != NUL
#ifdef FEAT_VISUAL
					  || (VIsual_active && *p_sel != 'o')
#endif
					  ;

#ifdef FEAT_VIRTUALEDIT
    curwin->w_coladd = 0;

    if ((addspaces || finetune)
	    && curwin->w_p_wrap
#ifdef FEAT_VERTSPLIT
	    && curwin->w_width != 0
#endif
	    && wcol >= (colnr_t)W_WIDTH(curwin)
	    && wcol < MAXCOL)
    {
	csize = linetabsize(ml_get_curline()) - 1;
	if (wcol / W_WIDTH(curwin) > (colnr_t)csize / W_WIDTH(curwin))
	{
	    /* In case of line wrapping don't move the cursor beyond the
	     * right screen edge. */
	    wcol = (csize / W_WIDTH(curwin) + 1) * W_WIDTH(curwin) - 1;
	}
    }
#endif

    /* try to advance to the specified column by counting char widths. */
    col = 0;
    line = ml_get_curline();

    if (wcol >= MAXCOL)
    {
	/*
	 * Commands like <End> give us a very big value to indicate they want
	 * to be as much on right side as possible.  Then stay in the column
	 * when moving up/down.
	 */
	idx = STRLEN(line) - 1;
#ifdef FEAT_VIRTUALEDIT
	if (addspaces || finetune)
	{
	    curwin->w_curswant = linetabsize(line) + one_more;
	    if (curwin->w_curswant > 0)
		--curwin->w_curswant;
	}
#endif
    }
    else
    {
	idx = -1;
	csize = 0;
	ptr = line;
	while (col <= wcol && *ptr != NUL)
	{
	    /* Count a tab for what it's worth (if list mode not on) */
	    csize = lbr_chartabsize_adv(&ptr, col);
	    col += csize;
	}
	idx = ptr - line - 1;

#ifdef FEAT_VIRTUALEDIT
	if (addspaces && (col != wcol + 1 || csize > 1))
	{
	    /*
	     * If line is too short, or a character doesn't allow to be at the
	     * wanted position, need to insert spaces.
	     */
	    if (col <= wcol)
	    {
		int	correct = wcol - col;
		char_u	*newline = alloc(idx + 1 + correct + 1);
		int	t;

		/*
		 * Line is too short: append spaces.
		 */
		if (newline == NULL)
		    return FAIL;
		for (t = 0; t < idx + 1; ++t)
		    newline[t] = line[t];

		for (t = 0; t < correct; ++t)
		    newline[t + idx + 1] = ' ';

		newline[idx + 1 + correct] = NUL;

		ml_replace(curwin->w_cursor.lnum, newline, FALSE);
		changed_bytes(curwin->w_cursor.lnum, (colnr_t)idx);
		idx += correct;
		col += correct;
	    }
	    else
	    {
		int	linelen = STRLEN(line);
		int	correct = wcol - col + 1; /* negative!! */
		char_u	*newline = alloc(linelen + csize);
		int	t, s = 0;
		int	v;

		/*
		 * break a tab
		 */
		if (newline == NULL || -correct > csize)
		    return FAIL;

		for (t = 0; t < linelen; t++)
		{
		    if (t != idx)
			newline[s++] = line[t];
		    else
			for (v = 0; v < csize; v++)
			    newline[s++] = ' ';
		}

		newline[linelen + csize - 1] = NUL;

		ml_replace(curwin->w_cursor.lnum, newline, FALSE);
		changed_bytes(curwin->w_cursor.lnum, idx);
		idx += (csize - 1 + correct);
		col += correct;
	    }
	}
#endif
    }

    /*
     * In Insert mode it is allowed to be one char beyond the end of the line.
     * Also in Visual mode, when 'selection' is not "old".
     */
    if (one_more && col <= wcol)
	++idx;
    if (idx < 0)
	curwin->w_cursor.col = 0;
    else
	curwin->w_cursor.col = idx;
#ifdef FEAT_MBYTE
    /* prevent cursor from moving on the trail byte */
    if (has_mbyte)
	mb_adjust_cursor();
#endif

#ifdef FEAT_VIRTUALEDIT
    if (finetune && wcol < MAXCOL)
    {
	int a = getviscol();
	int b = wcol - a;

	/* modify the real cursor position to make the cursor appear at the
	 * wanted column */
	if (b > 0 && b < (MAXCOL - 2 * W_WIDTH(curwin)))
             curwin->w_coladd = b;
    }
#endif

    if (col <= wcol)
	return FAIL;	    /* Couldn't reach column */
    else
	return OK;	    /* Reached column */
}

/*
 * inc(p)
 *
 * Increment the line pointer 'p' crossing line boundaries as necessary.
 * Return 1 when crossing a line (position points to NUL at the end of the
 * line), -1 when at end of file, 0 otherwise.
 */
    int
inc_cursor()
{
    return inc(&curwin->w_cursor);
}

    int
inc(lp)
    pos_t  *lp;
{
    char_u  *p = ml_get_pos(lp);

    if (*p != NUL)	/* still within line, move to next char (may be NUL) */
    {
#ifdef FEAT_MBYTE
	if (has_mbyte)
	{
	    int l = mb_ptr2len_check(p);

	    lp->col += l;
	    return ((p[l] != NUL) ? 0 : 1);
	}
#endif
	lp->col++;
	return ((p[1] != NUL) ? 0 : 1);
    }
    if (lp->lnum != curbuf->b_ml.ml_line_count)     /* there is a next line */
    {
	lp->col = 0;
	lp->lnum++;
	return 1;
    }
    return -1;
}

/*
 * incl(lp): same as inc(), but skip the NUL at the end of non-empty lines
 */
    int
incl(lp)
    pos_t    *lp;
{
    int	    r;

    if ((r = inc(lp)) == 1 && lp->col)
	r = inc(lp);
    return r;
}

/*
 * dec(p)
 *
 * Decrement the line pointer 'p' crossing line boundaries as necessary.
 * Return 1 when crossing a line, -1 when at start of file, 0 otherwise.
 */
    int
dec_cursor()
{
#ifdef FEAT_MBYTE
    return (has_mbyte ? mb_dec(&curwin->w_cursor) : dec(&curwin->w_cursor));
#else
    return dec(&curwin->w_cursor);
#endif
}

    int
dec(lp)
    pos_t  *lp;
{
    if (lp->col > 0)
    {		/* still within line */
	lp->col--;
	return 0;
    }
    if (lp->lnum > 1)
    {		/* there is a prior line */
	lp->lnum--;
	lp->col = STRLEN(ml_get(lp->lnum));
	return 1;
    }
    return -1;			/* at start of file */
}

/*
 * decl(lp): same as dec(), but skip the NUL at the end of non-empty lines
 */
    int
decl(lp)
    pos_t    *lp;
{
    int	    r;

    if ((r = dec(lp)) == 1 && lp->col)
	r = dec(lp);
    return r;
}

/*
 * Make sure curwin->w_cursor.lnum is valid.
 */
    void
check_cursor_lnum()
{
    if (curwin->w_cursor.lnum > curbuf->b_ml.ml_line_count)
	curwin->w_cursor.lnum = curbuf->b_ml.ml_line_count;
    if (curwin->w_cursor.lnum <= 0)
	curwin->w_cursor.lnum = 1;
}

/*
 * Make sure curwin->w_cursor.col is valid.
 */
    void
check_cursor_col()
{
    colnr_t len;

    len = STRLEN(ml_get_curline());
    if (len == 0)
	curwin->w_cursor.col = 0;
    else if (curwin->w_cursor.col >= len)
    {
	/* Allow cursor past end-of-line in Insert mode, restarting Insert
	 * mode or when in Visual mode and 'selection' isn't "old" */
	if (State & INSERT || restart_edit
#ifdef FEAT_VISUAL
		|| (VIsual_active && *p_sel != 'o')
#endif
		)
	    curwin->w_cursor.col = len;
	else
	    curwin->w_cursor.col = len - 1;
    }
}

/*
 * make sure curwin->w_cursor in on a valid character
 */
    void
adjust_cursor()
{
    check_cursor_lnum();
    check_cursor_col();
}

#if defined(FEAT_TEXTOBJ) || defined(PROTO)
/*
 * Make sure curwin->w_cursor is not on the NUL at the end of the line.
 * Allow it when in Visual mode and 'selection' is not "old".
 */
    void
adjust_cursor_col()
{
    if (curwin->w_cursor.col > 0
#ifdef FEAT_VISUAL
	    && (!VIsual_active || *p_sel == 'o')
#endif
	    && gchar_cursor() == NUL)
	--curwin->w_cursor.col;
}
#endif

/*
 * When curwin->w_leftcol has changed, adjust the cursor position.
 * Return TRUE if the cursor was moved.
 */
    int
leftcol_changed()
{
    long	lastcol;
    colnr_t	s, e;
    int		retval = FALSE;

    changed_cline_bef_curs();
    lastcol = curwin->w_leftcol + W_WIDTH(curwin) - curwin_col_off() - 1;
    validate_virtcol();

    /*
     * If the cursor is right or left of the screen, move it to last or first
     * character.
     */
    if (curwin->w_virtcol > (colnr_t)lastcol)
    {
	retval = TRUE;
	coladvance((colnr_t)lastcol);
    }
    else if (curwin->w_virtcol < curwin->w_leftcol)
    {
	retval = TRUE;
	(void)coladvance(curwin->w_leftcol);
    }

    /*
     * If the start of the character under the cursor is not on the screen,
     * advance the cursor one more char.  If this fails (last char of the
     * line) adjust the scrolling.
     */
    getvcol(curwin, &curwin->w_cursor, &s, NULL, &e);
    if (e > (colnr_t)lastcol)
    {
	retval = TRUE;
	coladvance(s - 1);
    }
    else if (s < curwin->w_leftcol)
    {
	retval = TRUE;
	if (coladvance(e + 1) == FAIL)	/* there isn't another character */
	{
	    curwin->w_leftcol = s;	/* adjust w_leftcol instead */
	    changed_cline_bef_curs();
	}
    }

    if (retval)
	curwin->w_set_curswant = TRUE;
    redraw_later(NOT_VALID);
    return retval;
}

/**********************************************************************
 * Various routines dealing with allocation and deallocation of memory.
 */

#if defined(MEM_PROFILE) || defined(PROTO)

# define MEM_SIZES  8200
static long_u mem_allocs[MEM_SIZES];
static long_u mem_frees[MEM_SIZES];
static long_u mem_allocated;
static long_u mem_freed;
static long_u mem_peak;
static long_u num_alloc;
static long_u num_freed;

static void mem_pre_alloc_s __ARGS((size_t *sizep));
static void mem_pre_alloc_l __ARGS((long_u *sizep));
static void mem_post_alloc __ARGS((void **pp, size_t size));
static void mem_pre_free __ARGS((void **pp));

    static void
mem_pre_alloc_s(sizep)
    size_t *sizep;
{
    *sizep += sizeof(size_t);
}

    static void
mem_pre_alloc_l(sizep)
    long_u *sizep;
{
    *sizep += sizeof(size_t);
}

    static void
mem_post_alloc(pp, size)
    void **pp;
    size_t size;
{
    if (*pp == NULL)
	return;
    size -= sizeof(size_t);
    *(long_u *)*pp = size;
    if (size <= MEM_SIZES-1)
	mem_allocs[size-1]++;
    else
	mem_allocs[MEM_SIZES-1]++;
    mem_allocated += size;
    if (mem_allocated - mem_freed > mem_peak)
	mem_peak = mem_allocated - mem_freed;
    num_alloc++;
    *pp = (void *)((char *)*pp + sizeof(size_t));
}

    static void
mem_pre_free(pp)
    void **pp;
{
    long_u size;

    *pp = (void *)((char *)*pp - sizeof(size_t));
    size = *(size_t *)*pp;
    if (size <= MEM_SIZES-1)
	mem_frees[size-1]++;
    else
	mem_frees[MEM_SIZES-1]++;
    mem_freed += size;
    num_freed++;
}

/*
 * called on exit via atexit()
 */
    void
vim_mem_profile_dump()
{
    int i, j;

    printf("\r\n");
    j = 0;
    for (i = 0; i < MEM_SIZES - 1; i++)
    {
	if (mem_allocs[i] || mem_frees[i])
	{
	    if (mem_frees[i] > mem_allocs[i])
		printf("\r\n%s", _("ERROR: "));
	    printf("[%4d / %4lu-%-4lu] ", i + 1, mem_allocs[i], mem_frees[i]);
	    j++;
	    if (j > 3)
	    {
		j = 0;
		printf("\r\n");
	    }
	}
    }

    i = MEM_SIZES - 1;
    if (mem_allocs[i])
    {
	printf("\r\n");
	if (mem_frees[i] > mem_allocs[i])
	    printf(_("ERROR: "));
	printf("[>%d / %4lu-%-4lu]", i, mem_allocs[i], mem_frees[i]);
    }

    printf("\r\n\n%s\r\n",
	    _("[bytes] total alloc-freed %lu-%lu, in use %lu, peak use %lu"),
	    mem_allocated, mem_freed, mem_allocated - mem_freed, mem_peak);
    printf("%s\r\n\n",
	    _("[calls] total re/malloc()'s %lu, total free()'s %lu"),
	    num_alloc, num_freed);
}

#endif /* MEM_PROFILE */

/*
 * Some memory is reserved for error messages and for being able to
 * call mf_release_all(), which needs some memory for mf_trans_add().
 */
#if defined(MSDOS) && !defined(DJGPP)
# define SMALL_MEM
# define KEEP_ROOM 8192L
#else
# define KEEP_ROOM (2 * 8192L)
#endif

static void vim_strup __ARGS((char_u *p));

/*
 * Note: if unsinged is 16 bits we can only allocate up to 64K with alloc().
 * Use lalloc for larger blocks.
 */
    char_u *
alloc(size)
    unsigned	    size;
{
    return (lalloc((long_u)size, TRUE));
}

/*
 * Allocate memory and set all bytes to zero.
 */
    char_u *
alloc_clear(size)
    unsigned	    size;
{
    char_u *p;

    p = (lalloc((long_u)size, TRUE));
    if (p != NULL)
	(void)vim_memset(p, 0, (size_t)size);
    return p;
}

/*
 * alloc() with check for maximum line length
 */
    char_u *
alloc_check(size)
    unsigned	    size;
{
#if !defined(UNIX) && !defined(__EMX__)
    if (sizeof(int) == 2 && size > 0x7fff)
    {
	/* Don't hide this message */
	emsg_silent = 0;
	msg_silent = 0;
	EMSG(_("Line is becoming too long"));
	return NULL;
    }
#endif
    return (lalloc((long_u)size, TRUE));
}

/*
 * Allocate memory like lalloc() and set all bytes to zero.
 */
    char_u *
lalloc_clear(size, message)
    long_u	size;
    int		message;
{
    char_u *p;

    p = (lalloc(size, message));
    if (p != NULL)
	(void)vim_memset(p, 0, (size_t)size);
    return p;
}

/*
 * Low level memory allocation function.
 * This is used often, KEEP IT FAST!
 */
    char_u *
lalloc(size, message)
    long_u	size;
    int		message;
{
    char_u	*p;		    /* pointer to new storage space */
    static int	releasing = FALSE;  /* don't do mf_release_all() recursive */
    int		try_again;
#if defined(HAVE_AVAIL_MEM) && !defined(SMALL_MEM)
    static long_u allocated = 0;    /* allocated since last avail check */
#endif

    /* Safety check for allocating zero bytes */
    if (size <= 0)
    {
	/* Don't hide this message */
	emsg_silent = 0;
	msg_silent = 0;
	EMSGN(_("Internal error: lalloc(%ld, )"), size);
	return NULL;
    }

#ifdef MEM_PROFILE
    mem_pre_alloc_l(&size);
#endif

#if defined(MSDOS) && !defined(DJGPP)
    if (size >= 0xfff0)		/* in MSDOS we can't deal with >64K blocks */
	p = NULL;
    else
#endif

    /*
     * Loop when out of memory: Try to release some memfile blocks and
     * if some blocks are released call malloc again.
     */
    for (;;)
    {
	/*
	 * Handle three kind of systems:
	 * 1. No check for available memory: Just return.
	 * 2. Slow check for available memory: call mch_avail_mem() after
	 *    allocating KEEP_ROOM amount of memory.
	 * 3. Strict check for available memory: call mch_avail_mem()
	 */
	if ((p = (char_u *)malloc((size_t)size)) != NULL)
	{
#ifndef HAVE_AVAIL_MEM
	    /* 1. No check for available memory: Just return. */
	    goto theend;
#else
# ifndef SMALL_MEM
	    /* 2. Slow check for available memory: call mch_avail_mem() after
	     *    allocating (KEEP_ROOM / 2) amount of memory. */
	    allocated += size;
	    if (allocated < KEEP_ROOM / 2)
		goto theend;
	    allocated = 0;
# endif
	    /* 3. check for available memory: call mch_avail_mem() */
	    if (mch_avail_mem(TRUE) < KEEP_ROOM && !releasing)
	    {
		vim_free((char *)p);	/* System is low... no go! */
		p = NULL;
	    }
	    else
		goto theend;
#endif
	}
	/*
	 * Remember that mf_release_all() is being called to avoid an endless
	 * loop, because mf_release_all() may call alloc() recursively.
	 */
	if (releasing)
	    break;
	releasing = TRUE;
	try_again = mf_release_all();
	releasing = FALSE;
	if (!try_again)
	    break;
    }

    if (message && p == NULL)
	do_outofmem_msg();

theend:
#ifdef MEM_PROFILE
    mem_post_alloc((void **)&p, (size_t)size);
#endif
    return p;
}

#if defined(MEM_PROFILE) || defined(PROTO)
/*
 * realloc() with memory profiling.
 */
    void *
mem_realloc(ptr, size)
    void *ptr;
    size_t size;
{
    void *p;

    mem_pre_free(&ptr);
    mem_pre_alloc_s(&size);

    p = realloc(ptr, size);

    mem_post_alloc(&p, size);

    return p;
}
#endif

/*
* Avoid repeating the error message many times (they take 1 second each).
* Did_outofmem_msg is reset when a character is read.
*/
    void
do_outofmem_msg()
{
    if (!did_outofmem_msg)
    {
	/* Don't hide this message */
	emsg_silent = 0;
	msg_silent = 0;
	EMSG(_(e_outofmem));
	did_outofmem_msg = TRUE;
    }
}

/*
 * copy a string into newly allocated memory
 */
    char_u *
vim_strsave(string)
    char_u	*string;
{
    char_u	*p;
    unsigned	len;

    len = STRLEN(string) + 1;
    p = alloc(len);
    if (p != NULL)
	mch_memmove(p, string, (size_t)len);
    return p;
}

    char_u *
vim_strnsave(string, len)
    char_u	*string;
    int		len;
{
    char_u	*p;

    p = alloc((unsigned)(len + 1));
    if (p != NULL)
    {
	STRNCPY(p, string, len);
	p[len] = NUL;
    }
    return p;
}

#if 0	/* not used */
/*
 * like vim_strnsave(), but remove backslashes from the string.
 */
    char_u *
vim_strnsave_esc(string, len)
    char_u	*string;
    int		len;
{
    char_u *p1, *p2;

    p1 = alloc((unsigned) (len + 1));
    if (p1 != NULL)
    {
	STRNCPY(p1, string, len);
	p1[len] = NUL;
	for (p2 = p1; *p2; ++p2)
	    if (*p2 == '\\' && *(p2 + 1) != NUL)
		STRCPY(p2, p2 + 1);
    }
    return p1;
}
#endif

/*
 * Same as vim_strsave(), but any characters found in esc_chars are preceded
 * by a backslash.
 */
    char_u *
vim_strsave_escaped(string, esc_chars)
    char_u	*string;
    char_u	*esc_chars;
{
    char_u	*p;
    char_u	*p2;
    char_u	*escaped_string;
    unsigned	length;
#ifdef FEAT_MBYTE
    int		l;
#endif

    /*
     * First count the number of backslashes required.
     * Then allocate the memory and insert them.
     */
    length = 1;				/* count the trailing '/' and NUL */
    for (p = string; *p; p++)
    {
#ifdef FEAT_MBYTE
	if (has_mbyte && (l = mb_ptr2len_check(p)) > 1)
	{
	    length += l;
	    p += l - 1;	/* skip multibyte */
	    continue;
	}
#endif
	if (vim_strchr(esc_chars, *p) != NULL)
	    ++length;			/* count a backslash */
	++length;			/* count an ordinary char */
    }
    escaped_string = alloc(length);
    if (escaped_string != NULL)
    {
	p2 = escaped_string;
	for (p = string; *p; p++)
	{
#ifdef FEAT_MBYTE
	    if (has_mbyte && (l = mb_ptr2len_check(p)) > 1)
	    {
		mch_memmove(p2, p, (size_t)l);
		p2 += l;
		p += l - 1;	/* skip multibyte char  */
		continue;
	    }
#endif
	    if (vim_strchr(esc_chars, *p) != NULL)
		*p2++ = '\\';
	    *p2++ = *p;
	}
	*p2 = NUL;
    }
    return escaped_string;
}

/*
 * Like vim_strsave(), but make all characters uppercase.
 * This uses ASCII lower-to-upper case translation, language independent.
 */
    char_u *
vim_strsave_up(string)
    char_u	*string;
{
    char_u *p1;

    p1 = vim_strsave(string);
    vim_strup(p1);
    return p1;
}

/*
 * Like vim_strnsave(), but make all characters uppercase.
 * This uses ASCII lower-to-upper case translation, language independent.
 */
    char_u *
vim_strnsave_up(string, len)
    char_u	*string;
    int		len;
{
    char_u *p1;

    p1 = vim_strnsave(string, len);
    vim_strup(p1);
    return p1;
}

/*
 * ASCII lower-to-upper case translation, language independent.
 */
    static void
vim_strup(p)
    char_u	*p;
{
    char_u  *p2;
    int	    c;

    if (p != NULL)
    {
	p2 = p;
	while ((c = *p2) != NUL)
#ifdef EBCDIC
	    *p2++ = isalpha(c) ? toupper(c) : c;
#else
	    *p2++ = (c < 'a' || c > 'z') ? c : (c - 0x20);
#endif
    }
}

/*
 * copy a space a number of times
 */
    void
copy_spaces(ptr, count)
    char_u	*ptr;
    size_t	count;
{
    size_t	i = count;
    char_u	*p = ptr;

    while (i--)
	*p++ = ' ';
}

#if defined(FEAT_VISUALEXTRA) || defined(PROTO)
/*
 * copy a character a number of times
 */
    void
copy_chars(ptr, count, c)
    char_u	*ptr;
    size_t	count;
    int		c;
{
    size_t	i = count;
    char_u	*p = ptr;

    while (i--)
	*p++ = c;
}
#endif

/*
 * delete spaces at the end of a string
 */
    void
del_trailing_spaces(ptr)
    char_u	*ptr;
{
    char_u	*q;

    q = ptr + STRLEN(ptr);
    while (--q > ptr && vim_iswhite(q[0]) && q[-1] != '\\' && q[-1] != Ctrl_V)
	*q = NUL;
}

/*
 * vim_strncpy()
 *
 * This is here because strncpy() does not guarantee successful results when
 * the to and from strings overlap.  It is only currently called from nextwild()
 * which copies part of the command line to another part of the command line.
 * This produced garbage when expanding files etc in the middle of the command
 * line (on my terminal, anyway) -- webb.
 */
    void
vim_strncpy(to, from, len)
    char_u *to;
    char_u *from;
    int len;
{
    int i;

    if (to <= from)
    {
	while (len-- && *from)
	    *to++ = *from++;
	if (len >= 0)
	    *to = *from;    /* Copy NUL */
    }
    else
    {
	for (i = 0; i < len; i++)
	{
	    to++;
	    if (*from++ == NUL)
	    {
		i++;
		break;
	    }
	}
	for (; i > 0; i--)
	    *--to = *--from;
    }
}

/*
 * Isolate one part of a string option where parts are separated with commas.
 * The part is copied into buf[maxlen].
 * "*option" is advanced to the next part.
 * The length is returned.
 */
    int
copy_option_part(option, buf, maxlen, sep_chars)
    char_u	**option;
    char_u	*buf;
    int		maxlen;
    char	*sep_chars;
{
    int	    len = 0;
    char_u  *p = *option;

    /* skip '.' at start of option part, for 'suffixes' */
    if (*p == '.')
	buf[len++] = *p++;
    while (*p && vim_strchr((char_u *)sep_chars, *p) == NULL)
    {
	/*
	 * Skip backslash before a separator character and space.
	 */
	if (p[0] == '\\' && vim_strchr((char_u *)sep_chars, p[1]) != NULL)
	    ++p;
	if (len < maxlen - 1)
	    buf[len++] = *p;
	++p;
    }
    buf[len] = NUL;

    p = skip_to_option_part(p);	/* p points to next file name */

    *option = p;
    return len;
}

/*
 * replacement for free() that ignores NULL pointers
 */
    void
vim_free(x)
    void *x;
{
    if (x != NULL)
    {
#ifdef MEM_PROFILE
	mem_pre_free(&x);
#endif
	free(x);
    }
}

#ifndef HAVE_MEMSET
    void *
vim_memset(ptr, c, size)
    void    *ptr;
    int	    c;
    size_t  size;
{
    char *p = ptr;

    while (size-- > 0)
	*p++ = c;
    return ptr;
}
#endif

#ifdef VIM_MEMCMP
/*
 * Return zero when "b1" and "b2" are the same for "len" bytes.
 * Return non-zero otherwise.
 */
    int
vim_memcmp(b1, b2, len)
    void    *b1;
    void    *b2;
    size_t  len;
{
    char_u  *p1 = (char_u *)b1, *p2 = (char_u *)b2;

    for ( ; len > 0; --len)
    {
	if (*p1 != *p2)
	    return 1;
	++p1;
	++p2;
    }
    return 0;
}
#endif

#ifdef VIM_MEMMOVE
/*
 * Version of memmove that handles overlapping source and destination.
 * For systems that don't have a function that is guaranteed to do that (SYSV).
 */
    void
mch_memmove(dst_arg, src_arg, len)
    void    *src_arg, *dst_arg;
    size_t  len;
{
    /*
     * A void doesn't have a size, we use char pointers.
     */
    char *dst = dst_arg, *src = src_arg;

					/* overlap, copy backwards */
    if (dst > src && dst < src + len)
    {
	src += len;
	dst += len;
	while (len-- > 0)
	    *--dst = *--src;
    }
    else				/* copy forwards */
	while (len-- > 0)
	    *dst++ = *src++;
}
#endif

#if (!defined(HAVE_STRCASECMP) && !defined(HAVE_STRICMP)) || defined(PROTO)
/*
 * Compare two strings, ignoring case.
 * return 0 for match, < 0 for smaller, > 0 for bigger
 */
    int
vim_stricmp(s1, s2)
    char	*s1;
    char	*s2;
{
    int		i;

    for (;;)
    {
	i = (int)TO_LOWER(*s1) - (int)TO_LOWER(*s2);
	if (i != 0)
	    return i;			    /* this character different */
	if (*s1 == NUL)
	    break;			    /* strings match until NUL */
	++s1;
	++s2;
    }
    return 0;				    /* strings match */
}
#endif

#if (!defined(HAVE_STRNCASECMP) && !defined(HAVE_STRNICMP)) || defined(PROTO)
/*
 * Compare two strings, for length "len", ignoring case.
 * return 0 for match, < 0 for smaller, > 0 for bigger
 */
    int
vim_strnicmp(s1, s2, len)
    char	*s1;
    char	*s2;
    size_t	len;
{
    int		i;

    while (len > 0)
    {
	i = (int)TO_LOWER(*s1) - (int)TO_LOWER(*s2);
	if (i != 0)
	    return i;			    /* this character different */
	if (*s1 == NUL)
	    break;			    /* strings match until NUL */
	++s1;
	++s2;
	--len;
    }
    return 0;				    /* strings match */
}
#endif

/*
 * Check if string "s2" appears somewhere in "s1" while ignoring case.
 * Return NULL if not, a pointer to the first occurrence if it does.
 */
    char_u *
vim_stristr(s1, s2)
    char_u	*s1;
    char_u	*s2;
{
    char_u	*p;
    int		len = STRLEN(s2);
    char_u	*end = s1 + STRLEN(s1) - len;

    for (p = s1; p <= end; ++p)
	if (STRNICMP(p, s2, len) == 0)
	    return p;
    return NULL;
}

/*
 * Version of strchr() and strrchr() that handle unsigned char strings
 * with characters above 128 correctly. Also it doesn't return a pointer to
 * the NUL at the end of the string.
 */
    char_u  *
vim_strchr(string, c)
    char_u	*string;
    int		c;
{
    char_u	*p;
    int		b;

    p = string;
#ifdef FEAT_MBYTE
    if (cc_utf8 && c >= 0x80)
    {
	while (*p != NUL)
	{
	    if (utf_ptr2char(p) == c)
		return p;
	    p += mb_ptr2len_check(p);
	}
	return NULL;
    }
    if (cc_dbcs && c > 255)
    {
	int	n2 = c & 0xff;

	c = ((unsigned)c >> 8) & 0xff;
	while ((b = *p) != NUL)
	{
	    if (b == c && p[1] == n2)
		return p;
	    p += mb_ptr2len_check(p);
	}
	return NULL;
    }
    if (has_mbyte)
    {
	while ((b = *p) != NUL)
	{
	    if (b == c)
		return p;
	    p += mb_ptr2len_check(p);
	}
	return NULL;
    }
#endif
    while ((b = *p) != NUL)
    {
	if (b == c)
	    return p;
	++p;
    }
    return NULL;
}

/*
 * Search for last occurrence of "c" in "string".
 * return NULL if not found.
 * Does not handle multi-byte!
 */
    char_u  *
vim_strrchr(string, c)
    char_u	*string;
    int		c;
{
    char_u	*retval = NULL;

    while (*string)
    {
	if (*string == c)
	    retval = string;
#ifdef FEAT_MBYTE
	if (has_mbyte)
	    string += mb_ptr2len_check(string);
	else
#endif
	    ++string;
    }
    return retval;
}

/*
 * Vim's version of strpbrk(), in case it's missing.
 * Don't generate a prototype for this, causes problems when it's not used.
 */
#ifndef PROTO
# ifndef HAVE_STRPBRK
#  ifdef vim_strpbrk
#   undef vim_strpbrk
#  endif
    char_u *
vim_strpbrk(s, charset)
    char_u	*s;
    char_u	*charset;
{
    while (*s)
    {
	if (vim_strchr(charset, *s) != NULL)
	    return s;
	++s;
    }
    return NULL;
}
# endif
#endif

/*
 * Vim has its own isspace() function, because on some machines isspace()
 * can't handle characters above 128.
 */
    int
vim_isspace(x)
    int	    x;
{
    return ((x >= 9 && x <= 13) || x == ' ');
}

/************************************************************************
 * Functions for hanlding growing arrays.
 */

/*
 * Clear an allocated growing array.
 */
    void
ga_clear(gap)
    garray_t *gap;
{
    vim_free(gap->ga_data);
    ga_init(gap);
}

#if defined(FEAT_EVAL) || defined(PROTO)
/*
 * Clear a growing array that contains a list of strings.
 */
    void
ga_clear_strings(gap)
    garray_t *gap;
{
    int		i;

    for (i = 0; i < gap->ga_len; ++i)
	vim_free(((char_u **)(gap->ga_data))[i]);
    ga_clear(gap);
}
#endif

/*
 * Initialize a growing array.	Don't forget to set ga_itemsize and
 * ga_growsize!  Or use ga_init2().
 */
    void
ga_init(gap)
    garray_t *gap;
{
    gap->ga_data = NULL;
    gap->ga_room = 0;
    gap->ga_len = 0;
}

    void
ga_init2(gap, itemsize, growsize)
    garray_t	*gap;
    int		itemsize;
    int		growsize;
{
    ga_init(gap);
    gap->ga_itemsize = itemsize;
    gap->ga_growsize = growsize;
}

/*
 * Make room in growing array "gap" for at least "n" items.
 * Return FAIL for failure, OK otherwise.
 */
    int
ga_grow(gap, n)
    garray_t	*gap;
    int		n;
{
    size_t	len;
    char_u	*pp;

    if (gap->ga_room < n)
    {
	if (n < gap->ga_growsize)
	    n = gap->ga_growsize;
	len = gap->ga_itemsize * (gap->ga_len + n);
	pp = alloc_clear((unsigned)len);
	if (pp == NULL)
	    return FAIL;
	gap->ga_room = n;
	if (gap->ga_data != NULL)
	{
	    mch_memmove(pp, gap->ga_data,
				      (size_t)(gap->ga_itemsize * gap->ga_len));
	    vim_free(gap->ga_data);
	}
	gap->ga_data = pp;
    }
    return OK;
}

#if defined(FEAT_EVAL) || defined(FEAT_CMDL_COMPL) || defined(PROTO)
/*
 * Concatenate a string to a growarray which contains characters.
 * Note: Does NOT copy the NUL at the end!
 */
    void
ga_concat(gap, s)
    garray_t	*gap;
    char_u	*s;
{
    size_t    len = STRLEN(s);

    if (ga_grow(gap, (int)len) == OK)
    {
	mch_memmove((char *)gap->ga_data + gap->ga_len, s, len);
	gap->ga_len += len;
	gap->ga_room -= len;
    }
}

/*
 * Append a character to a growarray which contains characters.
 */
    void
ga_append(gap, c)
    garray_t	*gap;
    int		c;
{
    if (ga_grow(gap, 1) == OK)
    {
	*((char *)gap->ga_data + gap->ga_len) = c;
	++gap->ga_len;
	--gap->ga_room;
    }
}
#endif

/************************************************************************
 * functions that use lookup tables for various things, generally to do with
 * special key codes.
 */

/*
 * Some useful tables.
 */

static struct modmasktable
{
    int	    mod_mask;	    /* Bit-mask for particular key modifier */
    char_u  name;	    /* Single letter name of modifier */
} mod_mask_table[] =
{
    {MOD_MASK_ALT,	(char_u)'M'},
    {MOD_MASK_CTRL,	(char_u)'C'},
    {MOD_MASK_SHIFT,	(char_u)'S'},
    {MOD_MASK_2CLICK,	(char_u)'2'},
    {MOD_MASK_3CLICK,	(char_u)'3'},
    {MOD_MASK_4CLICK,	(char_u)'4'},
#ifdef macintosh
    {MOD_MASK_CMD,      (char_u)'D'},
#endif
    {0x0,		NUL}
};

/*
 * Shifted key terminal codes and their unshifted equivalent.
 * Don't add mouse codes here, they are handled seperately!
 */
static char_u shifted_keys_table[] =
{
/*  shifted			unshifted  */
    '&', '9',			'@', '1',		/* begin */
    '&', '0',			'@', '2',		/* cancel */
    '*', '1',			'@', '4',		/* command */
    '*', '2',			'@', '5',		/* copy */
    '*', '3',			'@', '6',		/* create */
    '*', '4',			'k', 'D',		/* delete char */
    '*', '5',			'k', 'L',		/* delete line */
    '*', '7',			'@', '7',		/* end */
    '*', '9',			'@', '9',		/* exit */
    '*', '0',			'@', '0',		/* find */
    '#', '1',			'%', '1',		/* help */
    '#', '2',			'k', 'h',		/* home */
    '#', '3',			'k', 'I',		/* insert */
    '#', '4',			'k', 'l',		/* left arrow */
    '%', 'a',			'%', '3',		/* message */
    '%', 'b',			'%', '4',		/* move */
    '%', 'c',			'%', '5',		/* next */
    '%', 'd',			'%', '7',		/* options */
    '%', 'e',			'%', '8',		/* previous */
    '%', 'f',			'%', '9',		/* print */
    '%', 'g',			'%', '0',		/* redo */
    '%', 'h',			'&', '3',		/* replace */
    '%', 'i',			'k', 'r',		/* right arrow */
    '%', 'j',			'&', '5',		/* resume */
    '!', '1',			'&', '6',		/* save */
    '!', '2',			'&', '7',		/* suspend */
    '!', '3',			'&', '8',		/* undo */
    KS_EXTRA, (int)KE_S_UP,	'k', 'u',		/* up arrow */
    KS_EXTRA, (int)KE_S_DOWN,	'k', 'd',		/* down arrow */

    KS_EXTRA, (int)KE_S_XF1,	KS_EXTRA, (int)KE_XF1,	/* vt100 F1 */
    KS_EXTRA, (int)KE_S_XF2,	KS_EXTRA, (int)KE_XF2,
    KS_EXTRA, (int)KE_S_XF3,	KS_EXTRA, (int)KE_XF3,
    KS_EXTRA, (int)KE_S_XF4,	KS_EXTRA, (int)KE_XF4,

    KS_EXTRA, (int)KE_S_F1,	'k', '1',		/* F1 */
    KS_EXTRA, (int)KE_S_F2,	'k', '2',
    KS_EXTRA, (int)KE_S_F3,	'k', '3',
    KS_EXTRA, (int)KE_S_F4,	'k', '4',
    KS_EXTRA, (int)KE_S_F5,	'k', '5',
    KS_EXTRA, (int)KE_S_F6,	'k', '6',
    KS_EXTRA, (int)KE_S_F7,	'k', '7',
    KS_EXTRA, (int)KE_S_F8,	'k', '8',
    KS_EXTRA, (int)KE_S_F9,	'k', '9',
    KS_EXTRA, (int)KE_S_F10,	'k', ';',		/* F10 */

    KS_EXTRA, (int)KE_S_F11,	'F', '1',
    KS_EXTRA, (int)KE_S_F12,	'F', '2',
    KS_EXTRA, (int)KE_S_F13,	'F', '3',
    KS_EXTRA, (int)KE_S_F14,	'F', '4',
    KS_EXTRA, (int)KE_S_F15,	'F', '5',
    KS_EXTRA, (int)KE_S_F16,	'F', '6',
    KS_EXTRA, (int)KE_S_F17,	'F', '7',
    KS_EXTRA, (int)KE_S_F18,	'F', '8',
    KS_EXTRA, (int)KE_S_F19,	'F', '9',
    KS_EXTRA, (int)KE_S_F20,	'F', 'A',

    KS_EXTRA, (int)KE_S_F21,	'F', 'B',
    KS_EXTRA, (int)KE_S_F22,	'F', 'C',
    KS_EXTRA, (int)KE_S_F23,	'F', 'D',
    KS_EXTRA, (int)KE_S_F24,	'F', 'E',
    KS_EXTRA, (int)KE_S_F25,	'F', 'F',
    KS_EXTRA, (int)KE_S_F26,	'F', 'G',
    KS_EXTRA, (int)KE_S_F27,	'F', 'H',
    KS_EXTRA, (int)KE_S_F28,	'F', 'I',
    KS_EXTRA, (int)KE_S_F29,	'F', 'J',
    KS_EXTRA, (int)KE_S_F30,	'F', 'K',

    KS_EXTRA, (int)KE_S_F31,	'F', 'L',
    KS_EXTRA, (int)KE_S_F32,	'F', 'M',
    KS_EXTRA, (int)KE_S_F33,	'F', 'N',
    KS_EXTRA, (int)KE_S_F34,	'F', 'O',
    KS_EXTRA, (int)KE_S_F35,	'F', 'P',
    KS_EXTRA, (int)KE_S_F36,	'F', 'Q',
    KS_EXTRA, (int)KE_S_F37,	'F', 'R',

    KS_EXTRA, (int)KE_S_TAB,	KS_EXTRA, (int)KE_TAB,	/* TAB pseudo code*/

    NUL
};

static struct key_name_entry
{
    int	    key;	/* Special key code or ascii value */
    char_u  *name;	/* Name of key */
} key_names_table[] =
{
    {' ',		(char_u *)"Space"},
    {TAB,		(char_u *)"Tab"},
    {K_TAB,		(char_u *)"Tab"},
    {NL,		(char_u *)"NL"},
    {NL,		(char_u *)"NewLine"},	/* Alternative name */
    {NL,		(char_u *)"LineFeed"},	/* Alternative name */
    {NL,		(char_u *)"LF"},	/* Alternative name */
    {CR,		(char_u *)"CR"},
    {CR,		(char_u *)"Return"},	/* Alternative name */
    {K_BS,		(char_u *)"BS"},
    {K_BS,		(char_u *)"BackSpace"},	/* Alternative name */
    {ESC,		(char_u *)"Esc"},
    {CSI,		(char_u *)"CSI"},
    {K_CSI,		(char_u *)"xCSI"},
    {'|',		(char_u *)"Bar"},
    {'\\',		(char_u *)"Bslash"},
    {K_DEL,		(char_u *)"Del"},
    {K_DEL,		(char_u *)"Delete"},	/* Alternative name */
    {K_KDEL,		(char_u *)"kDel"},
    {K_UP,		(char_u *)"Up"},
    {K_DOWN,		(char_u *)"Down"},
    {K_LEFT,		(char_u *)"Left"},
    {K_RIGHT,		(char_u *)"Right"},

    {K_F1,		(char_u *)"F1"},
    {K_F2,		(char_u *)"F2"},
    {K_F3,		(char_u *)"F3"},
    {K_F4,		(char_u *)"F4"},
    {K_F5,		(char_u *)"F5"},
    {K_F6,		(char_u *)"F6"},
    {K_F7,		(char_u *)"F7"},
    {K_F8,		(char_u *)"F8"},
    {K_F9,		(char_u *)"F9"},
    {K_F10,		(char_u *)"F10"},

    {K_F11,		(char_u *)"F11"},
    {K_F12,		(char_u *)"F12"},
    {K_F13,		(char_u *)"F13"},
    {K_F14,		(char_u *)"F14"},
    {K_F15,		(char_u *)"F15"},
    {K_F16,		(char_u *)"F16"},
    {K_F17,		(char_u *)"F17"},
    {K_F18,		(char_u *)"F18"},
    {K_F19,		(char_u *)"F19"},
    {K_F20,		(char_u *)"F20"},

    {K_F21,		(char_u *)"F21"},
    {K_F22,		(char_u *)"F22"},
    {K_F23,		(char_u *)"F23"},
    {K_F24,		(char_u *)"F24"},
    {K_F25,		(char_u *)"F25"},
    {K_F26,		(char_u *)"F26"},
    {K_F27,		(char_u *)"F27"},
    {K_F28,		(char_u *)"F28"},
    {K_F29,		(char_u *)"F29"},
    {K_F30,		(char_u *)"F30"},

    {K_F31,		(char_u *)"F31"},
    {K_F32,		(char_u *)"F32"},
    {K_F33,		(char_u *)"F33"},
    {K_F34,		(char_u *)"F34"},
    {K_F35,		(char_u *)"F35"},
    {K_F36,		(char_u *)"F36"},
    {K_F37,		(char_u *)"F37"},

    {K_XF1,		(char_u *)"xF1"},
    {K_XF2,		(char_u *)"xF2"},
    {K_XF3,		(char_u *)"xF3"},
    {K_XF4,		(char_u *)"xF4"},

    {K_HELP,		(char_u *)"Help"},
    {K_UNDO,		(char_u *)"Undo"},
    {K_INS,		(char_u *)"Insert"},
    {K_INS,		(char_u *)"Ins"},	/* Alternative name */
    {K_KINS,		(char_u *)"kInsert"},
    {K_HOME,		(char_u *)"Home"},
    {K_KHOME,		(char_u *)"kHome"},
    {K_XHOME,		(char_u *)"xHome"},
    {K_END,		(char_u *)"End"},
    {K_KEND,		(char_u *)"kEnd"},
    {K_XEND,		(char_u *)"xEnd"},
    {K_PAGEUP,		(char_u *)"PageUp"},
    {K_PAGEDOWN,	(char_u *)"PageDown"},
    {K_KPAGEUP,		(char_u *)"kPageUp"},
    {K_KPAGEDOWN,	(char_u *)"kPageDown"},

    {K_KPLUS,		(char_u *)"kPlus"},
    {K_KMINUS,		(char_u *)"kMinus"},
    {K_KDIVIDE,		(char_u *)"kDivide"},
    {K_KMULTIPLY,	(char_u *)"kMultiply"},
    {K_KENTER,		(char_u *)"kEnter"},
    {K_KPOINT,		(char_u *)"kPoint"},

    {K_K0,		(char_u *)"k0"},
    {K_K1,		(char_u *)"k1"},
    {K_K2,		(char_u *)"k2"},
    {K_K3,		(char_u *)"k3"},
    {K_K4,		(char_u *)"k4"},
    {K_K5,		(char_u *)"k5"},
    {K_K6,		(char_u *)"k6"},
    {K_K7,		(char_u *)"k7"},
    {K_K8,		(char_u *)"k8"},
    {K_K9,		(char_u *)"k9"},

    {'<',		(char_u *)"lt"},

    {K_MOUSE,		(char_u *)"Mouse"},
    {K_LEFTMOUSE,	(char_u *)"LeftMouse"},
    {K_LEFTMOUSE_NM,	(char_u *)"LeftMouseNM"},
    {K_LEFTDRAG,	(char_u *)"LeftDrag"},
    {K_LEFTRELEASE,	(char_u *)"LeftRelease"},
    {K_LEFTRELEASE_NM,	(char_u *)"LeftReleaseNM"},
    {K_MIDDLEMOUSE,	(char_u *)"MiddleMouse"},
    {K_MIDDLEDRAG,	(char_u *)"MiddleDrag"},
    {K_MIDDLERELEASE,	(char_u *)"MiddleRelease"},
    {K_RIGHTMOUSE,	(char_u *)"RightMouse"},
    {K_RIGHTDRAG,	(char_u *)"RightDrag"},
    {K_RIGHTRELEASE,	(char_u *)"RightRelease"},
    {K_MOUSEDOWN,	(char_u *)"MouseDown"},
    {K_MOUSEUP,		(char_u *)"MouseUp"},
    {K_ZERO,		(char_u *)"Nul"},
    {0,			NULL}
};

#define KEY_NAMES_TABLE_LEN (sizeof(key_names_table) / sizeof(struct key_name_entry))

#ifdef FEAT_MOUSE
static struct mousetable
{
    int	    pseudo_code;	/* Code for pseudo mouse event */
    int	    button;		/* Which mouse button is it? */
    int	    is_click;		/* Is it a mouse button click event? */
    int	    is_drag;		/* Is it a mouse drag event? */
} mouse_table[] =
{
    {(int)KE_LEFTMOUSE,		MOUSE_LEFT,	TRUE,	FALSE},
#ifdef FEAT_GUI
    {(int)KE_LEFTMOUSE_NM,	MOUSE_LEFT,	TRUE,	FALSE},
#endif
    {(int)KE_LEFTDRAG,		MOUSE_LEFT,	FALSE,	TRUE},
    {(int)KE_LEFTRELEASE,	MOUSE_LEFT,	FALSE,	FALSE},
#ifdef FEAT_GUI
    {(int)KE_LEFTRELEASE_NM,	MOUSE_LEFT,	FALSE,	FALSE},
#endif
    {(int)KE_MIDDLEMOUSE,	MOUSE_MIDDLE,	TRUE,	FALSE},
    {(int)KE_MIDDLEDRAG,	MOUSE_MIDDLE,	FALSE,	TRUE},
    {(int)KE_MIDDLERELEASE,	MOUSE_MIDDLE,	FALSE,	FALSE},
    {(int)KE_RIGHTMOUSE,	MOUSE_RIGHT,	TRUE,	FALSE},
    {(int)KE_RIGHTDRAG,		MOUSE_RIGHT,	FALSE,	TRUE},
    {(int)KE_RIGHTRELEASE,	MOUSE_RIGHT,	FALSE,	FALSE},
    /* DRAG without CLICK */
    {(int)KE_IGNORE,		MOUSE_RELEASE,	FALSE,	TRUE},
    /* RELEASE without CLICK */
    {(int)KE_IGNORE,		MOUSE_RELEASE,	FALSE,	FALSE},
    {0,				0,		0,	0},
};
#endif /* FEAT_MOUSE */

/*
 * Return the modifier mask bit (MOD_MASK_*) which corresponds to the given
 * modifier name ('S' for Shift, 'C' for Ctrl etc).
 */
    int
name_to_mod_mask(c)
    int	    c;
{
    int	    i;

    if (c <= 255)	/* avoid TO_LOWER() with number > 255 */
    {
	for (i = 0; mod_mask_table[i].mod_mask; i++)
	    if (TO_LOWER(c) == TO_LOWER(mod_mask_table[i].name))
		return mod_mask_table[i].mod_mask;
    }
    return 0x0;
}

#if 0 /* not used */
/*
 * Decide whether the given key code (K_*) is a shifted special
 * key (by looking at mod_mask).  If it is, then return the appropriate shifted
 * key code, otherwise just return the character as is.
 */
    int
check_shifted_spec_key(c)
    int	    c;
{
    return simplify_key(c, &mod_mask);
}
#endif

/*
 * Check if if there is a special key code for "key" that includes the
 * modifiers specified.
 */
    int
simplify_key(key, modifiers)
    int	    key;
    int	    *modifiers;
{
    int	    i;
    int	    key0;
    int	    key1;

    if (*modifiers & MOD_MASK_SHIFT)
    {
	if (key == TAB)		/* TAB is a special case */
	{
	    *modifiers &= ~MOD_MASK_SHIFT;
	    return K_S_TAB;
	}
	key0 = KEY2TERMCAP0(key);
	key1 = KEY2TERMCAP1(key);
	for (i = 0; shifted_keys_table[i] != NUL; i += 4)
	    if (key0 == shifted_keys_table[i + 2] &&
					    key1 == shifted_keys_table[i + 3])
	    {
		*modifiers &= ~MOD_MASK_SHIFT;
		return TERMCAP2KEY(shifted_keys_table[i],
						   shifted_keys_table[i + 1]);
	    }
    }
    return key;
}

/*
 * Return a string which contains the name of the given key when the given
 * modifiers are down.
 */
    char_u *
get_special_key_name(c, modifiers)
    int	    c;
    int	    modifiers;
{
    static char_u string[MAX_KEY_NAME_LEN + 1];

    int	    i, idx;
    int	    table_idx;
    char_u  *s;

    string[0] = '<';
    idx = 1;

    /*
     * Translate shifted special keys into unshifted keys and set modifier.
     */
    if (IS_SPECIAL(c))
    {
	for (i = 0; shifted_keys_table[i]; i += 4)
	    if (       KEY2TERMCAP0(c) == shifted_keys_table[i]
		    && KEY2TERMCAP1(c) == shifted_keys_table[i + 1])
	    {
		modifiers |= MOD_MASK_SHIFT;
		c = TERMCAP2KEY(shifted_keys_table[i + 2],
						   shifted_keys_table[i + 3]);
		break;
	    }
    }

    /* try to find the key in the special key table */
    table_idx = find_special_key_in_table(c);

    /*
     * When not a known special key, and not a printable character, try to
     * extract modifiers.
     */
#ifdef FEAT_MBYTE
    if (mb_char2len(c) == 1)
#endif
    {
	if (table_idx < 0
		&& (!vim_isprintc(c) || (c & 0x7f) == ' ')
		&& (c & 0x80))
	{
	    c &= 0x7f;
	    modifiers |= MOD_MASK_ALT;
	    /* try again, to find the un-alted key in the special key table */
	    table_idx = find_special_key_in_table(c);
	}
	if (table_idx < 0 && !vim_isprintc(c) && c < ' ')
	{
#ifdef EBCDIC
	    c = CtrlChar(c);
#else
	    c += '@';
#endif
	    modifiers |= MOD_MASK_CTRL;
	}
    }

    /* translate the modifier into a string */
    for (i = 0; mod_mask_table[i].mod_mask; i++)
	if (modifiers & mod_mask_table[i].mod_mask)
	{
	    string[idx++] = mod_mask_table[i].name;
	    string[idx++] = (char_u)'-';
	}

    if (table_idx < 0)		/* unknown special key, may output t_xx */
    {
	if (IS_SPECIAL(c))
	{
	    string[idx++] = 't';
	    string[idx++] = '_';
	    string[idx++] = KEY2TERMCAP0(c);
	    string[idx++] = KEY2TERMCAP1(c);
	}
	/* Not a special key, only modifiers, output directly */
	else
	{
#ifdef FEAT_MBYTE
	    if (has_mbyte && mb_char2len(c) > 1)
		idx += mb_char2bytes(c, string + idx);
	    else
#endif
	    if (vim_isprintc(c))
		string[idx++] = c;
	    else
	    {
		s = transchar(c);
		while (*s)
		    string[idx++] = *s++;
	    }
	}
    }
    else		/* use name of special key */
    {
	STRCPY(string + idx, key_names_table[table_idx].name);
	idx = STRLEN(string);
    }
    string[idx++] = '>';
    string[idx] = NUL;
    return string;
}

/*
 * Try translating a <> name at (*srcp)[] to dst[].
 * Return the number of characters added to dst[], zero for no match.
 * If there is a match, srcp is advanced to after the <> name.
 * dst[] must be big enough to hold the result (up to six characters)!
 */
    int
trans_special(srcp, dst, keycode)
    char_u	**srcp;
    char_u	*dst;
    int		keycode; /* prefer key code, e.g. K_DEL instead of DEL */
{
    int	    modifiers;
    int	    key;
    int	    dlen = 0;

    key = find_special_key(srcp, &modifiers, keycode);
    if (key == 0)
	return 0;

    /* Put the appropriate modifier in a string */
    if (modifiers != 0)
    {
	dst[dlen++] = K_SPECIAL;
	dst[dlen++] = KS_MODIFIER;
	dst[dlen++] = modifiers;
    }

    if (IS_SPECIAL(key))
    {
	dst[dlen++] = K_SPECIAL;
	dst[dlen++] = KEY2TERMCAP0(key);
	dst[dlen++] = KEY2TERMCAP1(key);
    }
    else
	dst[dlen++] = key;

    return dlen;
}

/*
 * Try translating a <> name at (*srcp)[], return the key and modifiers.
 * srcp is advanced to after the <> name.
 * returns 0 if there is no match.
 */
    int
find_special_key(srcp, modp, keycode)
    char_u	**srcp;
    int		*modp;
    int		keycode; /* prefer key code, e.g. K_DEL instead of DEL */
{
    char_u  *last_dash;
    char_u  *end_of_name;
    char_u  *src;
    char_u  *bp;
    int	    modifiers;
    int	    bit;
    int	    key;

    src = *srcp;
    if (src[0] != '<')
	return 0;

    /* Find end of modifier list */
    last_dash = src;
    for (bp = src + 1; *bp == '-' || vim_isIDc(*bp); bp++)
    {
	if (*bp == '-')
	{
	    last_dash = bp;
	    if (bp[1] != NUL && bp[2] == '>')
		++bp;	/* anything accepted, like <C-?> */
	}
	if (bp[0] == 't' && bp[1] == '_' && bp[2] && bp[3])
	    bp += 3;	/* skip t_xx, xx may be '-' or '>' */
    }

    if (*bp == '>')	/* found matching '>' */
    {
	end_of_name = bp + 1;

	/* Which modifiers are given? */
	modifiers = 0x0;
	for (bp = src + 1; bp < last_dash; bp++)
	{
	    if (*bp != '-')
	    {
		bit = name_to_mod_mask(*bp);
		if (bit == 0x0)
		    break;	/* Illegal modifier name */
		modifiers |= bit;
	    }
	}

	/*
	 * Legal modifier name.
	 */
	if (bp >= last_dash)
	{
	    /*
	     * Modifier with single letter, or special key name.
	     */
	    if (modifiers != 0 && last_dash[2] == '>')
		key = last_dash[1];
	    else
		key = get_special_key_code(last_dash + 1);

	    /*
	     * get_special_key_code() may return NUL for invalid
	     * special key name.
	     */
	    if (key != NUL)
	    {
		/*
		 * Only use a modifier when there is no special key code that
		 * includes the modifier.
		 */
		key = simplify_key(key, &modifiers);

		if (!keycode)
		{
		    /* don't want keycode, use single byte code */
		    if (key == K_BS)
			key = BS;
		    else if (key == K_DEL || key == K_KDEL)
			key = DEL;
		}

		/*
		 * Normal Key with modifier: Try to make a single byte code.
		 */
		if (!IS_SPECIAL(key))
		{
		    if ((modifiers & MOD_MASK_SHIFT) && isalpha(key))
		    {
			key = TO_UPPER(key);
			modifiers &= ~MOD_MASK_SHIFT;
		    }
		    if ((modifiers & MOD_MASK_CTRL)
#ifdef EBCDIC
			    /*
			     * TODO: EBCDIC Better use:
			     * && (Ctrl(key) || key == '?')
			     * ???
			     */
			    && strchr("?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_", key)
								       != NULL
#else
			    && ((key >= '?' && key <= '_') || isalpha(key))
#endif
			    )
		    {
			if (key == '?')
			    key = DEL;
			else
#ifdef EBCDIC
			    key = Ctrl(key);
#else
			    key &= 0x1f;
#endif
			modifiers &= ~MOD_MASK_CTRL;
		    }
		    if ((modifiers & MOD_MASK_ALT) && key < 0x80)
		    {
			key |= 0x80;
			modifiers &= ~MOD_MASK_ALT;
		    }
		}

		*modp = modifiers;
		*srcp = end_of_name;
		return key;
	    }
	}
    }
    return 0;
}

/*
 * Try to find key "c" in the special key table.
 * Return the index when found, -1 when not found.
 */
    int
find_special_key_in_table(c)
    int	    c;
{
    int	    i;

    for (i = 0; key_names_table[i].name != NULL; i++)
	if (c == key_names_table[i].key)
	    break;
    if (key_names_table[i].name == NULL)
	i = -1;
    return i;
}

/*
 * Find the special key with the given name (the given string does not have to
 * end with NUL, the name is assumed to end before the first non-idchar).
 * If the name starts with "t_" the next two characters are interpreted as a
 * termcap name.
 * Return the key code, or 0 if not found.
 */
    int
get_special_key_code(name)
    char_u  *name;
{
    char_u  *table_name;
    char_u  string[3];
    int	    i, j;

    /*
     * If it's <t_xx> we get the code for xx from the termcap
     */
    if (name[0] == 't' && name[1] == '_' && name[2] != NUL && name[3] != NUL)
    {
	string[0] = name[2];
	string[1] = name[3];
	string[2] = NUL;
	if (add_termcap_entry(string, FALSE) == OK)
	    return TERMCAP2KEY(name[2], name[3]);
    }
    else
	for (i = 0; key_names_table[i].name != NULL; i++)
	{
	    table_name = key_names_table[i].name;
	    for (j = 0; vim_isIDc(name[j]) && table_name[j] != NUL; j++)
		if (TO_LOWER(table_name[j]) != TO_LOWER(name[j]))
		    break;
	    if (!vim_isIDc(name[j]) && table_name[j] == NUL)
		return key_names_table[i].key;
	}
    return 0;
}

#ifdef FEAT_CMDL_COMPL
    char_u *
get_key_name(i)
    int	    i;
{
    if (i >= KEY_NAMES_TABLE_LEN)
	return NULL;
    return  key_names_table[i].name;
}
#endif

#ifdef FEAT_MOUSE
/*
 * Look up the given mouse code to return the relevant information in the other
 * arguments.  Return which button is down or was released.
 */
    int
get_mouse_button(code, is_click, is_drag)
    int	    code;
    int	    *is_click;
    int	    *is_drag;
{
    int	    i;

    for (i = 0; mouse_table[i].pseudo_code; i++)
	if (code == mouse_table[i].pseudo_code)
	{
	    *is_click = mouse_table[i].is_click;
	    *is_drag = mouse_table[i].is_drag;
	    return mouse_table[i].button;
	}
    return 0;	    /* Shouldn't get here */
}

/*
 * Return the appropriate pseudo mouse event token (KE_LEFTMOUSE etc) based on
 * the given information about which mouse button is down, and whether the
 * mouse was clicked, dragged or released.
 */
    int
get_pseudo_mouse_code(button, is_click, is_drag)
    int	    button;	/* eg MOUSE_LEFT */
    int	    is_click;
    int	    is_drag;
{
    int	    i;

    for (i = 0; mouse_table[i].pseudo_code; i++)
	if (button == mouse_table[i].button
	    && is_click == mouse_table[i].is_click
	    && is_drag == mouse_table[i].is_drag)
	{
#ifdef FEAT_GUI
	    /* Trick: a non mappable left click and release has mouse_col < 0.
	     * Used for 'mousefocus' in gui_mouse_moved() */
	    if (mouse_col < 0)
	    {
		mouse_col = 0;
		if (mouse_table[i].pseudo_code == (int)KE_LEFTMOUSE)
		    return (int)KE_LEFTMOUSE_NM;
		if (mouse_table[i].pseudo_code == (int)KE_LEFTRELEASE)
		    return (int)KE_LEFTRELEASE_NM;
	    }
#endif
	    return mouse_table[i].pseudo_code;
	}
    return (int)KE_IGNORE;	    /* not recongnized, ignore it */
}
#endif /* FEAT_MOUSE */

/*
 * Return the current end-of-line type: EOL_DOS, EOL_UNIX or EOL_MAC.
 */
    int
get_fileformat(buf)
    buf_t	*buf;
{
    int		c = *buf->b_p_ff;

    if (buf->b_p_bin || c == 'u')
	return EOL_UNIX;
    if (c == 'm')
	return EOL_MAC;
    return EOL_DOS;
}

/*
 * Like get_fileformat(), but override 'fileformat' with "p" for "++opt=val"
 * argument.
 */
    int
get_fileformat_force(buf, eap)
    buf_t	*buf;
    exarg_t	*eap;
{
    int		c;

    if (eap != NULL && eap->force_ff != NULL)
	c = *eap->force_ff;
    else
    {
	if (buf->b_p_bin)
	    return EOL_UNIX;
	c = *buf->b_p_ff;
    }
    if (c == 'u')
	return EOL_UNIX;
    if (c == 'm')
	return EOL_MAC;
    return EOL_DOS;
}

/*
 * Set the current end-of-line type to EOL_DOS, EOL_UNIX or EOL_MAC.
 * Sets both 'textmode' and 'fileformat'.
 * Note: Does _not_ set global value of 'textmode'!
 */
    void
set_fileformat(t, local)
    int		t;
    int		local;		/* only change the local value */
{
    char	*p = NULL;

    switch (t)
    {
    case EOL_DOS:
	p = FF_DOS;
	curbuf->b_p_tx = TRUE;
	break;
    case EOL_UNIX:
	p = FF_UNIX;
	curbuf->b_p_tx = FALSE;
	break;
    case EOL_MAC:
	p = FF_MAC;
	curbuf->b_p_tx = FALSE;
	break;
    }
    if (p != NULL)
	set_string_option_direct((char_u *)"ff", -1, (char_u *)p,
					 OPT_FREE | (local ? 0 : OPT_GLOBAL));
#ifdef FEAT_WINDOWS
    check_status(curbuf);
#endif
}

/*
 * Return the default fileformat from 'fileformats'.
 */
    int
default_fileformat()
{
    switch (*p_ffs)
    {
	case 'm':   return EOL_MAC;
	case 'd':   return EOL_DOS;
    }
    return EOL_UNIX;
}

/*
 * Call shell.	Calls mch_call_shell, with 'shellxquote' added.
 */
    int
call_shell(cmd, opt)
    char_u	*cmd;
    int		opt;
{
    char_u	*ncmd;
    int		retval;

#ifdef FEAT_GUI_MSWIN
    /* Don't hide the pointer while executing a shell command. */
    gui_mch_mousehide(FALSE);
#endif
#ifdef FEAT_GUI
    ++hold_gui_events;
#endif
    /* The external command may update a tags file, clear cached tags. */
    tag_freematch();

    if (cmd == NULL || *p_sxq == NUL)
	retval = mch_call_shell(cmd, opt);
    else
    {
	ncmd = alloc((unsigned)(STRLEN(cmd) + STRLEN(p_sxq) * 2 + 1));
	if (ncmd != NULL)
	{
	    STRCPY(ncmd, p_sxq);
	    STRCAT(ncmd, cmd);
	    STRCAT(ncmd, p_sxq);
	    retval = mch_call_shell(ncmd, opt);
	    vim_free(ncmd);
	}
	else
	    retval = -1;
    }
#ifdef FEAT_GUI
    --hold_gui_events;
#endif

#ifdef FEAT_EVAL
    set_vim_var_nr(VV_SHELL_ERROR, (long)retval);
#endif

    /*
     * Check the window size, in case it changed while executing the
     * external command.
     */
    shell_resized_check();

    return retval;
}

/*
 * VISUAL and OP_PENDING State are never set, they are equal to NORMAL State
 * with a condition.  This function returns the real State.
 */
    int
get_real_state()
{
    if ((State & NORMAL))
    {
#ifdef FEAT_VISUAL
	if (VIsual_active)
	    return VISUAL;
	else
#endif
	    if (finish_op)
	    return OP_PENDING;
    }
    return State;
}

#if defined(FEAT_SESSION) || defined(MSWIN) || defined(FEAT_GUI_GTK) \
	|| defined(PROTO)
/*
 * Change to a file's directory.
 */
    int
vim_chdirfile(fname)
    char_u	*fname;
{
    char_u	temp_string[MAXPATHL];
    char_u	*p;
    char_u	*t;

    STRCPY(temp_string, fname);
    p = get_past_head(temp_string);
    t = gettail(temp_string);
    while (t > p && vim_ispathsep(t[-1]))
	--t;
    *t = NUL; /* chop off end of string */

    return mch_chdir((char *)temp_string);
}
#endif

#if defined(CURSOR_SHAPE) || defined(PROTO)

/*
 * Handling of cursor and mouse pointer shapes in various modes.
 */

cursorentry_t shape_table[SHAPE_IDX_COUNT] =
{
    /* The values will be filled in from the 'guicursor' and 'mouseshape'
     * defaults when Vim starts.
     * Adjust the SHAPE_IDX_ defines when making changes! */
    {0,	0, 0, 700L, 400L, 250L, 0, "n", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "v", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "i", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "r", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "c", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "ci", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "cr", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "o", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0, 700L, 400L, 250L, 0, "ve", SHAPE_CURSOR+SHAPE_MOUSE},
    {0,	0, 0,   0L,   0L,   0L, 0, "e", SHAPE_MOUSE},
    {0,	0, 0,   0L,   0L,   0L, 0, "s", SHAPE_MOUSE},
    {0,	0, 0,   0L,   0L,   0L, 0, "sd", SHAPE_MOUSE},
    {0,	0, 0,   0L,   0L,   0L, 0, "vs", SHAPE_MOUSE},
    {0,	0, 0,   0L,   0L,   0L, 0, "vd", SHAPE_MOUSE},
    {0,	0, 0,   0L,   0L,   0L, 0, "m", SHAPE_MOUSE},
    {0,	0, 0, 100L, 100L, 100L, 0, "sm", SHAPE_CURSOR},
};

#ifdef FEAT_MOUSESHAPE
/*
 * Table with names for mouse shapes.  Keep in sync with all the tables for
 * mch_set_mouse_shape()!.
 */
static char * mshape_names[] =
{
    "arrow",	/* default, must be the first one */
    "blank",	/* hidden */
    "beam",
    "updown",
    "udsizing",
    "leftright",
    "lrsizing",
    "busy",
    "no",
    "crosshair",
    "hand1",
    "hand2",
    "pencil",
    "question",
    "rightup-arrow",
    "up-arrow",
    NULL
};
#endif

/*
 * Parse the 'guicursor' option ("what" is SHAPE_CURSOR) or 'mouseshape'
 * ("what" is SHAPE_MOUSE).
 * Returns error message for an illegal option, NULL otherwise.
 */
    char_u *
parse_shape_opt(what)
    int		what;
{
    char_u	*modep;
    char_u	*colonp;
    char_u	*commap;
    char_u	*p, *endp;
    int		idx = 0;		/* init for GCC */
    int		all_idx;
    int		len;
    int		i;
    long	n;
    int		found_ve = FALSE;	/* found "ve" flag */

    /*
     * Repeat for all comma separated parts.
     */
#ifdef FEAT_MOUSESHAPE
    if (what == SHAPE_MOUSE)
	modep = p_mouseshape;
    else
#endif
	modep = p_guicursor;
    while (*modep)
    {
	colonp = vim_strchr(modep, ':');
	if (colonp == NULL)
	    return (char_u *)N_("Missing colon");
	commap = vim_strchr(modep, ',');

	/*
	 * Repeat for all mode's before the colon.
	 * For the 'a' mode, we loop to handle all the modes.
	 */
	all_idx = -1;
	while (modep < colonp || all_idx >= 0)
	{
	    if (all_idx < 0)
	    {
		/* Find the mode. */
		if (modep[1] == '-' || modep[1] == ':')
		    len = 1;
		else
		    len = 2;
		if (len == 1 && TO_LOWER(modep[0]) == 'a')
		    all_idx = SHAPE_IDX_COUNT - 1;
		else
		{
		    for (idx = 0; idx < SHAPE_IDX_COUNT; ++idx)
			if (STRNICMP(modep, shape_table[idx].name, len) == 0)
			    break;
		    if (idx == SHAPE_IDX_COUNT
			    || (shape_table[idx].used_for & what) == 0)
			return (char_u *)N_("Illegal mode");
		    if (len == 2 && modep[0] == 'v' && modep[1] == 'e')
			found_ve = TRUE;
		}
		modep += len + 1;
	    }

	    if (all_idx >= 0)
		idx = all_idx--;
#ifdef FEAT_MOUSESHAPE
	    else if (what == SHAPE_MOUSE)
	    {
		/* Set the default, for the missing parts */
		shape_table[idx].mshape = 0;
	    }
#endif
	    else
	    {
		/* Set the defaults, for the missing parts */
		shape_table[idx].shape = SHAPE_BLOCK;
		shape_table[idx].blinkwait = 700L;
		shape_table[idx].blinkon = 400L;
		shape_table[idx].blinkoff = 250L;
	    }

	    /* Parse the part after the colon */
	    for (p = colonp + 1; *p && *p != ','; )
	    {
#ifdef FEAT_MOUSESHAPE
		if (what == SHAPE_MOUSE)
		{
		    for (i = 0; ; ++i)
		    {
			if (mshape_names[i] == NULL)
			{
			    if (!isdigit(*p))
				return (char_u *)N_("Illegal mouseshape");
			    shape_table[idx].mshape =
					      getdigits(&p) + MSHAPE_NUMBERED;
			    break;
			}
			len = strlen(mshape_names[i]);
			if (STRNICMP(p, mshape_names[i], len) == 0)
			{
			    shape_table[idx].mshape = i;
			    p += len;
			    break;
			}
		    }
		}
		else /* if (what == SHAPE_MOUSE) */
#endif
		{
		    /*
		     * First handle the ones with a number argument.
		     */
		    i = *p;
		    len = 0;
		    if (STRNICMP(p, "ver", 3) == 0)
			len = 3;
		    else if (STRNICMP(p, "hor", 3) == 0)
			len = 3;
		    else if (STRNICMP(p, "blinkwait", 9) == 0)
			len = 9;
		    else if (STRNICMP(p, "blinkon", 7) == 0)
			len = 7;
		    else if (STRNICMP(p, "blinkoff", 8) == 0)
			len = 8;
		    if (len)
		    {
			p += len;
			if (!isdigit(*p))
			    return (char_u *)N_("digit expected");
			n = getdigits(&p);
			if (len == 3)   /* "ver" or "hor" */
			{
			    if (n == 0)
				return (char_u *)N_("Illegal percentage");
			    if (TO_LOWER(i) == 'v')
				shape_table[idx].shape = SHAPE_VER;
			    else
				shape_table[idx].shape = SHAPE_HOR;
			    shape_table[idx].percentage = n;
			}
			else if (len == 9)
			    shape_table[idx].blinkwait = n;
			else if (len == 7)
			    shape_table[idx].blinkon = n;
			else
			    shape_table[idx].blinkoff = n;
		    }
		    else if (STRNICMP(p, "block", 5) == 0)
		    {
			shape_table[idx].shape = SHAPE_BLOCK;
			p += 5;
		    }
		    else	/* must be a highlight group name then */
		    {
			endp = vim_strchr(p, '-');
			if (commap == NULL)		    /* last part */
			{
			    if (endp == NULL)
				endp = p + STRLEN(p);   /* find end of part */
			}
			else if (endp > commap || endp == NULL)
			    endp = commap;
			shape_table[idx].id = syn_check_group(p,
							     (int)(endp - p));
			p = endp;
		    }
		} /* if (what != SHAPE_MOUSE) */

		if (*p == '-')
		    ++p;
	    }
	}
	modep = p;
	if (*modep == ',')
	    ++modep;
    }

    /* If the 's' flag is not given, use the 'v' cursor for 's' */
    if (!found_ve)
    {
#ifdef FEAT_MOUSESHAPE
	if (what == SHAPE_MOUSE)
	{
	    shape_table[SHAPE_IDX_VE].mshape = shape_table[SHAPE_IDX_V].mshape;
	}
	else
#endif
	{
	    shape_table[SHAPE_IDX_VE].shape =
					      shape_table[SHAPE_IDX_V].shape;
	    shape_table[SHAPE_IDX_VE].percentage =
					 shape_table[SHAPE_IDX_V].percentage;
	    shape_table[SHAPE_IDX_VE].blinkwait =
					  shape_table[SHAPE_IDX_V].blinkwait;
	    shape_table[SHAPE_IDX_VE].blinkon =
					    shape_table[SHAPE_IDX_V].blinkon;
	    shape_table[SHAPE_IDX_VE].blinkoff =
					   shape_table[SHAPE_IDX_V].blinkoff;
	    shape_table[SHAPE_IDX_VE].id = shape_table[SHAPE_IDX_V].id;
	}
    }

    return NULL;
}

/*
 * Return the index into shape_table[] for the current mode.
 * When "mouse" is TRUE, consider indexes valid for the mouse pointer.
 */
    int
get_shape_idx(mouse)
    int	mouse;
{
#ifdef FEAT_MOUSESHAPE
    if (mouse && (State == HITRETURN || State == ASKMORE))
	return SHAPE_IDX_MORE;
    if (mouse && drag_status_line)
	return SHAPE_IDX_SDRAG;
# ifdef FEAT_VERTSPLIT
    if (mouse && drag_sep_line)
	return SHAPE_IDX_VDRAG;
# endif
#endif
    if (!mouse && State == SHOWMATCH)
	return SHAPE_IDX_SM;
    if (State == INSERT)
	return SHAPE_IDX_I;
    if (State == REPLACE)
	return SHAPE_IDX_R;
    if (State == VREPLACE)
	return SHAPE_IDX_R;
    if (State == CMDLINE)
    {
	if (cmdline_at_end())
	    return SHAPE_IDX_C;
	if (cmdline_overstrike())
	    return SHAPE_IDX_CR;
	return SHAPE_IDX_CI;
    }
    if (finish_op)
	return SHAPE_IDX_O;
#ifdef FEAT_VISUAL
    if (VIsual_active)
    {
	if (*p_sel == 'e')
	    return SHAPE_IDX_VE;
	else
	    return SHAPE_IDX_V;
    }
#endif
    return SHAPE_IDX_N;
}

# if defined(FEAT_MOUSESHAPE) || defined(PROTO)
static int old_mouse_shape = 0;

/*
 * Set the mouse shape:
 * If "shape" is -1, use shape depending on the current mode,
 * depending on the current state.
 * If "shape" is -2, only update the shape when it's CLINE or STATUS (used
 * when the mouse moves off the status or command line).
 */
    void
update_mouseshape(shape_idx)
    int	shape_idx;
{
    int new_mouse_shape;

    /* Only works in GUI mode. */
    if (!gui.in_use)
	return;
    if (shape_idx == -2
	    && old_mouse_shape != shape_table[SHAPE_IDX_CLINE].mshape
	    && old_mouse_shape != shape_table[SHAPE_IDX_STATUS].mshape
	    && old_mouse_shape != shape_table[SHAPE_IDX_VSEP].mshape)
	return;
    if (shape_idx < 0)
	new_mouse_shape = shape_table[get_shape_idx(TRUE)].mshape;
    else
	new_mouse_shape = shape_table[shape_idx].mshape;
    if (new_mouse_shape != old_mouse_shape)
    {
	mch_set_mouse_shape(new_mouse_shape);
	old_mouse_shape = new_mouse_shape;
    }
}
# endif

#endif /* CURSOR_SHAPE */


#ifdef FEAT_CRYPT
/*
 * Optional encryption suypport.
 * Mohsin Ahmed, mosh@sasi.com, 98-09-24
 * Based on zip/crypt sources.
 */

/*
 * NOTE FOR USA: It is unclear if exporting this code from the USA is allowed.
 * If you do not want to take any risk, remove this bit of the code (from
 * #ifdef FEAT_CRYPT to the matching #endif) and disable the crypt feature.
 * This code was originally created in Europe and India.
 */

/* from zip.h */

typedef unsigned short ush;	/* unsigned 16-bit value */
typedef unsigned long  ulg;	/* unsigned 32-bit value */

/*
 * from util.c
 * Table of CRC-32's of all single byte values (made by makecrc.c)
 */

ulg crc_32_tab[] =
{
    0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
    0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
    0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
    0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
    0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
    0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
    0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
    0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
    0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
    0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
    0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
    0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
    0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
    0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
    0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
    0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
    0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
    0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
    0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
    0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
    0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
    0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
    0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
    0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
    0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
    0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
    0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
    0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
    0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
    0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
    0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
    0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
    0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
    0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
    0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
    0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
    0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
    0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
    0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
    0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
    0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
    0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
    0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
    0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
    0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
    0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
    0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
    0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
    0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
    0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
    0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
    0x2d02ef8dL
};

#define CRC32(c, b) (crc_32_tab[((int)(c) ^ (b)) & 0xff] ^ ((c) >> 8))


static ulg keys[3]; /* keys defining the pseudo-random sequence */

/*
 * Return the next byte in the pseudo-random sequence
 */
    int
decrypt_byte()
{
    ush temp;

    temp = (ush)keys[2] | 2;
    return (int)(((unsigned)(temp * (temp ^ 1)) >> 8) & 0xff);
}

/*
 * Update the encryption keys with the next byte of plain text
 */
    int
update_keys(c)
    int c;			/* byte of plain text */
{
    keys[0] = CRC32(keys[0], c);
    keys[1] += keys[0] & 0xff;
    keys[1] = keys[1] * 134775813L + 1;
    keys[2] = CRC32(keys[2], (int)(keys[1] >> 24));
    return c;
}

/*
 * Initialize the encryption keys and the random header according to
 * the given password.
 * If "passwd" is NULL or empty, don't do anything.
 */
    void
crypt_init_keys(passwd)
    char_u *passwd;		/* password string with which to modify keys */
{
    if (passwd && *passwd)
    {
	keys[0] = 305419896L;
	keys[1] = 591751049L;
	keys[2] = 878082192L;
	while (*passwd != '\0')
	    update_keys((int)*passwd++);
    }
}

/*
 * Ask the user for a crypt key.
 * When "store" is TRUE, the new key in stored in the 'key' option, and the
 * 'key' option value is returned: Don't free it.
 * When "store" is FALSE, the typed key is returned in allocated memory.
 * Returns NULL on failure.
 */
    char_u *
get_crypt_key(store)
    int		store;
{
    char_u	*p;

    cmdline_crypt = TRUE;
    cmdline_row = msg_row;
    p = getcmdline_prompt(NUL, (char_u *)_("Enter encryption key: "), 0);
    cmdline_crypt = FALSE;

    /* since the user typed this, no need to wait for return */
    need_wait_return = FALSE;
    msg_didout = FALSE;
    if (p != NULL && store)
    {
	set_option_value((char_u *)"key", 0L, p, TRUE);
	return curbuf->b_p_key;
    }
    return p;
}

#endif /* FEAT_CRYPT */

/* TODO: make some #ifdef for this */
/*--------[ file searching ]-------------------------------------------------*/
/*
 * File searching functions for 'path', 'tags' and 'cdpath' options.
 * External visible functions:
 * vim_findfile_init()		initialize search context
 * vim_findfile_free_visited()	free list of visited files/dirs
 * vim_findfile()		find a file in the search context
 *
 * All static functions and variables start with 'ff_'
 *
 * Usage:
 * You always have to call vim_findfile_init() to initialize your search
 * context. Then you call vim_findfile() to do the search. vim_findfile() will
 * return the name of the found file or NULL if nothing is found.
 * To find all files that match the conditions call vim_findfile() again until
 * it returns NULL.
 *
 * More informations can be found in the comment of the function
 * vim_findfile_init().
 */

/*
 * type for the directory search stack
 */
typedef struct ff_stack
{
    struct ff_stack	*ff_prev;

    /* the fix part (no wildcards) and the part containing the wildcards
     * of the search path
     */
    char_u		*ff_fix_path;
#ifdef FEAT_PATH_EXTRA
    char_u		*ff_wc_path;
#endif

    /* files/dirs found in the above directory, matched by the first wildcard
     * of wc_part
     */
    char_u		**ff_filearray;
    int			ff_filearray_size;
    char_u		ff_filearray_cur;   /* needed for partly handled dirs */

    /* to store status of partly handled directories */
    int			ff_stage;

    /* How deep are we in the directory tree?
     * Counts backward from value of level parameter to vim_findfile_init
     */
    int			ff_level;

    /* Did we already expand '**' to an empty string? */
    int			ff_star_star_empty;
} ff_stack_t;

/*
 * type for already visited directories or files.
 */
typedef struct ff_visited
{
    struct ff_visited	*ffv_next;

#ifdef FEAT_PATH_EXTRA
    /* Visited directories are different if the wildcard string are
     * different. So we have to save it.
     */
    char_u		*ffv_wc_path;
#endif
    /* for unix use inode etc for comparison (needed because of links), else
     * use filename.
     */
#if defined(UNIX)
    struct stat		ffv_st;
#else
    /* The memory for this struct is allocated according to the length of
     * ffv_fname.
     */
    char_u		ffv_fname[1];	/* actually longer */
#endif
} ff_visited_t;

/*
 * We can several visited list during the search. This is expecially needed for
 * the tags option. If tags is set to (replace + with *)
 *      "./++/tags,./++/TAGS,inc/++/tags"
 * we need to search the directories for 'tags' and then search the same
 * directories for 'TAGS' again, but we shouldn't search them if we use the
 * third part of the tags option. So we need to store the old visited-list and
 * must be able to restore it later.
 */
typedef struct ff_visited_list_hdr
{
    struct ff_visited_list_hdr  *ffvl_next;

    /* the filename the attached visited list is for */
    char_u                      *ffvl_filename;

    ff_visited_t                *ffvl_visited_list;

} ff_visited_list_hdr_t;

/*
 * '**' can be expanded to several directory levels.
 * Set the default maximium depth.
 */
#define FF_MAX_STAR_STAR_EXPAND ((char_u)30)
/*
 * The search context:
 *   ff_stack_ptr:	the stack for the dirs to search
 *   ff_file_to_search: the file to search for
 *   ff_start_dir:	the starting directory, if search path was relative
 *
 * The following is needed to restart the search from the next upper directory
 * level. They are not set if no upward search is wanted.
 *   ff_fix_path:	the fix part of the given path (without wildcards)
 *   ff_wc_path:	the part of the given path containing wildcards
 *   ff_level:		how many levels of dirs to search downwards
 *   ff_stopdirs_v:	array of stop directories for upward search
 */
static ff_stack_t	        *ff_stack_ptr = NULL;
static ff_visited_list_hdr_t	*ff_visited_list = NULL;
static ff_visited_list_hdr_t    *ff_visited_lists_list = NULL;
static char_u		        *ff_file_to_search = NULL;
static char_u		        *ff_start_dir = NULL;
static char_u		        *ff_fix_path = NULL;
#ifdef FEAT_PATH_EXTRA
static char_u		        *ff_wc_path = NULL;
static int		        ff_level;
#endif
#ifdef FEAT_PATH_EXTRA
static char_u		        **ff_stopdirs_v = NULL;
#endif
static int		        ff_need_dir = FALSE;
static char_u                   *ff_expand_buffer = NULL;

/* locally needed functions */
#ifdef FEAT_PATH_EXTRA
static int ff_check_visited __ARGS((ff_visited_t **, char_u *, char_u *));
#else
static int ff_check_visited __ARGS((ff_visited_t **, char_u *));
#endif
static void ff_free_visited_list __ARGS((ff_visited_t *));
static ff_visited_list_hdr_t* ff_get_visited_list __ARGS((char_u*));

static void ff_push __ARGS((ff_stack_t *));
static ff_stack_t * ff_pop __ARGS((void));
static void ff_clear __ARGS((void));
static void ff_free_stack_element __ARGS((ff_stack_t *));
#ifdef FEAT_PATH_EXTRA
static ff_stack_t *ff_create_stack_element __ARGS((char_u *, char_u *, int, int));
#else
static ff_stack_t *ff_create_stack_element __ARGS((char_u *, int, int));
#endif
#ifdef FEAT_PATH_EXTRA
static int ff_path_in_stoplist __ARGS((char_u *, int, char_u **));
#endif

#ifdef FEAT_SEARCHPATH
static char_u *find_file_in_path_option __ARGS((char_u *ptr, int len, int options, int first, char_u *path_option, int need_dir));
#endif

#if 0
/*
 * if someone likes findfirst/findnext, here are the functions
 */
    char_u *
vim_findfirst(path, filename, level)
    char_u	*path;
    char_u	*filename;
    int		level;
{
    vim_findfile_init(path, filename, NULL, level, TRUE);
    return vim_findfile();
}

    char_u *
vim_findnext()
{
    return vim_findfile();
}
#endif

/*
 * Initialization routine for vim_findfile.
 * Returns OK or FAIL.
 *
 * Find the file 'filename' in the directory 'path'.
 * The parameter 'path' may contain wildcards. If so only search 'level'
 * directories deep. The parameter 'level' is the absolute maximum and is
 * not related to restricts given to the '**' wildcard. If 'level' is 100
 * and you use '**200' vim_findfile() will stop after 100 levels.
 *
 * If 'stopdirs' is not NULL and nothing is found downward, the search is
 * restarted on the next higher directory level. This is repeated until the
 * start-directory of a search is contained in 'stopdirs'. 'stopdirs' has the
 * format ";*<dirname>*\(;<dirname>\)*;\=$".
 *
 * If the 'path' is relative, the starting dir for the search is either VIM's
 * current dir or if the path starts with "./" the current files dir.
 * If the 'path' is absolut, the starting dir is that part of the path before
 * the first wildcard.
 *
 * Upward search is only done on the starting dir.
 *
 * If 'free_visited' is TRUE the list of already visited files/directories is
 * cleared. Set this to FALSE if you just want to search from another
 * directory, but want to be sure that no file from a previous search is found.
 * This is useful if you search for a file at different places.
 * The list of visited files/dirs can also be cleared with the function
 * vim_findfile_free_visited().
 *
 * This function silently ignores a few errors, vim_findfile() will have
 * limited functionality then.
 * On some errors it will return FAIL without cleaning up. This cleanup is the
 * first thing done, when the function is called again.
 *
 * Attention:
 * You call vim_findfile_init() and afterwards several times vim_findfile().
 * If someone else is calling vim_findfile_init() in between your search
 * context will be lost and you will search for his file.
 *
 * If you run into such a problem, you need to change vim_findfile_init() to
 * return a pointer to a search context and you need a function to destroy this
 * context afterwards. Also you need to find a solution to keep the list of
 * visited files/dirs for the new context if this is needed (this is definitely
 * needed for the 'tags' and 'path' option!).
 */
/*ARGSUSED*/
    int
vim_findfile_init(path, filename, stopdirs, level, free_visited)
    char_u	*path;
    char_u	*filename;
    char_u	*stopdirs;
    int		level;
    int		free_visited;
{
#ifdef FEAT_PATH_EXTRA
    char_u	*wc_part;
#endif
    ff_stack_t	*sptr;

    /* clear the stack */
    ff_clear();

    /* clear already visited files if wanted */
    if (free_visited == TRUE)
	vim_findfile_free_visited();
    else
    {
        ff_visited_list = ff_get_visited_list(filename);
        if (ff_visited_list == NULL)
            return FAIL;
    }

    if (ff_expand_buffer != NULL)
        ff_expand_buffer[0] = 0;
    else
    {
        ff_expand_buffer = alloc(MAXPATHL);
        if (ff_expand_buffer == NULL)
            return FAIL;
    }


    /* Store information on starting dir now if path is relative.
     * If path is absolute, we do that later.
     */
    if (path[0] == '.'
	    && (vim_ispathsep(path[1]) || path[1] == NUL)
	    && vim_strchr(p_cpo, CPO_DOTTAG) == NULL
	    && curbuf->b_fname != NULL)
    {
	ff_start_dir = vim_strnsave(curbuf->b_ffname,
			 (int)(gettail(curbuf->b_ffname) - curbuf->b_ffname));
	if (ff_start_dir == NULL)
	    return FAIL;
	if (*++path != NUL)
	    ++path;
    }
    else if (!mch_isFullName(path))
    {
        if (mch_dirname(ff_expand_buffer, MAXPATHL) == OK)
            ff_start_dir = vim_strsave(ff_expand_buffer);

	if (ff_start_dir == NULL)
	    return FAIL;
    }

#ifdef FEAT_PATH_EXTRA
    /*
     * If stopdirs are given, split them into an array of pointers.
     * If this fails (mem allocation), there is no upward search at all or a
     * stop directory is not recognized -> continue silently.
     * If stopdirs just contains a ";" or is empty, ff_stopdirs_v will only
     * contain a  NULL pointer. This is handled as unlimited upward search.
     * See function ff_path_in_stoplist() for details.
     */
    if (stopdirs != NULL)
    {
	char_u	*walker = stopdirs;
	int	dircount;

	while (*walker == ';')
	    walker++;

	dircount = 1;
	ff_stopdirs_v = (char_u **)alloc((unsigned)sizeof(char_u *));
	if (ff_stopdirs_v != NULL)
	{
	    do
	    {
		char_u	*helper;
		void	*ptr;

		helper = walker;
		ptr = vim_realloc(ff_stopdirs_v,
					       dircount+1 * sizeof(char_u *));
		if (ptr)
		    ff_stopdirs_v = ptr;
		else
		    /* ignore, keep what we have and continue */
		    break;
		walker = vim_strchr(walker, ';');
		if (walker)
		{
		    ff_stopdirs_v[dircount-1] = vim_strnsave(helper,
						      (int)(walker - helper));
		    walker++;
		}
		else
		    /* this might be "", which means ascent till top
		     * of directory tree.
		     */
		    ff_stopdirs_v[dircount-1] = vim_strsave(helper);

		dircount++;

	    } while (walker != NULL);
	    ff_stopdirs_v[dircount-1] = NULL;
	}
    }
#endif

#ifdef FEAT_PATH_EXTRA
    ff_level = level;

    /* split into:
     *  -fix path
     *  -wildcard_stuff (might be NULL)
     */
    wc_part = vim_strchr(path, '*');
    if (wc_part != NULL)
    {
	int	level;
	int	len;
	char_u	*errpt;

	ff_fix_path = vim_strnsave(path, (int)(wc_part - path));

	/*
	 * copy wc_path and add restricts to the '**' wildcard.
	 * The octett after a '**' is used as a (binary) counter.
	 * So '**3' is transposed to '**^C' ('^C' is ASCII value 3)
	 * or '**76' is transposed to '**N'( 'N' is ASCII value 76).
	 * If no restrict is given after '**' the default is used.
	 */
	len = 0;
	while (*wc_part != NUL)
	{
	    if (STRNCMP(wc_part, "**", 2) == 0)
	    {
		ff_expand_buffer[len++] = *wc_part++;
		ff_expand_buffer[len++] = *wc_part++;

		level = strtol((char *)wc_part, (char **)&errpt, 10);
		if (errpt != wc_part && level > 0 && level < 255)
		    ff_expand_buffer[len++] = level;
		else if (errpt != wc_part && level == 0)
		    /* restrict is 0 -> remove already added '**' */
		    len -= 2;
		else
		    ff_expand_buffer[len++] = FF_MAX_STAR_STAR_EXPAND;
		wc_part = errpt;
		if (*wc_part != PATHSEP && *wc_part != NUL)
		{
		    EMSG2(_("Invalid path: '**[number]' must be at the end of the path or be followed by '%s'. See :help path."), PATHSEPSTR);
		    return FAIL;
		}
	    }
	    else
		ff_expand_buffer[len++] = *wc_part++;
	}
	ff_expand_buffer[len] = NUL;
	ff_wc_path = vim_strsave(ff_expand_buffer);
	if (ff_wc_path == NULL)
	{
	    return FAIL;
	}
    }
    else
#endif
	ff_fix_path = vim_strsave(path);

    if (ff_start_dir == NULL)
    {
	/* store the fix part as startdir.
	 * This is needed if the parameter pathis fully qualified.
	 */
	ff_start_dir = vim_strsave(ff_fix_path);
	if (ff_start_dir)
	    ff_fix_path[0] = NUL;
    }

    /* create an absolute path */
    STRCPY(ff_expand_buffer, ff_start_dir);
    add_pathsep(ff_expand_buffer);
    STRCAT(ff_expand_buffer, ff_fix_path);
    add_pathsep(ff_expand_buffer);

    sptr = ff_create_stack_element(ff_expand_buffer,
#ifdef FEAT_PATH_EXTRA
	    ff_wc_path,
#endif
	    level, 0);

    if (sptr == NULL)
	return FAIL;

    ff_push(sptr);

    ff_file_to_search = vim_strsave(filename);
    if (ff_file_to_search == NULL)
	return FAIL;

    return OK;
}

#if defined(FEAT_PATH_EXTRA) || defined(PROTO)
/*
 * Get the stopdir string.  Check that ';' is not escaped.
 */
    char_u *
vim_findfile_stopdir(buf)
    char_u	*buf;
{
    char_u	*r_ptr = buf;

    while (*r_ptr != NUL && *r_ptr != ';')
    {
	if (r_ptr[0] == '\\' && r_ptr[1] == ';')
	{
	    /* overwrite the escape char,
	     * use STRLEN(r_ptr) to move the trailing '\0'
	     */
	    mch_memmove(r_ptr, r_ptr + 1, STRLEN(r_ptr));
	    r_ptr++;
	}
	r_ptr++;
    }
    if (*r_ptr == ';')
    {
	*r_ptr = 0;
	r_ptr++;
    }
    else if (*r_ptr == NUL)
	r_ptr = NULL;
    return r_ptr;
}
#endif

/*
 * Find a file in a search context.
 * The search context is created with vim_findfile_init() above.
 * Return a pointer to an allocated file name or NULL if nothing found.
 * To get all matching files call this function until you get NULL.
 *
 * The search algorithm is depth first. To change this replace the
 * stack with a list (don't forget to leave partly searched directories on the
 * top of the list).
 */
    char_u *
vim_findfile()
{
    char_u	*file_path;
#ifdef FEAT_PATH_EXTRA
    char_u	*rest_of_wildcards;
    char_u	*path_end = NULL;
#endif
    ff_stack_t	*ctx;
#if defined(FEAT_SEARCHPATH) || defined(FEAT_PATH_EXTRA)
    int		len;
#endif
    int		i;
    char_u	*p;
#ifdef FEAT_SEARCHPATH
    char_u	*suf;
#endif

    /*
     * If ff_file_to_search is NULL, no vim_findfile_init() was called, or its
     * return code was not checked. Anyway, there is nothing to search for.
     */
    if (ff_file_to_search == NULL)
    {
	emsg_silent = 0;
	msg_silent = 0;
	EMSG(_("Internal error: vim_findfile called without vim_findfile_init"));
        return NULL;
    }

    /*
     * filepath is used as buffer for various actions and as the storage to
     * return a found filename.
     */
    if ((file_path = alloc((int)MAXPATHL)) == NULL)
	return NULL;

#ifdef FEAT_PATH_EXTRA
    if (ff_start_dir != NULL)
	path_end = &ff_start_dir[STRLEN(ff_start_dir)];
#endif

#ifdef FEAT_PATH_EXTRA
    /* upward search loop */
    for (;;)
    {
#endif
	/* downward search loop */
	for (;;)
	{
	    /* check user */
	    ui_breakcheck();
	    if (got_int)
		break;

	    /* get directory to work on from stack */
	    ctx = ff_pop();
	    if (ctx == NULL)
		break;
	    /*
	     * TODO: decide if we leave this test in
	     *
	     * GOOD: don't search a directory(-tree) twice.
	     * BAD:  - check linked list for every new directory entered.
	     *       - check for double files also done below
	     *
	     * Here we check if we already searched this directory.
	     * We already searched a directory if:
	     * 1) The directory is the same.
	     * 2) We would use the same wildcard string.
	     *
	     * Good if you have links on same directory via several ways
	     *  or you have selfreferences in directories (e.g. SuSE Linux 6.3:
	     *  /etc/rc.d/init.d is linked to /etc/rc.d -> endless loop)
	     *
	     * This check is only needed for directories we work on for the
	     * first time (hence ctx->ff_filearray == NULL)
	     */
	    if (ctx->ff_filearray == NULL
		    && ff_check_visited(&ff_visited_list->ffvl_visited_list,
                        ctx->ff_fix_path
#ifdef FEAT_PATH_EXTRA
			, ctx->ff_wc_path
#endif
			) == FAIL)
	    {
#ifdef FF_VERBOSE
		if (p_verbose >= 5)
		{
		    /* always scroll up, don't overwrite */
		    msg_scroll = TRUE;
		    smsg((char_u *)"Already Searched: %s (%s)",
					   ctx->ff_fix_path, ctx->ff_wc_path);
		    /* don't overwrite this either */
		    msg_puts((char_u *)"\n");
		    cmdline_row = msg_row;
		}
#endif
		ff_free_stack_element(ctx);
		continue;
	    }
#ifdef FF_VERBOSE
	    else if (p_verbose >= 5)
	    {
		/* always scroll up, don't overwrite */
		msg_scroll = TRUE;
		smsg((char_u *)"Searching: %s (%s)", ctx->ff_fix_path,
			ctx->ff_wc_path);
		/* don't overwrite this either */
		msg_puts((char_u *)"\n");
		cmdline_row = msg_row;
	    }
#endif

	    /* check depth */
	    if (ctx->ff_level <= 0)
	    {
		ff_free_stack_element(ctx);
		continue;
	    }

	    file_path[0] = NUL;

	    /* if no filearray till now expand wildcards */
	    if (ctx->ff_filearray == NULL)
	    {
		char_u *dirptrs[2];

		dirptrs[0] = file_path;
		dirptrs[1] = NULL;

		if (!mch_isFullName(ctx->ff_fix_path) && ff_start_dir)
		{
		    STRCPY(file_path, ff_start_dir);
		    add_pathsep(file_path);
		}

		STRCAT(file_path, ctx->ff_fix_path);
		add_pathsep(file_path);

#ifdef FEAT_PATH_EXTRA
		len = STRLEN(file_path);
		rest_of_wildcards = ctx->ff_wc_path;
		if (*rest_of_wildcards != NUL)
		{
		    if (STRNCMP(rest_of_wildcards, "**", 2) == 0)
		    {
			char_u *p;

			/* pointer to the restrict byte
			 * The restrict byte is not a character!
			 */
			p = rest_of_wildcards + 2;

			if (*p > 0)
			{
			    (*p)--;
			    file_path[len++] = '*';
			}

			if (*p <= 0)
			{
			    /* remove '**<numb> from wildcards */
			    memmove(rest_of_wildcards, rest_of_wildcards + 3,
				    STRLEN(rest_of_wildcards + 3) + 1);
			}
			else
			    rest_of_wildcards += 3;

			if (ctx->ff_star_star_empty == 0)
			{
			    /* if not done before, expand '**' to empty */
			    ctx->ff_star_star_empty = 1;
			    dirptrs[1] = dirptrs[0];
			    dirptrs[0] = ctx->ff_fix_path;
			}
		    }
		    while (*rest_of_wildcards
			    && !vim_ispathsep(*rest_of_wildcards))
			file_path[len++] = *rest_of_wildcards++;

		    file_path[len] = NUL;
		    if (vim_ispathsep(*rest_of_wildcards))
			rest_of_wildcards++;
		}
#endif

		/*
		 * Expand wildcards like "*" and "$VAR".
		 */
		expand_wildcards((dirptrs[1] == NULL) ? 1 : 2, dirptrs,
			&ctx->ff_filearray_size,
			&ctx->ff_filearray,
			EW_FILE|EW_DIR|EW_ADDSLASH|EW_SILENT);

		ctx->ff_filearray_cur = 0;
		ctx->ff_stage = 0;
	    }
#ifdef FEAT_PATH_EXTRA
	    else
		rest_of_wildcards = &ctx->ff_wc_path[STRLEN(ctx->ff_wc_path)];
#endif

	    if (ctx->ff_stage == 0)
	    {
#ifdef FEAT_PATH_EXTRA
		if (*rest_of_wildcards == NUL)
#endif
		{
		    /*
		     * we don't have further wildwards to expand, so we have to
		     * check for the final file now
		     */
		    for (i = ctx->ff_filearray_cur; i < ctx->ff_filearray_size;
									  ++i)
		    {
			if (!mch_isdir(ctx->ff_filearray[i]))
			    continue;   /* not a directory */
			STRCPY(file_path, ctx->ff_filearray[i]);
			add_pathsep(file_path);
			STRCAT(file_path, ff_file_to_search);

			/*
			 * Try without extra suffix and then with suffixes
			 * from 'suffixesadd'.
			 */
#ifdef FEAT_SEARCHPATH
			len = STRLEN(file_path);
			suf = curbuf->b_p_sua;
			for (;;)
#endif
			{
			    /* if file exists and we didn't already find it */
			    if (mch_getperm(file_path) >= 0
				    && (!ff_need_dir || mch_isdir(file_path))
#ifndef FF_VERBOSE
				    && (ff_check_visited(
                                            &ff_visited_list->ffvl_visited_list,
					    file_path
#ifdef FEAT_PATH_EXTRA
					    , NULL
#endif
					    ) == OK)
#endif
			       )
			    {
#ifdef FF_VERBOSE
                                if (ff_check_visited(
                                            &ff_visited_list->ffvl_visited_list,
                                            file_path
#ifdef FEAT_PATH_EXTRA
                                            , NULL
#endif
                                                    ) == FAIL)
				{
				    if (p_verbose >= 5)
				    {
					/* always scroll up, don't overwrite */
					msg_scroll = TRUE;
					smsg((char_u *)"Already: %s", file_path);
					/* don't overwrite this either */
					msg_puts((char_u *)"\n");
					cmdline_row = msg_row;
				    }
				    continue;
				}
#endif

				/* repush dir to examine rest of subdirs */
				ctx->ff_filearray_cur = i + 1;
				ff_push(ctx);

				simplify_filename(file_path);
				if (mch_dirname(ff_expand_buffer, MAXPATHL) == OK)
				{
				    p = shorten_fname(file_path, ff_expand_buffer);
				    if (p != NULL)
					mch_memmove(file_path, p,
							       STRLEN(p) + 1);
				}
#ifdef FF_VERBOSE
				if (p_verbose >= 5)
				{
				    /* always scroll up, don't overwrite */
				    msg_scroll = TRUE;
				    smsg((char_u *)"HIT: %s", file_path);
				    /* don't overwrite this either */
				    msg_puts((char_u *)"\n");
				    cmdline_row = msg_row;
				}
#endif
				return file_path;
			    }

#ifdef FEAT_SEARCHPATH
			    /* Not found or found already, try next suffix. */
			    if (*suf == NUL)
				break;
			    copy_option_part(&suf, file_path + len,
							 MAXPATHL - len, ",");
#endif
			}
		    }
		}
#ifdef FEAT_PATH_EXTRA
		else
		{
		    /*
		     * still wildcards left, push the directories for further
		     * search
		     */
		    for (i = ctx->ff_filearray_cur; i < ctx->ff_filearray_size;
									  ++i)
		    {
			if (!mch_isdir(ctx->ff_filearray[i]))
			    continue;	/* not a directory */

			ff_push(ff_create_stack_element(ctx->ff_filearray[i],
				      rest_of_wildcards, ctx->ff_level - 1, 0));
		    }
		}
#endif
		ctx->ff_filearray_cur = 0;
		ctx->ff_stage = 1;
	    }

#ifdef FEAT_PATH_EXTRA
	    /*
	     * if wildcards contains '**' we have to descent till we reach the
	     * leaves of the directory tree.
	     */
	    if (STRNCMP(ctx->ff_wc_path, "**", 2) == 0)
	    {
		for (i = ctx->ff_filearray_cur; i < ctx->ff_filearray_size; ++i)
		{
		    if (STRCMP(ctx->ff_filearray[i], ctx->ff_fix_path) == 0)
			continue; /* don't repush same directory */
		    if (!mch_isdir(ctx->ff_filearray[i]))
			continue;   /* not a directory */
		    ff_push(ff_create_stack_element(ctx->ff_filearray[i],
				ctx->ff_wc_path, ctx->ff_level - 1, 1));
		}
	    }
#endif

	    /* we are done with the current directory */
	    ff_free_stack_element(ctx);

	}

#ifdef FEAT_PATH_EXTRA
	/* If we reached this, we didn't find anything downwards.
	 * Let's check if we should do an upward search.
	 */
	if (ff_start_dir && ff_stopdirs_v != NULL && !got_int)
	{
	    ff_stack_t  *sptr;

	    /* is the last starting directory in the stop list? */
	    if (ff_path_in_stoplist(ff_start_dir,
		       (int)(path_end - ff_start_dir), ff_stopdirs_v) == TRUE)
		break;

	    /* cut of last dir */
	    while (path_end > ff_start_dir && *path_end == PATHSEP)
		path_end--;
	    while (path_end > ff_start_dir && *(path_end-1) != PATHSEP)
		path_end--;
	    *path_end = 0;
	    path_end--;

	    if (*ff_start_dir == 0)
		break;

	    STRCPY(file_path, ff_start_dir);
	    add_pathsep(file_path);
	    STRCAT(file_path, ff_fix_path);

	    /* create a new stack entry */
	    sptr = ff_create_stack_element(file_path, ff_wc_path, ff_level, 0);
	    if (sptr == NULL)
		break;
	    ff_push(sptr);
	}
	else
	    break;
    }
#endif

    vim_free(file_path);
    return NULL;
}

/*
 * Free the list of visited files and directories
 */
    void
vim_findfile_free_visited()
{
    ff_visited_list_hdr_t *vp;

    while (ff_visited_lists_list != NULL)
    {
	vp = ff_visited_lists_list->ffvl_next;
        ff_free_visited_list(ff_visited_lists_list->ffvl_visited_list);
	vim_free(ff_visited_lists_list->ffvl_filename);
	vim_free(ff_visited_lists_list);
	ff_visited_lists_list = vp;
    }
    ff_visited_lists_list = NULL;
}

    static void
ff_free_visited_list(ff_visited_t *vl)
{
    ff_visited_t *vp;

    while (vl != NULL)
    {
	vp = vl->ffv_next;
	vim_free(vl);
	vl = vp;
    }
    vl = NULL;
}

    static ff_visited_list_hdr_t*
ff_get_visited_list(filename)
    char_u          *filename;
{
    ff_visited_list_hdr_t  *retptr = NULL;

    /* check if a visited list for the given filename exists */
    if (ff_visited_lists_list != NULL)
    {
        retptr = ff_visited_lists_list;
        while (retptr != NULL)
        {
            if (STRCMP(filename, retptr->ffvl_filename) == 0)
            {
#ifdef FF_VERBOSE
                if (p_verbose >= 5)
                {
                    /* always scroll up, don't overwrite */
                    msg_scroll = TRUE;
                    smsg((char_u *)"ff_get_visited_list: FOUND list for %s",
                            filename);
                    /* don't overwrite this either */
                    msg_puts((char_u *)"\n");
                    cmdline_row = msg_row;
                }
#endif
                return retptr;
            }
            retptr = retptr->ffvl_next;
        }
    }

#ifdef FF_VERBOSE
    if (p_verbose >= 5)
    {
        /* always scroll up, don't overwrite */
        msg_scroll = TRUE;
        smsg((char_u *)"ff_get_visited_list: new list for %s",
                filename);
        /* don't overwrite this either */
        msg_puts((char_u *)"\n");
        cmdline_row = msg_row;
    }
#endif

    /*
     * if we reach this we didn't find a list and we have to allocate new list
     */
    retptr = (ff_visited_list_hdr_t*)alloc(sizeof(*retptr));
    if (retptr == NULL)
        return NULL;

    retptr->ffvl_visited_list = NULL;
    retptr->ffvl_filename = vim_strsave(filename);
    if (retptr->ffvl_filename == NULL)
    {
        vim_free(retptr);
        return NULL;
    }
    retptr->ffvl_next = ff_visited_lists_list;
    ff_visited_lists_list = retptr;

    return retptr;
}

/*
 * check if two wildcard pathes are equal. Returns TRUE or FALSE.
 * They are equal if:
 *  - both pathes are NULL
 *  - they have the same length
 *  - char by char comparison is OK
 *  - the only differences are in the counters behind a '**', so
 *    '**\20' is equal to '**\24'
 */
    static int
ff_wc_equal(char_u *s1, char_u * s2)
{
    int     i;

    if (s1 == s2)
        return TRUE;

    if (s1 == NULL || s2 == NULL)
        return FALSE;

    if (STRLEN(s1) != STRLEN(s2))
        return FAIL;

    for (i = 0; s1[i] != NUL && s2[i] != NUL; i++)
    {
        if (s1[i] != s2[i])
        {
            if (i >= 2)
                if (s1[i-1] == '*' && s1[i-2] == '*')
                    continue;
                else
                    return FAIL;
            else
                    return FAIL;
        }
    }
    return TRUE;
}

/*
 * maintains the list of already visited files and dirs
 * returns FAIL if the given file/dir is already in the list
 * returns OK if it is newly added
 *
 * TODO: What to do on memory allocation problems?
 *	 -> return TRUE - Better the file is found several times instead of
 *	    never.
 */
    static int
ff_check_visited(visited_list, fname
#ifdef FEAT_PATH_EXTRA
	, wc_path
#endif
	)
    ff_visited_t	**visited_list;
    char_u		*fname;
#ifdef FEAT_PATH_EXTRA
    char_u		*wc_path;
#endif
{
    ff_visited_t	*vp;
#ifdef UNIX
    struct stat		st;
#endif

#if defined(UNIX)
    if (mch_stat((char *)fname, &st) < 0)
#else
    if (mch_FullName(fname, ff_expand_buffer, MAXPATHL, TRUE) == FAIL)
#endif
	return FAIL;

    /* check against list of already visited files */
    for (vp = *visited_list; vp != NULL; vp = vp->ffv_next)
    {
#if defined(UNIX)
	if (vp->ffv_st.st_dev == st.st_dev
		&& vp->ffv_st.st_ino == st.st_ino)
#else
	if (fnamecmp(vp->ffv_fname, ff_expand_buffer) == 0)
#endif
	{
#ifdef FEAT_PATH_EXTRA
            /* are the wildcard parts equal */
            if (ff_wc_equal(vp->ffv_wc_path, wc_path) == TRUE)
#endif
		break;
	}
    }

    if (vp != NULL)
    {
	/* already visited */
	return FAIL;
    }

    /*
     * New file/dir.  Add it to the list of visited files/dirs.
     */
#ifdef UNIX
    vp = (ff_visited_t *)alloc((unsigned)sizeof(ff_visited_t));
#else
    vp = (ff_visited_t *)alloc((unsigned)(sizeof(ff_visited_t)
						       + STRLEN(ff_expand_buffer)));
#endif

    if (vp != NULL)
    {
#ifdef UNIX
	vp->ffv_st = st;
#else
	STRCPY(vp->ffv_fname, ff_expand_buffer);
#endif
#ifdef FEAT_PATH_EXTRA
	if (wc_path != NULL)
	    vp->ffv_wc_path = vim_strsave(wc_path);
	else
	    vp->ffv_wc_path = NULL;
#endif

	vp->ffv_next = *visited_list;
	*visited_list = vp;
    }

    return OK;
}

/*
 * create stack element from given path pieces
 */
    static ff_stack_t *
ff_create_stack_element(fix_part,
#ifdef FEAT_PATH_EXTRA
	wc_part,
#endif
	level, star_star_empty)
    char_u	*fix_part;
#ifdef FEAT_PATH_EXTRA
    char_u	*wc_part;
#endif
    int		level;
    int		star_star_empty;
{
    ff_stack_t	*new;

    new = (ff_stack_t *)alloc((unsigned)sizeof(ff_stack_t));
    if (new == NULL)
	return NULL;

    new->ff_prev	   = NULL;
    new->ff_filearray	   = NULL;
    new->ff_filearray_size = 0;
    new->ff_filearray_cur  = 0;
    new->ff_stage	   = 0;
    new->ff_level	   = level;
    new->ff_star_star_empty = star_star_empty;;

    /* the following saves NULL pointer checks in vim_findfile */
    if (fix_part == NULL)
	fix_part = (char_u *)"";
    new->ff_fix_path = vim_strsave(fix_part);

#ifdef FEAT_PATH_EXTRA
    if (wc_part == NULL)
	wc_part  = (char_u *)"";
    new->ff_wc_path = vim_strsave(wc_part);
#endif

    if (new->ff_fix_path == NULL
#ifdef FEAT_PATH_EXTRA
	    || new->ff_wc_path == NULL
#endif
	    )
    {
	ff_free_stack_element(new);
	new = NULL;
    }

    return new;
}

/*
 * push a dir on the directory stack
 */
    static void
ff_push(ctx)
    ff_stack_t *ctx;
{
    /* check for NULL pointer, not to return an error to the user, but
     * to prevent a crash
     */
    if (ctx != NULL)
    {
	ctx->ff_prev   = ff_stack_ptr;
	ff_stack_ptr = ctx;
    }
}

/*
 * pop a dir from the directory stack
 * returns NULL if stack is empty
 */
    static ff_stack_t *
ff_pop()
{
    ff_stack_t  *sptr;

    sptr = ff_stack_ptr;
    if (ff_stack_ptr != NULL)
	ff_stack_ptr = ff_stack_ptr->ff_prev;

    return sptr;
}

/*
 * free the given stack element
 */
    static void
ff_free_stack_element(ctx)
    ff_stack_t  *ctx;
{
    /* vim_free handles possible NULL pointers */
    vim_free(ctx->ff_fix_path);
#ifdef FEAT_PATH_EXTRA
    vim_free(ctx->ff_wc_path);
#endif

    if (ctx->ff_filearray != NULL)
	FreeWild(ctx->ff_filearray_size, ctx->ff_filearray);

    vim_free(ctx);
}

/*
 * delete (free) the search context
 */
    static void
ff_clear()
{
    ff_stack_t   *sptr;

    /* clear up stack */
    while ((sptr = ff_pop()) != NULL)
	ff_free_stack_element(sptr);

    vim_free(ff_file_to_search);
    vim_free(ff_start_dir);
    vim_free(ff_fix_path);
#ifdef FEAT_PATH_EXTRA
    vim_free(ff_wc_path);
#endif

#ifdef FEAT_PATH_EXTRA
    if (ff_stopdirs_v != NULL)
    {
	int  i = 0;

	while (ff_stopdirs_v[i] != NULL)
	{
	    vim_free(ff_stopdirs_v[i]);
	    i++;
	}
	vim_free(ff_stopdirs_v);
    }
    ff_stopdirs_v = NULL;
#endif

    /* restet everything */
    ff_file_to_search = ff_start_dir = ff_fix_path = NULL;
#ifdef FEAT_PATH_EXTRA
    ff_wc_path = NULL;
    ff_level = 0;
#endif
}

#ifdef FEAT_PATH_EXTRA
/*
 * check if the given path is in the stopdirs
 * returns TRUE if yes else FALSE
 */
    static int
ff_path_in_stoplist(path, path_len, stopdirs_v)
    char_u	*path;
    int		path_len;
    char_u	**stopdirs_v;
{
    int		i = 0;

    /* eat up trailing path separators, except the first */
    while (path_len > 1 && path[path_len - 1] == PATHSEP)
	path_len--;

    /* if no path consider it as match */
    if (path_len == 0)
	return TRUE;

    for (i = 0; stopdirs_v[i] != NULL; i++)
    {
	if ((int)STRLEN(stopdirs_v[i]) > path_len)
	{
	    /* match for parent directory. So '/home' also matches
	     * '/home/rks'. Check for PATHSEP in stopdirs_v[i], else
	     * '/home/r' would also match '/home/rks'
	     */
	    if (STRNCMP(stopdirs_v[i], path, path_len) == 0
		    && stopdirs_v[i][path_len] == PATHSEP)
		return TRUE;
	}
	else
	{
	    if (STRCMP(stopdirs_v[i], path) == 0)
		return TRUE;
	}
    }
    return FALSE;
}
#endif

#if defined(FEAT_SEARCHPATH) || defined(PROTO)
/*
 * Find the file name "ptr[len]" in the path.
 *
 * On the first call set the parameter 'first' to TRUE to initialize
 * the search.  For repeating calls to FALSE.
 *
 * Repeating calls will return other files called 'ptr[len]' from the path.
 *
 * Only on the first call 'ptr' and 'len' are used.  For repeating calls they
 * don't need valid values.
 *
 * If nothing found on the first call the option FNAME_MESS will issue the
 * message:
 *	    'Can't find file "<file>" in path'
 * On repeating calls:
 *	    'No more file "<file>" found in path'
 *
 * options:
 * FNAME_MESS	    give error message when not found
 *
 * Uses NameBuff[]!
 *
 * Returns an allocated string for the file name.  NULL for error.
 *
 */
    char_u *
find_file_in_path(ptr, len, options, first)
    char_u	*ptr;		/* file name */
    int		len;		/* length of file name */
    int		options;
    int		first;		/* use count'th matching file name */
{
    return find_file_in_path_option(ptr, len, options, first, p_path, FALSE);
}

/*
 * Find the directory name "ptr[len]" in the path.
 *
 * options:
 * FNAME_MESS	    give error message when not found
 *
 * Uses NameBuff[]!
 *
 * Returns an allocated string for the file name.  NULL for error.
 */
    char_u *
find_directory_in_path(ptr, len, options)
    char_u	*ptr;		/* file name */
    int		len;		/* length of file name */
    int		options;
{
    return find_file_in_path_option(ptr, len, options, TRUE, p_cdpath, TRUE);
}

    static char_u *
find_file_in_path_option(ptr, len, options, first, path_option, need_dir)
    char_u	*ptr;		/* file name */
    int		len;		/* length of file name */
    int		options;
    int		first;		/* use count'th matching file name */
    char_u	*path_option;	/* p_path or p_cdpath */
    int		need_dir;	/* looking for directory name */
{
    static char_u	*dir;
    static char_u	*file_to_find = NULL;
    static int		did_findfile_init = FALSE;
    char_u		save_char;
    char_u		*file_name = NULL;
    char_u		*buf = NULL;

    if (first == TRUE)
    {
	/* copy file name into NameBuff, expanding environment variables */
	save_char = ptr[len];
	ptr[len] = NUL;
	expand_env(ptr, NameBuff, MAXPATHL);
	ptr[len] = save_char;

	vim_free(file_to_find);
	file_to_find = vim_strsave(NameBuff);
	if (file_to_find == NULL)	/* out of memory */
	    return NULL;
    }

    if (mch_isFullName(file_to_find))
#if 0
            ||
            (file_to_find[0] == '.' && (file_to_find[1] == PATHSEP || file_to_find[1] == '.')))
#endif
    {
	/*
	 * Absolute path, no need to use "path_option".
	 * If this is not a first call, return NULL.  We already returned a
	 * filename on the first call.
	 * When the file doesn't exist, try adding parts of 'suffixesadd'.
	 */
	if (first == TRUE)
	{
	    buf = curbuf->b_p_sua;
	    len = STRLEN(NameBuff);
	    for (;;)
	    {
		if (mch_getperm(NameBuff) >= 0
			&& (!need_dir || mch_isdir(NameBuff)))
		    return vim_strsave(NameBuff);
		if (*buf == NUL)
		    break;
		copy_option_part(&buf, NameBuff + len, MAXPATHL - len, ",");
	    }
	}
    }
    else
    {
	/*
	 * Loop over all paths in the 'path' or 'cdpath' option.
	 * When "first" is set, first setup to the start of the option.
	 * Otherwise continue to find the next match.
	 */
	if (first == TRUE)
	{
	    vim_findfile_free_visited();
	    dir = path_option;
	    did_findfile_init = FALSE;
	}

	for (;;)
	{
	    if (did_findfile_init)
	    {
		ff_need_dir = need_dir;
		file_name = vim_findfile();
		ff_need_dir = FALSE;
		if (file_name != NULL)
		    break;

		did_findfile_init = FALSE;
	    }
	    else
	    {
		char_u  *r_ptr;

		if (dir == NULL || *dir == NUL)
		    break;

		if ((buf = alloc((int)(MAXPATHL))) == NULL)
		    break;

		/* copy next path */
		buf[0] = 0;
		copy_option_part(&dir, buf, MAXPATHL, " ,");

#ifdef FEAT_PATH_EXTRA
		/* get the stopdir string */
		r_ptr = vim_findfile_stopdir(buf);
#else
		r_ptr = NULL;
#endif
		if (vim_findfile_init(buf, file_to_find, r_ptr, 100, FALSE)
									== OK)
		    did_findfile_init = TRUE;
		vim_free(buf);
	    }
	}
    }
    if (file_name == NULL && (options & FNAME_MESS))
    {
	if (first == TRUE)
	{
	    if (need_dir)
		EMSG2(_("Can't find directory \"%s\" in cdpath"), file_to_find);
	    else
		EMSG2(_("Can't find file \"%s\" in path"), file_to_find);
	}
	else
	{
	    if (need_dir)
		EMSG2(_("No more directory \"%s\" found in cdpath"), file_to_find);
	    else
		EMSG2(_("No more file \"%s\" found in path"), file_to_find);
	}
    }

    return file_name;
}

#endif /* FEAT_SEARCHPATH */

/*
 * Change directory to "new_dir".  If FEAT_SEARCHPATH is defined, search 'cdpath'
 * for relative directory names, otherwise just mch_chdir().
 */
    int
vim_chdir(new_dir)
    char_u	*new_dir;
{
#ifndef FEAT_SEARCHPATH
    return mch_chdir((char *)new_dir);
#else
    char_u	*dir_name;

    dir_name = find_directory_in_path(new_dir, (int)STRLEN(new_dir),
								  FNAME_MESS);
    if (dir_name == NULL)
	return -1;
    return mch_chdir((char *)dir_name);
#endif
}

/*
 * Get user name from machine-specific function and cache it.
 * Returns the user name in "buf[len]".
 * Some systems are quite slow in obtaining the user name (Windows NT).
 * Returns OK or FAIL.
 */
    int
get_user_name(buf, len)
    char_u	*buf;
    int		len;
{
    static char_u	*name = NULL;

    if (name == NULL)
    {
	if (mch_get_user_name(buf, len) == FAIL)
	    return FAIL;
	name = vim_strsave(buf);
    }
    else
	STRNCPY(buf, name, len);
    return OK;
}

#ifndef HAVE_QSORT
/*
 * Our own qsort(), for systems that don't have it.
 * It's simple and slow.  From the K&R C book.
 */
    void
qsort(base, elm_count, elm_size, cmp)
    void	*base;
    size_t	elm_count;
    size_t	elm_size;
    int (*cmp) __ARGS((const void *, const void *));
{
    char_u	*buf;
    char_u	*p1;
    char_u	*p2;
    int		i, j;
    int		gap;

    buf = alloc((unsigned)elm_size);
    if (buf == NULL)
	return;

    for (gap = elm_count / 2; gap > 0; gap /= 2)
	for (i = gap; i < elm_count; ++i)
	    for (j = i - gap; j >= 0; j -= gap)
	    {
		/* Compare the elements. */
		p1 = (char_u *)base + j * elm_size;
		p2 = (char_u *)base + (j + gap) * elm_size;
		if ((*cmp)((void *)p1, (void *)p2) <= 0)
		    break;
		/* Exchange the elemets. */
		mch_memmove(buf, p1, elm_size);
		mch_memmove(p1, p2, elm_size);
		mch_memmove(p2, buf, elm_size);
	    }

    vim_free(buf);
}
#endif

