/* vi:set ts=8 sts=4 sw=4:
 *
 * VIM - Vi IMproved	by Bram Moolenaar
 * Multibyte extensions by Sung-Hoon Baek
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */
/*
 *	file : multbyte.c
 */

#include "vim.h"
#include "globals.h"
#include "proto.h"
#include "option.h"
#ifdef WIN32
# include <windows.h>
#ifndef __MINGW32__
# include <winnls.h>
#endif
#endif
#ifdef USE_GUI_X11
# include <X11/Intrinsic.h>
#endif

#if defined(MULTI_BYTE) || defined(PROTO)

/*
 * Is 'c' a lead byte of multi-byte character?
 */
    int
IsLeadByte(c)
    int		c;
{
#ifdef WIN32
    /* is_dbcs is set by setting 'fileencoding'.  It becomes a Windows
     * CodePage identifier, which we can pass directly in to Windows API*/
    return IsDBCSLeadByteEx(is_dbcs, (BYTE)c);
#else
    return (c & 0x80);
#endif
}

/*
 * Is *p a trail byte of multi-byte character?  base : string pointer to line
 */
    int
IsTrailByte(base, p)
    char_u *base;
    char_u *p;
{
    int lbc = 0;    /* lead byte count*/

    if (base >= p)
	return 0;

    while (p > base)
    {
	if (!IsLeadByte(*(--p)))
	    break;
	lbc++;
    }

    return (lbc & 1);
}

/*
 * if the cursor moves on an trail byte, set the cursor on the lead byte.
 */
    int
AdjustCursorForMultiByteCharacter()
{
    char_u *p;

    if (curwin->w_cursor.col > 0 )
    {
	p = ml_get(curwin->w_cursor.lnum);
	if (IsTrailByte(p, p + curwin->w_cursor.col))
	{
	    --curwin->w_cursor.col;
	    return 1;
	}
    }
    return 0;
}

/*
 * count the length of the str which has multi-byte characters.  two-byte
 * character counts as one character.
 */
    int
MultiStrLen(str)
    char_u	*str;
{
    int count;

    if (str == NULL)
	return 0;
    for (count = 0; *str != NUL; count++)
    {
	if (IsLeadByte(*str))
	{
	    str++;
	    if (*str != NUL)
		str++;
	}
	else
	    str++;
    }
    return count;
}

    int
mb_dec(lp)
    FPOS  *lp;
{
    char_u *p = ml_get(lp->lnum);

    if (lp->col > 0)
    {		/* still within line */
	lp->col--;
	if ( lp->col > 0 && IsTrailByte(p, p + lp->col))
	    lp->col--;
	return 0;
    }
    if (lp->lnum > 1)
    {		/* there is a prior line */
	lp->lnum--;
	lp->col = STRLEN(ml_get(lp->lnum));
	if ( lp->col > 0 && IsTrailByte(p, p + lp->col))
	    lp->col--;
	return 1;
    }
    return -1;			/* at start of file */
}

    int
mb_isbyte1(buf, x)
    char_u	*buf;
    int		x;
{
    register int xx = x;

    if (*(buf + x) & 0x80)
    {
	do
	    x--;
	while (x >= 0 && (*(buf + x) & 0x80));
	if ((xx - x) % 2 == 1)
	    return 1;
    }
    return 0;
}

    int
mb_isbyte2(buf, x)
    char_u	*buf;
    int		x;
{
    register int xx = x;

    if (*(buf + x) & 0x80)
    {
	do
	    x--;
	while (x >= 0 && (*(buf + x) & 0x80));
	if ((xx - x) % 2 == 0)
	    return 1;
    }
    return 0;
}
#endif /* MULTI_BYTE */

#if defined(USE_XIM) || defined(PROTO)

static int	xim_has_focus = 0;
#ifdef USE_GUI_X11
static XIMStyle	input_style;
static XIM	xim;
#endif

    void
xim_set_focus(int focus)
{
    if (!xic)
	return;

    if (focus)
    {
	if (!xim_has_focus)
	{
	    xim_has_focus = 1;
#ifdef USE_GUI_GTK
	    gdk_im_begin(xic, gui.drawarea->window);
#else
	    XSetICFocus(xic);
#endif
	}
    }
    else
    {
	if (xim_has_focus)
	{
	    xim_has_focus = 0;
#ifdef USE_GUI_GTK
	    gdk_im_end();
#else
	    XUnsetICFocus(xic);
#endif
	}
    }
}

    void
xim_set_preedit()
{
    if (!xic)
	return;

    xim_set_focus(!(State & NORMAL));

#ifdef USE_GUI_GTK
    if (/*GTK_WIDGET_HAS_FOCUS(gui.drawarea) &&*/ gdk_im_ready() &&
	(gdk_ic_get_style(xic) & GDK_IM_PREEDIT_POSITION))
    {
	GdkICAttributesType mask = 0;

	if (!xim_has_focus)
	{
	    if (xic_attr->spot_location.y >= 0)
	    {
		xic_attr->spot_location.x = 0;
		xic_attr->spot_location.y = -100;
		mask |= GDK_IC_SPOT_LOCATION;
	    }
	}
	else
	{
	    if (xic_attr->spot_location.x != TEXT_X(gui.col) ||
		xic_attr->spot_location.y != TEXT_Y(gui.row))
	    {
		xic_attr->spot_location.x = TEXT_X(gui.col);
		xic_attr->spot_location.y = TEXT_Y(gui.row);
		mask |= GDK_IC_SPOT_LOCATION;
	    }
	}

	if (xim_fg_color < 0)
	{
	    xim_fg_color = gui.def_norm_pixel;
	    xim_bg_color = gui.def_back_pixel;
	}
	if (xic_attr->preedit_foreground.pixel != xim_fg_color)
	{
	    xic_attr->preedit_foreground.pixel = xim_fg_color;
	    mask |= GDK_IC_PREEDIT_FOREGROUND;
	}
	if (xic_attr->preedit_background.pixel != xim_bg_color)
	{
	    xic_attr->preedit_background.pixel = xim_bg_color;
	    mask |= GDK_IC_PREEDIT_BACKGROUND;
	}

	if (mask)
	    gdk_ic_set_attr(xic, xic_attr, mask);
    }
#else
    {
	XVaNestedList attr_list;
	XRectangle spot_area;
	XPoint over_spot;
	int line_space;

	if (!xim_has_focus)
	{
	    /* hide XIM cursor */
	    over_spot.x = 0;
	    over_spot.y = -100; /* arbitrary invisible position */
	    attr_list = (XVaNestedList) XVaCreateNestedList(0,
							    XNSpotLocation,
							    &over_spot,
							    NULL);
	    XSetICValues(xic, XNPreeditAttributes, attr_list, NULL);
	    XFree(attr_list);
	    return;
	}

	if (input_style & XIMPreeditPosition)
	{
	    if (xim_fg_color < 0)
	    {
		xim_fg_color = gui.def_norm_pixel;
		xim_bg_color = gui.def_back_pixel;
	    }
	    over_spot.x = gui.col * gui.char_width;
	    over_spot.y = gui_get_base_height() + gui.row * gui.char_height
			  + gui.char_ascent;
	    spot_area.x = 0;
	    spot_area.y = gui_get_base_height();
	    spot_area.height = gui_get_base_height() + gui.char_height * Rows;
	    spot_area.width  = gui.char_width * Columns;
	    line_space = gui.char_height;
	    attr_list = (XVaNestedList) XVaCreateNestedList(0,
					    XNSpotLocation, &over_spot,
					    XNForeground, (Pixel) xim_fg_color,
					    XNBackground, (Pixel) xim_bg_color,
					    XNArea, &spot_area,
					    XNLineSpace, line_space,
					    NULL);
	    if (XSetICValues(xic, XNPreeditAttributes, attr_list, NULL))
		fprintf(stderr, "Cannot set IC values\n");
	    XFree(attr_list);
	}
    }
#endif
}

    void
xim_set_status_area()
{
#if !defined(USE_GUI_GTK)
    XVaNestedList preedit_list = 0, status_list = 0, list = 0;
    XRectangle pre_area, status_area;

    if (!xic)
	return;

    if (input_style & XIMPreeditArea)   /* off-the-spot */
    {
	preedit_list = (XVaNestedList) XVaCreateNestedList(0, XNArea,
							   &pre_area, NULL);
	XGetICValues(xic, XNPreeditAttributes, preedit_list, NULL);
	pre_area.y = gui_get_base_height() + gui.char_height * Rows;
    }

    if (input_style & XIMStatusArea)
    {
	status_area.x = 0;
	status_area.y = gui_get_base_height() + gui.char_height * Rows;
	status_area.width = gui.char_width * Columns;
	status_area.height = gui.char_height;
	status_list = (XVaNestedList) XVaCreateNestedList(0, XNArea,
							  &status_area, NULL);
    }

    if (input_style & XIMPreeditPosition)   /* over-the-spot */
    {
	pre_area.x = 0;
	pre_area.y = gui_get_base_height();
	pre_area.height = gui_get_base_height() + gui.char_height;
	pre_area.width	= gui.char_width * Columns;
	preedit_list = (XVaNestedList)XVaCreateNestedList(0, XNArea,
							  &pre_area, NULL);
    }

    if (preedit_list && status_list)
	list = (XVaNestedList) XVaCreateNestedList(0, XNPreeditAttributes,
						   preedit_list,
						   XNStatusAttributes,
						   status_list, NULL);
    else if (preedit_list)
	list = (XVaNestedList) XVaCreateNestedList(0, XNPreeditAttributes,
						   preedit_list, NULL);
    else if (status_list)
	list = (XVaNestedList) XVaCreateNestedList(0, XNStatusAttributes,
						   status_list, NULL);
    else
	list = NULL;

    if (list)
    {
	XSetICValues(xic, XNVaNestedList, list, NULL);
	if (status_list)
	    XFree(status_list);
	if (preedit_list)
	    XFree(preedit_list);
	XFree(list);
    }
#endif
}

#if 0	/* not used? */
    XIMStyle
get_xim_style()
{
    return input_style;
}
#endif

#if defined(USE_GUI_X11) || defined(PROTO)
# if defined(XtSpecificationRelease) && XtSpecificationRelease >= 6
#  define USE_X11R6_XIM
# endif

static int xim_real_init __ARGS((Window x11_window, Display *x11_display));

#ifdef USE_X11R6_XIM
static void xim_instantiate_cb __ARGS((Display *display, XPointer client_data, XPointer	call_data));
static void xim_destroy_cb __ARGS((XIM im, XPointer client_data, XPointer call_data));

    static void
xim_instantiate_cb(display, client_data, call_data)
    Display	*display;
    XPointer	client_data;
    XPointer	call_data;
{
    Window	x11_window;
    Display	*x11_display;

    gui_get_x11_windis(&x11_window, &x11_display);
    if (display != x11_display)
	return;

    xim_real_init(x11_window, x11_display);
    if (xim != NULL)
	XUnregisterIMInstantiateCallback(x11_display, NULL, NULL, NULL,
					 xim_instantiate_cb, NULL);
}

    static void
xim_destroy_cb(im, client_data, call_data)
    XIM		im;
    XPointer	client_data;
    XPointer	call_data;
{
    Window	x11_window;
    Display	*x11_display;

    gui_get_x11_windis(&x11_window, &x11_display);

    xim = NULL;
    xic = NULL;

    XRegisterIMInstantiateCallback(x11_display, NULL, NULL, NULL,
				   xim_instantiate_cb, NULL);
}
#endif

    void
xim_init()
{
    Window	x11_window;
    Display	*x11_display;

    gui_get_x11_windis(&x11_window, &x11_display);

    xim = NULL;
    xic = NULL;

    if (xim_real_init(x11_window, x11_display))
	return;

#ifdef USE_X11R6_XIM
    XRegisterIMInstantiateCallback(x11_display, NULL, NULL, NULL,
				   xim_instantiate_cb, NULL);
#endif
}

    static int
xim_real_init(x11_window, x11_display)
    Window  x11_window;
    Display *x11_display;
{
    int		i;
    char	*p,
		*s,
		*ns,
		*end,
		tmp[1024],
		buf[32];
    XIMStyles	*xim_styles;
    XIMStyle	this_input_style = 0;
    Boolean	found;
    XPoint	over_spot;
    XVaNestedList preedit_list, status_list;

#ifdef FONTSET
    if (!gui.fontset != NULL)
    {
	fprintf(stderr, "XIM requires guifontset setting\n");
	return FALSE;
    }
#endif
    if (xim == NULL)
    {
	if (!gui.input_method || !*gui.input_method)
	{
	    if ((p = XSetLocaleModifiers("")) != NULL && *p)
		xim = XOpenIM(x11_display, NULL, NULL, NULL);
	}
	else
	{
	    strcpy(tmp, gui.input_method);
	    for (ns=s=tmp; ns && *s;)
	    {
		while (*s && isspace((unsigned char)*s))
		    s++;
		if (!*s)
		    break;
		if ((ns = end = strchr(s, ',')) == 0)
		    end = s + strlen(s);
		while (isspace((unsigned char)*end))
		    end--;
		*end = '\0';

		strcpy(buf, "@im=");
		strcat(buf, s);
		if ((p = XSetLocaleModifiers(buf)) != NULL
		    && *p
		    && (xim = XOpenIM(x11_display, NULL, NULL, NULL)) != NULL)
		    break;

		s = ns + 1;
	    }
	}
    }

    if (xim == NULL && (p = XSetLocaleModifiers("")) != NULL && *p)
	xim = XOpenIM(x11_display, NULL, NULL, NULL);

    if (!xim)
	xim = XOpenIM(x11_display, NULL, NULL, NULL);

    if (!xim)
    {
	fprintf(stderr, "Failed to open input method\n");
	return FALSE;
    }

#ifdef USE_X11R6_XIM
    {
	XIMCallback destroy_cb;

	destroy_cb.callback = xim_destroy_cb;
	destroy_cb.client_data = NULL;
	if (XSetIMValues(xim, XNDestroyCallback, &destroy_cb, NULL))
	    fprintf(stderr, "Warning: Could not set destroy callback to IM\n");
    }
#endif

    if (XGetIMValues(xim, XNQueryInputStyle, &xim_styles, NULL) || !xim_styles)
    {
	fprintf(stderr, "input method doesn't support any style\n");
	XCloseIM(xim);
	return FALSE;
    }

    found = False;
    strcpy(tmp, gui.preedit_type);
    for (s = tmp; s && !found; )
    {
	while (*s && isspace((unsigned char)*s))
	    s++;
	if (!*s)
	    break;
	if ((ns = end = strchr(s, ',')) != 0)
	    ns++;
	else
	    end = s + strlen(s);
	while (isspace((unsigned char)*end))
	    end--;
	*end = '\0';

	if (!strcmp(s, "OverTheSpot"))
	    this_input_style = (XIMPreeditPosition | XIMStatusArea);
	else if (!strcmp(s, "OffTheSpot"))
	    this_input_style = (XIMPreeditArea | XIMStatusArea);
	else if (!strcmp(s, "Root"))
	    this_input_style = (XIMPreeditNothing | XIMStatusNothing);

	for (i = 0; (unsigned short)i < xim_styles->count_styles; i++)
	{
	    if (this_input_style == xim_styles->supported_styles[i])
	    {
		found = True;
		break;
	    }
	}

	s = ns;
    }
    XFree(xim_styles);

    if (!found)
    {
	fprintf(stderr, "input method doesn't support my preedit type\n");
	XCloseIM(xim);
	return FALSE;
    }

    over_spot.x = gui.col * gui.char_width;
    over_spot.y = gui_get_base_height() + gui.row * gui.char_height
	          + gui.char_ascent;
    input_style = this_input_style;
    preedit_list = (XVaNestedList)
	XVaCreateNestedList(0,
			    XNSpotLocation, &over_spot,
			    XNForeground, (Pixel)gui.def_norm_pixel,
			    XNBackground, (Pixel)gui.def_back_pixel,
			    XNFontSet, (XFontSet)gui.norm_font,
			    NULL);
    status_list = (XVaNestedList)
	XVaCreateNestedList(0,
			    XNForeground, (Pixel)gui.def_norm_pixel,
			    XNBackground, (Pixel)gui.def_back_pixel,
			    XNFontSet, (XFontSet)gui.norm_font,
			    NULL);
    xic = XCreateIC(xim,
		    XNInputStyle, input_style,
		    XNClientWindow, x11_window,
		    XNFocusWindow, x11_window,
		    XNPreeditAttributes, preedit_list,
		    XNStatusAttributes, status_list,
		    NULL);
    XFree(status_list);
    XFree(preedit_list);
    if (xic && (input_style & XIMStatusArea))
	xim_set_status_area();

    if (!xic)
    {
	fprintf(stderr,"Failed to create input context\n");
	XCloseIM(xim);
	return FALSE;
    }

    return TRUE;
}

#elif defined(USE_GUI_GTK) || defined(PROTO)
    void
xim_init()
{
    if (!gdk_im_ready())
    {
	EMSG("Your GTK+ does not support XIM");
	return;
    }
    if ((xic_attr = gdk_ic_attr_new()) != NULL)
    {
	gint width, height;
	GdkEventMask mask;
	GdkColormap *colormap;
	GdkICAttr *attr = xic_attr;
	GdkICAttributesType attrmask = GDK_IC_ALL_REQ;
	GdkIMStyle style;
	GtkWidget *widget = gui.drawarea;
	GdkIMStyle supported_style = GDK_IM_PREEDIT_NONE |
				     GDK_IM_PREEDIT_NOTHING |
				     GDK_IM_PREEDIT_POSITION |
				     GDK_IM_STATUS_NONE |
				     GDK_IM_STATUS_NOTHING;

#ifdef USE_FONTSET
	if (widget->style
		&& gui.fontset != NULL
		&& gui.fontset->type != GDK_FONT_FONTSET)
	    supported_style &= ~GDK_IM_PREEDIT_POSITION;
#endif

	attr->style = style = gdk_im_decide_style (supported_style);
	attr->client_window = widget->window;

	if ((colormap = gtk_widget_get_colormap (widget)) !=
	    gtk_widget_get_default_colormap ())
	{
	    attrmask |= GDK_IC_PREEDIT_COLORMAP;
	    attr->preedit_colormap = colormap;
	}
	attrmask |= GDK_IC_PREEDIT_FOREGROUND;
	attrmask |= GDK_IC_PREEDIT_BACKGROUND;
	attr->preedit_foreground = widget->style->fg[GTK_STATE_NORMAL];
	attr->preedit_background = widget->style->base[GTK_STATE_NORMAL];

	switch (style & GDK_IM_PREEDIT_MASK)
	{
	    case GDK_IM_PREEDIT_POSITION:
#ifdef USE_FONTSET
		if (widget->style
			&& gui.fontset != NULL
			&& gui.fontset->type != GDK_FONT_FONTSET)
		{
		    g_warning ("over-the-spot style requires fontset");
		    break;
		}
#endif

		gdk_window_get_size (widget->window, &width, &height);

		attrmask |= GDK_IC_PREEDIT_POSITION_REQ;
		attr->spot_location.x = 0;
		attr->spot_location.y = -1;
		attr->preedit_area.x = 0;
		attr->preedit_area.y = 0;
		attr->preedit_area.width = width;
		attr->preedit_area.height = height;
#ifdef USE_FONTSET
		attr->preedit_fontset = gui.fontset;
#endif

		break;
	}
	xic = gdk_ic_new (attr, attrmask);

	if (xic == NULL)
	    g_warning ("Can't create input context.");
	else
	{
	    mask = gdk_window_get_events (widget->window);
	    mask |= gdk_ic_get_events (xic);
	    gdk_window_set_events (widget->window, mask);
	}
    }
}
#endif /* USE_GUI_GTK */

#endif /* USE_XIM */
