/* vi:set ts=8 sts=4 sw=4:
 *
 * VIM - Vi IMproved	by Bram Moolenaar
 * Multibyte extensions by Sung-Hoon Baek
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */
/*
 *	file : multbyte.c
 */

#include "vim.h"
#include "globals.h"
#include "proto.h"
#include "option.h"
#ifdef WIN32
# include <windows.h>
#ifndef __MINGW32__
# include <winnls.h>
#endif
#endif

#if defined(MULTI_BYTE) || defined(PROTO)

/*
 * Is 'c' a lead byte of multi-byte character?
 */
    int
IsLeadByte(c)
    int		c;
{
#ifdef WIN32
    /* is_dbcs is set by setting 'fileencoding'.  It becomes a Windows
     * CodePage identifier, which we can pass directly in to Windows API*/
    return IsDBCSLeadByteEx(is_dbcs, (BYTE)c);
#else
    return (c & 0x80);
#endif
}

/*
 * Is *p a trail byte of multi-byte character?  base : string pointer to line
 */
    int
IsTrailByte(base, p)
    char_u *base;
    char_u *p;
{
    int lbc = 0;    /* lead byte count*/

    if (base >= p)
	return 0;

    while (p > base)
    {
	if (!IsLeadByte(*(--p)))
	    break;
	lbc++;
    }

    return (lbc & 1);
}

/*
 * if the cursor moves on an trail byte, set the cursor on the lead byte.
 */
    int
AdjustCursorForMultiByteCharacter()
{
    char_u *p;

    if (curwin->w_cursor.col > 0 )
    {
	p = ml_get(curwin->w_cursor.lnum);
	if (IsTrailByte(p, p + curwin->w_cursor.col))
	{
	    --curwin->w_cursor.col;
	    return 1;
	}
    }
    return 0;
}

/*
 * count the length of the str which has multi-byte characters.  two-byte
 * character counts as one character.
 */
    int
MultiStrLen(str)
    char_u	*str;
{
    int count;

    if (str == NULL)
	return 0;
    for (count = 0; *str != NUL; count++)
    {
	if (IsLeadByte(*str))
	{
	    str++;
	    if (*str != NUL)
		str++;
	}
	else
	    str++;
    }
    return count;
}

    int
mb_dec(lp)
    FPOS  *lp;
{
    char_u *p = ml_get(lp->lnum);

    if (lp->col > 0)
    {		/* still within line */
	lp->col--;
	if ( lp->col > 0 && IsTrailByte(p, p + lp->col))
	    lp->col--;
	return 0;
    }
    if (lp->lnum > 1)
    {		/* there is a prior line */
	lp->lnum--;
	lp->col = STRLEN(ml_get(lp->lnum));
	if ( lp->col > 0 && IsTrailByte(p, p + lp->col))
	    lp->col--;
	return 1;
    }
    return -1;			/* at start of file */
}

    int
mb_isbyte1(buf, x)
    char_u	*buf;
    int		x;
{
    register int xx = x;

    if (*(buf + x) & 0x80)
    {
	do
	    x--;
	while (x >= 0 && (*(buf + x) & 0x80));
	if ((xx - x) % 2 == 1)
	    return 1;
    }
    return 0;
}

    int
mb_isbyte2(buf, x)
    char_u	*buf;
    int		x;
{
    register int xx = x;

    if (*(buf + x) & 0x80)
    {
	do
	    x--;
	while (x >= 0 && (*(buf + x) & 0x80));
	if ((xx - x) % 2 == 0)
	    return 1;
    }
    return 0;
}
#endif /* MULTI_BYTE */

#if defined(USE_XIM) || defined(PROTO)

static int xim_has_focus = 0;
static XIMStyle	input_style;

    void
xim_set_focus(int focus)
{
    if (!xic)
	return;

    if (focus)
    {
	if (!xim_has_focus)
	{
	    xim_has_focus = 1;
	    XSetICFocus(xic);
	}
    }
    else
    {
	if (xim_has_focus)
	{
	    xim_has_focus = 0;
	    XUnsetICFocus(xic);
	}
    }
}

    void
xim_set_preedit()
{
    XVaNestedList attr_list;
    XRectangle spot_area;
    XPoint over_spot;
    int line_space;

    if (!xic)
	return;

    xim_set_focus(!(State & NORMAL));
    if (!xim_has_focus)
    {
	/* hide XIM cursor */
	over_spot.x = 0;
	over_spot.y = -100; /* arbitrary invisible position */
	attr_list = (XVaNestedList) XVaCreateNestedList(0,
						XNSpotLocation, &over_spot,
						NULL);
	XSetICValues(xic, XNPreeditAttributes, attr_list, NULL);
	XFree(attr_list);
	return;
    }

    if (input_style & XIMPreeditPosition)
    {
	if (xim_fg_color == -1)
	{
	    xim_fg_color = gui.def_norm_pixel;
	    xim_bg_color = gui.def_back_pixel;
	}
	over_spot.x = TEXT_X(gui.col);
	over_spot.y = TEXT_Y(gui.row);
	spot_area.x = 0;
	spot_area.y = 0;
	spot_area.height = gui.char_height * Rows;
	spot_area.width  = gui.char_width * Columns;
	line_space = gui.char_height;
	attr_list = (XVaNestedList) XVaCreateNestedList(0,
					XNSpotLocation, &over_spot,
					XNForeground, (Pixel) xim_fg_color,
					XNBackground, (Pixel) xim_bg_color,
					XNArea, &spot_area,
					XNLineSpace, line_space,
					NULL);
	XSetICValues(xic, XNPreeditAttributes, attr_list, NULL);
	XFree(attr_list);
    }
}

    void
xim_set_status_area()
{
    XVaNestedList preedit_list = 0, status_list = 0, list = 0;
    XRectangle pre_area, status_area;

    if (!xic)
	return;

    if (input_style & XIMPreeditArea)   /* off-the-spot */
    {
	preedit_list = (XVaNestedList) XVaCreateNestedList(0, XNArea,
							   &pre_area, NULL);
	XGetICValues(xic, XNPreeditAttributes, preedit_list, NULL);
	pre_area.y = gui.char_height * Rows;
    }

    if (input_style & XIMStatusArea)
    {
	status_area.x = 0;
	status_area.y = gui.char_height * Rows;
	status_area.width = gui.char_width * Columns;
	status_area.height = gui.char_height;
	status_list = (XVaNestedList) XVaCreateNestedList(0, XNArea,
							  &status_area, NULL);
    }

    if (input_style & XIMPreeditPosition)   /* over-the-spot */
    {
	pre_area.x = 0;
	pre_area.y	= 0;
	pre_area.height = gui.char_height;
	pre_area.width	= gui.char_width * Columns;
	preedit_list = (XVaNestedList)XVaCreateNestedList(0, XNArea,
							  &pre_area, NULL);
    }

    if (preedit_list && status_list)
	list = (XVaNestedList) XVaCreateNestedList(0, XNPreeditAttributes,
						   preedit_list,
						   XNStatusAttributes,
						   status_list, NULL);
    else if (preedit_list)
	list = (XVaNestedList) XVaCreateNestedList(0, XNPreeditAttributes,
						   preedit_list, NULL);
    else if (status_list)
	list = (XVaNestedList) XVaCreateNestedList(0, XNStatusAttributes,
						   status_list, NULL);
    else
	list = NULL;

    if (list)
    {
	XSetICValues(xic, XNVaNestedList, list, NULL);
	if (status_list)
	    XFree(status_list);
	if (preedit_list)
	    XFree(preedit_list);
	XFree(list);
    }
}

#if defined(USE_GUI_X11) || defined(PROTO)
    void
xim_init()
{
    int		i;
    char	*p,
		*s,
		*ns,
		*end,
		tmp[1024],
		buf[32];
    XIMStyles	*xim_styles;
    XIMStyle	this_input_style = 0;
    Boolean	found;
    XPoint	over_spot;
    XVaNestedList preedit_list, status_list;
    Window	x11_window;
    Display	*x11_display;

    xic = NULL;
    gui_get_x11_windis(&x11_window, &x11_display);

    if (xim == NULL)
    {
	if (!gui.input_method || !*gui.input_method)
	{
	    if ((p = XSetLocaleModifiers("")) != NULL && *p)
		xim = XOpenIM(x11_display, NULL, NULL, NULL);
	}
	else
	{
	    strcpy(tmp, gui.input_method);
	    for (ns=s=tmp; ns && *s;)
	    {
		while (*s && isspace((unsigned char)*s))
		    s++;
		if (!*s)
		    break;
		if ((ns = end = strchr(s, ',')) == 0)
		    end = s + strlen(s);
		while (isspace((unsigned char)*end))
		    end--;
		*end = '\0';

		strcpy(buf, "@im=");
		strcat(buf, s);
		if ((p = XSetLocaleModifiers(buf)) != NULL
		    && *p
		    && (xim = XOpenIM(x11_display, NULL, NULL, NULL)) != NULL)
		    break;

		s = ns + 1;
	    }
	}
    }

    if (xim == NULL && (p = XSetLocaleModifiers("")) != NULL && *p)
	xim = XOpenIM(x11_display, NULL, NULL, NULL);

    if (!xim)
	xim = XOpenIM(x11_display, NULL, NULL, NULL);

    if (!xim)
    {
	fprintf(stderr, "Failed to open input method\n");
	return;
    }

    if (XGetIMValues(xim, XNQueryInputStyle, &xim_styles, NULL) || !xim_styles)
    {
	fprintf(stderr, "input method doesn't support any style\n");
	XCloseIM(xim);
	return;
    }

    found = False;
    strcpy(tmp, gui.preedit_type);
    for (s = tmp; s && !found; )
    {
	while (*s && isspace((unsigned char)*s))
	    s++;
	if (!*s)
	    break;
	if ((ns = end = strchr(s, ',')) != 0)
	    ns++;
	else
	    end = s + strlen(s);
	while (isspace((unsigned char)*end))
	    end--;
	*end = '\0';

	if (!strcmp(s, "OverTheSpot"))
	    this_input_style = (XIMPreeditPosition | XIMStatusArea);
	else if (!strcmp(s, "OffTheSpot"))
	    this_input_style = (XIMPreeditArea | XIMStatusArea);
	else if (!strcmp(s, "Root"))
	    this_input_style = (XIMPreeditNothing | XIMStatusNothing);

	for (i = 0; (unsigned short)i < xim_styles->count_styles; i++)
	{
	    if (this_input_style == xim_styles->supported_styles[i])
	    {
		found = True;
		break;
	    }
	}

	s = ns;
    }
    XFree(xim_styles);

    if (!found)
    {
	fprintf(stderr, "input method doesn't support my preedit type\n");
	XCloseIM(xim);
	return;
    }

    over_spot.x = TEXT_X(0);
    over_spot.y = TEXT_Y(0);
    input_style = this_input_style;
    preedit_list = (XVaNestedList)
	XVaCreateNestedList(0,
			    XNSpotLocation, &over_spot,
			    XNForeground, (Pixel)gui.def_norm_pixel,
			    XNBackground, (Pixel)gui.def_back_pixel,
			    XNFontSet, (XFontSet)gui.norm_font,
			    NULL);
    status_list = (XVaNestedList)
	XVaCreateNestedList(0,
			    XNForeground, (Pixel)gui.def_norm_pixel,
			    XNBackground, (Pixel)gui.def_back_pixel,
			    XNFontSet, (XFontSet)gui.norm_font,
			    NULL);
    xic = XCreateIC(xim,
		    XNInputStyle, input_style,
		    XNClientWindow, x11_window,
		    XNFocusWindow, x11_window,
		    XNPreeditAttributes, preedit_list,
		    XNStatusAttributes, status_list,
		    NULL);
    XFree(status_list);
    XFree(preedit_list);
    if (xic && (input_style & XIMStatusArea))
	xim_set_status_area();

    if (!xic)
    {
	fprintf(stderr,"Failed to create input context\n");
	XCloseIM(xim);
    }

    return;
}
#endif

#endif /* USE_XIM */
