/* vi:set ts=8 sts=4 sw=4:
 * VIM - Vi IMproved		by Bram Moolenaar
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */

/*
 * Porting to GTK+ was done by:
 *
 * (C) 1998,1999 by Marcin Dalecki <dalecki@cs.net.pl>
 * with GREAT support and continuous encouragements by Andy Kahn and of
 * course Bram Moolenaar!
 *
 *	"I'm a one-man software company. If you have anything UNIX, net or
 *	embedded systems related, which seems to cause some serious trouble for
 *	your's in-house developers, maybe we need to talk badly with each other
 *	:-) <dalecki@cs.net.pl> (My native language is polish and I speak
 *	native grade german too. I'm living in Gttingen.de.)
 *	--mdcki"
 */

#include "gui_gtk_to.h"
#include "gui_gtk_f.h"

#ifdef MIN
# undef MIN
#endif
#ifdef MAX
# undef MAX
#endif

#include "vim.h"

#include "../pixmaps/alert.xpm"
#include "../pixmaps/error.xpm"
#include "../pixmaps/generic.xpm"
#include "../pixmaps/info.xpm"
#include "../pixmaps/quest.xpm"

/*
 * Icons used by the toolbar code.
 */
#include "../pixmaps/tb_new.xpm"
#include "../pixmaps/tb_open.xpm"
#include "../pixmaps/tb_close.xpm"
#include "../pixmaps/tb_save.xpm"
#include "../pixmaps/tb_print.xpm"
#include "../pixmaps/tb_cut.xpm"
#include "../pixmaps/tb_copy.xpm"
#include "../pixmaps/tb_paste.xpm"
#include "../pixmaps/tb_find.xpm"
#include "../pixmaps/tb_find_next.xpm"
#include "../pixmaps/tb_find_prev.xpm"
#include "../pixmaps/tb_find_help.xpm"
#include "../pixmaps/tb_exit.xpm"
#include "../pixmaps/tb_undo.xpm"
#include "../pixmaps/tb_redo.xpm"
#include "../pixmaps/tb_help.xpm"
#include "../pixmaps/tb_macro.xpm"
#include "../pixmaps/tb_make.xpm"
#include "../pixmaps/tb_save_all.xpm"
#include "../pixmaps/tb_jump.xpm"
#include "../pixmaps/tb_ctags.xpm"
#include "../pixmaps/tb_load_session.xpm"
#include "../pixmaps/tb_save_session.xpm"
#include "../pixmaps/tb_new_session.xpm"
#include "../pixmaps/tb_blank.xpm"
#include "../pixmaps/tb_maximize.xpm"
#include "../pixmaps/tb_split.xpm"
#include "../pixmaps/tb_minimize.xpm"
#include "../pixmaps/tb_shell.xpm"
#include "../pixmaps/tb_replace.xpm"

#include <gdk/gdkkeysyms.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>

#include <gtk/gtk.h>

/*
 * Flags used to distinguish the different contexts in which the
 * find/replace callback may be called.
 */
#define FR_DIALOGTERM	1
#define FR_FINDNEXT	2
#define FR_R_FINDNEXT	3
#define FR_REPLACE	4
#define FR_REPLACEALL	5


extern GtkWidget *main_window;
extern GtkWidget *form_window;
extern GtkWidget *drawing_area;
extern GtkWidget *menu_bar;
extern GtkWidget *tool_bar;

#ifdef GTK_HAVE_FEATURES_1_1_0
extern GtkAccelGroup *accel_group;
#endif

#if defined(USE_BROWSE) || defined(PROTO)
static void open_ok_callback(GtkWidget * widget, gpointer data);
static void open_cancel_callback(GtkWidget * widget, gpointer data);
static gboolean open_destroy_callback(GtkWidget * widget);
#endif

#ifdef GUI_DIALOG
static void butproc(GtkWidget * widget, gpointer data);
#endif

static void entry_activate_cb(GtkWidget *widget, GtkWidget *with);
static void entry_changed_cb(GtkWidget *entry, GtkWidget *dialog);
static void find_direction_cb(GtkWidget *widget, gpointer data);
static void find_replace_cb(GtkWidget *widget, gint flags);
static void gui_mch_recurse_tearoffs(VimMenu *menu, int val);
static void menu_item_callback(GtkWidget *widget, gpointer data);
static void exact_match_cb(GtkWidget *widget, gpointer data);
static void repl_dir_cb(GtkWidget * widget, gpointer data);
static void find_replace_dialog_create(int do_replace);
static void pixmap_create_blank(GdkPixmap **pixmap, GdkBitmap **mask);
static void pixmap_create_by_name(char *name, GdkPixmap **, GdkBitmap **mask);
static void pixmap_create_by_num(int pixmap_num, GdkPixmap **, GdkBitmap **);
static void pixmap_create_by_dir(char *name, GdkPixmap **, GdkBitmap **mask);
static void get_pixmap(char *menuname, GdkPixmap **pixmap, GdkBitmap **mask);


/*
 * Create a highly customized menu item by hand instead of by using:
 *
 * gtk_menu_item_new_with_label(menu->dname);
 *
 * This is neccessary, since there is no other way in GTK+ to get the
 * not automatically parsed accellerator stuff right.
 */
static void
menu_item_new(VimMenu *menu, GtkWidget *parent_widget, int sub_menu)
{
    char *name;
    char *tmp;
    GtkWidget *widget;
#ifdef GTK_HAVE_FEATURES_1_1_0
    GtkWidget *bin;
    GtkWidget *label;
    guint accel_key;
    GtkAccelGroup *parent_accel_group = NULL;
    GSList *tmp_list;

    tmp_list = gtk_accel_groups_from_object(GTK_OBJECT(parent_widget));
    if (tmp_list)
	parent_accel_group = tmp_list->data;

    widget = gtk_widget_new(GTK_TYPE_MENU_ITEM,
			    "GtkWidget::visible", TRUE,
			    "GtkWidget::sensitive", TRUE,
			    /* "GtkWidget::parent", parent->submenu_id, */
			    NULL);
    bin = gtk_widget_new(GTK_TYPE_HBOX,
			 "GtkWidget::visible", TRUE,
			 "GtkWidget::parent", widget,
			 "GtkBox::spacing", 16,
			 NULL);
    label = gtk_widget_new(GTK_TYPE_ACCEL_LABEL,
			   "GtkWidget::visible", TRUE,
			   "GtkWidget::parent", bin,
			   "GtkAccelLabel::accel_widget", widget,
			   "GtkMisc::xalign", 0.0,
			   NULL);
    if (menu->actext)
	gtk_widget_new(GTK_TYPE_LABEL,
		       "GtkWidget::visible", TRUE,
		       "GtkWidget::parent", bin,
		       "GtkLabel::label", menu->actext,
		       "GtkMisc::xalign", 1.0,
			NULL);
    if (sub_menu)
	name = g_strdup((const gchar *)(menu->name));
    else
	name = g_strdup((const gchar *)(menu->dname));

    /* Translate VIM accelerator tagging into GTK+'s */
    for (tmp = name; *tmp; ++tmp) {
	if (*tmp == '&')
	    *tmp = '_';
    }
    accel_key = gtk_label_parse_uline(GTK_LABEL(label), name);

    g_free(name);
    if ((accel_key != GDK_VoidSymbol) && GTK_IS_MENU_BAR(parent_widget)) {
	gtk_widget_add_accelerator(widget,
				   "activate_item",
				   accel_group,
				   accel_key, GDK_MOD1_MASK,
				   GTK_ACCEL_LOCKED);
    }
    if ((accel_key != GDK_VoidSymbol) && parent_accel_group) {
	gtk_widget_add_accelerator(widget,
				   "activate_item",
				   parent_accel_group,
				   accel_key, 0,
				   GTK_ACCEL_LOCKED);
    }

    menu->id = widget;
#else
    char *tmp2;

    name = g_strdup((const gchar *)(menu->name));
    for (tmp = (char *)(menu->name), tmp2 = name; *tmp; tmp++) {
	if (*tmp != '&') {
	    *tmp2 = *tmp;
	    tmp2++;
	}
    }
    *tmp2 = '\0';
    widget = gtk_menu_item_new_with_label(name);
    g_free(name);
    menu->id = widget;
#endif
}


/*ARGSUSED*/
void
gui_mch_add_menu(VimMenu * menu, VimMenu * parent, int idx)
{
    if (popup_menu(menu->name)) {
	menu->submenu_id = gtk_menu_new();
	return;
    }

    if (menubar_menu(menu->name) == 0 ||
	(parent != NULL && parent->submenu_id == 0)) {
	return;
    }

    if (parent == NULL)
	menu_item_new(menu, menu_bar, TRUE);
    else
	menu_item_new(menu, parent->submenu_id, TRUE);

    if (menu->id == NULL)
	return;			/* failed */

    if (parent == NULL)
	gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), menu->id);
    else
	gtk_menu_append(GTK_MENU(parent->submenu_id), menu->id);

    menu->submenu_id = gtk_menu_new();
#ifdef GTK_HAVE_FEATURES_1_1_0
    gtk_menu_set_accel_group(GTK_MENU(menu->submenu_id),
			     accel_group);
#endif
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu->id), menu->submenu_id);

#ifdef GTK_HAVE_FEATURES_1_1_0
    menu->tearoff_handle = tearoff_menu_item_new();
    gtk_menu_append(GTK_MENU(menu->submenu_id), menu->tearoff_handle);
#endif

#if 0
    XmNmnemonic, p_wak[0] == 'n' ? NUL : menu->mnemonic,
#endif

    if (menu->submenu_id == NULL)	/* failed */
	return;
    /*
     * The "Help" menu is a special case, and should be placed at the far right
     * hand side of the menu-bar.
     * FIXME: this isn't  proper for i18n.
     */
    if (parent == NULL && STRCMP((char *) menu->dname, "Help") == 0)
	gtk_menu_item_right_justify(GTK_MENU_ITEM(menu->id));
}

/*
 * Those are the pixmaps used for the default buttons.
 */
struct NameToPixmap {
    char *name;
    char **xpm;
};

static const struct NameToPixmap built_in_pixmaps[] =
{
    {"New", tb_new_xpm},
    {"Open", tb_open_xpm},
    {"Save", tb_save_xpm},
    {"Undo", tb_undo_xpm},
    {"Redo", tb_redo_xpm},
    {"Cut", tb_cut_xpm},
    {"Copy", tb_copy_xpm},
    {"Paste", tb_paste_xpm},
    {"Print", tb_print_xpm},
    {"Help", tb_help_xpm},
    {"Find", tb_find_xpm},
    {"SaveAll",	tb_save_all_xpm},
    {"SaveSesn", tb_save_session_xpm},
    {"NewSesn", tb_new_session_xpm},
    {"LoadSesn", tb_load_session_xpm},
    {"RunScript", tb_macro_xpm},
    {"Replace",	tb_replace_xpm},
    {"WinClose", tb_close_xpm},
    {"WinMax",	tb_maximize_xpm},
    {"WinMin", tb_minimize_xpm},
    {"WinSplit", tb_split_xpm},
    {"Shell", tb_shell_xpm},
    {"FindPrev", tb_find_prev_xpm},
    {"FindNext", tb_find_next_xpm},
    {"FindHelp", tb_find_help_xpm},
    {"Make", tb_make_xpm},
    {"TagJump", tb_jump_xpm},
    {"RunCtags", tb_ctags_xpm},
    {"Exit", tb_exit_xpm},
    { NULL, NULL} /* end tag */
};


/*
 * do ":h toolbar" for details on the order of things searched to
 * find the toolbar pixmap.
 */
static void
get_pixmap(char *menuname, GdkPixmap **pixmap, GdkBitmap **mask)
{
    int builtin_num;

    *pixmap = NULL;
    *mask = NULL;
    if (strncmp(menuname, "BuiltIn", 7) == 0) {
	if (isdigit((int)menuname[7]) && isdigit((int)menuname[8])) {
	    builtin_num = atoi(menuname + 7);
	    pixmap_create_by_num(builtin_num, pixmap, mask);
	} else {
	    pixmap_create_blank(pixmap, mask);
	}
    } else {
	pixmap_create_by_dir(menuname, pixmap, mask);
	if (*pixmap == NULL) {
	    pixmap_create_by_name(menuname, pixmap, mask);
	    if (*pixmap == NULL)
		pixmap_create_blank(pixmap, mask);
	}
    }
}


/*
 * creates a blank pixmap using tb_blank
 */
static void
pixmap_create_blank(GdkPixmap **pixmap, GdkBitmap **mask)
{
    *pixmap = gdk_pixmap_create_from_xpm_d(
				    main_window->window,
				    mask,
				    &main_window->style->bg[GTK_STATE_NORMAL],
				    tb_blank_xpm);
}


/*
 * creates a pixmap using one of the built-in pixmap names
 */
static void
pixmap_create_by_name(char *name, GdkPixmap **pixmap, GdkBitmap **mask)
{
    const struct NameToPixmap *tmp = built_in_pixmaps;

    for (tmp = built_in_pixmaps; tmp->name; tmp++) {
	if (!STRCMP(tmp->name, name))
	    break;
    }

    *pixmap = gdk_pixmap_create_from_xpm_d(
				main_window->window,
				mask,
				&main_window->style->bg[GTK_STATE_NORMAL],
				(tmp->xpm) ? tmp->xpm : tb_blank_xpm);
} /* pixmap_create_by_name */


/*
 * creates a pixmap by using a built-in number
 */
static void
pixmap_create_by_num(int pixmap_num, GdkPixmap **pixmap, GdkBitmap **mask)
{
    int num_pixmaps;

    num_pixmaps = (sizeof(built_in_pixmaps) / sizeof(built_in_pixmaps[0])) - 1;

    if (pixmap_num < 0 || pixmap_num >= num_pixmaps)
	*pixmap = gdk_pixmap_create_from_xpm_d(
				    main_window->window,
				    mask,
				    &main_window->style->bg[GTK_STATE_NORMAL],
				    tb_blank_xpm);
    else
	*pixmap = gdk_pixmap_create_from_xpm_d(
				    main_window->window,
				    mask,
				    &main_window->style->bg[GTK_STATE_NORMAL],
				    built_in_pixmaps[pixmap_num].xpm);
}


/*
 * creates a pixmap by using the pixmap name found in $VIM/bitmaps/
 *
 * MAYBE: if the custom pixmap found in the directory is "big", should we scale
 * the bitmap like the help documentation says so or leave it alone?  i think
 * it's user error if there is a bitmap of the wrong size.
 */
static void
pixmap_create_by_dir(char *name, GdkPixmap **pixmap, GdkBitmap **mask)
{
    char_u *full_pathname;

    if ((full_pathname = (char_u *)alloc(MAXPATHL+1)) == NULL) {
	pixmap_create_blank(pixmap, mask);
    } else {
	expand_env((char_u *)"$VIM/bitmaps/",
		   full_pathname,
		   (MAXPATHL + 1) - strlen(name) - 5);
	STRCAT(full_pathname, name);
	STRCAT(full_pathname, ".xpm");
	if (access((const char *)full_pathname, F_OK) == 0)
	    *pixmap = gdk_pixmap_create_from_xpm(
				    main_window->window,
				    mask,
				    &main_window->style->bg[GTK_STATE_NORMAL],
				    (const char *)full_pathname);
	else
	    *pixmap = NULL;

	vim_free(full_pathname);
    }
}


/*ARGSUSED*/
void
gui_mch_add_menu_item(VimMenu * menu, VimMenu * parent, int idx)
{
    if (toolbar_menu(parent->name)) {
	if (is_menu_separator(menu->name)) {
	    gtk_toolbar_append_space(GTK_TOOLBAR(tool_bar));
	} else {
	    GdkPixmap *pixmap;
	    GdkBitmap *mask;

	    get_pixmap((char *)(menu->name), &pixmap, &mask);
	    if (pixmap == NULL)
		return; /* should at least have blank pixmap, but if not... */

	    menu->id = gtk_toolbar_append_item(
				    GTK_TOOLBAR(tool_bar),
				    (char *)(menu->dname),
				    (char *)(menu->strings[MENU_INDEX_TIP]),
				    (char *)(menu->dname),
				    gtk_pixmap_new(pixmap, mask),
				    GTK_SIGNAL_FUNC(menu_item_callback),
				    (gpointer) menu);
	}
	menu->submenu_id = NULL;
	return;
    } /* toolbar menu item */

    /* No parent, must be a non-menubar menu */
    if (parent->submenu_id == 0)
	return;

    if (is_menu_separator(menu->name)) {
	/* just add it */
	menu->id = gtk_menu_item_new();
	gtk_widget_show(menu->id);
	gtk_menu_append(GTK_MENU(parent->submenu_id), menu->id);

	return; /* that's all */
    }
    /* Menu mnemonics don't work currently in submenues, we just
       ignore them therefore currently */
#if 0
    label = XmStringCreate((char *) menu->dname, STRING_TAG);
    if (label == NULL)
	return;
    menu->submenu_id = 0;
    menu->id = XtVaCreateWidget("subMenu",
	    , parent->submenu_id,
	    XmNlabelString, label,
	    XmNmnemonic, menu->mnemonic,
	    XmNpositionIndex, idx,
	    NULL);
    XmStringFree(label);
#endif

    menu_item_new(menu, parent->submenu_id, FALSE);
    gtk_widget_show(menu->id);
    gtk_menu_append(GTK_MENU(parent->submenu_id), menu->id);

    if (menu->id != 0)
	gtk_signal_connect(GTK_OBJECT(menu->id), "activate",
		GTK_SIGNAL_FUNC(menu_item_callback), (gpointer) menu);
}


void
gui_mch_set_text_area_pos(int x, int y, int w, int h)
{
    gtk_form_move_resize(GTK_FORM(form_window), drawing_area, x, y, w, h);
}

/*
 * Nothing to be done here, since we have started to manage our windows with
 * abstract management widgets, which is much better in fact.
 */
/*ARGSUSED*/
void
gui_mch_set_menu_pos(int x, int y, int w, int h)
{
}

/*
 * Enable or disable mnemonics for the toplevel menus.
 */
/*ARGSUSED*/
void
gui_gtk_set_mnemonics(int enable)
{
#if 0
    VimMenu *menu;
    /*  FIXME: imeplemnt this later */
    for (menu = root_menu; menu != NULL; menu = menu->next)
	if (menu->id != 0)
	    XtVaSetValues(menu->id,
		    XmNmnemonic, enable ? menu->mnemonic : NUL,
		    NULL);
#endif
}

void
gui_mch_toggle_tearoffs(int enable)
{
    gui_mch_recurse_tearoffs(root_menu, enable);
}


/*
 * Destroy the machine specific menu widget.
 */
void
gui_mch_destroy_menu(VimMenu * menu)
{
    if (menu->submenu_id != 0) {
	gtk_widget_destroy(menu->submenu_id);
	menu->submenu_id = 0;
    }
    if (menu->id != 0) {
	/* parent = gtk_widget_get_parent(menu->id); */
	gtk_widget_destroy(menu->id);
	menu->id = 0;
    }
}


/*
 * Scrollbar stuff.
 */

void
gui_mch_set_scrollbar_thumb(GuiScrollbar * sb, int val, int size, int max)
{
    if (sb->id != 0) {
	GtkAdjustment *adjustment = GTK_RANGE(sb->id)->adjustment;
	adjustment->lower = 0;
	adjustment->value = val;
	adjustment->upper = max + 1;
	adjustment->page_size = size;
	adjustment->page_increment = (size > 2 ? size - 2 : 1);
	adjustment->step_increment = 1;
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_adjustment_changed(adjustment);
#endif
    }
}

void
gui_mch_set_scrollbar_pos(GuiScrollbar * sb, int x, int y, int w, int h)
{
    if (!sb->id)
	return;
    gtk_form_move_resize(GTK_FORM(form_window), sb->id, x, y, w, h);
}

/*
 * Take action upon scrollbar dragging.
 */
static void
adjustment_value_changed(GtkAdjustment * adjustment, gpointer data)
{
    GuiScrollbar *sb;
    long value;

    sb = gui_find_scrollbar((long) data);
    value = adjustment->value;

    /*
     * We just ignore the dragging argument, since otherwise
     * the scrollbar size will not be adjusted properly in
     * synthetic scrolls.
     */
    gui_drag_scrollbar(sb, value, FALSE);
    if (gtk_main_level() > 0)
	gtk_main_quit();
}

/* SBAR_VERT or SBAR_HORIZ */
void
gui_mch_create_scrollbar(GuiScrollbar * sb, int orient)
{
    if (orient == SBAR_HORIZ) {
	sb->id = gtk_hscrollbar_new(NULL);
	GTK_WIDGET_UNSET_FLAGS(sb->id, GTK_CAN_FOCUS);
	gtk_widget_show(sb->id);
	gtk_form_put(GTK_FORM(form_window), sb->id, 0, 0);
    }
    if (orient == SBAR_VERT) {
	sb->id = gtk_vscrollbar_new(NULL);
	GTK_WIDGET_UNSET_FLAGS(sb->id, GTK_CAN_FOCUS);
	gtk_widget_show(sb->id);
	gtk_form_put(GTK_FORM(form_window), sb->id, 0, 0);
    }
    if (sb->id != NULL) {
	GtkAdjustment *adjustment;

	adjustment = gtk_range_get_adjustment(
		GTK_RANGE((GtkScrollbar *)sb->id));
	gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		(GtkSignalFunc) adjustment_value_changed,
			   (gpointer) sb->ident);
    }
}

void
gui_mch_destroy_scrollbar(GuiScrollbar * sb)
{
    if (sb->id != 0) {
	gtk_widget_destroy(sb->id);
	sb->id = 0;
    }
}

#if defined(USE_BROWSE) || defined(PROTO)
/*
 * Imeplementation of the file selector related stuff
 */
static GtkWidget *open_file_dialog = NULL;
static char *browse_fname = NULL;

/*
 * Put up a file requester.
 * Returns the selected name in allocated memory, or NULL for Cancel.
 * saving,			select file to write
 * title			title for the window
 * dflt				default name
 * ext				not used (extension added)
 * initdir			initial directory, NULL for current dir
 * filter			not used (file name filter)
 */
/*ARGSUSED*/
char_u *
gui_mch_browse(int saving,
	       char_u * title,
	       char_u * dflt,
	       char_u * ext,
	       char_u * initdir,
	       char_u * filter)
{
    GtkFileSelection *fs;	/* shortcut */
    char_u dirbuf[MAXPATHL];

    open_file_dialog = gtk_file_selection_new((const gchar *)title);
    fs = GTK_FILE_SELECTION(open_file_dialog);
    gtk_container_border_width(GTK_CONTAINER(fs), 4);
    gtk_file_selection_hide_fileop_buttons(fs);
#ifdef GTK_HAVE_FEATURES_1_1_0
    gtk_window_set_modal(GTK_WINDOW(open_file_dialog), TRUE);
#endif
    gtk_window_position(GTK_WINDOW(open_file_dialog), GTK_WIN_POS_MOUSE);

    /*
     * Initial directory should be set someway here!
     */

    gtk_signal_connect(GTK_OBJECT(fs->ok_button),
		       "clicked", GTK_SIGNAL_FUNC(open_ok_callback),
		       NULL);
    gtk_signal_connect_object(GTK_OBJECT(open_file_dialog),
		       "destroy", GTK_SIGNAL_FUNC(open_destroy_callback),
			      GTK_OBJECT(open_file_dialog));
    gtk_signal_connect(GTK_OBJECT(fs->cancel_button),
		       "clicked", GTK_SIGNAL_FUNC(open_cancel_callback),
		       NULL);
    if (dflt == NULL)
	dflt = (char_u *) "";
    if (initdir == NULL || *initdir == NUL) {
	mch_dirname(dirbuf, MAXPATHL);
	strcat((char *)dirbuf, "/");	/* make sure this is a directory */
	initdir = dirbuf;
    }
    gtk_file_selection_set_filename(fs, (const gchar *)initdir);

    gtk_widget_show(open_file_dialog);
    while (open_file_dialog)
	gtk_main_iteration_do(TRUE);

    if (gtk_main_level() > 0)
	gtk_main_quit();
    if (browse_fname == NULL)
	return NULL;

    return vim_strsave((char_u *) browse_fname);
} /* gui_mch_browse */

/*ARGSUSED*/
static void
open_ok_callback(GtkWidget * widget, gpointer data)
{
    if (browse_fname != NULL) {
	g_free(browse_fname);
	browse_fname = NULL;
    }

    browse_fname = g_strdup(gtk_file_selection_get_filename(
					GTK_FILE_SELECTION(open_file_dialog)));

    gtk_widget_destroy(open_file_dialog);
    open_file_dialog = NULL;
}

/*ARGSUSED*/
static void
open_cancel_callback(GtkWidget * widget, gpointer data)
{
    if (browse_fname != NULL) {
	g_free(browse_fname);
	browse_fname = NULL;
    }
    gtk_widget_destroy(open_file_dialog);
    open_file_dialog = NULL;
}

/*ARGSUSED*/
static gboolean
open_destroy_callback(GtkWidget * widget)
{
    /* This is needed to get out of gtk_main */
    if (gtk_main_level() > 0)
	gtk_main_quit();

    return FALSE;
}

#endif	/* USE_BROWSE */

#ifdef GUI_DIALOG

static int dialogStatus;
static GtkWidget *dialogbb = NULL;

/* ARGSUSED */
int
gui_mch_dialog(int type,
	       char_u * title,
	       char_u * message,
	       char_u * buttons,
	       int dfltbutton)
{
    char_u *buts;
    char_u *p, *next;
    int butcount, *intp;

    GtkWidget *vbox = NULL;
    GtkWidget *table = NULL;
    GtkWidget *pixmap;
    GtkWidget *dialogmessage = NULL;
    GtkWidget *action_area = NULL;
    GtkWidget *separator = NULL;

    GdkPixmap *icon = NULL;
    GdkBitmap *mask = NULL;
    char **icon_data = NULL;

#define MAXBUT 10
    GtkWidget *dialogButton[MAXBUT];

    if (title == NULL)
	title = (char_u *) "Vim dialog";

    if ((type < 0) || (type > VIM_LAST_TYPE))
	type = VIM_GENERIC;

    /* if our pointer is currently hidden, then we should show it. */
    gui_mch_mousehide(FALSE);

    dialogbb = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(dialogbb), (const gchar *)title);
    gtk_window_position(GTK_WINDOW(dialogbb), GTK_WIN_POS_MOUSE);
    gtk_grab_add(dialogbb);
    gtk_widget_realize(dialogbb);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(dialogbb), vbox);
    gtk_widget_show(vbox);

    table = gtk_table_new(1, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 4);
    gtk_table_set_col_spacings(GTK_TABLE(table), 8);
    gtk_container_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(vbox), table, 4, 4, 0);
    gtk_widget_show(table);

    /* Add pixmap */
    switch (type) {
    case VIM_GENERIC:
	icon_data = generic_xpm;
	break;
    case VIM_ERROR:
	icon_data = error_xpm;
	break;
    case VIM_WARNING:
	icon_data = alert_xpm;
	break;
    case VIM_INFO:
	icon_data = info_xpm;
	break;
    case VIM_QUESTION:
	icon_data = quest_xpm;
	break;
    default:
	icon_data = generic_xpm;
    };
    icon = gdk_pixmap_create_from_xpm_d(main_window->window, &mask,
				     &dialogbb->style->white, icon_data);
    if (icon) {
	pixmap = gtk_pixmap_new(icon, mask);
	/* gtk_misc_set_alignment(GTK_MISC(pixmap), 0.5, 0.5); */
	gtk_table_attach_defaults(GTK_TABLE(table), pixmap, 0, 1, 0, 1);
	gtk_widget_show(pixmap);
    }
    /* Add label */
    dialogmessage = gtk_label_new((const gchar *)message);
    gtk_table_attach_defaults(GTK_TABLE(table), dialogmessage, 1, 2, 0, 1);
    gtk_widget_show(dialogmessage);

    /* for some reason this doesn't work properly under kwm */
    /* gdk_window_set_decorations(dialogbb->window, GDK_DECOR_BORDER);
     * gdk_window_set_functions(dialogbb->window, 0);
     */

    action_area = gtk_hbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(action_area), 4);
    gtk_box_pack_end(GTK_BOX(vbox), action_area, FALSE, TRUE, 0);
    gtk_widget_show(action_area);

    /* make a copy, so that we can insert NULs */
    buts = vim_strsave(buttons);
    if (buts == NULL)
	return -1;

    /*
     * Create the buttons.
     */
    p = buts;
    for (butcount = 0; butcount < MAXBUT; ++butcount) {
	for (next = p; *next; ++next) {
	    if (*next == DLG_HOTKEY_CHAR)
		mch_memmove(next, next + 1, STRLEN(next));
	    if (*next == DLG_BUTTON_SEP) {
		*next++ = NUL;
		break;
	    }
	}

	dialogButton[butcount] = gtk_button_new_with_label((const gchar *)p);
	GTK_WIDGET_SET_FLAGS(dialogButton[butcount], GTK_CAN_DEFAULT);
	gtk_widget_show(dialogButton[butcount]);
	intp = (int *)g_malloc(sizeof(int));
	*intp = butcount;
	gtk_signal_connect_object(GTK_OBJECT(dialogButton[butcount]), "clicked",
			  GTK_SIGNAL_FUNC(butproc), (gpointer)intp);

	if (*next == NUL) {
	    gtk_box_pack_end(GTK_BOX(action_area), dialogButton[butcount],
			     FALSE, FALSE, 0);
	    break;
	} else {
	    gtk_box_pack_start(GTK_BOX(action_area), dialogButton[butcount],
			       FALSE, FALSE, 0);
	}
	p = next;
    }
    ++butcount;
    vim_free(buts);
    if (dfltbutton < 1)
	dfltbutton = 1;
    if (dfltbutton > butcount)
	dfltbutton = butcount;
    /* XtVaSetValues(dialogbb, XmNdefaultButton, dialogButton[dfltbutton - 1], NULL); */

    gtk_widget_grab_focus(dialogButton[dfltbutton - 1]);
    gtk_widget_grab_default(dialogButton[dfltbutton - 1]);

    separator = gtk_hseparator_new();
    gtk_box_pack_end(GTK_BOX(vbox), separator, FALSE, TRUE, 0);
    gtk_widget_show(separator);

    dialogStatus = -1;
    gtk_widget_show(dialogbb);
    gtk_main();

    return dialogStatus;
}

/* ARGSUSED */
static void
butproc(GtkWidget * widget, gpointer data)
{
    int *butcount = (int *)data;
    dialogStatus = *butcount + 1;

    gtk_widget_destroy(dialogbb);
    if (gtk_main_level() > 0)
	gtk_main_quit();
}

#endif	/* GUI_DIALOG */

void
gui_mch_show_popupmenu(VimMenu * menu)
{
    gtk_menu_popup(GTK_MENU(menu->submenu_id),
		   NULL, NULL, NULL, NULL, 3, GDK_CURRENT_TIME);
}


/*
 * Don't create it twice!
 */

typedef struct _SharedFindReplace {
    GtkWidget *dialog;	/* the main dialog widget */
    GtkWidget *exact;	/* 'Exact match' check button */
    GtkWidget *up;	/* search direction 'Up' radio button */
    GtkWidget *down;    /* search direction 'Down' radio button */
    GtkWidget *what;	/* 'Find what' entry text widget */
    GtkWidget *with;	/* 'Replace with' entry text widget */
    GtkWidget *find;	/* 'Find Next' action button */
    GtkWidget *replace;	/* 'Replace With' action button */
    GtkWidget *all;	/* 'Replace All' action button */
} SharedFindReplace;

static SharedFindReplace find_widgets;
static SharedFindReplace repl_widgets;


/*ARGSUSED*/
void
gui_mch_find_dialog(char_u * arg)
{
    find_replace_dialog_create(FALSE);
}


/*ARGSUSED*/
void
gui_mch_replace_dialog(char_u * arg)
{
    find_replace_dialog_create(TRUE);
}


/*
 * Synchronize all gui elements, which are dependant upon the
 * main text font used. Those are in esp. the find/replace dialogs.
 * If You don't understand why this should be needed, please try to
 * search for "pi" in iso8859-2.
 */
void gui_gtk_synch_fonts(void)
{
    SharedFindReplace *frdp;
    int do_replace;

    /* OK this loop is a bit tricky... */
    for (do_replace = 0; do_replace <= 1; ++do_replace) {
	frdp = (do_replace) ? (&repl_widgets) : (&find_widgets);
	if (frdp->dialog) {
	    GtkStyle *style;

	    /* synch the font with whats used by the text itself */
	    style = gtk_style_copy(gtk_widget_get_style(frdp->what));
	    gdk_font_unref(style->font);
	    style->font = gui.norm_font;
	    gdk_font_ref(style->font);
	    gtk_widget_set_style(frdp->what, style);
	    gtk_style_unref(style);
	    if (do_replace) {
		style = gtk_style_copy(gtk_widget_get_style(frdp->with));
		gdk_font_unref(style->font);
		style->font = gui.norm_font;
		gdk_font_ref(style->font);
		gtk_widget_set_style(frdp->with, style);
		gtk_style_unref(style);
	    }
	}
    }
}

static void
find_replace_dialog_create(int do_replace)
{
    GtkWidget *hbox;		/* main top down box */
    GtkWidget *actionarea;
    GtkWidget *table;
    GtkWidget *tmp;
    GtkWidget *vbox;
    gboolean sensitive;
    SharedFindReplace *frdp;
    char *entry_text;

    frdp = (do_replace) ? (&repl_widgets) : (&find_widgets);

    if (frdp->dialog) {
	gui_gtk_synch_fonts();
	if (!GTK_WIDGET_VISIBLE(frdp->dialog)) {
	    gtk_widget_grab_focus(frdp->what);
	    gtk_widget_show(frdp->dialog);
	}

	gdk_window_raise(frdp->dialog->window);

	return;
    }

    frdp->dialog = gtk_window_new(GTK_WINDOW_DIALOG);
    if (do_replace) {
	gtk_window_set_wmclass(GTK_WINDOW(frdp->dialog), "searchrepl", "gvim");
	gtk_window_set_title(GTK_WINDOW(frdp->dialog), "VIM - Search and Replace...");
    } else {
	gtk_window_set_wmclass(GTK_WINDOW(frdp->dialog), "search", "gvim");
	gtk_window_set_title(GTK_WINDOW(frdp->dialog), "VIM - Search...");
    }

    gtk_window_position(GTK_WINDOW(frdp->dialog), GTK_WIN_POS_MOUSE);
    gtk_widget_realize(frdp->dialog);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frdp->dialog), hbox);

    if (do_replace)
	table = gtk_table_new(1024, 4, FALSE);
    else
	table = gtk_table_new(1024, 3, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);
    gtk_container_border_width(GTK_CONTAINER(table), 4);

    tmp = gtk_label_new("Find what:");
    gtk_misc_set_alignment(GTK_MISC(tmp), 1.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), tmp, 0, 1, 0, 1,
		     GTK_FILL, GTK_EXPAND, 2, 2);
    frdp->what = gtk_entry_new();
    entry_text = gtk_entry_get_text(GTK_ENTRY(frdp->what));
    sensitive = (strlen(entry_text) != 0);
    gtk_entry_set_text(GTK_ENTRY(frdp->what), entry_text);
    gtk_signal_connect(GTK_OBJECT(frdp->what), "changed",
		       GTK_SIGNAL_FUNC(entry_changed_cb), frdp->dialog);
    gtk_table_attach(GTK_TABLE(table), frdp->what, 1, 1024, 0, 1,
		     GTK_EXPAND | GTK_FILL, GTK_EXPAND, 2, 2);

    if (do_replace) {
	tmp = gtk_label_new("Replace with:");
	gtk_misc_set_alignment(GTK_MISC(tmp), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), tmp, 0, 1, 1, 2,
			 GTK_FILL, GTK_EXPAND, 2, 2);
	frdp->with = gtk_entry_new();
	gtk_signal_connect(GTK_OBJECT(frdp->with), "activate",
			   GTK_SIGNAL_FUNC(find_replace_cb),
			   (gpointer) FR_R_FINDNEXT);
	gtk_table_attach(GTK_TABLE(table), frdp->with, 1, 1024, 1, 2,
			 GTK_EXPAND | GTK_FILL, GTK_EXPAND, 2, 2);

	/*
	 * Make the entry activation only change the input focus onto the
	 * with item.
	 */
	gtk_signal_connect(GTK_OBJECT(frdp->what), "activate",
			   GTK_SIGNAL_FUNC(entry_activate_cb), frdp->with);
    } else {
	/*
	 * Make the entry activation do the search.
	 */
	gtk_signal_connect(GTK_OBJECT(frdp->what), "activate",
			   GTK_SIGNAL_FUNC(find_replace_cb),
			   (gpointer) FR_FINDNEXT);
    }

    /* exact match only button */
    frdp->exact = gtk_check_button_new_with_label("Match exact word only");
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(frdp->exact), FALSE);
    gtk_signal_connect(GTK_OBJECT(frdp->exact), "clicked",
		       GTK_SIGNAL_FUNC(exact_match_cb), NULL);
    if (do_replace)
	gtk_table_attach(GTK_TABLE(table), frdp->exact, 0, 1023, 3, 4,
			 GTK_FILL, GTK_EXPAND, 2, 2);
    else
	gtk_table_attach(GTK_TABLE(table), frdp->exact, 0, 1023, 2, 3,
			 GTK_FILL, GTK_EXPAND, 2, 2);

    tmp = gtk_frame_new("Direction");
    if (do_replace)
	gtk_table_attach(GTK_TABLE(table), tmp, 1023, 1024, 2, 4,
			 GTK_FILL, GTK_FILL, 2, 2);
    else
	gtk_table_attach(GTK_TABLE(table), tmp, 1023, 1024, 1, 3,
			 GTK_FILL, GTK_FILL, 2, 2);
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(vbox), 0);
    gtk_container_add(GTK_CONTAINER(tmp), vbox);

    /* 'up' and 'down' buttons */
    frdp->up = gtk_radio_button_new_with_label(NULL, "Up");
    gtk_box_pack_start(GTK_BOX(vbox), frdp->up, TRUE, TRUE, 0);
    frdp->down = gtk_radio_button_new_with_label(
			gtk_radio_button_group(GTK_RADIO_BUTTON(frdp->up)),
			"Down");
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(frdp->down), TRUE);
    if (do_replace)
	gtk_signal_connect(GTK_OBJECT(frdp->down), "clicked",
			   GTK_SIGNAL_FUNC(repl_dir_cb), NULL);
    else
	gtk_signal_connect(GTK_OBJECT(frdp->down), "clicked",
			   GTK_SIGNAL_FUNC(find_direction_cb), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), frdp->down, TRUE, TRUE, 0);

    /* vbox to hold the action buttons */
    actionarea = gtk_vbutton_box_new();
    gtk_container_border_width(GTK_CONTAINER(actionarea), 2);
    if (do_replace) {
	gtk_button_box_set_layout(GTK_BUTTON_BOX(actionarea),
				  GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(actionarea), 0);
    }
    gtk_box_pack_end(GTK_BOX(hbox), actionarea, FALSE, FALSE, 0);

    /* 'find next' button */
    frdp->find = gtk_button_new_with_label("Find Next");
    gtk_widget_set_sensitive(frdp->find, sensitive);
    if (do_replace)
	gtk_signal_connect(GTK_OBJECT(frdp->find), "clicked",
			   GTK_SIGNAL_FUNC(find_replace_cb),
			   (gpointer) FR_R_FINDNEXT);
    else
	gtk_signal_connect(GTK_OBJECT(frdp->find), "clicked",
			   GTK_SIGNAL_FUNC(find_replace_cb),
			   (gpointer) FR_FINDNEXT);
    GTK_WIDGET_SET_FLAGS(frdp->find, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(actionarea), frdp->find, FALSE, FALSE, 0);
    gtk_widget_grab_default(frdp->find);

    if (do_replace) {
	/* 'replace' button */
	frdp->replace = gtk_button_new_with_label("Replace");
	gtk_widget_set_sensitive(frdp->replace, sensitive);
	GTK_WIDGET_SET_FLAGS(frdp->replace, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(actionarea), frdp->replace, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(frdp->replace), "clicked",
			   GTK_SIGNAL_FUNC(find_replace_cb),
			   (gpointer) FR_REPLACE);

	/* 'replace all' button */
	frdp->all = gtk_button_new_with_label("Replace All");
	gtk_widget_set_sensitive(frdp->all, sensitive);
	GTK_WIDGET_SET_FLAGS(frdp->all, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(actionarea), frdp->all, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(frdp->all), "clicked",
			   GTK_SIGNAL_FUNC(find_replace_cb),
			   (gpointer) FR_REPLACEALL);
    }

    /* 'cancel' button */
    tmp = gtk_button_new_with_label("Cancel");
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_box_pack_end(GTK_BOX(actionarea), tmp, FALSE, FALSE, 0);
    gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		       GTK_SIGNAL_FUNC(find_replace_cb),
		       (gpointer) FR_DIALOGTERM);
    gtk_signal_connect(GTK_OBJECT(tmp), "delete_event",
		       GTK_SIGNAL_FUNC(find_replace_cb),
		       (gpointer) FR_DIALOGTERM);
    gtk_signal_connect_object(GTK_OBJECT(tmp),
			      "clicked", GTK_SIGNAL_FUNC(gtk_widget_hide),
			      GTK_OBJECT(frdp->dialog));
    gtk_signal_connect_object(GTK_OBJECT(frdp->dialog),
			      "delete_event", GTK_SIGNAL_FUNC(gtk_widget_hide),
			      GTK_OBJECT(frdp->dialog));

    tmp = gtk_vseparator_new();
    gtk_box_pack_end(GTK_BOX(hbox), tmp, FALSE, TRUE, 0);
    gtk_widget_grab_focus(frdp->what);

    gui_gtk_synch_fonts();

    gtk_widget_show_all(frdp->dialog);
}

/*
 * Convenience function.
 * creates a button with a label, and packs it into the box specified by the
 * parameter 'parent'.
 */
GtkWidget *
gui_gtk_button_new_with_label(
    char *labelname,
    GtkSignalFunc cbfunc,
    gpointer cbdata,
    GtkWidget *parent,
    int connect_object,
    gboolean expand,
    gboolean fill
    )
{
    GtkWidget *tmp;

    tmp = gtk_button_new_with_label(labelname);
    if (connect_object)
	gtk_signal_connect_object(GTK_OBJECT(tmp), "clicked",
				  GTK_SIGNAL_FUNC(cbfunc), GTK_OBJECT(cbdata));
    else
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
			   GTK_SIGNAL_FUNC(cbfunc), cbdata);
    gtk_box_pack_end(GTK_BOX(parent), tmp, expand, fill, 0);
    return tmp;
}


/*** private function definitions ***/

static void
gui_mch_recurse_tearoffs(VimMenu * menu, int val)
{
#ifdef GTK_HAVE_FEATURES_1_1_0
    while (menu != NULL) {
	if (!popup_menu(menu->name)) {
	    if (menu->submenu_id != 0) {
		if (val)
		    gtk_widget_show(menu->tearoff_handle);
		else
		    gtk_widget_hide(menu->tearoff_handle);
	    }
	    gui_mch_recurse_tearoffs(menu->children, val);
	}
	menu = menu->next;
    }
#endif
}

/*ARGSUSED*/
static void
menu_item_callback(GtkWidget * widget, gpointer data)
{
    gui_menu_cb((VimMenu *) data);

    /* make sure the menu action is taken immediately */
    if (gtk_main_level() > 0)
	gtk_main_quit();
}

/*
 * Callback for actions of the find and replace dialogs
 */
/*ARGSUSED*/
static void
find_replace_cb(GtkWidget * widget, gint flags)
{
    gchar *entry_text;
    gchar *with_text;
    char_u cmd[600];		/* XXX kludge */
    gboolean direction_down = TRUE;
    gboolean exact_match = FALSE;

    /*
     * Get the strings which we will handle if needed.
     */
    if (flags == FR_FINDNEXT) {
	entry_text = gtk_entry_get_text(GTK_ENTRY(find_widgets.what));
	with_text = NULL;
	direction_down = GTK_TOGGLE_BUTTON(find_widgets.down)->active;
	exact_match = GTK_TOGGLE_BUTTON(find_widgets.exact)->active;
    } else if (flags == FR_R_FINDNEXT || flags == FR_REPLACE ||
	       flags == FR_REPLACEALL) {
	entry_text = gtk_entry_get_text(GTK_ENTRY(repl_widgets.what));
	with_text = gtk_entry_get_text(GTK_ENTRY(repl_widgets.with));
	direction_down = GTK_TOGGLE_BUTTON(repl_widgets.down)->active;
	exact_match = GTK_TOGGLE_BUTTON(repl_widgets.exact)->active;
    } else {
	entry_text = NULL;
	with_text = NULL;
    }

    if (State & INSERT)
	cmd[0] = Ctrl('O');
    else if ((State | NORMAL) == 0 && State != CONFIRM)
	cmd[0] = ESC;
    else
	cmd[0] = NUL;
    cmd[1] = NUL;

    /*
     * Synthesize the input corresponding to the different actions.
     */
    if (flags == FR_DIALOGTERM) {
	if (State == CONFIRM) {
	    add_to_input_buf((char_u *)"q", 1);
	    return;
	}
    }
    if (flags == FR_FINDNEXT || flags == FR_R_FINDNEXT) {
	if (State == CONFIRM) {
	    STRCAT(cmd, "n");
	} else {
	    if (direction_down)
		STRCAT(cmd, "/");
	    else
		STRCAT(cmd, "?");

	    if (exact_match)
		STRCAT(cmd, "\\<");
	    STRCAT(cmd, entry_text);
	    if (exact_match)
		STRCAT(cmd, "\\>");

	    STRCAT(cmd, "\r");
	}
    } else if (flags == FR_REPLACE) {
	if (State == CONFIRM) {
	    STRCAT(cmd, "y");
	} else {
	    STRCAT(cmd, ":%sno/");
	    STRCAT(cmd, entry_text);
	    STRCAT(cmd, "/");
	    STRCAT(cmd, with_text);
	    STRCAT(cmd, "/gc\r");
	}
	/*
	 * Give main window the focus back: this is to allow
	 * handling of the confirmation y/n/a/q stuff.
	 */
	/*(void)SetFocus(s_hwnd); */
    } else if (flags == FR_REPLACEALL) {
	if (State == CONFIRM) {
	    STRCAT(cmd, "a");
	} else {
	    STRCAT(cmd, ":%sno/");
	    STRCAT(cmd, entry_text);
	    STRCAT(cmd, "/");
	    STRCAT(cmd, with_text);
	    STRCAT(cmd, "/g\r");
	}
    }
    if (*cmd) {
	/*
	 * Is there a better way to throw multiline commands in ?
	 * (--mdcki)
	 */
	add_to_input_buf(cmd, STRLEN(cmd));
	if (gtk_main_level() > 0)
	    gtk_main_quit();	/* make sure if will be handled immediately */
    }
}

/* our usual callback function */
/*ARGSUSED*/
static void
entry_activate_cb(GtkWidget * widget, GtkWidget * with)
{
    gtk_widget_grab_focus(with);
}


/*
 * The following are used to synchronize the direction setting
 * between the search and the replace dialog.
 */
/*ARGSUSED*/
static void
find_direction_cb(GtkWidget * widget, gpointer data)
{
    gboolean direction_down = GTK_TOGGLE_BUTTON(widget)->active;

    if (repl_widgets.dialog) {
	GtkWidget *w;
	w = direction_down ? repl_widgets.down : repl_widgets.up;

	if (!GTK_TOGGLE_BUTTON(w)->active)
	    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(w), TRUE);
    }
}

/*ARGSUSED*/
static void
repl_dir_cb(GtkWidget *widget, gpointer data)
{
    gboolean direction_down = GTK_TOGGLE_BUTTON(widget)->active;

    if (find_widgets.dialog) {
	GtkWidget *w;
	w = direction_down ? find_widgets.down : find_widgets.up;

	if (!GTK_TOGGLE_BUTTON(w)->active)
	    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(w), TRUE);
    }
}

/*ARGSUSED*/
static void
exact_match_cb(GtkWidget * widget, gpointer data)
{
    gboolean exact_match = GTK_TOGGLE_BUTTON(widget)->active;

    if (find_widgets.dialog)
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(find_widgets.exact),
				    exact_match);
    if (repl_widgets.dialog)
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(repl_widgets.exact),
				    exact_match);
}

static void
entry_changed_cb(GtkWidget * entry, GtkWidget * dialog)
{
    gchar *entry_text;
    gboolean nonempty;

    entry_text = gtk_entry_get_text(GTK_ENTRY(entry));

    if (!entry_text)
	return;			/* shouldn't happen */

    nonempty = (strlen(entry_text) != 0);

    if (dialog == find_widgets.dialog) {
	gtk_widget_set_sensitive(find_widgets.find, nonempty);
	if (repl_widgets.dialog) {
	    gtk_widget_set_sensitive(repl_widgets.find, nonempty);
	    gtk_widget_set_sensitive(repl_widgets.replace, nonempty);
	    gtk_widget_set_sensitive(repl_widgets.all, nonempty);
	    if (strcmp(entry_text,
		       gtk_entry_get_text(GTK_ENTRY(repl_widgets.what))))
		gtk_entry_set_text(GTK_ENTRY(repl_widgets.what), entry_text);
	}
    }
    if (dialog == repl_widgets.dialog) {
	gtk_widget_set_sensitive(repl_widgets.find, nonempty);
	gtk_widget_set_sensitive(repl_widgets.replace, nonempty);
	gtk_widget_set_sensitive(repl_widgets.all, nonempty);
	if (find_widgets.dialog) {
	    gtk_widget_set_sensitive(find_widgets.find, nonempty);
	    if (strcmp(entry_text,
		       gtk_entry_get_text(GTK_ENTRY(find_widgets.what))))
		gtk_entry_set_text(GTK_ENTRY(find_widgets.what), entry_text);
	}
    }
}

/*
 * Routines which manipulate a dialog window which lists all the buffers seen
 * from the output of the ":buffers" command.
 */

#define BUFLIST_NUM_COLUMNS 4

static void buflist_close __ARGS((GtkWidget *wgt, gpointer cbdata));
static void buflist_destroy __ARGS((GtkWidget *wgt, gpointer cbdata));
static void buflist_def_cb __ARGS((GtkWidget *wgt, gpointer cbdata));
static void buflist_list_build __ARGS((void));
static void buflist_win_create __ARGS((void));
static void buflist_select __ARGS((GtkWidget *wgt, int row, int column,
				   GdkEventButton *event, gpointer cbdata));
static int  buflist_timeout_update(gpointer cbdata);

static GtkWidget *buflist_toplev = NULL;
static GtkWidget *buflist_data = NULL;
static int buflist_update_id = -1;

void
gtk_buffer_list(void)
{
    if (buflist_toplev) {

	/*
	 * We should be refreshing it's contents here!
	 */
	gdk_window_raise(buflist_toplev->window);
	return;
    }

    buflist_win_create();
    buflist_list_build();
    /*
     * we connect the signal here instead of inside buflist_list_build()
     * because buflist_list_build() is used in more than one place.  it is
     * only when initially creating the dialog that the signal needs to be
     * connected.
     */
    gtk_signal_connect(GTK_OBJECT(buflist_data), "select_row",
		       GTK_SIGNAL_FUNC(buflist_select), NULL);
}

/*
 * Build's a list of buffers using a GTK CList.  the CList is the global
 * variable 'buflist_data'.
 */
    static void
buflist_list_build()
{
    BUF	*buf;
    char *rownfo[BUFLIST_NUM_COLUMNS];
    char numstr[8];
    char flags[4];
    char linestr[16];
    int added = 0;

    gtk_clist_freeze(GTK_CLIST(buflist_data));
    for (buf = firstbuf; buf != NULL && !got_int; buf = buf->b_next)
    {
	sprintf(numstr, "%d", buf->b_fnum);
	rownfo[0] = numstr;

	sprintf(flags, "%c%c%c",
		buf == curbuf ? '%' :
			(curwin->w_alt_fnum == buf->b_fnum ? '#' : ' '),
		buf->b_ml.ml_mfp == NULL ? '-' :
			(buf->b_nwindows == 0 ? 'h' : ' '),
		buf_changed(buf) ? '+' : ' ');
	rownfo[1] = flags;

	sprintf(linestr, "%ld",
		buf == curbuf ? curwin->w_cursor.lnum :
				(long)buflist_findlnum(buf));
	rownfo[2] = linestr;

	if (buf->b_fname == NULL)
	    rownfo[3] = g_strdup("No File");
	else
	{
	    home_replace(buf, buf->b_fname, NameBuff, MAXPATHL, TRUE);
	    rownfo[3] = g_strdup((char *)NameBuff);
	}

	gtk_clist_append(GTK_CLIST(buflist_data), rownfo);
	g_free(rownfo[3]);

	if (buf == curbuf)
	    gtk_clist_select_row(GTK_CLIST(buflist_data), added, 0);

	added++;
    }

    gtk_clist_thaw(GTK_CLIST(buflist_data));

    if (buflist_update_id != -1)
	gtk_timeout_remove(buflist_update_id);

    buflist_update_id = (int)gtk_timeout_add(60000,
					    (GtkFunction)buflist_timeout_update,
					    NULL);
}


/*
 * Callback for the "Close" button.  simply destroys the top level widget and
 * sets it to NULL.
 */
/*ARGSUSED*/
    static void
buflist_close(GtkWidget *wgt, gpointer cbdata)
{
    if (!buflist_toplev)
	return;

    gtk_widget_destroy(buflist_toplev);
    buflist_toplev = NULL;
}


/*
 * Called only from do_bufdel() when the current buffer has been deleted or
 * unloaded.
 */
    void
buflist_delete_entry(int bufnum)
{
    int i;
    char *bnum;

    if (!buflist_data)
	return;

    for (i = 0; i < GTK_CLIST(buflist_data)->rows; i++)
    {
	(void)gtk_clist_get_text(GTK_CLIST(buflist_data), i, 0, &bnum);
	if (atoi(bnum) == bufnum)
	    break;
    }

    gtk_clist_freeze(GTK_CLIST(buflist_data));
    if (i == (GTK_CLIST(buflist_data)->rows - 1))
    {
	gtk_clist_remove(GTK_CLIST(buflist_data), i);
	gtk_clist_select_row(GTK_CLIST(buflist_data), i - 1, 0);
    }
    else
    {
	gtk_clist_remove(GTK_CLIST(buflist_data), i);
	gtk_clist_select_row(GTK_CLIST(buflist_data), i, 0);
    }

    gtk_clist_thaw(GTK_CLIST(buflist_data));
    if (gtk_main_level() > 0)
	gtk_main_quit();
}


/*
 * Updates the clist data by zapping everything, and then rebuiling the list.
 * This gets automatically done every 60 seconds.
 */
/*ARGSUSED*/
    static void
buflist_update(GtkWidget *wgt, gpointer cbdata)
{
    int currow;
    char *bnum;

    for (currow = 0; currow < GTK_CLIST(buflist_data)->rows; currow++)
    {
	(void)gtk_clist_get_text(GTK_CLIST(buflist_data), currow, 0, &bnum);
	if (atoi(bnum) == curbuf->b_fnum)
	    break;
    }

    gtk_clist_freeze(GTK_CLIST(buflist_data));
    while (GTK_CLIST(buflist_data)->rows > 0)
	gtk_clist_remove(GTK_CLIST(buflist_data), 0);

    buflist_list_build();
    gtk_clist_moveto(GTK_CLIST(buflist_data), currow, 0, 0.5, 0);
    gtk_clist_thaw(GTK_CLIST(buflist_data));
}


/*
 * The timeout callback invoked every 60 seconds to update the clist data.
 */
/*ARGSUSED*/
    static int
buflist_timeout_update(gpointer cbdata)
{
    if (buflist_toplev)
	buflist_update(NULL, NULL);
    return 1;
}


/*
 * 'destroy' signal callback
 */
/*ARGSUSED*/
    static void
buflist_destroy(GtkWidget *wgt, gpointer cbdata)
{
    if (buflist_update_id != -1)
	gtk_timeout_remove(buflist_update_id);

    buflist_toplev = NULL;
    if (gtk_main_level() > 0)
	gtk_main_quit();
}


/*
 * Default callback for buttons on the buflist dialog window.  Simply executes
 * an ex command that was passed as the callback data.
 */
/*ARGSUSED*/
    static void
buflist_def_cb(GtkWidget *wgt, gpointer cbdata)
{
    char_u *cmd = (char_u *)cbdata;

    if (cmd)
    {
	do_cmdline(cmd, NULL, NULL, DOCMD_NOWAIT);
	/* make sure we still know what we are handling */
	buflist_update(NULL, NULL);
    }
}


/*
 * callback invoked when selecting a row in the buflist.  goes to the buffer
 * selected by invoking the ex command :buf.
 */
/*ARGSUSED*/
    static void
buflist_select(GtkWidget *wgt, int row, int column, GdkEventButton *event,
	       gpointer cbdata)
{
    char *bufnum, cmd[16];

    if (!buflist_data)
	return;

    (void)gtk_clist_get_text(GTK_CLIST(buflist_data), row, 0, &bufnum);

    g_snprintf(cmd, 16, ":buf %s", bufnum);
    do_cmdline((char_u *)cmd, NULL, NULL, DOCMD_NOWAIT);
    gtk_main_iteration_do(FALSE);
    if (gtk_main_level() > 0)
	gtk_main_quit();
}


/*
 * creates the buflist dialog window
 */
    static void
buflist_win_create()
{
    GtkWidget *tmp, *vbox, *hbox;
    char *titles[] = { " # ", " Flags ", " Line # ", " File name " };

    buflist_toplev = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(buflist_toplev), "VIM - Buffers List...");
    gtk_signal_connect(GTK_OBJECT(buflist_toplev), "destroy",
		       GTK_SIGNAL_FUNC(buflist_destroy), buflist_toplev);

    vbox = gtk_vbox_new(FALSE, 4);
    gtk_container_add(GTK_CONTAINER(buflist_toplev), vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox), 5);
    gtk_widget_show(vbox);

    buflist_data = gtk_clist_new_with_titles(BUFLIST_NUM_COLUMNS, titles);
    gtk_clist_column_titles_passive(GTK_CLIST(buflist_data));
#ifdef GTK_HAVE_FEATURES_1_1_0
    gtk_clist_set_column_auto_resize(GTK_CLIST(buflist_data), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(buflist_data), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(buflist_data), 2, TRUE);
    gtk_clist_set_column_min_width(GTK_CLIST(buflist_data), 3, 64);
#else
    gtk_clist_set_column_width(GTK_CLIST(buflist_data), 0, 25);
    gtk_clist_set_column_width(GTK_CLIST(buflist_data), 1, 45);
    gtk_clist_set_column_width(GTK_CLIST(buflist_data), 2, 45);
    gtk_clist_set_column_width(GTK_CLIST(buflist_data), 3, 110);
#endif
    gtk_clist_set_column_justification(GTK_CLIST(buflist_data), 0,
				       GTK_JUSTIFY_RIGHT);
    gtk_clist_set_column_justification(GTK_CLIST(buflist_data), 1,
				       GTK_JUSTIFY_CENTER);
    gtk_clist_set_column_justification(GTK_CLIST(buflist_data), 2,
				       GTK_JUSTIFY_RIGHT);

    tmp = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(tmp),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(tmp), buflist_data);
    gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);

    hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_EDGE);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
    gtk_container_border_width(GTK_CONTAINER(hbox), 5);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    tmp = gui_gtk_button_new_with_label("Close",
					GTK_SIGNAL_FUNC(buflist_def_cb),
					":bdel", hbox, FALSE, FALSE, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    tmp = gui_gtk_button_new_with_label("Save",
					GTK_SIGNAL_FUNC(buflist_def_cb),
					":conf w", hbox, FALSE, FALSE, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    tmp = gui_gtk_button_new_with_label("Print",
					GTK_SIGNAL_FUNC(buflist_def_cb),
					":w !lpr", hbox, FALSE, FALSE, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    tmp = gui_gtk_button_new_with_label("Update",
					GTK_SIGNAL_FUNC(buflist_update),
					NULL, hbox, FALSE, FALSE, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    tmp = gui_gtk_button_new_with_label("Cancel",
					GTK_SIGNAL_FUNC(buflist_close),
					NULL, hbox, FALSE, FALSE, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(tmp);

    gtk_widget_set_usize(buflist_toplev, -1, 350);
    gtk_widget_grab_focus(buflist_data);
    gtk_widget_show_all(buflist_toplev);
}


static void helpfind_ok(GtkWidget *wgt, gpointer cbdata);
static void helpfind_entry_changed(GtkWidget *entry, gpointer cbdata);

static GtkWidget *helpfind = NULL;
static GtkWidget *help_entry = NULL;

    void
do_helpfind()
{
    GtkWidget *vbox, *hbox, *tmp;

    if (helpfind)
    {
	if (!GTK_WIDGET_VISIBLE(helpfind))
	{
	    gtk_widget_grab_focus(help_entry);
	    gtk_widget_show(helpfind);
	}

	gdk_window_raise(helpfind->window);
	return;
    }

    helpfind = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_position(GTK_WINDOW(helpfind), GTK_WIN_POS_MOUSE);
    gtk_window_set_title(GTK_WINDOW(helpfind), "VIM - Help on what?");
    gtk_signal_connect(GTK_OBJECT(helpfind), "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed), &helpfind);

    vbox = gtk_vbox_new(FALSE, 6);
    gtk_container_add(GTK_CONTAINER(helpfind), vbox);
    gtk_container_border_width(GTK_CONTAINER(helpfind), 6);

    hbox = gtk_hbox_new(FALSE, 6);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

    tmp = gtk_label_new("Topic: ");
    gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 0);

    help_entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox), help_entry, TRUE, TRUE, 0);

    tmp = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(vbox), tmp,  FALSE, TRUE, 0);

    hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_EDGE);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 0);
    gtk_container_border_width(GTK_CONTAINER(hbox), 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    tmp = gui_gtk_button_new_with_label("Ok",
					GTK_SIGNAL_FUNC(helpfind_ok),
					help_entry, hbox, FALSE, FALSE, FALSE);
    gtk_signal_connect(GTK_OBJECT(help_entry), "changed",
		       GTK_SIGNAL_FUNC(helpfind_entry_changed), tmp);
    gtk_signal_connect(GTK_OBJECT(help_entry), "activate",
		       GTK_SIGNAL_FUNC(helpfind_ok), help_entry);
    gtk_widget_set_sensitive(tmp, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);

    tmp = gui_gtk_button_new_with_label("Cancel",
					GTK_SIGNAL_FUNC(gtk_widget_destroy),
					helpfind, hbox, TRUE, FALSE, FALSE);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(tmp);

    gtk_widget_grab_focus(help_entry);
    gtk_widget_show_all(helpfind);
}


    static void
helpfind_entry_changed(GtkWidget *entry, gpointer cbdata)
{
    GtkWidget *ok = (GtkWidget *)cbdata;
    char *txt = gtk_entry_get_text(GTK_ENTRY(entry));

    if (txt == NULL)
	return;

    gtk_widget_set_sensitive(ok, strlen(txt));
}


/*ARGSUSED*/
    static void
helpfind_ok(GtkWidget *wgt, gpointer cbdata)
{
    char *txt, *cmd;
    GtkEntry *entry = GTK_ENTRY(cbdata);

    if ((txt = gtk_entry_get_text(entry)) == NULL)
	return;

    if ((cmd = (char *)alloc(strlen(txt) + 6)) == NULL)
	return;

    g_snprintf(cmd, strlen(txt) + 5, ":h %s\r", txt);
    do_cmdline((char_u *)cmd, NULL, NULL, DOCMD_NOWAIT);
    gtk_widget_destroy(helpfind);
    vim_free(cmd);
}
