/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * (C) 1998 by Marcin Dalecki <dalecki@cs.net.pl>
 *
 * Special form widgets which alwos not only to control randomly
 * the positioning of the widget, but which is handling the
 * childs sizes too.
 */
#include <gtk/gtk.h>
#include "gui_gtk_form.h"


static void gtk_form_class_init    (GtkFormClass    *klass);
static void gtk_form_init          (GtkForm         *form);
static void gtk_form_map           (GtkWidget        *widget);
static void gtk_form_realize       (GtkWidget        *widget);
static void gtk_form_size_request  (GtkWidget        *widget,
				     GtkRequisition   *requisition);
static void gtk_form_size_allocate (GtkWidget        *widget,
				     GtkAllocation    *allocation);
static void gtk_form_paint         (GtkWidget        *widget,
				     GdkRectangle     *area);
static void gtk_form_draw          (GtkWidget        *widget,
				     GdkRectangle     *area);
static gint gtk_form_expose        (GtkWidget        *widget,
				     GdkEventExpose   *event);
static void gtk_form_add           (GtkContainer     *container,
				     GtkWidget        *widget);
static void gtk_form_remove        (GtkContainer     *container,
				     GtkWidget        *widget);
#ifdef GTK_HAVE_FEATURES_1_1_0
static void gtk_form_forall        (GtkContainer     *container,
				     gboolean	       include_internals,
				     GtkCallback       callback,
				     gpointer          callback_data);
static GtkType gtk_form_child_type (GtkContainer     *container);
#endif


static GtkContainerClass *parent_class = NULL;


GtkType
gtk_form_get_type (void)
{
  static GtkType form_type = 0;

  if (!form_type)
    {
      GtkTypeInfo form_info =
      {
	"GtkForm",
	sizeof (GtkForm),
	sizeof (GtkFormClass),
	(GtkClassInitFunc) gtk_form_class_init,
	(GtkObjectInitFunc) gtk_form_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
#ifdef GTK_HAVE_FEATURES_1_1_0
        (GtkClassInitFunc) NULL,
#endif
      };

      form_type = gtk_type_unique (GTK_TYPE_CONTAINER, &form_info);
    }

  return form_type;
}

static void
gtk_form_class_init (GtkFormClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;

  parent_class = gtk_type_class (GTK_TYPE_CONTAINER);

  widget_class->map = gtk_form_map;
  widget_class->realize = gtk_form_realize;
  widget_class->size_request = gtk_form_size_request;
  widget_class->size_allocate = gtk_form_size_allocate;
  widget_class->draw = gtk_form_draw;
  widget_class->expose_event = gtk_form_expose;

  container_class->add = gtk_form_add;
  container_class->remove = gtk_form_remove;
#ifdef GTK_HAVE_FEATURES_1_1_0
  container_class->forall = gtk_form_forall;
  container_class->child_type = gtk_form_child_type;
#endif
}

#ifdef GTK_HAVE_FEATURES_1_1_0
static GtkType
gtk_form_child_type (GtkContainer     *container)
{
  return GTK_TYPE_WIDGET;
}
#endif

static void
gtk_form_init (GtkForm *form)
{
  GTK_WIDGET_UNSET_FLAGS (form, GTK_NO_WINDOW);

  form->children = NULL;
}

GtkWidget*
gtk_form_new (void)
{
  GtkForm *form;

  form = gtk_type_new (GTK_TYPE_FORM);
  return GTK_WIDGET (form);
}

void
gtk_form_put (GtkForm       *form,
               GtkWidget      *widget,
               gint16         x,
               gint16         y)
{
  GtkFormChild *child_info;

  g_return_if_fail (form != NULL);
  g_return_if_fail (GTK_IS_FORM (form));
  g_return_if_fail (widget != NULL);

  child_info = g_new (GtkFormChild, 1);
  child_info->widget = widget;
  child_info->x = x;
  child_info->y = y;

  gtk_widget_set_parent (widget, GTK_WIDGET (form));

  form->children = g_list_append (form->children, child_info);

  if (GTK_WIDGET_REALIZED (form) && !GTK_WIDGET_REALIZED (widget))
    gtk_widget_realize (widget);

  if (GTK_WIDGET_MAPPED (form) && !GTK_WIDGET_MAPPED (widget))
    gtk_widget_map (widget);

  if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_VISIBLE (form))
    gtk_widget_queue_resize (GTK_WIDGET (form));
}

void
gtk_form_move_resize (GtkForm       *form,
                      GtkWidget      *widget,
                      gint16         x,
                      gint16         y,
                      gint16	     w,
                      gint16	     h)
{
  GtkFormChild *child;
  GList *children;
  GtkRequisition *requisition = &GTK_WIDGET(form)->requisition;
  gboolean need_resize = FALSE;

  g_return_if_fail (form != NULL);
  g_return_if_fail (GTK_IS_FORM (form));
  g_return_if_fail (widget != NULL);

  children = form->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      if (child->widget == widget)
        {
          child->x = x;
          child->y = y;
          child->w = w;
          child->h = h;

	  widget->requisition.width = w;
	  widget->requisition.height = h;

	  if (child->x + child->w > requisition->width) {
	    requisition->width  = child->x + child->w;
	    need_resize = TRUE;
	  }
	  if (child->y + child->h > requisition->height) {
	    requisition->height = child->y + child->h;
	    need_resize = TRUE;
	  }

          if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_VISIBLE (form))
            gtk_widget_queue_resize (GTK_WIDGET (form));

          break;
        }
    }

  /*
   * Make sure we can accomodate all our childs.
   * This is the only reason we need a special widget for the handling
   * of this.
   * The second reason will become obvious when we start to try to resize the
   * subwindows independently by mouse drags of some kind of ruler, which still
   * needs to be introduced.
   */
  if (need_resize && GTK_WIDGET_VISIBLE(form)) {
    gtk_widget_queue_resize(GTK_WIDGET(form));
  }
}

static void
gtk_form_map (GtkWidget *widget)
{
  GtkForm *form;
  GtkFormChild *child;
  GList *children;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FORM (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);
  form = GTK_FORM (widget);

  gdk_window_show (widget->window);

  children = form->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      if (GTK_WIDGET_VISIBLE (child->widget) &&
	  !GTK_WIDGET_MAPPED (child->widget))
	gtk_widget_map (child->widget);
    }
}

static void
gtk_form_realize (GtkWidget *widget)
{
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FORM (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes,
				   attributes_mask);
  gdk_window_set_user_data (widget->window, widget);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}

static void
gtk_form_size_request (GtkWidget      *widget,
			GtkRequisition *requisition)
{
  GtkForm *form;
  GtkFormChild *child;
  GList *children;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FORM (widget));
  g_return_if_fail (requisition != NULL);

  form = GTK_FORM (widget);
  requisition->width = 0;
  requisition->height = 0;

  children = form->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      if (GTK_WIDGET_VISIBLE (child->widget))
	{
          gtk_widget_size_request (child->widget, &child->widget->requisition);

          requisition->height = MAX (requisition->height,
                                     child->y +
                                     child->widget->requisition.height);
          requisition->width = MAX (requisition->width,
                                    child->x +
                                    child->widget->requisition.width);
	}
    }

  requisition->height += GTK_CONTAINER (form)->border_width * 2;
  requisition->width += GTK_CONTAINER (form)->border_width * 2;

  /* printf("BANG %d %d\n", requisition->height, requisition->width); */
}

static void
gtk_form_size_allocate (GtkWidget     *widget,
			 GtkAllocation *allocation)
{
  GtkForm *form;
  GtkFormChild *child;
  GtkAllocation child_allocation;
  GList *children;
  guint16 border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FORM(widget));
  g_return_if_fail (allocation != NULL);

  form = GTK_FORM (widget);

  widget->allocation = *allocation;
  if (GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
			    allocation->x,
			    allocation->y,
			    allocation->width,
			    allocation->height);

  border_width = GTK_CONTAINER (form)->border_width;

  children = form->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      if (GTK_WIDGET_VISIBLE (child->widget))
	{
	  child_allocation.x = child->x + border_width;
	  child_allocation.y = child->y + border_width;
	  child_allocation.width = child->widget->requisition.width;
	  child_allocation.height = child->widget->requisition.height;
	  gtk_widget_size_allocate (child->widget, &child_allocation);
	}
    }
}

static void
gtk_form_paint (GtkWidget    *widget,
		 GdkRectangle *area)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FORM (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
      gdk_window_clear_area (widget->window,
			     area->x, area->y,
			     area->width, area->height);
}

static void
gtk_form_draw (GtkWidget    *widget,
		GdkRectangle *area)
{
  GtkForm *form;
  GtkFormChild *child;
  GdkRectangle child_area;
  GList *children;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FORM (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      form = GTK_FORM (widget);
      gtk_form_paint (widget, area);

      children = form->children;
      while (children)
	{
	  child = children->data;
	  children = children->next;

	  if (gtk_widget_intersect (child->widget, area, &child_area))
	    gtk_widget_draw (child->widget, &child_area);
	}
    }
}

static gint
gtk_form_expose (GtkWidget      *widget,
		  GdkEventExpose *event)
{
  GtkForm *form;
  GtkFormChild *child;
  GdkEventExpose child_event;
  GList *children;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_FORM (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      form = GTK_FORM (widget);

      child_event = *event;

      children = form->children;
      while (children)
	{
	  child = children->data;
	  children = children->next;

	  if (GTK_WIDGET_NO_WINDOW (child->widget) &&
	      gtk_widget_intersect (child->widget, &event->area,
				    &child_event.area))
	    gtk_widget_event (child->widget, (GdkEvent*) &child_event);
	}
    }

  return FALSE;
}

static void
gtk_form_add (GtkContainer *container,
	       GtkWidget    *widget)
{
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_FORM (container));
  g_return_if_fail (widget != NULL);

  gtk_form_put (GTK_FORM (container), widget, 0, 0);
}

static void
gtk_form_remove (GtkContainer *container,
		  GtkWidget    *widget)
{
  GtkForm *form;
  GtkFormChild *child;
  GList *children;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_FORM (container));
  g_return_if_fail (widget != NULL);

  form = GTK_FORM (container);

  children = form->children;
  while (children)
    {
      child = children->data;

      if (child->widget == widget)
	{
	  gboolean was_visible = GTK_WIDGET_VISIBLE (widget);

	  gtk_widget_unparent (widget);

	  form->children = g_list_remove_link (form->children, children);
	  g_list_free (children);
	  g_free (child);

	  if (was_visible && GTK_WIDGET_VISIBLE (container))
	    gtk_widget_queue_resize (GTK_WIDGET (container));

	  break;
	}

      children = children->next;
    }
}

#ifdef GTK_HAVE_FEATURES_1_1_0
static void
gtk_form_forall (GtkContainer *container,
		  gboolean	include_internals,
		  GtkCallback   callback,
		  gpointer      callback_data)
{
  GtkForm *form;
  GtkFormChild *child;
  GList *children;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_FORM (container));
  g_return_if_fail (callback != NULL);

  form = GTK_FORM (container);

  children = form->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      (* callback) (child->widget, callback_data);
    }
}
#endif
