/* vi:set ts=8 sts=4 sw=4:
 *
 * VIM - Vi IMproved	by Bram Moolenaar
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */

#include "vim.h"

/*
 * os_riscos.c
 *
 * Thomas Leonard <tal197@ecs.soton.ac.uk>
 */

#ifdef USE_GUI
int window_handle;
int task_handle = 0;		/* Zero means we are not yet a Wimp task */
int *redraw_block = NULL;	/* NULL means not in a redraw loop. */

int fg_colour = 0; 
int bg_colour = 0xffffff00;
#endif

int windowed;		/* Flag - are we running inside a text window? */
int WinLeft, WinTop;	/* We might be started inside a text window */
int ScrollTop;		/* Make cursor movements relative to ScrollTop. */

extern int r0;
extern int r1;
extern int r2;
extern int r3;
extern int r4;
extern int r5;
extern int r6;

int old_escape_state = -1;
int old_cursor_state = -1;

#define NORMAL_FG 0x00000000
#define NORMAL_BG 0xffffffff

#define rgb(r,g,b) ((b<<24) + (g<<16) + (r<<8))

/* Convert a DOS colour number to an RGB palette entry.
 * Mappings from X11 rgb/txt file.
 */
    int
map_colour(dos)
    int dos;		/* Standard DOS colour number. */
{
    switch (dos)
    {
	case 0: return 0;			/* Black */
	case 1: return rgb(0,0,139);		/* DarkBlue */
	case 2: return rgb(0,100,0);		/* DarkGreen */
	case 3: return rgb(0,139,139);		/* DarkCyan */
	case 4: return rgb(139,0,0);		/* DarkRed */
	case 5: return rgb(139,0,139);		/* DarkMagenta */
	case 6: return rgb(165,42,42);		/* Brown */
	case 7: return rgb(211,211,211);	/* LightGray, LightGrey, Gray, Grey */
	case 8: return rgb(169,169,169);	/* DarkGray, DarkGrey */
	case 9: return rgb(173,216,230);	/* Blue, LightBlue */
	case 10: return rgb(144,238,144);	/* Green, LightGreen */
	case 11: return rgb(224,255,255);	/* Cyan, LightCyan */
	case 12: return rgb(255,0,0);		/* Red, LightRed */
	case 13: return rgb(255,0,255);		/* Magenta, LightMagenta */
	case 14: return rgb(255,255,0);		/* Yellow */
	case 15: return rgb(255,255,255);	/* White */
    }
    return rgb(100,100,100);
}

    void
text_fg(fg)
    int fg;		/* Foregound colour in the form &BBGGRR00 */
{
    swi(XOS_Bit | ColourTrans_SetTextColour, fg, 0, 0, 0);
}

    void
text_bg(bg)
    int bg;		/* Backgound colour in the form &BBGGRR00 */
{
    swi(XOS_Bit | ColourTrans_SetTextColour, bg, 0, 0, 1<<7);
}

#define OUT_NORMAL 0
#define OUT_NUMBER 1

    void
mch_write(s, len)
    char_u  *s;
    int	    len;
{
    static int mode = OUT_NORMAL;
    static int x, y;			/* For reading numbers in. */

    while (len--)
    {
	char_u c = *s++;
	switch (mode)
	{
	    case OUT_NUMBER:
		if (c < '0' || c > '9')
		{
		    mode = OUT_NORMAL;
		}
		else
		{
		    x = (x * 10) + c - '0';
		    continue;
		}

	    /* note: no break here! */

	    case OUT_NORMAL:
		switch (c)
		{
		    case 1:
			/* Number (in decimal) follows. */
			mode = OUT_NUMBER;
			y = x;
			x = 0;
			break;
		    case 2:
			/* Position cursor. */
			swi(OS_WriteI + 31);
			swi(OS_WriteC, x);
			swi(OS_WriteC, y - ScrollTop);
			break;
		    case 3:
			/* Set scroll region. */
			if (x == Rows -1 && y == 0 && !windowed)
			{
			    /* Whole screen - remove text window.
			     * This is MUCH faster.
			     */
			    swi(OS_WriteI + 26);
			}
			else
			{
			    /* Create a text window. */
			    swi(OS_WriteI + 28);
			    swi(OS_WriteC, WinLeft);
			    swi(OS_WriteC, WinTop + x);
			    swi(OS_WriteC, WinLeft + Columns - 1);
			    swi(OS_WriteC, WinTop + y);
			}
			ScrollTop = y;
			break;
		    case 4:
			/* Normal mode. */
			text_fg(NORMAL_FG);
			text_bg(NORMAL_BG);
			break;
		    case 5:
			/* Reverse mode. */
			text_fg(NORMAL_BG);
			text_bg(NORMAL_FG);
			break;
		    case 10:
			swi(OS_NewLine);
			break;
		    case 14:
			/* Cursor invisible. */
			swi(OS_WriteN,
			     "\027\001\000\000\000\000\000\000\000\000",
			     10);
			break;
		    case 15:
			/* Cursor visible. */
			swi(OS_WriteN,
			     "\027\001\002\000\000\000\000\000\000\000",
			     10);
			break;
		    case 16:
			/* Cursor very visible (flash) */
			swi(OS_WriteN,
			     "\027\001\003\000\000\000\000\000\000\000",
			     10);
		    case 17:
			/* Set foreground colour. */
			text_fg(map_colour(x));
			break;
		    case 18:
			/* Set background colour. */
			text_bg(map_colour(x));
			break;
		    case 19:
			/* Scroll text down. */
			swi(OS_WriteN,
			     "\027\007\000\002\000\000\000\000\000\000",
			     10);
			break;
		    default:
			swi(OS_WriteC, c);
		}
		continue;

	    default:
		printf("[output error]");
		mode = OUT_NORMAL;
	}
    }
}

/*
 * mch_inchar(): low level input funcion.
 * Get a characters from the keyboard.
 * Return the number of characters that are available.
 * If wtime == 0 do not wait for characters.
 * If wtime == n wait n msecs for characters (not with GUI ?).
 * If wtime == -1 wait forever for characters.
 */
    int
mch_inchar(buf, maxlen, wtime)
    char_u  *buf;
    int	    maxlen;
    long    wtime;	    /* don't use "time", MIPS cannot handle it */
{
    int got=0;
    unsigned int start_time = clock();

    /*
     * OK, here's the plan:
     *
     * 1) Wait until wtime expires or we get a key
     * 2) Get keys until the keyboard buffer is empty or buf is full
     */

    while (swi(OS_Byte,145,0) & c_flag)
    {
    	/* Nothing at all in the keyboard buffer.
    	 * Has our time expired yet?
    	 */
	if ( (wtime != -1) && (clock() - start_time) >= wtime )
	    return 0;		/* Nothing read - giving up */
    }

    /* We've got one char (in r2) - are there any more? */

    while (got < maxlen)
    {
/* printf("{%o}",r2); */    
	buf[got++] = r2;

	if (swi(OS_Byte,145,0) & c_flag)
	    return got;		/* Keyboard buffer empty */
    }
    return got;			/* buf is full */
}

/*
 * return non-zero if a character is available
 */
    int
mch_char_avail()
{
    if (swi(OS_Byte, 152, 0) & c_flag)
	return 0;
    return 1;
}

/* Find out how much free memory we have.
 * I don't know how to work this out exactly but, since we can claim
 * more memory from the OS, let's just report the free pool size.
 * Dynamic area 6 doesn't exist pre 3.6 according to StrongHelp, so
 * we'll use Wimp_SlotSize. If that fails (outside the desktop?)
 * then just return a big number and hope.
 */
    long_u
mch_avail_mem(special)
    int special;
{
    if (swi(XOS_Bit | Wimp_SlotSize, -1, -1) & v_flag)
	return 0x7fffffff;
    return r2;
}

    void
mch_delay(msec, ignoreinput)
    long	msec;
    int		ignoreinput;
{
    clock_t	end_time;

    end_time=clock()+(msec*CLOCKS_PER_SEC)/1000;
    while (clock() < end_time)
	;
}

/*
 * If the machine has job control, use it to suspend the program,
 * otherwise fake it by starting a new shell.
 */
    void
mch_suspend()
{
    (void)mch_call_shell(NULL, SHELL_COOKED);
}

    void
mch_windinit()
{
    /*
     * Read window size first. Calls to mch_get_winsize() will
     * simply return these values in future so that setting the
     * text window (used for scrolling) won't give strange results.
     */

    int buf[7] = {132, 135, 256, 257, 1, 2, -1};
    /* Command windows are no longer forced open, since if we are
     * in the desktop then we'll use the GUI version.
     * Opening a command window here messes up the GUI version startup
     */
#ifndef USE_GUI
    swi(XOS_Bit | OS_WriteI);
#endif
    swi(OS_ReadVduVariables, buf, buf);
    WinLeft = buf[0];
    WinTop  = buf[1];
    Columns = buf[2];
    Rows    = buf[3] + 1;	/* Seems to be one off (VduVars wrong?) */
    ScrollTop = 0;

    /* Are we running in a textwindow? */
    if (Rows == buf[5] + 1 && Columns == buf[4] + 1)
	windowed = 0;
    else
	windowed = 1;

    /* Choose a nice colour scheme. */
    text_fg(NORMAL_FG);
    text_bg(NORMAL_BG);
}

/*
 * Check_win checks whether we have an interactive stdout.
 */
/* ARGSUSED */
    int
mch_check_win(argc, argv)
    int	    argc;
    char    **argv;
{
    return OK;
}

/*
 * Return TRUE if the input comes from a terminal, FALSE otherwise.
 */
    int
mch_input_isatty()
{
    if (swi(XOS_Bit | OS_ChangeRedirection, -1, -1) & v_flag)
	return TRUE;		/* Error - TRUE is probably correct though */
    if (r0 == 0)
	return TRUE;
    return FALSE;
}

    int
mch_can_restore_title()
{
    return FALSE;
}

    int
mch_can_restore_icon()
{
    return FALSE;
}

/*
 * Set the window title and icon.
 */
    void
mch_settitle(title, icon)
    char_u *title;
    char_u *icon;
{
    return;
}

/*
 * Restore the window/icon title.
 * "which" is one of:
 *  1  only restore title
 *  2  only restore icon
 *  3  restore title and icon
 */
    void
mch_restore_title(which)
    int which;
{
    return;
}

/*
 * Insert user name in s[len].
 * Return OK if a name found.
 */
    int
mch_get_user_name(s, len)
    char_u  *s;
    int	    len;
{
    /* RISC OS doesn't support user names. */
    return FAIL;
}

/*
 * Insert host name in s[len].
 */

    void
mch_get_host_name(s, len)
    char_u  *s;
    int	    len;
{
    char_u *name;
    int rlen;
    if (swi(XOS_Bit | OS_ReadVarVal, "Machine$Name", s, len, 0, 3) & v_flag)
    {
	/* Variable does not exist (normal operation) */
	STRNCPY(s, "(unknown)", len);
    }
}

/*
 * return process ID
 */
    long
mch_get_pid()
{
    if (swi(XOS_Bit | Wimp_ReadSysInfo, 5) & v_flag)
	return 0;
    return r0;
}

/*
 * Get name of current directory into buffer 'buf' of length 'len' bytes.
 * Return OK for success, FAIL for failure.
 */
    int
mch_dirname(buf, len)
    char_u  *buf;
    int	    len;
{
    if (swi(XOS_Bit | OS_FSControl, 37, "@", buf, 0, 0, len) & v_flag)
    	return FAIL;
    return OK;
}

/*
 * Get absolute file name into buffer 'buf' of length 'len' bytes.
 *
 * return FAIL for failure, OK for success
 */
    int
mch_FullName(fname, buf, len, force)
    char_u *fname, *buf;
    int len;
    int	force;		/* Also expand when already absolute path name.
    			 * Not used under RISC OS.
    			 */
{
    if (fname == NULL)
	return FAIL;    
    if (swi(XOS_Bit | OS_FSControl, 37, fname, buf, 0, 0, len) & v_flag)
    	return FAIL;
    return OK;
}

/*
 * return TRUE is fname is an absolute path name
 */
    int
mch_isFullName(fname)
    char_u	*fname;
{
    if (vim_strchr(fname, (char_u) '$') != NULL)
	return TRUE;
    return FALSE;
}

/*
 * Get file permissions for 'name'.
 * Returns -1 when it doesn't exist.
 */
    long
mch_getperm(name)
    char_u *name;
{
    struct stat statb;

    if (stat((char *)name, &statb))
	return -1;
    return statb.st_mode;
}

/*
 * set file permission for 'name' to 'perm'
 *
 * return FAIL for failure, OK otherwise
 */
    int
mch_setperm(name, perm)
    char_u  *name;
    long    perm;
{
    return (chmod((char *)name, (mode_t)perm) == 0 ? OK : FAIL);
}

/*
 * Set hidden flag for "name".
 */
/* ARGSUSED */
    void
mch_hide(name)
    char_u	*name;
{
    /* can't hide a file */
}

/*
 * return TRUE if "name" is a directory
 * return FALSE if "name" is not a directory
 * return FALSE for error
 */
    int
mch_isdir(name)
    char_u *name;
{
    if (swi(XOS_Bit | OS_File, 17, name) & v_flag)
	return FALSE;
    if (r0 == 2 || r0 == 3)
	return TRUE;		/* Count image files as directories. */
    return FALSE;
}

    void
mch_windexit(r)
    int r;
{
    settmode(TMODE_COOK);
    exiting = TRUE;
    out_flush();
    ml_close_all(TRUE);		/* remove all memfiles */
#ifdef USE_GUI
    if (gui.in_use)
	gui_exit(r);
#endif
    swi(XOS_Bit | OS_NewLine);
    if (old_escape_state != -1)
	swi(XOS_Bit | OS_Byte, 229, old_escape_state, 0);
    if (old_cursor_state != -1)
	swi(XOS_Bit | OS_Byte, 4, old_cursor_state);
    exit(r);
}

    void
mch_settmode(tmode)
    int		tmode;
{
    /* Block cursor. */
    swi(XOS_Bit | OS_WriteN,
	"\027\000\012\000\000\000\000\000\000\000",
	10);

    /* Disable the standard cursor keys actions. */
    swi(XOS_Bit | OS_Byte, 4, 1);
    if (old_cursor_state == -1)
	old_cursor_state = r1;

    /* Stop Escape from quitting Vim! */
    swi(XOS_Bit | OS_Byte, 229, 1, 0);
    if (old_escape_state == -1)
	old_escape_state = r1;

}

/*
 * set mouse clicks on or off (only works for xterms)
 */
    void
mch_setmouse(on)
    int	    on;
{
}

/*
 * set screen mode, always fails.
 */
/* ARGSUSED */
    int
mch_screenmode(arg)
    char_u   *arg;
{
    EMSG("Screen mode setting not supported");
    return FAIL;
}

/*
 * Try to get the current window size.
 * Return OK when size could be determined, FAIL otherwise.
 * Simply return results stored by mch_windinit().
 */
    int
mch_get_winsize()
{
    /* if size changed: screenalloc will allocate new screen buffers */
    return OK;
}

    void
mch_set_winsize()
{
/*    printf("mch_set_winsize \n"); */
}

    int
mch_call_shell(cmd, options)
    char_u  *cmd;
    int	    options;	/* SHELL_FILTER if called by do_filter() */
			/* SHELL_COOKED if term needs cooked mode */
			/* SHELL_EXPAND if called by mch_expand_wildcards() */
{
    int		retval;

    if (options & SHELL_COOKED)
	settmode(TMODE_COOK);		/* set to normal mode */
    if (cmd == NULL)
	cmd = (char_u *) "GOS";

#ifdef USE_GUI
    if (gui.in_use)
    {
        if (swi(Wimp_StartTask, cmd) & v_flag)
	    retval = -1;
        else
        {
	    /* [ block until finished ] */
	    /* [ read return value ] */
	    retval = 0;
        }
        return retval;
    }
#else
    MSG_PUTS("\n");

   /* I don't even want to think about what UnixLib must
    * be doing to allow this to work...
    */
    retval = system(cmd);
    if (retval)
	EMSG(strerror(EOPSYS));		/* Doesn't seem to set errno? */

    swi(OS_Byte, 229, 1, 0);		/* Re-disable escape */
#endif
    settmode(TMODE_RAW);		/* set to raw mode */
    return retval;
}

/*
 * Check for Escape being pressed right now.
 */
    void
mch_breakcheck()
{
    if (swi(OS_Byte, 121, 0xf0) & v_flag)
	return;
    if (r1 == 0xff)
    {
	got_int = TRUE;
	swi(OS_Byte, 15, 1);	/* Flush input buffer */
    }
}

/*
 * Recursively expand one path component into all matching files and/or
 * directories.
 * Return the number of matches found.
 */
    int
mch_expandpath(gap, path, flags)
    struct growarray	*gap;	/* Grow array for results. */
    char_u		*path;
    int			flags;	/* Add dirs/files/missing objects. */
{
    int			got;	/* Number of matches. */
    char_u		*pattern;

   /* Plan:
    *
    * 1) Get first part of path - no wildcards
    * 2) Get next path element (wildcarded)
    * 3) Get rest of path
    *
    * If (3) is nothing then only the leaf is wildcarded - add to gap
    * Otherwise call recursively for each path in (2), passing (3)
    *
    * This is just the header function.
    */

    /* We must be able to modifiy path, so make a copy */
    pattern = vim_strsave(path);
    if (pattern == NULL)
	return 0;
    got = expand_section(gap, (char_u *)"", pattern, flags);
    vim_free(pattern);
    return got;
}

/*
 * expand_section(gap, "$.Dir1.Dir2", "ABBA*.myleaf##")
 *
 * calls expand_section(gap, "$.Dir1.Dir2.ABBA_Gold", "myleaf##")
 *   and expand_section(gap, "$.Dir1.Dir2.ABBA_Live", "myleaf##")
 *
 * If rest is just a leaf then all matches are added to gap.
 *
 * Returns number of items added to gap.
 */
    int
expand_section(gap, root, rest, flags)
    struct growarray	*gap;
    char_u		*root;	/* Non-wildcarded path to search */
    char_u		*rest;	/* Wildcarded remainder of path */
    int			flags;	/* Add dirs/files/missing objects. */
{
    static char_u buf[MAXPATHL];	/* Temporary buffer. */
    char_u dir[MAXPATHL];
    int start_element = -1;		/* Start of wildcarded element */
    char_u c;
    int i;
    int got, dir_pos;
    int buflen;			/* Chars used in buf[] */
    int colon = 0;		/* Dir ends in ':' */

    buflen = strlen(root);
    STRNCPY(buf, root, buflen);	/* Copy root into buffer. */

   /*
    * Find end of nonwildcarded section.
    * Count ':' as a path sep since Vim:Bug* is a valid pathname.
    */

    for (i = 0; c = rest[i]; i++)
    {
	if (c == PATHSEP)
	{
	    start_element = i;
	    colon = 0;
	}
	if (c == ':')
	{
	    start_element = i + 1;
	    colon = 1;
	}
	if (c == '#' || c == '*')
	    break;
    }
    if (c == 0)
	start_element = i;

   /*
    * start_element +> terminator for non-wildcarded section.
    * Transfer this bit into buf.
    */
    if (buflen + start_element + 4 >= MAXPATHL)
       return 0;			/* Buffer full */
    if (start_element >= 0)
    {
	if (*root && !colon)
	    buf[buflen++] = PATHSEP;
	strncpy(buf + buflen, rest, start_element);
	buflen += start_element;
    }
    buf[buflen] = 0;

   /*
    * Did we reach the end of the string without hitting any wildcards?
    */
    if (c == 0)
    {
	/* Yes - add combined path to grow array and return. */
	addfile(gap, buf, flags);
	return 1;
    }

    if (start_element < 0 || !colon)
	start_element++;
    rest += start_element;

   /*
    * rest does contain wildcards if we get here.
    *
    * Now : have we reached the leaf names part yet?
    * If so, add all matches (files and dirs) to gap.
    * If not, get next path element and scan all matching directories.
    */

    start_element = -1;
    for (i = 0; rest[i]; i++)
    {
	if (rest[i] == '.')
	{
	    start_element = i;
	    rest[i] = 0;		/* Break string here. */
	    break;
	}
    }

    /* If start_element is -1 then we are matching leaf names */

    r3 = 0;			/* Number of objs read. */
    dir_pos = 0;		/* Position through directory. */
    got = 0;			/* Files added so far. */
    while (dir_pos != -1)
    {
	buf[buflen] = 0;
	if (swi(XOS_Bit | OS_GBPB, 9,
		buf,				/* Directory to scan. */
		buf + buflen + (1 - colon),	/* Buffer for result. */
		1,			/* Number of objects to read. */
		dir_pos,		/* Search position. */
		MAXPATHL - 2 - buflen,	/* Size of result buffer. */
		rest)			/* Wildcarded leafname. */
			& v_flag)
	{
	    EMSG(r0 + 4);
	    r4 = -1;
	}
	dir_pos = r4;		/* r4 corrupted by addfile() */
	if (r3 > 0)
	{
	    char_u *path = buf;
	    if (buflen == 0)
		path++;			/* Don't do '.File' */
	    else if (!colon)
		buf[buflen] = '.';		/* Join path and leaf */

	   /* Path -> full path of object found */
	    if (start_element == -1)
	    {
		addfile(gap, path, flags);
		got++;
	    }
	    else
	    {
	       /* Scan into subdirectories and images; ignore files */
		swi(OS_File, 17, path);
		if (r0 == 2 || r0 == 3)
		    got += expand_section(gap,
						path,
						rest + start_element + 1,
						flags);
	    }
	}
    }

    /* Restore the dot if we removed it. */
    if (start_element >= 0)
	rest[start_element] = '.';
    return got;
}

/*
 * mch_expand_wildcards() - this code does wild-card pattern matching using
 * the shell
 *
 * return OK for success, FAIL for error (you may lose some memory) and put
 * an error message in *file.
 *
 * num_pat is number of input patterns
 * pat is array of pointers to input patterns
 * num_file is pointer to number of matched file names
 * file is pointer to array of pointers to matched file names
 */
    int
mch_expand_wildcards(num_pat, pat, num_file, file, flags)
    int		    num_pat;
    char_u	  **pat;
    int		   *num_file;
    char_u	 ***file;
    int		    flags;
{
    /* This doesn't get called unless SPECIAL_WILDCHAR is defined. */
    return FAIL;
}

    int			/* TRUE if string contains wildcards. */
mch_has_wildcard(p)
    char_u  *p;		/* String to check. */
{
    if (vim_strpbrk((char_u *)"*#", p))
	return TRUE;
    return FALSE;
}

    int			/* see Unix unlink(2) */
mch_remove(file)
    char_u *file;	/* Name of file to delete. */
{
    if (swi(XOS_Bit | OS_FSControl, 27, file, 0, 0) & v_flag)
	return EXIT_FAILURE;
    return EXIT_SUCCESS;
}

/* Try to make existing scripts work without modification.
 *
 * Two main cases:
 * - Absolute : $VIM/syntax/help.vim
 * - Relative : Adfs::4.$.!Vim.Resources.Syntax/help.vim
 */
    void
mch_munge_fname(fname)
    char_u *fname;
{
    char_u c;
    int len;

    if (strncmp(fname, "$VIM/", 5) == 0)
    {
	strncpy(fname, "Vim:", 4);
	for (fname += 5; c = *fname; fname++)
	{
	    if (c == '.')
		break;
	    if (c == '/')
		fname[-1] = '.';
	    else
		fname[-1] = c;
	}
	fname[-1] = '\0';
    }
    else
    {
	len = strlen(fname);
	if (strcmp(fname + len - 4, ".vim") == 0)
	{
	    fname[len - 4] = '\0';
	    for (; c = *fname; fname++)
	    {
		if (c == '/')
		    *fname = '.';
	    }
	}
    }
}


/* GUI specific stuff. */

#ifdef USE_GUI
/*
 * Parse the GUI related command-line arguments.  Any arguments used are
 * deleted from argv, and *argc is decremented accordingly.  This is called
 * when vim is started, whether or not the GUI has been started.
 */
    void
gui_mch_prepare(int *argc, char **argv)
{
}

/* Fatal error on initialisation - report it and die. */
void ro_die(char_u *error)
{
    swi(Wimp_ReportError, error, 5, "GVim");
    exit(EXIT_FAILURE);
}

/*
 * Initialise the RISC OS GUI.
 * Create all the windows.
 * Returns OK for success, FAIL when the GUI can't be started.
 */
    int
gui_mch_init()
{
    int name[3];
    char *window;
    char *data;

    if (swi(XOS_Bit | Wimp_Initialise, 310, 0x4b534154, "GVim", 0) & v_flag)
	return FAIL;
    task_handle = r1;
    if (swi(XOS_Bit | Wimp_OpenTemplate, 0, "Vim:Templates") & v_flag)
	ro_die( (char *) r0);
    strcpy( (char *) name, "editor");
    if (swi(XOS_Bit | Wimp_LoadTemplate, 0, 0, 0, 0, -1, name, 0) & v_flag)
	ro_die( (char *) r0);
    window = malloc(r1);	/* Don't print text messages from alloc() */
    data = malloc(r2);
    if (window == NULL || data == NULL)
	ro_die("\0\0\0\0Out of memory - Can't load templates");
    swi(Wimp_LoadTemplate, 0,
				window,		/* Temp block */
				data,		/* Icon data */
				data + r2 + 1,	/* End of icon data */
				-1,		/* No fonts */
				name, 0);	/* First match */
    if (r6 == 0)
	ro_die("\0\0\0\0Can't find window in Templates file");
    if (swi(XOS_Bit | Wimp_CreateWindow, 0, window) & v_flag)
	ro_die( (char *) r0);
    window_handle = r0;
    vim_free(window);

    /* Set default foreground and background colours. */
    gui.norm_pixel = gui.def_norm_pixel;
    gui.back_pixel = gui.def_back_pixel;

    /* Get the colors from the "Normal" and "Menu" group (set in syntax.c or
     * in a vimrc file) */
    set_normal_colors();

    /*
     * Check that none of the colors are the same as the background color
     */
    gui_check_colors();

    /* [ set geometry ] */
    return OK;
}

/*
 * Called when the foreground or background color has been changed.
 */
    void
gui_mch_new_colors()
{
}

/*
 * Open the GUI window which was created by a call to gui_mch_init().
 */
    int
gui_mch_open(void)
{
    int block[8];
    block[0] = window_handle;
    swi(Wimp_GetWindowState, 0, block);
    block[7] = -1;
    swi(Wimp_OpenWindow, 0, block);
    gui.num_cols = (block[3] - block[1]) / gui.char_width;
    gui.num_rows = (block[4] - block[2]) / gui.char_height;
    swi(Wimp_SetCaretPosition, window_handle, -1, 0, 0, -1, -1);
    return OK;
}

    void
gui_mch_exit(int rc)
{
    exit(rc);
}

    void
gui_mch_set_winsize(width, height,
			min_width, min_height,
			base_width, base_height)
    int width;
    int height;
    int min_width;
    int min_height;
    int base_width;
    int base_height;
{
}

/*
 * Allow 10 pixels for horizontal borders, 30 for vertical borders.
 * Is there no way in X to find out how wide the borders really are?
 */
    void
gui_mch_get_screen_dimensions(int *screen_w, int *screen_h)
{
}

/*
 * Initialise vim to use the font with the given name.
 * Return FAIL if the font could not be loaded, OK otherwise.
 */
    int
gui_mch_init_font(char_u *font_name)
{
    return OK;
}

/*
 * Get a font structure for highlighting.
 */
    GuiFont
gui_mch_get_font(char_u *name, int giveErrorIfMissing)
{
}

/*
 * Set the current text font.
 */
    void
gui_mch_set_font(GuiFont font)
{
    gui.char_width = 16;
    gui.char_height = 32;
}

/*
 * Return TRUE if the two fonts given are equivalent.
 */
    int
gui_mch_same_font(GuiFont f1, GuiFont f2)
{
    return FALSE;
}

/*
 * If a font is not going to be used, free its structure.
 */
    void
gui_mch_free_font(GuiFont font)
{
}

/*
 * Return the Pixel value(color) for the given color name.
 * Return -1 for error.
 */
    GuiColor
gui_mch_get_color(char_u *name)
{
    int		i;
    struct colour
    {
    	char_u	*name;
    	int	value;
    } colours[] =
    {
	{ "Black",		rgb(0,		0,	0)	},
	{ "DarkBlue",		rgb(0,		0,	139)	},
	{ "DarkGreen",		rgb(0,		100,	0)	},
	{ "DarkCyan",		rgb(0,		139,	139)	},
	{ "DarkRed",		rgb(139,	0,	0)	},
	{ "DarkMagenta",	rgb(139,	0,	139)	},
	{ "Brown",		rgb(165,	42,	42)	},
	{ "LightGray",		rgb(211,	211,	211)	},
	{ "LightGrey",		rgb(211,	211,	211)	},
	{ "Gray",		rgb(211,	211,	211)	},
	{ "Grey",		rgb(211,	211,	211)	},
	{ "DarkGray",		rgb(169,	169,	169)	},
	{ "DarkGrey",		rgb(169,	169,	169)	},
	{ "Blue",		rgb(173,	216,	230)	},
	{ "LightBlue",		rgb(173,	216,	230)	},
	{ "Green",		rgb(144,	238,	144)	},
	{ "LightGreen",		rgb(144,	238,	144)	},
	{ "Cyan",		rgb(224,	255,	255)	},
	{ "LightCyan",		rgb(224,	255,	255)	},
	{ "Red",		rgb(255,	0,	0)	},
	{ "LightRed",		rgb(255,	0,	0)	},
	{ "Magenta",		rgb(255,	0,	255)	},
	{ "LightMagenta",	rgb(255,	0,	255)	},
	{ "Yellow",		rgb(255,	255,	0)	},
	{ "White",		rgb(255,	255,	255)	},
	{NULL, 0}
    };

    for (i = 0; ; i++)
    {
	if (colours[i].name == NULL)
	    return (GuiColor) -1;
	if (STRICMP(name, colours[i].name) == 0)
	    return colours[i].value;
    }
    printf("[ missing colour %s ]\n", name);
    return (GuiColor) -1;
}

/*
 * Set the current text foreground color.
 */
    void
gui_mch_set_fg_color(GuiColor colour)
{
    swi(ColourTrans_ReturnGCOL, colour);
    fg_colour = r0;
}

/*
 * Set the current text background color.
 */
    void
gui_mch_set_bg_color(GuiColor colour)
{
    swi(ColourTrans_ReturnGCOL, colour);
    bg_colour = r0;
}

    void
gui_mch_draw_string(int row, int col, char_u *s, int len, int flags)
{
    int x, y;		/* Workarea x,y */

    x = col * gui.char_width;
    y = -row * gui.char_height;

    if (redraw_block)
    {
	swi(ColourTrans_SetColour, fg_colour, 0, 0, 0, 0);
	swi(ColourTrans_SetColour, bg_colour, 0, 0, 1 << 7, 0);
	swi(OS_Plot, 4,			/* Move the drawing cursor */
		x + redraw_block[1],
		y + redraw_block[4] - redraw_block[6] - 1);
	swi(OS_WriteN, s, len);
    }
    else
    {
        int block[44];
        block[0] = window_handle;
        block[1] = x;
        block[2] = y - gui.char_height;
        block[3] = (col + len) * gui.char_width;
        block[4] = y;
        swi(Wimp_UpdateWindow, 0, block);
        while (r0)
        {
	    swi(ColourTrans_SetColour, fg_colour, 0, 0, 0, 0);
	    swi(ColourTrans_SetColour, bg_colour, 0, 0, 1 << 7, 0);
	    swi(OS_WriteI + 16);
	    swi(OS_Plot, 4,
	    		x + block[1],
	    		y + block[4] - block[6] - 1);
	    swi(OS_WriteN, s, len);
	    swi(Wimp_GetRectangle, 0, block);
        }
    }
}

/*
 * Return OK if the key with the termcap name "name" is supported.
 */
    int
gui_mch_haskey(char_u *name)
{
    return FAIL;
}

    void
gui_mch_beep(void)
{
    swi(XOS_Bit | OS_WriteI + 7);
}

/*
 * Visual bell.
 */
    void
gui_mch_flash(void)
{
}

/*
 * Invert a rectangle from row r, column c, for nr rows and nc columns.
 */
    void
gui_mch_invert_rectangle(int r, int c, int nr, int nc)
{
}

/*
 * Iconify the GUI window.
 */
    void
gui_mch_iconify(void)
{
}

/*
 * Draw a cursor without focus.
 */
    void
gui_mch_draw_hollow_cursor(GuiColor color)
{
}

/*
 * Draw part of a cursor, "w" pixels wide, and "h" pixels high, using
 * color "color".
 */
    void
gui_mch_draw_part_cursor(int w, int h, GuiColor color)
{
}

/*
 * Catch up with any queued events.  This may put keyboard input into the
 * input buffer, call resize call-backs, trigger timers etc.  If there is
 * nothing in the event queue(& no timers pending), then we return
 * immediately.
 */
    void
gui_mch_update(void)
{
    int block[64];
    swi(Wimp_Poll, 0, block);
    if (r0)
    	process_event(r0, block);
}

    void
redraw_window(block)
    int *block;
{
    int x, y;			/* Vim workarea coords */

    swi(Wimp_RedrawWindow, 0, block);
    redraw_block = block;
    while (r0)
    {
    	swi(OS_WriteI + 16);
    	x = block[7] - block[1];
    	y = block[4] - block[10] - block[6];
    	gui_redraw(x , y, block[9] - block[7], block[10] - block[8]);
    	swi(Wimp_GetRectangle, 0, block);
    }
    redraw_block = NULL;
}

    void
process_event(event, block)
    int event;
    int *block;
{
    switch (event)
    {
	case 1:
	    redraw_window(block);
	    break;
	case 2:
	    swi(Wimp_OpenWindow, 0, block);
	    break;
	case 3:
	    gui_exit(1);
	    break;
	case 6:		/* Mouse click. */
	    swi(Wimp_SetCaretPosition, window_handle, -1, 0, 0, -1, -1);
	    break;
	case 8:
	    add_to_input_buf(&block[6], 1);
	    break;
	case 17:
	case 18:
	    if (block[4] == 0)
	        exit(1);
	    break;
    }
}

/*
 * GUI input routine called by gui_wait_for_chars().  Waits for a character
 * from the keyboard.
 *  wtime == -1	    Wait forever.
 *  wtime == 0	    This should never happen.
 *  wtime > 0	    Wait wtime milliseconds for a character.
 * Returns OK if a character was found to be available within the given time,
 * or FAIL otherwise.
 */
    int
gui_mch_wait_for_chars(long wtime)
{
    int block[64];
    if (wtime == -1)
    	swi(Wimp_Poll, 1, block);
    else
    {
	swi(OS_ReadMonotonicTime);
	swi(Wimp_PollIdle, 0, block, r0 + (wtime / 10));
    }
    if (r0)
	process_event(r0, block);
    return vim_is_input_buf_empty() ? FAIL : OK;
}

/* Flush any output to the screen */
    void
gui_mch_flush(void)
{
}

/*
 * Clear a rectangular region of the screen from text pos(row1, col1) to
 *(row2, col2) inclusive.
 */
    void
gui_mch_clear_block(int row1, int col1, int row2, int col2)
{
    if (redraw_block)
    {
	swi(OS_Plot, 4, TEXT_X(col1), -TEXT_Y(row2 + 1));
	swi(OS_Plot, 96 + 7, TEXT_X(col2 + 1), -TEXT_Y(row1));
    }
    else
    {
	int block[44];
	block[0] = window_handle;
	block[1] = TEXT_X(col1);
	block[2] = -TEXT_Y(row2 + 1);
	block[3] = TEXT_X(col2 + 1);
	block[4] = -TEXT_Y(row1);
	swi(Wimp_UpdateWindow, 0, block);
	while (r0)
	{
	    swi(OS_WriteI + 16);
	    swi(Wimp_GetRectangle, 0, block);
	}
    }
}

    void
gui_mch_clear_all(void)
{
    if (redraw_block)
    {
	swi(OS_WriteI + 16);
    }
    else
    {
	int block[44];
	block[0] = window_handle;
	block[1] = 0;
	block[2] = -gui.num_rows * gui.char_height;
	block[3] = gui.num_cols * gui.char_width;
	block[4] = 0;
	swi(Wimp_UpdateWindow, 0, block);
	while (r0)
	{
	    swi(OS_WriteI + 16);
	    swi(Wimp_GetRectangle, 0, block);
        }
    }
}

/*
 * Delete the given number of lines from the given row, scrolling up any
 * text further down within the scroll region.
 */
    void
gui_mch_delete_lines(int row, int num_lines)
{
    int top_from = -row - num_lines;
    int bot_from = -gui.scroll_region_bot - 1;
    int bot_to   = bot_from + num_lines;

    swi(Wimp_BlockCopy, window_handle,
    			0,
    			bot_from * gui.char_height,
    			gui.num_cols * gui.char_width,
    			top_from * gui.char_height,

    			0,
    			bot_to * gui.char_height);
    gui_mch_clear_block(gui.scroll_region_bot - num_lines + 1,
    			0,
    			gui.scroll_region_bot,
    			gui.num_cols - 1);
}

/*
 * Insert the given number of lines before the given row, scrolling down any
 * following text within the scroll region.
 */
    void
gui_mch_insert_lines(int row, int num_lines)
{
    int top_from = -row;
    int bot_to   = -gui.scroll_region_bot - 1;
    int bot_from = bot_to + num_lines;

    swi(Wimp_BlockCopy, window_handle,
    			0,
    			bot_from * gui.char_height,
    			gui.num_cols * gui.char_width,
    			top_from * gui.char_height,

    			0,
    			bot_to * gui.char_height);
    gui_mch_clear_block(row,
    			0,
    			row + num_lines - 1,
    			gui.num_cols - 1);
}

    void
clip_mch_request_selection(void)
{
}

    void
clip_mch_lose_selection(void)
{
}

    int
clip_mch_own_selection(void)
{
    return FALSE;
}

/*
 * Send the current selection to the clipboard.  Do nothing for X because we
 * will fill in the selection only when requested by another app.
 */
    void
clip_mch_set_selection(void)
{
}

/*
 * Make a menu either grey or not grey.
 */
    void
gui_mch_menu_grey(GuiMenu *menu, int grey)
{
}

/*
 * Make menu item hidden or not hidden
 */
    void
gui_mch_menu_hidden(GuiMenu *menu, int hidden)
{
}

/*
 * This is called after setting all the menus to grey/hidden or not.
 */
    void
gui_mch_draw_menubar(void)
{
}

    void
gui_mch_enable_scrollbar(GuiScrollbar *sb, int flag)
{
}

    void
gui_mch_set_blinking(long waittime, long on, long off)
{
}

/*
 * Stop the cursor blinking.  Show the cursor if it wasn't shown.
 */
    void
gui_mch_stop_blink(void)
{
}

/*
 * Start the cursor blinking.  If it was already blinking, this restarts the
 * waiting time and shows the cursor.
 */
    void
gui_mch_start_blink(void)
{
}

/*
 * Return the lightness of a pixel.  White is 255.
 * Uses CIE luminance weights as given in the ChangeFSI help file.
 */
    int
gui_mch_get_lightness(GuiColor pixel)
{
    int red   = (pixel >> 24) & 0xff;
    int green = (pixel >> 16) & 0xff;
    int blue  = (pixel >> 8 ) & 0xff;
    return ((red * 299) + (green * 587) + (blue * 114)) / 1000;
}

/*
 * Return the RGB value of a pixel as "#RRGGBB".
 */
    char_u *
gui_mch_get_rgb(GuiColor pixel)
{
    static char_u retval[10];
    sprintf((char *)retval, "#%02x%02x%02x",
	    (unsigned) ((pixel >> 24) & 0xff),
	    (unsigned) ((pixel >> 16) & 0xff),
	    (unsigned) ((pixel >> 8)  & 0xff));
    return retval;
}

    void
gui_mch_set_text_area_pos(int x, int y, int w, int h)
{
}

    void
gui_mch_enable_menu(int flag)
{
}

    void
gui_mch_set_menu_pos(int x, int y, int w, int h)
{
}

    void
gui_mch_add_menu(GuiMenu *menu, GuiMenu *parent)
{
}

    void
gui_mch_add_menu_item(GuiMenu *menu, GuiMenu *parent)
{
}

    void
gui_mch_new_menu_colors(void)
{
}

    void
gui_mch_destroy_menu(GuiMenu *menu)
{
}

    void
gui_mch_set_scrollbar_thumb(GuiScrollbar *sb, int val, int size, int max)
{
}

    void
gui_mch_set_scrollbar_pos(GuiScrollbar *sb, int x, int y, int w, int h)
{
}

    void
gui_mch_create_scrollbar(GuiScrollbar *sb, int orient)
{
}

    void
gui_mch_destroy_scrollbar(GuiScrollbar *sb)
{
}

    void
gui_mch_set_scrollbar_colors(GuiScrollbar *sb)
{
}

/*
 * Get current y mouse coordinate in text window.
 * Return -1 when unknown.
 */
    int
gui_mch_get_mouse_y()
{
    return -1;
}

/* MouseTo(x, y) */
    void
gui_mch_setmouse(x, y)
    int		x;
    int		y;
{
}

    void
gui_mch_show_popupmenu(GuiMenu *menu)
{
    /* TODO */
}

    void
gui_mch_toggle_tearoffs(enable)
    int		enable;
{
    /* no tearoff menus */
}
#endif
