/*
 * SniffVim.c Interface between Vim and SNiFF+
 *	      derived from Sniff_Interface.c for elvis written by obi.
 *
 * $Id: SniffVim.c,v 1.1.1.11 1997/11/20 09:23:58 toni Exp $
 */

#include "vim.h"
#include "os_unixx.h"

int fd_from_sniff;
int sniff_connected;
int sniff_request_waiting=0;
int want_sniff_request=0;

typedef struct cmd_tab CMD_TAB;
struct cmd_tab
{
    char *longname;
    char cmd_char;
};

static CMD_TAB sniff_cmds[] =
{
    { "toggle",       'e' },
    { "superclass",   's' },
    { "overridden",   'm' },
    { "retrieve",     'r' },
    { "retrieve-next",'R' },
    { "goto-symbol",  'g' },
    { "find-symbol",  'f' },
    { "browse-class", 'w' },
    { "hierarchy",    't' },
    { "restr-hier",   'T' },
    { "xref-to",      'x' },
    { "xref-by",      'X' },
    { "xref-has",     'c' },
    { "xref-used-by", 'C' },
    { "show-docu",    'd' },
    { "gen-docu",     'D' },
    { "connect",      'y' },
    { "disconnect",   'q' },
    { "font-info",    'z' },
    { "update",       'u' },
    { NULL,	      '\0'}
};

static char *SniffEmacs[2] = {"sniffemacs", (char *)NULL};
static int fd_to_sniff;
static int sniff_will_disconnect=0;


/* Function Prototypes */
static void vi_msg(char *);
static void vi_error_msg(char *);
static char *vi_symbol_under_cursor();
static long vi_symbol_location(char *);
static void vi_open_file(char *);
static char *vi_buffer_name();
static BUF  *vi_find_buffer(char *);
static void vi_exec_cmd(char *);
static void vi_set_writability(BUF *, int);
static void vi_set_cursor_pos(long char_nr);
static long vi_cursor_pos();


/* Connect to Sniff: returns 1 on error */
int ConnectToSniffEmacs()
{
    int pid;
    int ToSniffEmacs[2], FromSniffEmacs[2];

    pipe(ToSniffEmacs);
    pipe(FromSniffEmacs);

    /* fork */
    if ( (pid = fork()) == 0)
    {
	/* child */

	/* prepare communication pipes */
	close(ToSniffEmacs[1]);
	close(FromSniffEmacs[0]);

	dup2(ToSniffEmacs[0],fileno(stdin));	/* we write to ToSniffEmacs[1] */
	dup2(FromSniffEmacs[1],fileno(stdout)); /* and read from FromSniffEmacs[0] */

	close(ToSniffEmacs[0]);
	close(FromSniffEmacs[1]);

	/* start sniffemacs */
	execvp (SniffEmacs[0], SniffEmacs);

	vi_error_msg("Error connecting to SNiFF+");

	exit(1);
	return 1;  /* never reached */
    }
    else if (pid>0)
    {
	/* parent process */
	close(ToSniffEmacs[0]);
	fd_to_sniff  = ToSniffEmacs[1];
	close(FromSniffEmacs[1]);
	fd_from_sniff = FromSniffEmacs[0];
	sniff_connected = 1;
	return 0;
    }
    else   /* error in fork() */
    {
	vi_error_msg("Error during fork");
	return 1;
    }
}



void HandleSniffRequest( char* buffer )
{
    char VICommand[256];
    char command;
    char *arguments, *curr_file;
    char file[256], new_path[256];
    int  position, writable, tab_width;

    static  char* SetTab     = "set tabstop=%d";
    static  char* SelectFile = "buf %s";
    static  char* DeleteBuf  = "bd %s";
    static  char* GotoLine   = "%d";

    curr_file = vi_buffer_name();
    command   = buffer[0];
    arguments = &buffer[1];

    switch (command)
    {
	case 'o' : /* visit file at char pos */
	case 'O' : /* visit file at line number */
	    sscanf(arguments, "%s %d %d", file, &position, &writable);
	    if (curr_file && !strcmp(file, curr_file))
	    {
		sprintf(VICommand, SelectFile, file);
		vi_exec_cmd(VICommand);
	    }
	    else
		vi_open_file(file);
	    vi_set_writability(curbuf, writable);
	    if (command == 'o')
		vi_set_cursor_pos((long)position);
	    else
	    {
		sprintf(VICommand, GotoLine, position);
		vi_exec_cmd(VICommand);
	    }
	    break;

	case 'p' : /* path of file has changed */
	    {
		BUF *buf;
		sscanf(arguments, "%s %s", file, new_path);
		buf = vi_find_buffer(file);
		if (!buf)
		{
		    return;
		}
		if (!buf->b_changed)
		{
			sprintf(VICommand, DeleteBuf, file);
			vi_exec_cmd(VICommand);
		}
		vi_open_file(new_path);
		break;
	    }

	case 'w' : /* writability has changed */
	    {
		BUF *buf;
		sscanf(arguments, "%s %d", file, &writable);
		buf = vi_find_buffer(file);
		if (!buf)	/* file not loaded */
		    return;
		vi_set_writability(buf, writable);
		break;
	    }

	case 'h' : /* highlight info */
	    break;   /* not implemented */

	case 't' : /* Set tab width */
	    sscanf(arguments, "%s %d", file, &tab_width);
	    sprintf(VICommand, SelectFile, file);
	    vi_exec_cmd(VICommand);
	    sprintf(VICommand, SetTab, tab_width);
	    vi_exec_cmd(VICommand);
	    break;

	case 'A' : /* Warning/Info msg */
	    vi_msg(arguments);
	    break;
	case 'a' : /* Error msg */
	    vi_error_msg(arguments);
	    if (!strncmp(arguments, "Cannot", 6))  /* Cannot connect ... */
		sniff_connected = 0;
	    break;

	case '\0':
	    break;
	default :
	    sprintf(VICommand, "Unrecognized sniff request [%s]", buffer );
	    vi_error_msg(VICommand);
	    break;
    }
}

/* read string from fd up to next newline, */
/* returns 0 if no data available or no complete line */
int get_request(int fd, char *buf, int maxlen)
{
    static char inbuf[1024];
    static int pos=0, bytes=0;
    register int len;
#ifdef HAVE_SELECT
    struct timeval tval;
    fd_set rfds;
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    tval.tv_sec  = 0;
    tval.tv_usec = 0;
#else
    struct pollfd fds;
    fds.fd = fd;
    fds.events = POLLIN;
#endif

    for(len=0; len<maxlen-1; len++)
    {
	if (pos>=bytes)
	{
#ifdef HAVE_SELECT
	    if (select(fd+1, &rfds, NULL, NULL, &tval)>0)
#else
	    if (poll(&fds, 1, 0)>0)
#endif
	    {
		bytes = read(fd, inbuf, sizeof(inbuf));
		if (bytes<=0)
		    return -1;
		pos = 0;
	    }
	    else
	    {
		pos = pos-len;
		buf[0] = '\0';
		return 0;
	    }
	}
	buf[len] = inbuf[pos++];
	if (buf[len]=='\n') break;
    }
    buf[++len] = '\0';
    return len;
}


void ProcessSniffRequests()
{
    char buf[256];
    int len;

     for(;;)
    {
	if (!sniff_connected) return;

	len = get_request(fd_from_sniff, buf, sizeof(buf));
	if (len<0)
	{
	    vi_error_msg("Sniff: Error during read. Disconnected");
	    sniff_connected = 0;
	    break;
	}
	else if (len>0)
	{
	    buf[len-1] = '\0';		/* delete trailing newline */
	    HandleSniffRequest( buf );
	}
	else
	    break;
    }

    if (sniff_will_disconnect)
    {
	sniff_connected = 0;
	sniff_will_disconnect = 0;
    }
}

void WriteToSniff( char *str )
{
    int bytes;
    bytes = write(fd_to_sniff, str, strlen(str));
    if (bytes<0)
    {
	vi_msg("Sniff: Error during write. Disconnected");
	sniff_connected = 0;
    }
}

void SendContextRequest( char command, char* symbol )
{
    char cmdstr[512];
    char *buffer_name;
    long symbol_location;

    buffer_name     = vi_buffer_name();
    if (!buffer_name)
    {
	vi_error_msg("Not a legal buffer");
	return;
    }
    if (!symbol)
	sprintf(cmdstr, "%c%s\n", command, buffer_name);
    else
    {
	symbol_location = vi_symbol_location(symbol);
	sprintf(cmdstr, "%c%s %ld %s\n", command, buffer_name, symbol_location, symbol);
    }
    WriteToSniff( cmdstr );
}

void SendSimpleRequest( char command )
{
    char cmdstr[]=" \n";

    cmdstr[0] = command;
    WriteToSniff( cmdstr );
}

void SendRequest(char command, char* symbol )
{
    char *msg_str="";
    char msg_txt[256];

    if (!command) return;

    if (!sniff_connected)
    {
	if (command=='y')
	{
	    if (ConnectToSniffEmacs())
		vi_error_msg("Error connecting to SNiFF+");
	    else
		vi_exec_cmd("source $VIM/sniff.vim");
	}
	else if (command!='z' && command!='u' && command!='q')
	    vi_error_msg("SNiFF+ not connected");
	return;
    }

    if (strchr("Rzu", command))
    {			    /* requests with buffer name only */
	switch(command)
	{
	    case 'R' : /* retrieve next */
		msg_str = "Retrieve next symbol";
		break;
	    case 'z':  /* ask for font info */
		if (!curbuf->b_sniff)	/* not a Sniff buffer */
		    return;
		break;
	    case 'u' : /* update file (reparse) after write */
		if (!curbuf->b_sniff)
		    return;
		break;
	}
	vi_msg(msg_str);
	SendContextRequest(command, NULL);
    }
    else if (strchr("emrsfwgtTxXcCdD", command))
    {			     /* context requests with symbol */
	switch(command)
	{
	    case 'e' : /* toggle btw. implementation and definition */
		msg_str = "Toggle";
		symbol	= " ";
		break;
	    case 'm' : /* show overridden member function */
		msg_str = "Show overridden member function";
		symbol	= " ";
		break;
	    case 'r' : /* retrieve symbol */
		msg_str = "Retrieve:";
		break;
	    case 's':  /* superclass */
		msg_str = "Show base class of:";
		break;
	    case 'f' : /* find symbol */
		msg_str = "Find symbol:";
		break;
	    case 'w' : /* browse class */
		msg_str = "Browse class:";
		break;
	    case 'g' : /* show source of a symbol */
		msg_str = "Show source of:";
		break;
	    case 't' : /* show a class in the hierarchy-browser */
		msg_str = "Show class in hierarchy:" ;
		break;
	    case 'T' : /* show a class in the restricted hierarchy */
		msg_str = "Show class in restricted hierarchy: ";
		break;
	    case 'x' : /* xref to */
		msg_str = "Xref to:";
		break;
	    case 'X' : /* xref by */
		msg_str = "Xref by:";
		break;
	    case 'c' : /* xref has a */
		msg_str = "Xref has a:";
		break;
	    case 'C' : /* xref used by */
		msg_str = "Xref used by:";
		break;
	    case 'd' : /* show docu */
		msg_str = "Show docu of:";
		break;
	    case 'D' : /* generate docu */
		msg_str = "Generate docu for:";
		break;
	}
	if (!symbol && !(symbol = vi_symbol_under_cursor()))
	    return;
	sprintf(msg_txt, "%s %s", msg_str, symbol);
	vi_msg(msg_txt);
	SendContextRequest(command, symbol);
    }
    else if (command=='q')
    {
	SendSimpleRequest(command);
	sniff_will_disconnect = 1;
	sleep(2);		/* Sleep. Incoming msg could disturb edit */
    }
}

/* Do :sniff command */
    void
do_sniff(char *arg)
{
    char *symbol=NULL;
    char cmd;
    int  len_cmd=0;
    int  i;
    int  print_cmds=FALSE;

    if (ends_excmd((char_u) *arg))
    {					/* no request: print available commands */
	print_cmds = TRUE;
	msg_start();
	msg_outtrans_attr((char_u *)"-- SNiFF+ commands --", highlight_attr[HLF_T]);
    }
    else
    {
	for (symbol=arg; !vim_iswhite(*symbol) && !ends_excmd(*symbol); symbol++);
	len_cmd = (int)(symbol-arg);
	symbol = (char *)skipwhite((char_u *)symbol);
	if (ends_excmd(*symbol))
	    symbol = NULL;
    }

    for(i=0; sniff_cmds[i].longname; i++)
    {
	if (print_cmds)
	{
	    msg_putchar('\n');
	    msg_outtrans((char_u *)":sniff ");
	    msg_outtrans((char_u *)sniff_cmds[i].longname);
	}
	else
	    if (!strncmp(arg, sniff_cmds[i].longname, len_cmd))
		break;
    }
    if (print_cmds)
    {
	msg_end();
	return;
    }
    cmd = sniff_cmds[i].cmd_char;
    if (cmd)
	SendRequest(cmd, symbol);
    else
	vi_error_msg("Sniff: Unrecognized command");
}



void vi_msg(char *str)
{
    if (str && *str!='\0')
	MSG((char_u *)str);
}

void vi_error_msg(char *str)
{
    if (str && *str!='\0')
	EMSG((char_u *)str);
}

void vi_open_file(char *fname)
{
    ++no_wait_return;
    do_ecmd(0, (char_u *)fname, NULL, NULL, (linenr_t)1,
					ECMD_HIDE+ECMD_OLDBUF);
    curbuf->b_sniff = TRUE;
    --no_wait_return;					/* [ex_docmd.c] */
}

BUF *vi_find_buffer(char *fname)
{			    /* derived from buflist_findname() [buffer.c] */
    BUF		*buf;

    for (buf = firstbuf; buf != NULL; buf = buf->b_next)
	if (buf->b_sfname != NULL && fnamecmp(fname, buf->b_sfname) == 0)
	    return (buf);
    return NULL;
}

void vi_set_writability(BUF *buf, int writable)
{
    buf->b_p_ro = !writable;
}

char *vi_symbol_under_cursor()
{
    int    len, i;
    char   *symbolp;
    static char sniff_symbol[256];

    len = find_ident_under_cursor((char_u **)&symbolp, FIND_IDENT);
							/* [normal.c] */
    if (len <= 0)
	return NULL;
    for(i=0;i<len;i++)
	sniff_symbol[i] = *(symbolp+i);
    sniff_symbol[i] = '\0';
    return sniff_symbol;
}

/*ARGSUSED*/
long vi_symbol_location(char *symbol)
{
    return vi_cursor_pos();
}

char *vi_buffer_name()
{
    return (char *)curbuf->b_sfname;  /* global variable */
}

void vi_exec_cmd( char *vicmd )
{
    do_cmdline((char_u *)vicmd, NULL, NULL, DOCMD_NOWAIT);  /* [ex_docmd.c] */
}

/*
 * Set cursor on character position
 * derived from cursor_pos_info() [buffer.c]
 */
void vi_set_cursor_pos( long char_pos )
{
    linenr_t	    lnum;
    long	    char_count=1;  /* first position = 1 */
    int		    line_size;
    int		    eol_size;

    if (curbuf->b_p_tx)
	eol_size = 2;
    else
	eol_size = 1;
    for (lnum = 1; lnum <= curbuf->b_ml.ml_line_count; ++lnum)
    {
	line_size = STRLEN(ml_get(lnum)) + eol_size;
	if (char_count+line_size > char_pos) break;
	char_count += line_size;
    }
    curwin->w_cursor.lnum = lnum;
    curwin->w_cursor.col  = char_pos - char_count;
}

long vi_cursor_pos()
{
    linenr_t	    lnum;
    long	    char_count=1;  /* sniff starts with pos 1 */
    int		    line_size;
    int		    eol_size;

    if (curbuf->b_p_tx)
	eol_size = 2;
    else
	eol_size = 1;
    for (lnum = 1; lnum < curwin->w_cursor.lnum; ++lnum)
    {
	line_size = STRLEN(ml_get(lnum)) + eol_size;
	char_count += line_size;
    }
    return char_count + curwin->w_cursor.col;
}
