///////////////////////////////////////////////////////////////////
//
// Tcl plugin install.js script for XPI
//
// The plugin is structured as the NPAPI dll (nptclXY.dll) and a
// sibling directory (nptcl/) that contains all other necessary
// parts - notably the plugin library, which must be "native" to
// allow for runtime modification and sharing with other binaries.
//

var SOFTWARE_NAME  = "Tcl/Tk Plugin";
var VERSION        = "3.0";
var PLID_BASE      = "@mozilla.org/TclPlugin";
var PLID           = PLID_BASE + ",version=" + VERSION;

// This information is passed in by the script that generates
// the appropriate install.js for each platform's XPI
var NPAPIDLL       = "nptcl30.dll";
var PLUGHOSTDLL    = "pluginhostctrl.dll";
var NPTCLDIR       = "nptcl";
// Specified in bytes
var SPACE_REQD     = 3720443;

///////////////////////////////////////////////////////////////////

// Invoke initInstall to start the installation
err = initInstall(SOFTWARE_NAME, PLID, VERSION);
if (err == BAD_PACKAGE_NAME)
{
    // HACK: Mozilla 1.1 has a busted PLID parser which doesn't like
    // the equals sign
    PLID = PLID_BASE;
    err = initInstall(SOFTWARE_NAME, PLID, VERSION);
}

if (err == SUCCESS)
{
    err = doAddFiles();
}
else
{
    logComment("Install failed at initInstall level with " + err);
}

if (err == SUCCESS)
{
    err = performInstall();
    if (err == SUCCESS)
    {
	// Refresh plugin list and reload pages (1)
        refreshPlugins(1);
    }
}
else
{
    cancelInstall(err);
}

/**
 * Function for preinstallation of plugin (FirstInstall).
 *
 * @param dirPath	directory path from getFolder
 * @param spaceRequired	required space in bytes
 * 
 **/
function verifyDiskSpace(dirPath, spaceRequired)
{
    // Get the available disk space on the given path in bytes
    var spaceAvailable = fileGetDiskSpaceAvailable(dirPath);

    logComment("Disk space required: " + parseInt(spaceRequired / 1024) + "K");
    // do the verification
    if (spaceAvailable < spaceRequired)
    {
        logComment("Insufficient disk space: " + dirPath);
        logComment("  required : " + parseInt(spaceRequired / 1024) + "K");
        logComment("  available: " + parseInt(spaceAvailable / 1024) + "K");
        return INSUFFICIENT_DISK_SPACE;
    }

    return SUCCESS;
}

function doAddFiles()
{
    // Install plugin files
    var folder = getFolder("Plugins");
    // "Current User" is actually the current profile directory
    var userFolder = getFolder("Current User", "../../plugins");
    if (getPlatform() == "win")
    {
	// Windows has extra /Profiles/ path component
	userFolder = getFolder("Current User", "../../../plugins");
    }

    err = verifyDiskSpace(folder, SPACE_REQD);
    if (err == SUCCESS)
    {
	err = addFile(PLID, VERSION, NPAPIDLL, folder, null);
	if ((err == ACCESS_DENIED) && (folder != userFolder)
	    && (verifyDiskSpace(userFolder, SPACE_REQD) == SUCCESS)
	    && confirm("Unable to install to \"" + folder + "\".\n" +
		       "Attempt install to user-specific plugins folder \""
		       + userFolder + "\"?"))
	{
	    File.dirCreate(userFolder); // we don't care about result
	    folder = userFolder;
	    err = addFile(PLID, VERSION, NPAPIDLL, folder, null);
	}
	if (err == SUCCESS)
	{
	    // We can't remove the folder first as it may contain modified
	    // configuration files, but we need more checks on that.
	    // Also, File.dirRemove ordering appears broken, in that the
	    // installer (for FF1.5 at least) won't recreate this directory
	    // and we get a -214 error.
	    err = addDirectory(PLID, VERSION, NPTCLDIR, folder, NPTCLDIR,
			       true);
	}
	if (err != SUCCESS)
	{
	    alert("Installation of " + NPAPIDLL +
		  " failed.\nError code " + err);
	    logComment("Adding file " + NPAPIDLL + " to " + folder +
		       " failed. Error code: " + err);
	    return err;
	}
	if ((getPlatform() == "win") && (PLUGHOSTDLL != ""))
	{
	    var msg =
		"Register ActiveX NPAPI host control for Internet Explorer?\n"
		+ "Admin privileges are required.  Any Firefox NPAPI "
		+ "will be hosted in Internet Explorer!\n\n"
		+ "To unregister, you can call:\n"
		+ "\"regsvr32 -u " + folder + PLUGHOSTDLL + "\"";
	    var confirmed;
	    if ("BUTTON_POS_0" in Install) {
		confirmed = confirm(msg,
				    "Register ActiveX NPAPI Host Control?",
				    STD_YES_NO_BUTTONS);
	    } else {
		msg += "\n(OK to register, Cancel to continue without ActiveX control)";
		confirmed = confirm(msg);
	    }
	    if (confirmed) {
		var hostctrl = getFolder("file:///", folder + PLUGHOSTDLL);
		err = addFile(PLID, VERSION, PLUGHOSTDLL, folder, null);
		if (err == SUCCESS) {
		    err = File.windowsRegisterServer(hostctrl);
		}
	    }
	}
    }
    else
    {
	logComment("Cancelling current browser install due to lack of space...");
    }
    return err;
}

function getPlatform()
{
    var platformStr;
    var platformNode;
    if ('platform' in Install)
    {
        platformStr = new String(Install.platform);
        if (!platformStr.search(/^Macintosh/))
            platformNode = 'mac';
        else if (!platformStr.search(/^Win/))
            platformNode = 'win';
        else
            platformNode = 'unix';
    }
    else
    {
        var fOSMac  = getFolder("Mac System");
        var fOSWin  = getFolder("Win System");
        logComment("fOSMac: "  + fOSMac);
        logComment("fOSWin: "  + fOSWin);
        if (fOSMac != null)
            platformNode = 'mac';
        else if(fOSWin != null)
            platformNode = 'win';
        else
            platformNode = 'unix';
    }
    return platformNode;
}
