/*

  ssh-f-configd.h

  Author: Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Main header of Secure Shell Configuration Daemon for F-Secure FSMA.

*/

#ifndef SSH_F_CONFIGD_H
#define SSH_F_CONFIGD_H

#include "sshbuffer.h"
#include "sshadt_strmap.h"

#include "ssh-f-configd-vers.h"
#include "ssh-f-configd-fsma.h"
#include "ssh-f-configd-oid.h"

#if 0
#define SSH_F_CONFIGD_FSMA_STRING_VALUE_NONE "-*-*-FSMA-NONE-*-*-"
#else
#define SSH_F_CONFIGD_FSMA_STRING_VALUE_NONE ""
#endif

#ifdef ETCDIR
#  define SSH_F_CONFIGD_SSHD2_CONFIG_FILE_DEFAULT ETCDIR "/sshd2_config"
#  define SSH_F_CONFIGD_SSH2_CONFIG_FILE_DEFAULT ETCDIR "/ssh2_config"
#  define SSH_F_CONFIGD_OID_MAPPING_DIR_DEFAULT ETCDIR "/fsma-oids"
#else /* ETCDIR */
#  define SSH_F_CONFIGD_SSHD2_CONFIG_FILE_DEFAULT "/etc/ssh2/sshd2_config"
#  define SSH_F_CONFIGD_SSH2_CONFIG_FILE_DEFAULT "/etc/ssh2/ssh2_config"
#  define SSH_F_CONFIGD_OID_MAPPING_DIR_DEFAULT "/etc/ssh2/fsma-oids"
#endif /* ETCDIR */

#ifdef HAVE_VAR_RUN
#  define SSH_F_CONFIGD_SSHD2_PID_DIR_DEFAULT "/var/run"
#else /* HAVE_VAR_RUN */
#  ifdef ETCDIR
#    define SSH_F_CONFIGD_SSHD2_PID_DIR_DEFAULT ETCDIR
#  else /* ETCDIR */
#    define SSH_F_CONFIGD_SSHD2_PID_DIR_DEFAULT "/etc/ssh2"
#  endif /* ETCDIR */
#endif /* HAVE_VAR_RUN */

#define SSH_F_CONFIGD_SSHD2_PID_FILE_DEFAULT \
                SSH_F_CONFIGD_SSHD2_PID_DIR_DEFAULT "/" "sshd2_22.pid"
#define SSH_F_CONFIGD_PID_FILE_DEFAULT \
                SSH_F_CONFIGD_SSHD2_PID_DIR_DEFAULT "/" "ssh_f_configd.pid"

#define SSH_F_CONFIGD_FSMA_FATAL_EXITVAL 42
#define SSH_F_CONFIGD_POLICY_VAR_MAX_LEN 0x200

typedef enum SshFConfigDPolicyTransferStatus {
    SSH_F_CONFIGD_POLICY_CHANGED = 0,
    SSH_F_CONFIGD_POLICY_NOT_CHANGED,
    SSH_F_CONFIGD_POLICY_TRANSFER_FAILED
} SshFConfigDPolicyTransferStatus;

typedef struct SshFConfigDConf *SshFConfigDConf;
typedef struct SshFConfigDProg *SshFConfigDProg;
typedef struct SshFConfigDOidKeyPair *SshFConfigDOidKeyPair;

typedef SshFConfigDPolicyTransferStatus (*SshFConfigDOIDHandler)(
                                                 SshFConfigDConf conf,
                                                 SshFConfigDProg prog,
                                                 SshFConfigDOidKeyPair pair,
                                                 DfPolicyVar var);

typedef struct SshFConfigDProg {
  char *name;
  char *config_file;
  char *config_file_bak;
  char *oid_pattern;
  SshADTContainer oid_map;
  DfPolicyVar oid_pattern_var;
  char *oid_mapping_file;
  char *oid_pattern_file;
  SshBufferStruct config_data[1];
  char *pid_file;
  pid_t pid;
  Boolean configuration_valid;
  Boolean configuration_disable;
} SshFConfigDProgRec;

typedef struct SshFConfigDConf {
  char *av0;
  int exit_signal;
  Boolean log_entry_on_termination;
  char *oid_mapping_dir;
  FSMASESSION fsma_session;
  Boolean fsma_session_active;
  FSMAMESSAGEPUMP fsma_pump;
  char *fsma_module_oid;
  char *pid_file;
  Boolean pid_file_written;
  DfPolicyVar fsma_module_oid_var;
  Boolean batch_mode;
  Boolean terminating;
  Boolean debug_mode;
  SshADTContainer prog_map;
} SshFConfigDConfRec;

typedef struct SshFConfigDOidKeyPair {
  char *oid;
  char *key;
  char *val;
  SshFConfigDOIDHandler oid_handler;
  char *val_raw;
  Boolean val_is_raw;
  SshTime timestamp;
  Boolean updated;
} SshFConfigDOidKeyPairRec;

#ifndef UNICODE
#define SSH_F_CONFIGD_CHAR_TYPE char
#else /* UNICODE */
#define SSH_F_CONFIGD_CHAR_TYPE w_char_t
#endif /* UNICODE */

void ssh_f_configd_terminate(SshFConfigDConf conf, int exit_value);

SshFConfigDConf ssh_f_configd_conf_alloc(const char *av0);
void ssh_f_configd_conf_free(SshFConfigDConf conf);
SshFConfigDOidKeyPair ssh_f_configd_oid_key_pair_alloc(const char *oid,
                                                       const char *key);
SshFConfigDProg ssh_f_configd_prog_alloc(SshFConfigDConf conf);
void ssh_f_configd_prog_free(SshFConfigDConf conf, SshFConfigDProg prog);

void ssh_f_configd_ssh_pid_refresh(SshFConfigDConf conf,
                                   SshFConfigDProg prog);
void ssh_f_configd_ssh_reconfigure(SshFConfigDConf conf,
                                   SshFConfigDProg prog);
void ssh_f_configd_ssh_config_data_clear(SshFConfigDConf conf,
                                         SshFConfigDProg prog);
void ssh_f_configd_ssh_config_data_begin(SshFConfigDConf conf,
                                         SshFConfigDProg prog);
void ssh_f_configd_ssh_config_data_add_string(SshFConfigDConf conf,
                                              SshFConfigDProg prog,
                                              const char *str);
void ssh_f_configd_ssh_config_data_add_param(SshFConfigDConf conf,
                                             SshFConfigDProg prog,
                                             const char *key,
                                             const char *arg);
void ssh_f_configd_ssh_config_data_end(SshFConfigDConf conf,
                                       SshFConfigDProg prog);
void ssh_f_configd_ssh_config_data_activate(SshFConfigDConf conf,
                                            SshFConfigDProg prog);
void ssh_f_configd_ssh_config_data_dump(SshFConfigDConf conf,
                                        SshFConfigDProg prog);
SshFConfigDPolicyTransferStatus ssh_f_configd_config_data_collect_generic(
                                                        SshFConfigDConf conf,
                                                        SshFConfigDProg prog);
void ssh_f_configd_ssh_config_data_process(SshFConfigDConf conf,
                                           SshFConfigDProg prog);

void ssh_f_configd_reconfigure_programs(SshFConfigDConf conf);

void ssh_f_configd_fsma_init(SshFConfigDConf conf);
void ssh_f_configd_fsma_uninit(SshFConfigDConf conf);

void ssh_f_configd_log_stderr(SshLogFacility facility, 
                              SshLogSeverity severity, 
                              const char *msg, 
                              void *context);

void ssh_f_configd_log(SshLogFacility facility, 
                       SshLogSeverity severity, 
                       const char *msg, 
                       void *context);

/* Special OID handlers. */
SshFConfigDOIDHandler ssh_f_configd_find_handler(const char *oid);
SshFConfigDPolicyTransferStatus sshd_conf_handle_1_3_6_1_4_1_2213_36_1_4(
                                                 SshFConfigDConf conf,
                                                 SshFConfigDProg prog,
                                                 SshFConfigDOidKeyPair pair,
                                                 DfPolicyVar var);

#endif /* SSH_F_CONFIGD_H */
/* eof (ssh-f-configd.h) */
