/*
 *
 * Authors: Tero Kivinen <kivinen@iki.fi>
 *              Santeri Paavolainen <santtu@ssh.com>
 *
 * Copyright (c) 2000 SSH Communications Security Oy <info@ssh.fi>
 *
 * Implementations of sshmutex.h, sshcondition.h and sshthread.h APIs
 * based on the pthreads threading API.
 */

/*
 *        Program: Util Lib
 *        $Source: /ssh/CVS/src/lib/sshutil/ssheloop/sshmt_pthreads.c,v $
 *        $Author: vsuontam $
 *
 *        Creation          : 21:21 Feb 19 2000 kivinen
 *        Last Modification : 00:21 Feb 24 2000 kivinen
 *        Last check in     : $Date: 2001/10/01 09:48:17 $
 *        Revision number   : $Revision: 1.5 $
 *        State             : $State: Exp $
 *        Version           : 1.166
 *        
 *
 *        Description       : Posix mutexes and multithread
 *                            environment support.
 *
 *
 *        $Log: sshmt_pthreads.c,v $ *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        $EndLog$
 */

/* Tru64 pthreads does not work if XOPEN_SOURCE is defined. */

#ifdef _XOPEN_SOURCE_EXTENDED
#undef _XOPEN_SOURCE_EXTENDED
#define SAVED_XOPEN_SOURCE_EXTENDED
#endif /* _XOPEN_SOURCE_EXTENDED */

#ifdef _XOPEN_SOURCE
#undef _XOPEN_SOURCE
#define SAVED_XOPEN_SOURCE
#endif /* _XOPEN_SOURCE */

#include <pthread.h>
/* We must include strings.h here because otherwise we get conflict later */
#include <strings.h>

#ifdef SAVED_XOPEN_SOURCE_EXTENDED
#undef _XOPEN_SOURCE_EXTENDED
#define _XOPEN_SOURCE_EXTENDED 1
#undef SAVED_XOPEN_SOURCE_EXTENDED
#endif /* SAVED_XOPEN_SOURCE_EXTENDED */

#ifdef SAVED_XOPEN_SOURCE
#undef _XOPEN_SOURCE
#define _XOPEN_SOURCE 1
#undef SAVED_XOPEN_SOURCE
#endif /* SAVED_XOPEN_SOURCE */

#include "sshincludes.h"
#include "sshthread.h"
#include "sshmutex.h"
#include "sshcondition.h"

/* NOTE: Do not use debug prints here, because the debugging system might be
   using mutexes to protect itself. */

#undef SSH_DEBUG_MODULE

/* Mutex type, the actual contents is system dependent */
struct SshMutexRec {
  char *name;
  pthread_mutex_t mutex;
};

struct SshConditionRec {
  char *name;
  pthread_cond_t condition;
};

/*****************************  Mutexes   **********************************/

/* Allocate mutex and initialize it to unlocked state. Currently no flags
   defined. Name is the name of the mutex, it is only used for debugging. This
   function will take a copy of the name. The name can also be NULL. */
SshMutex ssh_mutex_create(const char *name, SshUInt32 flags)
{
  SshMutex mutex;
  int err;
  mutex = ssh_xcalloc(1, sizeof(*mutex));
  if (name)
    mutex->name = ssh_xstrdup(name);
  err = pthread_mutex_init(&(mutex->mutex), NULL);
  if (err != 0)
    ssh_fatal("Mutex init failed: %s", strerror(err));
  return mutex;
}

/* Destroy mutex. It is fatal error to call this if mutex is locked. */
void ssh_mutex_destroy(SshMutex mutex)
{
  int err;
  err = pthread_mutex_destroy(&(mutex->mutex));
  if (err != 0)
    ssh_fatal("Mutex destroy failed: %s", strerror(err));
  ssh_xfree(mutex->name);
  ssh_xfree(mutex);
}

/* Locks the mutex. If the mutex is already locked then this will block until
   the mutex is unlocked. */
void ssh_mutex_lock(SshMutex mutex)
{
  int err;

  err = pthread_mutex_lock(&(mutex->mutex));
  if (err != 0)
    ssh_fatal("Mutex lock failed: %s", strerror(err));
}

/* Unlocks the mutex. It is fatal error to call this function if the mutex is
   already unlocked. Also only the original thread that took the lock is
   allowed to unlock it. */
void ssh_mutex_unlock(SshMutex mutex)
{
  int err;

  err = pthread_mutex_unlock(&(mutex->mutex));
  if (err != 0)
    ssh_fatal("Mutex lock failed: %s", strerror(err));
}

/* Returns the name of the mutex. This returns NULL if the mutex does not have
   name. */
const char *ssh_mutex_get_name(SshMutex mutex)
{
  return mutex->name;
}

/*************************** Condition variables ***************************/

/* Create a condition variable */
SshCondition ssh_condition_create (const char *name, SshUInt32 flags)
{
  SshCondition cond;
  cond = ssh_xcalloc(1, sizeof(*cond));

  if (name)
    cond->name = ssh_xstrdup(name);

  pthread_cond_init(&cond->condition, NULL);

  return cond;
}

/* Destroy a condition variable */
void ssh_condition_destroy(SshCondition cond)
{
  pthread_cond_destroy(&cond->condition);
  ssh_xfree(cond->name);
  ssh_xfree(cond);
}

/* Signal a thread blocked on a condition variable */
void ssh_condition_signal(SshCondition cond)
{
  pthread_cond_signal(&cond->condition);
}

/* Signal all threads blocked on a condition variable */
void ssh_condition_broadcast(SshCondition cond)
{
  pthread_cond_broadcast(&cond->condition);
}

/* Wait on a condition variable */
void ssh_condition_wait(SshCondition cond, SshMutex mutex)
{
  pthread_cond_wait(&cond->condition, &mutex->mutex);
}

/* Returns the name assigned to a condition variable */
const char *ssh_condition_get_name(SshCondition cond)
{
  return cond->name;
}

/*******************************  Threads **********************************/

/* XXX remove this old code: it conformed to the old sshthread.h api,
   which was not well designed (see the difference in code sizes of
   the implementation) */

#if 0
struct SshThreadRec {
  /* Thread id of the this thread */
  pthread_t thread_id;

  /* Mutex protecting the internal thread structure state */
  pthread_mutex_t mutex;

  /* Count of threads blocking on destroy_cond if this thread is being
     destroyed. The thread which receives the signal and decrements
     this count to 0 is the one responsible for actual thread state
     destruction and resource freeing. */
  SshUInt32 destroyed;

  /* TRUE if this thread is destructing itself.. */
  Boolean self_destruct;

  /* TRUE if the thread is currently running, eg. has acquired
     func_cb, done_cb and context parameters successfully from
     ssh_thread_execute */
  Boolean running;

  /* The started thread will block on this condition variable to get
     the callback etc. arguments. */
  pthread_cond_t start_cond;

  /* The current thread will broadcast this condition if it notices
     the `destroyed' is non-zero when it is exiting. Notice that the
     listener to this signal which then decrements `destroyed' to 0
     must handle the actual resource freeing. */
  pthread_cond_t destroy_cond;

  /* Callbacks and the context argument */
  SshThreadFuncCB func_cb;
  SshThreadDoneCB done_cb;
  void *context;
};

static pthread_once_t once_control = { PTHREAD_ONCE_INIT };
static pthread_key_t thread_key;

static void* ssh_thread_internal_start(void *);

/* Initialize the thread_key, which is used to store the current
   thread's SshThread pointer */
static void ssh_thread_internal_once()
{
  pthread_key_create(&thread_key, NULL);
}

SshThread ssh_thread_create()
{
  SshThread thread;
  int err;

  /* This will ensure the ssh_thread_internal_once is called only once. */
  pthread_once(&once_control, ssh_thread_internal_once);

  thread = ssh_xcalloc(1, sizeof(*thread));
  pthread_mutex_init(&thread->mutex, NULL);
  pthread_cond_init(&thread->start_cond, NULL);
  pthread_cond_init(&thread->destroy_cond, NULL);

  pthread_mutex_lock(&thread->mutex);
  err = pthread_create(&thread->thread_id, NULL,
                       ssh_thread_internal_start, thread);

  if (err != 0)
    {
      pthread_mutex_unlock(&thread->mutex);
      pthread_mutex_destroy(&thread->mutex);
      pthread_cond_destroy(&thread->start_cond);
      pthread_cond_destroy(&thread->destroy_cond);
      ssh_xfree(thread);
      return NULL;
    }

  /* The started thread will now wait on the thread->wait parameter */
  return thread;
}

Boolean ssh_thread_execute(SshThread thread, SshThreadFuncCB func_cb,
                           SshThreadDoneCB done_cb, void *context)
{
  pthread_mutex_lock(&thread->mutex);
  if (thread->running)
    {
      pthread_mutex_unlock(&thread->mutex);
      return FALSE;
    }

  thread->func_cb = func_cb;
  thread->done_cb = done_cb;
  thread->context = context;

  pthread_mutex_unlock(&thread->mutex);
  pthread_cond_signal(&thread->start_cond);

  return TRUE;
}

static void ssh_thread_internal_destroy(SshThread thread)
{
  pthread_mutex_unlock(&thread->mutex);
  pthread_mutex_destroy(&thread->mutex);
  pthread_cond_destroy(&thread->start_cond);
  pthread_cond_destroy(&thread->destroy_cond);
  ssh_xfree(thread);
}

void ssh_thread_destroy(SshThread thread)
{
  pthread_mutex_lock(&thread->mutex);

  if (thread->thread_id == pthread_self())
    {
      /* count our own destruction only once, since we don't block */
      if (!thread->destroyed)
        thread->destroyed++;
      thread->self_destruct = TRUE;
    }
  else
    {
      thread->destroyed++;
      pthread_cond_signal(&thread->start_cond);
      pthread_cond_wait(&thread->destroy_cond, &thread->mutex);
      thread->destroyed--;

      if (thread->destroyed == 0)
        ssh_thread_internal_destroy(thread);
    }
}

SshThread ssh_thread_current()
{
  return (SshThread) pthread_getspecific(thread_key);
}

static void *ssh_thread_internal_start(void *ctx)
{
  SshThread thread = (SshThread)ctx;

  pthread_detach(pthread_self());
  pthread_setspecific(thread_key, thread);

  while (1)
    {
      pthread_mutex_lock(&thread->mutex);
      /* Either we get the cb values, or the thread was destructed
         while waiting for those.. */

      while (thread->func_cb == NULL && thread->destroyed == 0)
        pthread_cond_wait(&thread->start_cond, &thread->mutex);

      if (thread->func_cb != NULL && thread->destroyed == 0)
        {
          pthread_mutex_unlock(&thread->mutex);
          (*thread->func_cb)(thread->context);
          pthread_mutex_lock(&thread->mutex);

          if (thread->destroyed != 0)
            {
              pthread_mutex_unlock(&thread->mutex);
              (*thread->done_cb)(thread->context);
              pthread_mutex_lock(&thread->mutex);
            }

          thread->func_cb = NULL;
          thread->done_cb = NULL;
          thread->context = NULL;
        }

      if (thread->destroyed > 0)
        {
          pthread_cond_broadcast(&thread->destroy_cond);

          if (thread->self_destruct)
            thread->destroyed--;

          if (thread->destroyed == 0)
            {
              ssh_thread_internal_destroy(thread);
              return NULL;
            }

          pthread_mutex_unlock(&thread->mutex);
          return NULL;
        }
    }
}
#endif

/* pthread_t is either void pointer or "unsigned int" -- a pointer
   (SshThread) should be always large enough */
#define SSH_THREAD_TO_PTHREAD(X)        ((pthread_t) (X))
#define SSH_PTHREAD_TO_THREAD(X)        ((SshThread) (X))

SshThread ssh_thread_create(SshThreadFuncCB func_cb, void *context)
{
  pthread_t tid;
  int err;

  err = pthread_create(&tid, NULL, func_cb, context);

  if (err != 0)
    return NULL;

  return SSH_PTHREAD_TO_THREAD(tid);
}

void *ssh_thread_join(SshThread thread)
{
  void *ret;
  int err;

  err = pthread_join(SSH_THREAD_TO_PTHREAD(thread), &ret);

  if (err != 0)
    ssh_fatal("invalid thread given to ssh_thread_join");

  return ret;
}

void ssh_thread_detach(SshThread thread)
{
  pthread_detach(SSH_THREAD_TO_PTHREAD(thread));
}

SshThread ssh_thread_current()
{
  return SSH_PTHREAD_TO_THREAD(pthread_self());
}
