/*

sigchld.c

  Author: Tatu Ylonen   <ylo@ssh.fi>
          Sami Lehtinen <sjl@ssh.fi>
          
  Copyright (c) 1997,2002 SSH Communications Security, Finland
                          All rights reserved
  
  Generic SIGCHLD handler.  Allows one to register SIGCHLD handler based
  on the pid of the waited process.

*/

#include "sshincludes.h"
#include "sigchld.h"
#include "ssheloop.h"
#include "sshadt_list.h"
#include "sshadt_conv.h"

#define SSH_DEBUG_MODULE "SshSigChld"

typedef struct SshSigChldNodeRec {
  pid_t pid;
  SshSigChldHandler callback;
  void *context;
  struct SshSigChldNodeRec *next;
} *SshSigChldNode;

#define SSH_SIGCHLD_ORPHAN_MAX_AGE 10

typedef struct SshSigChldOrphanRec {
  pid_t pid;
  int status;
  SshTime exit_time;
} SshSigChldOrphanStruct, *SshSigChldOrphan;

/* List of registered sigchld handlers. */
SshSigChldNode ssh_sigchld_handlers = NULL;
SshADTContainer ssh_sigchld_orphan_list = NULL;

/* Indicates whether the subsystem has been initialized and the real
   SIGCHLD handler registered. */
Boolean ssh_sigchld_initialized = FALSE;

/* Forward declaration. */
void ssh_sigchld_real_callback(int signal, void *context);

/* Calls any callbacks for the given pid. */

void ssh_sigchld_process_pid(pid_t pid, int status)
{
#ifdef HAVE_SIGNAL
  int exitcode;
  SshSigChldNode node;
  SshSigChldOrphan orphan;
  SshADTHandle h, h2;
  SshTime cur_time = ssh_time();
  
  /* Translated status to our format. */
  if (WIFEXITED(status))
    exitcode = WEXITSTATUS(status);
  else
    exitcode = -WTERMSIG(status);

  /* Loop over all handlers. */
  for (node = ssh_sigchld_handlers; node; node = node->next)
    {
      /* Continue until the correct handler found. */
      if (node->pid != pid)
        continue;

      /* Call the handler. */
      SSH_DEBUG(1, ("calling handler pid %d code %d", (int)pid, exitcode));
      (*node->callback)(pid, exitcode, node->context);

      /* Remove the handler.  Note that the callback might already have
         removed the handler. */
      ssh_sigchld_unregister(pid);
      return;
    }
  SSH_DEBUG(8, ("no handler for pid %d code %d, adding orphan record",
                (int)pid, exitcode));
  
  /* Clean up too old orphans. */
  for (h = ssh_adt_enumerate_start(ssh_sigchld_orphan_list);
       h != SSH_ADT_INVALID;
       /* NOTHING*/)
    {
      h2 = ssh_adt_enumerate_next(ssh_sigchld_orphan_list, h);
      orphan = ssh_adt_get(ssh_sigchld_orphan_list, h);
      if (cur_time - orphan->exit_time > SSH_SIGCHLD_ORPHAN_MAX_AGE)
        {
          SSH_DEBUG(9, ("Cleaning up old orphan (age: %ld, pid: %d).",
                        (unsigned long)(cur_time - orphan->exit_time),
                        (int)orphan->pid));
          ssh_adt_delete(ssh_sigchld_orphan_list, h);
        }
      h = h2;
    }
  
  /* Add entry to the orphan list. */
  orphan = ssh_xcalloc(1, sizeof(*orphan));
  orphan->pid = pid;
  orphan->status = status;
  orphan->exit_time = cur_time;

  h = ssh_adt_insert(ssh_sigchld_orphan_list, orphan);
  if (h == SSH_ADT_INVALID)
    ssh_fatal("ssh_sigchld_process_pid: couldn't allocate memory for orphan "
              "record (pid: %d)", (int)pid);
#else  /* HAVE_SIGNAL */
  /*XXX*/
#endif /* HAVE_SIGNAL */
}

/* This callback is called by the event loop whenever a SIGCHLD signal
   is received.  This will wait for any terminated processes, and
   will call the appropriate sigchld handlers for them.  If a process
   does not have a handler, its return status is silently ignored. */

void ssh_sigchld_real_callback(int signal, void *context)
{

#ifdef HAVE_SIGNAL
  pid_t pid;
  int status;

  SSH_DEBUG(3, ("SIGCHLD received."));
  
#ifdef HAVE_WAITPID
  for (;;)
    {
      /* This is to insure we didn't break of the waitpid() because of
         signal etc. */
      while ((pid = waitpid(-1, &status, WNOHANG)) < 0)
        if (errno != EINTR)
          {
            SSH_HEAVY_DEBUG(20, ("errno: %d, strerror: %s.",
                                 errno, strerror(errno)));
            break;
          }
      if (pid <= 0)
        break;
      
      if (WIFSTOPPED(status))
        continue;
      ssh_sigchld_process_pid(pid, status);
    }
#else /* HAVE_WAITPID */
  while ((pid = wait(&status)) < 0)
    if (errno != EINTR)
      break;
  
  if (pid > 0 && !WIFSTOPPED(status))
    ssh_sigchld_process_pid(pid, status);
#endif /* HAVE_WAITPID */
  /* Problem seems to be, that signal(2) has different behaviour in
     different systems, and in e.g. Linux, if a signal is defaulted
     with signal() and then restored, it retains the property of being
     "persistent" (as in BSD), which is set with sigaction()
     et. al. In Solaris, however, this property is lost, and when the
     handler is called after being restored with signal(), the handler
     will return to SIG_DFL, and the next SIGCHLD won't be
     caught. Which is not nice. With this, if we catch a SIGCHLD which we
     don't have a handler for, we re-register the signal-handler to
     eventloop (which will reset the signal mask), and the next
     SIGCHLD can be caught.

     So the behaviour of ssh_sigchld_register() is now the same on
     different platforms.  */
  ssh_register_signal(SIGCHLD, ssh_sigchld_real_callback, context);
#else  /* HAVE_SIGNAL */
  /*XXX*/
#endif /* HAVE_SIGNAL */
}

/* Initializes the sigchld handler subsystem.  It is permissible to call
   this multiple times; only one initialization will be performed.
   It is guaranteed that after this has been called, it is safe to fork and
   call ssh_sigchld_register (in the parent) for the new process as long
   as the process does not return to the event loop in the meanwhile. */

void ssh_sigchld_initialize(void)
{

#ifdef HAVE_SIGNAL
  if (ssh_sigchld_initialized)
    return;

  ssh_sigchld_initialized = TRUE;
  ssh_register_signal(SIGCHLD, ssh_sigchld_real_callback, NULL);

  ssh_sigchld_orphan_list =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  if (ssh_sigchld_orphan_list == NULL)
    ssh_fatal("ssh_sigchld_initialize: couldn't allocate orphan list");
  
#else  /* HAVE_SIGNAL */
  /*XXX*/
#endif /* HAVE_SIGNAL */
}

void ssh_sigchld_uninitialize(void)
{
#ifdef HAVE_SIGNAL
  if (!ssh_sigchld_initialized)
    return;

  ssh_sigchld_initialized = FALSE;
  ssh_unregister_signal(SIGCHLD);
  
  ssh_adt_destroy(ssh_sigchld_orphan_list);
  ssh_sigchld_orphan_list = NULL;
#else  /* HAVE_SIGNAL */
  /*XXX*/
#endif /* HAVE_SIGNAL */  
}

/* Registers the given function to be called when the specified
   process terminates.  Only one callback can be registered for any
   process; any older callbacks for the process are erased when a new
   one is registered. */

void ssh_sigchld_register(pid_t pid, SshSigChldHandler callback,
                          void *context)
{

#ifdef HAVE_SIGNAL
  SshSigChldNode node;
  SshSigChldOrphan orphan;  
  SshADTHandle h;

  SSH_DEBUG(2, ("Registering handler for pid %d.", (int)pid));
  /* Clear any old callback for the pid. */
  ssh_sigchld_unregister(pid);

  /* Add a new sigchld handler record in the list. */
  node = ssh_xmalloc(sizeof(*node));
  node->pid = pid;
  node->callback = callback;
  node->context = context;
  node->next = ssh_sigchld_handlers;
  ssh_sigchld_handlers = node;


  /* Check for a matching orphan, and if it is new enough, call the
     callback from the bottom of the event loop and remove the orphan
     from the list. */
  for (h = ssh_adt_enumerate_start(ssh_sigchld_orphan_list);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(ssh_sigchld_orphan_list, h))
    {
      orphan = ssh_adt_get(ssh_sigchld_orphan_list, h);
      SSH_INVARIANT(orphan != NULL);
      
      if (orphan->pid == pid)
        {
          (void)ssh_adt_detach(ssh_sigchld_orphan_list, h);
          SSH_DEBUG(1, ("Calling SIGCHLD handler callback for orphan %d.",
                        (int) pid));
          ssh_sigchld_process_pid(orphan->pid, orphan->status);
          ssh_xfree(orphan);
          return;
        }
    }

#else  /* HAVE_SIGNAL */
  /*XXX*/
#endif /* HAVE_SIGNAL */
}

/* Unregisters the given SIGCHLD callback. */

void ssh_sigchld_unregister(pid_t pid)
{

#ifdef HAVE_SIGNAL
  SshSigChldNode node, *nodep;

  for (nodep = &ssh_sigchld_handlers; *nodep; nodep = &(*nodep)->next)
    if ((*nodep)->pid == pid)
      {
        node = *nodep;
        *nodep = node->next;
        ssh_xfree(node);
        return;
      }
#else  /* HAVE_SIGNAL */
  /*XXX*/
#endif /* HAVE_SIGNAL */
}
