/*

  gentest.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1996, 2000, 2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Fri Nov  1 05:37:55 1996 [mkojo]

  Testing those gen- prefixed files.

  */
#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshtimemeasure.h"
#include "readfile.h"
#include "sshmp.h"
#include "sshdsprintf.h"
#include "sshcryptocore/namelist.h"
#include "t-gentest.h"

/**************************** PKCS tests *******************************/

#ifdef SSHDIST_CRYPT_GENPKCS

void my_progress_func(SshCryptoProgressID id,
                      unsigned int time_value, void *context)
{
  if (verbose)
    {
      switch (id)
        {
        case SSH_CRYPTO_PRIME_SEARCH:
          printf("\rPrime search: %c", "-\\|/"[time_value%4]);
          fflush(stdout);
          break;
        default:
          printf("\rOperation %d: %dth value.", id, time_value);
          fflush(stdout);
          break;
        }
    }
}

typedef struct
{
  char *key_type;
  char *sign; char *encrypt; char *dh;
  char *predefined;
  char *passphrase;
  char *cipher;
  int   size;
  int   entropy;
} PkcsInfo;

PkcsInfo pkcs_info[] =
{
  { "if-modn", "rsa-pkcs1-sha1", "rsa-pkcs1v2-oaep", NULL, NULL,
    "hello", "rc2-cbc", 1024, 0 },
  { "if-modn", "rsa-pkcs1-md5", "rsa-pkcs1-none", NULL, NULL,
    "hello", "rc2-cbc", 1024, 0 },






  { "dl-modp", "dsa-nist-sha1", NULL, NULL, NULL,
    "hello", "rc2-cbc", 1024, 0 },

  /* Number of DH tests. */
  { "dl-modp", NULL, NULL, "plain", NULL,
    "hello", "rc2-cbc", 1024, 160 },
  { "dl-modp", NULL, NULL, "plain", NULL,
    "hello", "rc2-cbc", 1024, 256 },

  /* The lowest should have equivalent speed to the first. */
  { "dl-modp", NULL, NULL, "plain", "ietf-ike-grp-modp-1024",
    "hello", "rc2-cbc", 1024, 160 },
  { "dl-modp", NULL, NULL, "plain", "ietf-ike-grp-modp-1024",
    "hello", "rc2-cbc", 1024, 256 },
  { "dl-modp", NULL, NULL, "plain", "ietf-ike-grp-modp-1024",
    "hello", "rc2-cbc", 1024, 512 },
  { "dl-modp", NULL, NULL, "plain", "ietf-ike-grp-modp-1024",
    "hello", "rc2-cbc", 1024, 768 },

  /* Combination. */
  { "dl-modp", "dsa-nist-sha1", NULL, "plain", NULL,
    "hello", "rc2-cbc", 1024, 0 },


















  { NULL }
};


SshPrivateKey pkcs_make_prvkey(PkcsInfo *info)
{
  char *buf, *tmp[10], *k;
  SshPrivateKey prv;
  SshCryptoStatus status;
  int i;

  ssh_dsprintf(&buf, "%s{", info->key_type);
  printf(" -- key type = %s\n", info->key_type);

  /* Generate the key. */
  i = 0;
  if (info->sign)
    {
      ssh_dsprintf(&tmp[i], "sign{%s}", info->sign);
      printf(" -- signature algorithm = %s\n", info->sign);
      i++;
    }
  if (info->encrypt)
    {
      ssh_dsprintf(&tmp[i], "encrypt{%s}", info->encrypt);
      printf(" -- encryption algorithm = %s\n", info->encrypt);
      i++;
    }
  if (info->dh)
    {
      ssh_dsprintf(&tmp[i], "dh{%s}", info->dh);
      printf(" -- Diffie-Hellman type = %s\n", info->dh);
      i++;
    }
  tmp[i] = NULL;

  for (i = 0; tmp[i]; i++)
    {
      if (i > 0)
        ssh_dsprintf(&k, "%s,%s", buf, tmp[i]);
      else
        ssh_dsprintf(&k, "%s%s", buf, tmp[i]);
      ssh_xfree(buf);
      ssh_xfree(tmp[i]);
      buf = k;
    }
  ssh_dsprintf(&k, "%s}", buf);
  ssh_xfree(buf);
  buf = k;

  printf(" -- combined name = %s\n", buf);

  if (info->predefined)
    {
      printf(" -- predefined parameters = %s\n", info->predefined);
      if (info->entropy)
        {
          printf(" -- randomizer entropy = %u bits\n", info->entropy);

          printf(" -- generating...\n");
          status =
            ssh_private_key_generate(&prv,
                                     buf,
                                     SSH_PKF_PREDEFINED_GROUP,
                                     info->predefined,
                                     SSH_PKF_RANDOMIZER_ENTROPY,
                                     info->entropy,
                                     SSH_PKF_END);
          goto end;
        }
      else
        {
          printf(" -- generating...\n");
          status =
            ssh_private_key_generate(&prv,
                                     buf,
                                     SSH_PKF_PREDEFINED_GROUP,
                                     info->predefined,
                                     SSH_PKF_END);
          goto end;
        }
    }
  else
    {
      printf(" -- private key size = %u bits\n", info->size);

      if (info->entropy)
        {
          printf(" -- randomizer entropy = %u bits\n", info->entropy);

          printf(" -- searching...\n");
          status =
            ssh_private_key_generate(&prv,
                                     buf,
                                     SSH_PKF_SIZE, info->size,
                                     SSH_PKF_RANDOMIZER_ENTROPY,
                                     info->entropy,
                                     SSH_PKF_END);
          printf("\n");
          goto end;
        }
      else
        {
          printf(" -- searching...\n");
          status =
            ssh_private_key_generate(&prv,
                                     buf,
                                     SSH_PKF_SIZE, info->size,
                                     SSH_PKF_END);
          printf("\n");
          goto end;
        }
    }

end:

  ssh_xfree(buf);

  if (status != SSH_CRYPTO_OK)
    return NULL;

  return prv;
}

void pkcs_tests_export_import(SshPrivateKey private_key,
                              SshPublicKey public_key,
                              PkcsInfo *info)
{
  /* Test pointers. */
  unsigned char *a, *b;
  size_t a_len, b_len;
  unsigned char *signature, *signature2;
  size_t signature_len, signature_len_out;


  /* Export and import tests. */

  /* Sign a bit of data with the generated key. After the key has
     been imported/exported, verify the signature with the new
     keys to see if import/export really work (the keys remain the
     same, not just return proper values) .*/

  if (info->sign)
    {
      signature_len = ssh_private_key_max_signature_output_len(private_key);
      signature = ssh_xmalloc(signature_len);
      ssh_private_key_sign(private_key,
                           (unsigned char *) info->passphrase,
                           strlen(info->passphrase),
                           signature, signature_len, &signature_len_out);
      if (!ssh_public_key_verify_signature(public_key,
                                           signature, signature_len_out,
                                           (unsigned char *) info->passphrase,
                                           strlen(info->passphrase)))
        ssh_fatal("error: can not verify signature for import/export test");
    }

  if (ssh_public_key_export(public_key, &a, &a_len) != SSH_CRYPTO_OK)
    ssh_fatal("error: public key %s export failed.", info->key_type);

  if (verbose) printf(" -- Public key exported.\n");

  if (ssh_private_key_export_with_passphrase(private_key,
                                             info->cipher,
                                             info->passphrase,
                                             &b, &b_len) != SSH_CRYPTO_OK)
    ssh_fatal("error: private key %s export failed.", info->key_type);

  if (verbose) printf(" -- Private key exported with passphrase.\n");

  if (ssh_public_key_import(a, a_len, &public_key) != SSH_CRYPTO_OK)
    ssh_fatal("error: public key %s import failed.", info->key_type);

  if (verbose) printf(" -- Public key imported.\n");

  if (ssh_private_key_import_with_passphrase(b,
                                             b_len,
                                             info->passphrase,
                                             &private_key)
      != SSH_CRYPTO_OK)
    ssh_fatal("error: private key %s import failed.", info->key_type);

  if (verbose) printf(" -- Private key imported with passphrase.\n");

  ssh_xfree(a);
  ssh_xfree(b);

  if (info->sign)
    {
      if (!ssh_public_key_verify_signature(public_key,
                                           signature, signature_len_out,
                                           (unsigned char *) info->passphrase,
                                           strlen(info->passphrase)))
        ssh_fatal("error: can not verify signature using imported public key");

      signature2 = ssh_xmalloc(signature_len);
      ssh_private_key_sign(private_key,
                           (unsigned char *) info->passphrase,
                           strlen(info->passphrase),
                           signature2, signature_len, &signature_len_out);
      if (!ssh_public_key_verify_signature(public_key,
                                           signature2, signature_len_out,
                                           (unsigned char *) info->passphrase,
                                           strlen(info->passphrase)))
        ssh_fatal("error: can not verify signature "
                  "made using imported private key with imported public key");

      ssh_xfree(signature2);
      ssh_xfree(signature);
    }

  ssh_public_key_free(public_key);
  ssh_private_key_free(private_key);
}

void pkcs_tests(void)
{
  struct SshTimeMeasureRec tmit = SSH_TIME_MEASURE_INITIALIZER;
  int i, info_index, cnt, total;
  SshPrivateKey private_key;
  SshPublicKey  public_key;

  SshPkGroupDHSecret secret_one, secret_two;
  SshPkGroup pk_group_one, pk_group_two;

  /* Test pointers. */
  unsigned char *a, *b, *c, *d;
  size_t a_len, b_len, c_len, d_len, len;


  /* Register a progress monitoring function. */
  ssh_crypto_library_register_progress_func(my_progress_func, NULL);
  for (info_index = 0; pkcs_info[info_index].key_type; info_index++)
    {
      PkcsInfo *info = &pkcs_info[info_index];

      printf("\n");
      tstart(&tmit, "generating key type = %s.", info->key_type);
      private_key = pkcs_make_prvkey(info);
      tstop(&tmit, "key type = %s generated.", info->key_type);
      if (private_key == NULL)
        ssh_fatal("error: key generation failed.");

      tstart(&tmit, "deriving public key.");
      public_key = ssh_private_key_derive_public_key(private_key);
      tstop(&tmit, "public key derived.");
      if (public_key == NULL)
        ssh_fatal("error: public key derivation failed.");

      tstart(&tmit, "export/import tests.");
      pkcs_tests_export_import(private_key, public_key, info);
      tstop(&tmit, "export/import tests finished.");

      tstart(&tmit, "precomputation test (private key).");
      ssh_private_key_precompute(private_key);
      tstop(&tmit, "precomputation test (private key).");

      tstart(&tmit, "precomputation test (public key).");
      ssh_public_key_precompute(public_key);
      tstop(&tmit, "precomputation test (public key).");

      /* Encryption tests. */

      a_len = ssh_public_key_max_encrypt_input_len(public_key);
      if (a_len != 0)
        {
          b_len = ssh_public_key_max_encrypt_output_len(public_key);

          if (a_len == -1)
            a_len = 1024;
          if (b_len == -1)
            b_len = a_len;

          a = ssh_xmalloc(a_len);
          b = ssh_xmalloc(b_len);

          for (i = 0; i < a_len; i++)
            {
              a[i] = i & 0xff;
            }

          cnt = PKCS_CNT;
          total = 0;
        retry1:

          tstartn(&tmit, total, "encryption test.");

          for (i = 0; i < cnt; i++)
            {
              if (ssh_public_key_encrypt(public_key, a, a_len, b, b_len,
                                         &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: pkcs %s encryption error.", info->key_type);
            }

          if (tstopn(&tmit, total + cnt, "encryption test."))
            {
              total += cnt;
              cnt    = (cnt + cnt/2);
              goto retry1;
            }

          if (len > b_len)
            ssh_fatal("error: pkcs %s outputed ciphertext too long.",
                      info->key_type);

          if (len > ssh_private_key_max_decrypt_input_len(private_key))
            ssh_fatal("error: pkcs %s ciphertext length incompatible.",
                      info->key_type);

          c_len = ssh_private_key_max_decrypt_output_len(private_key);
          if (c_len == -1)
            c_len = b_len;
          c = ssh_xmalloc(c_len);

          cnt = PKCS_CNT;
          total = 0;

        retry2:

          tstartn(&tmit, total, "decryption test.");

          for (i = 0; i < cnt; i++)
            {
              if (ssh_private_key_decrypt(private_key,
                                          b, b_len, c,
                                          c_len, &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: pkcs %s decryption error.", info->key_type);

            }

          if (tstopn(&tmit, cnt + total, "decryption test."))
            {
              total += cnt;
              cnt    = (cnt + cnt/2);
              goto retry2;
            }

          if (len > c_len)
            ssh_fatal("error: pkcs %s outputed plaintext too long.",
                      info->key_type);

          if (len != a_len)
            ssh_fatal("error: pkcs %s plaintext length incompatible.",
                      info->key_type);

          ssh_xfree(b);
          ssh_xfree(a);

          c_len = len;

          for (i = 0; i < c_len; i++)
            {
              if (c[i] != (i & 0xff))
                {
                  ssh_fatal("error: pkcs %s decryption failed.", info->key_type);
                }
            }
          ssh_xfree(c);
        }

      /* Signature tests. */

      /* Randomizers! */

      a_len = ssh_private_key_max_signature_input_len(private_key);
      if (a_len != 0)
        {
          b_len = ssh_private_key_max_signature_output_len(private_key);

          if (a_len == -1)
            a_len = 1024;
          if (b_len == -1)
            b_len = a_len;

          a = ssh_xmalloc(a_len);
          b = ssh_xmalloc(b_len);

          for (i = 0; i < a_len; i++)
            {
              a[i] = i & 0xf;
            }

          cnt = PKCS_CNT;
          total = 0;

        retry3:

          tstartn(&tmit, total, "signature test.");

          for (i = 0; i < cnt; i++)
            {
              if (ssh_private_key_sign(private_key, a, a_len,
                                       b, b_len, &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: pkcs %s sign error.", info->key_type);
            }

          if (tstopn(&tmit, total + cnt, "signature test."))
            {
              total += cnt;
              cnt    = (cnt + cnt/2);
              goto retry3;
            }

          if (len > b_len)
            ssh_fatal("error: pkcs %s outputed signature too long.",
                      info->key_type);

          cnt = PKCS_CNT;
          total = 0;

        retry4:

          tstartn(&tmit, total, "verification test.");

          for (i = 0; i < cnt; i++)
            {
              if (ssh_public_key_verify_signature(public_key,
                                                  b, len,
                                                  a, a_len) == FALSE)
                ssh_fatal("error: %s signature not correct.", info->key_type);
            }

          if (tstopn(&tmit, total + cnt, "verification test."))
            {
              total += cnt;
              cnt    = (cnt + cnt/2);
              goto retry4;
            }

          ssh_xfree(a);
          ssh_xfree(b);
        }

      pk_group_one = ssh_public_key_derive_pk_group(public_key);
      pk_group_two = ssh_private_key_derive_pk_group(private_key);

      if (pk_group_one && pk_group_two)
        {
          a_len =
            ssh_pk_group_dh_setup_max_output_length(pk_group_one);
          b_len =
            ssh_pk_group_dh_setup_max_output_length(pk_group_two);

          /* Not capable for diffie hellman. */
          if (a_len == 0 || b_len == 0)
            {
              printf("Method not capable of performing Diffie-Hellman.\n");
              goto end_dh;
            }

          a = ssh_xmalloc(a_len);
          b = ssh_xmalloc(b_len);


          c_len =
            ssh_pk_group_dh_agree_max_output_length(pk_group_one);
          d_len =
            ssh_pk_group_dh_agree_max_output_length(pk_group_two);

          if (c_len == 0 || d_len == 0)
            ssh_fatal("error: could not continue to agree.");

          c = ssh_xmalloc(c_len);
          d = ssh_xmalloc(d_len);

          secret_one = NULL;

          cnt = PKCS_CNT;
          total = 0;

          if (ssh_pk_group_dh_setup(pk_group_two,
                                                &secret_two,
                                                b, b_len,
                                                &len) != SSH_CRYPTO_OK)
            ssh_fatal("error: could not do Diffie-Hellman setup. (2)");

        retry_dh1:

          tstartn(&tmit, total, "Diffie-Hellman test (doing both setup and agree).");

          for (i = 0; i < cnt; i++)
            {
              if (ssh_pk_group_dh_setup(pk_group_one,
                                                    &secret_one,
                                                    a, a_len,
                                                    &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: could not do Diffie-Hellman setup. (1)");
              if (len != a_len)
                ssh_fatal("error: len != a_len!");

              if (ssh_pk_group_dh_agree(pk_group_one,
                                                    secret_one,
                                                    b, b_len,
                                                    c, c_len,
                                                    &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: could not do Diffie-Hellman agree. (1)");

              if (len != c_len)
                ssh_fatal("error: size mismatch in Diffie-Hellman.\n");
            }

          if (tstopn(&tmit, total + cnt, "Diffie-Hellman test."))
            {
              total += cnt;
              cnt    = (cnt + cnt/2);
              goto retry_dh1;
            }

          if (ssh_pk_group_dh_agree(pk_group_two,
                                                secret_two,
                                                a, a_len,
                                                d, d_len,
                                                &len) != SSH_CRYPTO_OK)
            ssh_fatal("error: could not do Diffie-Hellman agree. (2)");

          if (d_len != len)
            ssh_fatal("error: size mismatch.\n");

          if (d_len != c_len)
            ssh_fatal("error: not correct agreement.\n");

          if (memcmp(d, c, d_len) != 0)
            ssh_fatal("error: incorrect result.\n");

          ssh_xfree(a);
          ssh_xfree(b);
          ssh_xfree(c);
          ssh_xfree(d);

        end_dh:
          ssh_pk_group_free(pk_group_one);
          ssh_pk_group_free(pk_group_two);
        }

      /* Free contexts. */

      ssh_public_key_free(public_key);
      ssh_private_key_free(private_key);
    }
}

void pkcs_random_tests(void)
{
  char *namelist = ssh_public_key_get_supported();
  const char *tmp_namelist = namelist;
  char *pkcs_name = NULL;
  int i, cnt;
  size_t len;
  static int use_randomizers = 1;
  int size = 1024;
  char *passphrase = "ssh-communications-security-finland-passphrase";
  char *cipher_name = "blowfish-cfb";

  /* Test pointers. */
  unsigned char *a, *b, *c, *d;
  size_t a_len, b_len, c_len, d_len;
  struct SshTimeMeasureRec tmit = SSH_TIME_MEASURE_INITIALIZER;
  unsigned char *signature, *signature2;
  size_t signature_len, signature_len_out;

  SshPublicKey public_key;
  SshPrivateKey private_key;

  SshPkGroupDHSecret secret_one, secret_two;
  SshPkGroup pk_group_one, pk_group_two;

  /* Register a progress monitoring function. */
  ssh_crypto_library_register_progress_func(my_progress_func,
                                            NULL);

  while ((pkcs_name = ssh_name_list_get_name(tmp_namelist)) != NULL)
    {
      /* Allocation of the private key. */

      printf("Public key method %s in testing.\n", pkcs_name);

      tstart(&tmit, "key generation (private).");

      if (memcmp(pkcs_name, "e", 1) == 0)
        {
          tmp_namelist = ssh_name_list_step_forward(tmp_namelist);
          ssh_free(pkcs_name);
          continue;
        }

      if (memcmp(pkcs_name, (const unsigned char *)"dl-", 3) == 0)
        {
          if (ssh_private_key_generate(&private_key,
                                       pkcs_name,
                                       SSH_PKF_SIZE, size,
                                       SSH_PKF_RANDOMIZER_ENTROPY, 160,
                                       SSH_PKF_END) != SSH_CRYPTO_OK)
            ssh_fatal("error: pkcs %s generate keys failed.", pkcs_name);
        }
      else if (memcmp(pkcs_name, (const unsigned char *)"if-", 3) == 0)
        {
          if (ssh_private_key_generate(&private_key,
                                       pkcs_name,
                                       SSH_PKF_SIZE, size,
                                       SSH_PKF_END) != SSH_CRYPTO_OK)
            ssh_fatal("error: pkcs %s generate keys failed.", pkcs_name);
        }























      else
        ssh_fatal("error: pkcs %s key type did not match.", pkcs_name);

      tstop(&tmit, "key generation (private).");

      tstart(&tmit, "key derivation (public)");
      public_key = ssh_private_key_derive_public_key(private_key);
      tstop(&tmit, "key derivation (public)");

      {
        char *type;
        size_t size;
        SshMPIntC p, q;
        int status;

        ssh_mp_init(p);
        ssh_mp_init(q);

        status = ssh_private_key_get_info(private_key,
                                          SSH_PKF_KEY_TYPE, &type,
                                          SSH_PKF_SIZE, &size,
                                          SSH_PKF_PRIME_Q, q,
                                          SSH_PKF_END);

        if (status) ssh_fatal("can not get private key info");

        status = ssh_public_key_get_info(public_key,
                                         SSH_PKF_KEY_TYPE, &type,
                                         SSH_PKF_SIZE, &size,
                                         SSH_PKF_END);
        if (status) ssh_fatal("can not get public key info");

        ssh_mp_clear(p);
        ssh_mp_clear(q);
      }


      /* Test different signature schemes */
      if (ssh_private_key_max_signature_input_len(private_key) != 0)
        {
          char *p, *q, *r;
          SshCryptoStatus status;
          unsigned char *data_buf;
          size_t data_buf_len;

          p = pkcs_name;
          while ((p = strchr(p, '{')) != NULL)
            {
              if (strncmp(p - 4, "sign{", 5) == 0)
                break;
              p++;
            }

          if (p == NULL)
            ssh_fatal("No signature algorithms found");

          r = strchr(p, '}');
          if (r == NULL)
            ssh_fatal("Invalid name format no '}' found");

          p++;
          q = ssh_xmemdup(p, r - p);

          p = q;
          do {
            r = strchr(p, ',');
            if (r != NULL)
              *r++ = '\0';
            else
              r = NULL;

            status = ssh_public_key_select_scheme(public_key,
                                                  SSH_PKF_SIGN, p,
                                                  SSH_PKF_END);
            if (status)
              ssh_fatal("ssh_public_key_select_scheme failed");

            status = ssh_private_key_select_scheme(private_key,
                                                   SSH_PKF_SIGN, p,
                                                   SSH_PKF_END);
            if (status)
              ssh_fatal("ssh_private_key_select_scheme failed");

            signature_len =
              ssh_private_key_max_signature_output_len(private_key);
            signature = ssh_xmalloc(signature_len);
            data_buf_len =
              ssh_private_key_max_signature_input_len(private_key);
            if (data_buf_len == -1)
              data_buf_len = 1024;
            data_buf = ssh_xmalloc(data_buf_len);
            for(i = 0; i < data_buf_len; i++)
              data_buf[i] = ssh_random_get_byte();

            tstart(&tmit, "sign & verify: %s", p);
            if (ssh_private_key_max_signature_input_len(private_key) != -1)
              {
                ssh_private_key_sign_digest(private_key,
                                            data_buf, data_buf_len,
                                            signature, signature_len,
                                            &signature_len_out);
                if (!ssh_public_key_verify_signature_with_digest(public_key,
                                                                 signature,
                                                                 signature_len_out,
                                                                 data_buf,
                                                                 data_buf_len))
                  ssh_fatal("error: can not verify signature for scheme test");
              }
            else
              {
                ssh_private_key_sign(private_key,
                                     data_buf, data_buf_len,
                                     signature, signature_len,
                                     &signature_len_out);
                if (!ssh_public_key_verify_signature(public_key,
                                                     signature,
                                                     signature_len_out,
                                                     data_buf,
                                                     data_buf_len))
                  ssh_fatal("error: can not verify signature for scheme test");
              }
            tstop(&tmit, "sign & verify: %s", p);

            ssh_xfree(data_buf);
            ssh_xfree(signature);
            p = r;
          } while (p);

          status = ssh_public_key_select_scheme(public_key,
                                                SSH_PKF_SIGN, q,
                                                SSH_PKF_END);
          if (status)
            ssh_fatal("ssh_public_key_select_scheme failed");

          status = ssh_private_key_select_scheme(private_key,
                                                 SSH_PKF_SIGN, q,
                                                 SSH_PKF_END);
          if (status)
            ssh_fatal("ssh_private_key_select_scheme failed");
          ssh_xfree(q);
        }

      /* Test different encryption schemes */
      if (ssh_private_key_max_decrypt_input_len(private_key) != 0)
        {
          char *p, *q, *r;
          SshCryptoStatus status;
          unsigned char *data_buf, *data_buf2, *encryption_buf;
          size_t data_buf_len, data_buf2_len, encryption_buf_len, return_val;

          p = pkcs_name;
          while ((p = strchr(p, '{')) != NULL)
            {
              if (strncmp(p - 7, "encrypt{", 8) == 0)
                break;
              p++;
            }

          if (p == NULL)
            ssh_fatal("No encryption algorithms found");

          r = strchr(p, '}');
          if (r == NULL)
            ssh_fatal("Invalid name format no '}' found");

          p++;
          q = ssh_xmemdup(p, r - p);

          p = q;
          do {
            r = strchr(p, ',');
            if (r != NULL)
              *r++ = '\0';
            else
              r = NULL;

            status = ssh_public_key_select_scheme(public_key,
                                                  SSH_PKF_ENCRYPT, p,
                                                  SSH_PKF_END);
            if (status)
              ssh_fatal("ssh_public_key_select_scheme failed");

            status = ssh_private_key_select_scheme(private_key,
                                                   SSH_PKF_ENCRYPT, p,
                                                   SSH_PKF_END);
            if (status)
              ssh_fatal("ssh_private_key_select_scheme failed");

            encryption_buf_len =
              ssh_public_key_max_encrypt_output_len(public_key);
            encryption_buf = ssh_xmalloc(encryption_buf_len);

            data_buf_len =
              ssh_public_key_max_encrypt_input_len(public_key);
            data_buf = ssh_xmalloc(data_buf_len);

            data_buf2_len =
              ssh_private_key_max_decrypt_output_len(private_key);
            data_buf2 = ssh_xmalloc(data_buf2_len);

            data_buf[0] = 0;
            for(i = 1; i < data_buf_len; i++)
              data_buf[i] = ssh_random_get_byte();

            tstart(&tmit, "encrypt & decrypt (public): %s", p);
            status = ssh_public_key_encrypt(public_key,
                                            data_buf, data_buf_len,
                                            encryption_buf, encryption_buf_len,
                                            &return_val);
            if (status)
              ssh_fatal("ssh_public_key_encrypt failed");
            if (return_val != encryption_buf_len)
              ssh_fatal("Invalid length returned from the ssh_public_key_encrypt");

            status = ssh_private_key_decrypt(private_key,
                                             encryption_buf,
                                             encryption_buf_len,
                                             data_buf2,
                                             data_buf2_len,
                                             &return_val);
            tstop(&tmit, "encrypt & decrypt (public): %s", p);

            if (status)
              ssh_fatal("ssh_private_key_decrypt failed");
            if (return_val != data_buf_len)
              ssh_fatal("Invalid length returned from the ssh_private_key_decrypt");

            if (memcmp(data_buf, data_buf2, data_buf_len) != 0)
              ssh_fatal("Encryption failed");

            tstart(&tmit, "encrypt & decrypt (private): %s", p);
            status = ssh_private_key_encrypt(private_key,
                                             data_buf, data_buf_len,
                                             encryption_buf,
                                             encryption_buf_len,
                                             &return_val);
            if (status == SSH_CRYPTO_OK)
              {
                if (return_val != encryption_buf_len)
                  ssh_fatal("Invalid length returned from the ssh_private_key_encrypt");

                status = ssh_public_key_decrypt(public_key,
                                                encryption_buf,
                                                encryption_buf_len,
                                                data_buf2,
                                                data_buf2_len,
                                                &return_val);
                if (status)
                  ssh_fatal("ssh_public_key_decrypt failed");
                if (return_val != data_buf_len)
                  ssh_fatal("Invalid length returned from the ssh_public_key_decrypt");

                if (memcmp(data_buf, data_buf2, data_buf_len) != 0)
                  ssh_fatal("Encryption failed");
              }
            else if (status == SSH_CRYPTO_UNSUPPORTED)
              {
                if (verbose)
                  printf("Scheme %s does not support "
                         "private key encrypt and public key decrypt "
                         "operations, ignored\n", p);
              }
            else
              ssh_fatal("ssh_private_key_encrypt failed");

            tstop(&tmit, "encrypt & decrypt (private): %s", p);

            ssh_xfree(data_buf);
            ssh_xfree(data_buf2);
            ssh_xfree(encryption_buf);
            p = r;
          } while (p);

          status = ssh_public_key_select_scheme(public_key,
                                                SSH_PKF_ENCRYPT, q,
                                                SSH_PKF_END);
          if (status)
            ssh_fatal("ssh_public_key_select_scheme failed");

          status = ssh_private_key_select_scheme(private_key,
                                                 SSH_PKF_ENCRYPT, q,
                                                 SSH_PKF_END);
          if (status)
            ssh_fatal("ssh_private_key_select_scheme failed");
          ssh_xfree(q);
        }


      /* Export and import tests. */

      /* Sign a bit of data with the generated key. After the key has
         been imported/exported, verify the signature with the new
         keys to see if import/export really work (the keys remain the
         same, not just return proper values) .*/

      signature_len = ssh_private_key_max_signature_output_len(private_key);
      signature = ssh_xmalloc(signature_len);
      ssh_private_key_sign(private_key,
                           (unsigned char *) passphrase, strlen(passphrase),
                           signature, signature_len, &signature_len_out);
      if (!ssh_public_key_verify_signature(public_key,
                                           signature, signature_len_out,
                                           (unsigned char *) passphrase,
                                           strlen(passphrase)))
        ssh_fatal("error: can not verify signature for import/export test");


      if (ssh_public_key_export(public_key, &a, &a_len) != SSH_CRYPTO_OK)
        ssh_fatal("error: public key %s export failed.", pkcs_name);

      if (verbose) printf("Public key exported.\n");

      if (ssh_private_key_export_with_passphrase(private_key,
                                                 cipher_name,
                                                 passphrase,
                                                 &b, &b_len) != SSH_CRYPTO_OK)
        ssh_fatal("error: private key %s export failed.", pkcs_name);

      if (verbose) printf("Private key exported with passphrase.\n");

      ssh_public_key_free(public_key);
      ssh_private_key_free(private_key);

      if (ssh_public_key_import(a, a_len, &public_key) != SSH_CRYPTO_OK)
        ssh_fatal("error: public key %s import failed.", pkcs_name);

      if (verbose) printf("Public key imported.\n");

      if (ssh_private_key_import_with_passphrase(b,
                                                 b_len,
                                                 passphrase,
                                                 &private_key)
          != SSH_CRYPTO_OK)
        ssh_fatal("error: private key %s import failed.", pkcs_name);

      if (verbose) printf("Private key imported with passphrase.\n");

      ssh_xfree(a);
      ssh_xfree(b);

      if (!ssh_public_key_verify_signature(public_key,
                                           signature, signature_len_out,
                                           (unsigned char *) passphrase,
                                           strlen(passphrase)))
        ssh_fatal("error: can not verify signature using imported public key");

      signature2 = ssh_xmalloc(signature_len);
      ssh_private_key_sign(private_key,
                           (unsigned char *) passphrase, strlen(passphrase),
                           signature2, signature_len, &signature_len_out);
      if (!ssh_public_key_verify_signature(public_key,
                                           signature2, signature_len_out,
                                           (unsigned char *) passphrase,
                                           strlen(passphrase)))
        ssh_fatal("error: can not verify signature made using imported private key with imported public key");

      ssh_xfree(signature2);
      ssh_xfree(signature);


      /* Testing of public key groups. */

      pk_group_one = ssh_public_key_derive_pk_group(public_key);
      if (pk_group_one)
        {
          if (verbose) printf("Testing public key group import/export. \n");

          if (ssh_pk_group_export(pk_group_one,
                                  &a, &a_len) != SSH_CRYPTO_OK)
            ssh_fatal("error: cannot export public key group.");

          ssh_pk_group_free(pk_group_one);

          if (ssh_pk_group_import(a, a_len,
                                  &pk_group_one) != SSH_CRYPTO_OK)
            ssh_fatal("error: cannot import public key group.");

          ssh_xfree(a);
          if (verbose) printf("Import/export runned.\n");

          /* Randomizers. */

          if (use_randomizers)
            {
              if (verbose) printf("Generating randomizers.\n");
              for (i = 0; i < 20; i++)
                ssh_pk_group_generate_randomizer(pk_group_one);

              if (verbose) printf("Exporting randomizers.\n");

              if (ssh_pk_group_count_randomizers(pk_group_one) != 20)
                ssh_fatal("error: generated incorrect amount of randomizers.");

              if (ssh_pk_group_export_randomizers(pk_group_one,
                                                  &a, &a_len) != SSH_CRYPTO_OK)
                ssh_fatal("error: cannot export randomizers.");

              if (ssh_pk_group_import_randomizers(pk_group_one,
                                                  a, a_len)
                  != SSH_CRYPTO_OK)
                ssh_fatal("error: cannot import randomizers.");

              ssh_xfree(a);
            }

          ssh_pk_group_free(pk_group_one);
        }

      /* Do precomputation. */

      if (verbose) printf("Precomputation tests.\n");

      tstart(&tmit, "precomputation (group 1)");
      ssh_private_key_precompute(private_key);
      ssh_public_key_precompute(public_key);
      tstop(&tmit, "precomputation (group 1)");

      /* Encryption tests. */

      if (verbose) printf("Encryption tests.\n");

      a_len = ssh_public_key_max_encrypt_input_len(public_key);
      if (a_len != 0)
        {
          b_len = ssh_public_key_max_encrypt_output_len(public_key);

          if (a_len == -1)
            a_len = 1024;
          if (b_len == -1)
            b_len = a_len;

          a = ssh_xmalloc(a_len);
          b = ssh_xmalloc(b_len);

          for (i = 0; i < a_len; i++)
            {
              a[i] = i & 0xff;
            }

          cnt = PKCS_CNT;

        retry1:

          ssh_time_measure_reset(&tmit);
          ssh_time_measure_start(&tmit);

          for (i = 0; i < cnt; i++)
            {
              if (ssh_public_key_encrypt(public_key, a, a_len, b, b_len,
                                         &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: pkcs %s encryption error.", pkcs_name);
            }

          ssh_time_measure_stop(&tmit);

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) <= TEST_TIME_MIN
              && cnt < 100000)
            {
              if (ssh_time_measure_get(&tmit,
                                       SSH_TIME_GRANULARITY_MILLISECOND) < 10)
                cnt *= 128;
              else
                cnt = (int) (cnt * TEST_TIME_OPTIMUM
                             / ssh_time_measure_get(
                                        &tmit,
                                        SSH_TIME_GRANULARITY_MILLISECOND));
              if (verbose)
                printf("  - %s encrypt was too fast, retrying...\n", pkcs_name);
              goto retry1;
            }

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) >= TEST_TIME_MIN)
            printf("%s -- " TEST_FMT " times/sec (" TEST_FMT " ms/encrypt).\n",
                   pkcs_name, ((double) cnt) /
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0),
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0) / (double) cnt * 1000);
          else
            printf("  - timing could not be performed for %s.\n", pkcs_name);

          printf("Encrypted with public key.\n");

          if (len > b_len)
            ssh_fatal("error: pkcs %s outputed ciphertext too long.",
                      pkcs_name);

          if (len > ssh_private_key_max_decrypt_input_len(private_key))
            ssh_fatal("error: pkcs %s ciphertext length incompatible.",
                      pkcs_name);

          c_len = ssh_private_key_max_decrypt_output_len(private_key);
          if (c_len == -1)
            c_len = b_len;
          c = ssh_xmalloc(c_len);

          cnt = PKCS_CNT;

        retry2:

          ssh_time_measure_reset(&tmit);
          ssh_time_measure_start(&tmit);

          for (i = 0; i < cnt; i++)
            {
              if (ssh_private_key_decrypt(private_key,
                                          b, b_len, c,
                                          c_len, &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: pkcs %s decryption error.", pkcs_name);

            }

          ssh_time_measure_stop(&tmit);

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) <= TEST_TIME_MIN && cnt < 100000)
            {
              if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_MILLISECOND) < 10)
                cnt *= 128;
              else
                cnt = (int) (cnt * TEST_TIME_OPTIMUM
                             / ssh_time_measure_get(
                                        &tmit,
                                        SSH_TIME_GRANULARITY_MILLISECOND));
              if (verbose)
                printf("  - %s decrypt was too fast, retrying...\n", pkcs_name);
              goto retry2;
            }

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) >= TEST_TIME_MIN)
            printf("%s -- " TEST_FMT " times/sec (" TEST_FMT " ms/decrypt).\n",
                   pkcs_name, ((double)cnt) /
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0),
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0) / (double) cnt * 1000);
          else
            printf("  - timing could not be performed for %s.\n", pkcs_name);

          printf("Decrypted with the private key.\n");

          if (len > c_len)
            ssh_fatal("error: pkcs %s outputed plaintext too long.",
                      pkcs_name);

          if (len != a_len)
            ssh_fatal("error: pkcs %s plaintext length incompatible.",
                      pkcs_name);

          ssh_xfree(b);
          ssh_xfree(a);

          c_len = len;

          for (i = 0; i < c_len; i++)
            {
              if (c[i] != (i & 0xff))
                {
                  ssh_fatal("error: pkcs %s decryption failed.", pkcs_name);
                }
            }
          ssh_xfree(c);
        }
      else
        {
          printf("Method not capable for encryption.\n");
        }

      /* Signature tests. */

      printf("Signature tests.\n");

      /* Randomizers! */

      a_len = ssh_private_key_max_signature_input_len(private_key);
      if (a_len != 0)
        {
          b_len = ssh_private_key_max_signature_output_len(private_key);

          if (a_len == -1)
            a_len = 1024;
          if (b_len == -1)
            b_len = a_len;

          a = ssh_xmalloc(a_len);
          b = ssh_xmalloc(b_len);

          for (i = 0; i < a_len; i++)
            {
              a[i] = i & 0xf;
            }

          cnt = PKCS_CNT;

        retry3:

          ssh_time_measure_reset(&tmit);
          ssh_time_measure_start(&tmit);

          for (i = 0; i < cnt; i++)
            {
              if (ssh_private_key_sign(private_key, a, a_len,
                                       b, b_len, &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: pkcs %s sign error.", pkcs_name);
            }

          ssh_time_measure_stop(&tmit);

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) <= TEST_TIME_MIN && cnt < 100000)
            {
              if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_MILLISECOND) < 10)
                cnt *= 128;
              else
                cnt = (int) (cnt * TEST_TIME_OPTIMUM
                             / ssh_time_measure_get(
                                        &tmit,
                                        SSH_TIME_GRANULARITY_MILLISECOND));
              if (verbose)
                printf("  - %s signing was too fast, retrying...\n", pkcs_name);
              goto retry3;
            }

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) >= TEST_TIME_MIN)
            printf("%s -- " TEST_FMT " times/sec (" TEST_FMT " ms/sign).\n", pkcs_name,
                   ((double)cnt) /
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0),
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0) / (double) cnt * 1000);
          else
            printf("  - timing could not be performed for %s.\n", pkcs_name);

          printf("Signed with the private key.\n");

          if (len > b_len)
            ssh_fatal("error: pkcs %s outputed signature too long.",
                      pkcs_name);

          cnt = PKCS_CNT;

        retry4:

          ssh_time_measure_reset(&tmit);
          ssh_time_measure_start(&tmit);

          for (i = 0; i < cnt; i++)
            {
              if (ssh_public_key_verify_signature(public_key,
                                                  b, len,
                                                  a, a_len) == FALSE)
                ssh_fatal("error: %s signature not correct.", pkcs_name);
            }

          ssh_time_measure_stop(&tmit);

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) <= TEST_TIME_MIN && cnt < 100000)
            {
              if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_MILLISECOND) < 10)
                cnt *= 128;
              else
                cnt = (int) (cnt * TEST_TIME_OPTIMUM
                             / ssh_time_measure_get(
                                        &tmit,
                                        SSH_TIME_GRANULARITY_MILLISECOND));
              if (verbose)
                printf("  - %s signing verifying was too fast, retrying...\n", pkcs_name);
              goto retry4;
            }

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) >= TEST_TIME_MIN)
            printf("%s -- " TEST_FMT " times/sec (" TEST_FMT " ms/verify).\n",
                   pkcs_name, ((double)cnt) /
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0),
                   ((double) ssh_time_measure_get(&tmit,
                                                  SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0) / (double) cnt * 1000);
          else
            printf("  - timing could not be performed for %s.\n",
                   pkcs_name);

          printf("Verified with the public key.\n");

          ssh_xfree(a);
          ssh_xfree(b);
        }
      else
        printf("Method not capable of signing.\n");

      pk_group_one = ssh_public_key_derive_pk_group(public_key);
      pk_group_two = ssh_private_key_derive_pk_group(private_key);

      if (pk_group_one && pk_group_two)
        {
          printf("Derived groups.\n");

          a_len =
            ssh_pk_group_dh_setup_max_output_length(pk_group_one);
          b_len =
            ssh_pk_group_dh_setup_max_output_length(pk_group_two);

          /* Not capable for diffie hellman. */
          if (a_len == 0 || b_len == 0)
            {
              printf("Method not capable of performing Diffie-Hellman.\n");
              goto end_dh;
            }

          a = ssh_xmalloc(a_len);
          b = ssh_xmalloc(b_len);

          c_len =
            ssh_pk_group_dh_agree_max_output_length(pk_group_one);
          d_len =
            ssh_pk_group_dh_agree_max_output_length(pk_group_two);

          if (c_len == 0 || d_len == 0)
            ssh_fatal("error: could not continue to agree.");

          c = ssh_xmalloc(c_len);
          d = ssh_xmalloc(d_len);

          secret_one = NULL;


          if (ssh_pk_group_dh_setup(pk_group_two,
                                    &secret_two,
                                    b, b_len,
                                    &len) != SSH_CRYPTO_OK)
            ssh_fatal("error: could not do Diffie-Hellman setup. (2)");

          cnt = 10;

        retry_dh1:

          ssh_time_measure_reset(&tmit);
          ssh_time_measure_start(&tmit);

          for (i = 0; i < cnt; i++)
            {
              if (ssh_pk_group_dh_setup(pk_group_one,
                                        &secret_one,
                                        a, a_len,
                                        &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: could not do Diffie-Hellman setup. (1)");
              if (len != a_len)
                ssh_fatal("error: len != a_len!");
              if (ssh_pk_group_dh_agree(pk_group_one,
                                        secret_one,
                                        b, b_len,
                                        c, c_len,
                                        &len) != SSH_CRYPTO_OK)
                ssh_fatal("error: could not do Diffie-Hellman agree. (1)");

              if (len != c_len)
                ssh_fatal("error: minor detail.\n");
            }

          ssh_time_measure_stop(&tmit);

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND)
              <= TEST_TIME_MIN && cnt < 100000)
            {
              if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_MILLISECOND)
                  < 10)
                cnt *= 128;
              else
                cnt = (int)
                  (cnt * TEST_TIME_OPTIMUM
                   / ssh_time_measure_get(&tmit,
                                          SSH_TIME_GRANULARITY_MILLISECOND));
              if (verbose)
                printf("  - %s dh setup was too fast, retrying...\n",
                       pkcs_name);
              goto retry_dh1;
            }

          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND)
              >= TEST_TIME_MIN)
            printf("%s -- " TEST_FMT " times/s (" TEST_FMT " ms/dh setup).\n",
                   pkcs_name,
                   ((double)cnt) /
                   ((double)ssh_time_measure_get(&tmit,
                                                 SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0),
                   ((double)ssh_time_measure_get(&tmit,
                                                 SSH_TIME_GRANULARITY_MICROSECOND)
                    / 1000000.0) / (double) cnt * 1000);
          else
            printf("  - timing could not be performed for %s.\n", pkcs_name);

          if (ssh_pk_group_dh_agree(pk_group_two, secret_two,
                                    a, a_len,
                                    d, d_len,
                                    &len) != SSH_CRYPTO_OK)
            ssh_fatal("error: could not do Diffie-Hellman agree. (2)");

          if (d_len != len)
            ssh_fatal("error: minor detail.\n");

          if (d_len != c_len)
            ssh_fatal("error: not correct agreement.\n");

          if (memcmp(d, c, d_len) != 0)
            ssh_fatal("error: incorrect result.\n");

          printf("Diffie-Hellman key agreement was a success.\n");
          ssh_xfree(a);
          ssh_xfree(b);
          ssh_xfree(c);
          ssh_xfree(d);

          ssh_pk_group_free(pk_group_one);
          ssh_pk_group_free(pk_group_two);

        end_dh:
          ;                     /* OSF cc cannot compile this file if this
                                   empty statement is not here. */
        }
      else
        printf("Method not capable of extracting groups.\n");

      /* Free contexts. */

      ssh_public_key_free(public_key);
      ssh_private_key_free(private_key);

      ssh_xfree(pkcs_name);
      tmp_namelist = ssh_name_list_step_forward(tmp_namelist);

      printf("\n");
    }

  /* Remove progress monitoring functions from use. */
  ssh_crypto_library_register_progress_func(NULL_FNPTR, NULL);

  ssh_xfree(namelist);
}

/* This function runs some basic tests on the Diffie-Hellman predefined groups. The
   Diffie-Hellman computations tests are found in t-dh.c. Only testing dl-modp groups. 
   Elliptic curves groups can be tested later if anyone feels that such a test would be 
   worthwhile. */
void predefined_groups_tests(void)
{
  char *namelist = ssh_public_key_get_predefined_groups("dl-modp");
  const char *tmp_namelist = namelist;
  SshPkGroup group;
  char *group_name = NULL;
  /* Test pointers. */
  unsigned char *a;
  size_t a_len;
  int i, status, size;

  /* Register a progress monitoring function. */
  ssh_crypto_library_register_progress_func(my_progress_func,
                                            NULL);

  while ((group_name = ssh_name_list_get_name(tmp_namelist)) != NULL)
    {
      printf("Testing predefined group %s.\n", group_name);
      /* Allocation of the predefined group. */
      {
        char *type;
        SshMPIntegerStruct p, q, g;
        int ent, entropy;

        /* Put the randomizer entropy to 200. */
        ent = 200;
        status = ssh_pk_group_generate(&group, "dl-modp",
                                       SSH_PKF_PREDEFINED_GROUP, group_name,
                                       SSH_PKF_RANDOMIZER_ENTROPY, ent,
                                       SSH_PKF_END);

        if (status) ssh_fatal("can not generate the group %s.\n", group_name);

        ssh_mprz_init(&p);
        ssh_mprz_init(&q);
        ssh_mprz_init(&g);
        
        status = ssh_pk_group_get_info(group,
                                       SSH_PKF_KEY_TYPE, &type,
                                       SSH_PKF_SIZE, &size,
                                       SSH_PKF_PRIME_Q, &q,
                                       SSH_PKF_PRIME_P, &p, 
                                       SSH_PKF_GENERATOR_G, &g, 
                                       SSH_PKF_RANDOMIZER_ENTROPY, &entropy,
                                       SSH_PKF_END); 
        
        if (status) ssh_fatal("can not get the group info");

        if (group == NULL) ssh_fatal("the generated group is NULL");    
        if (size == 0) ssh_fatal("the group has zero elements");
        
        ssh_mp_clear(&p);
        ssh_mp_clear(&q);
        ssh_mp_clear(&g);
      }
      
      if (verbose) printf("Testing the group import/export. \n");
      
      if (ssh_pk_group_export(group, &a, &a_len) != SSH_CRYPTO_OK)
        ssh_fatal("error: cannot export public key group.");
      
      ssh_pk_group_free(group);
      
      if (ssh_pk_group_import(a, a_len, &group) != SSH_CRYPTO_OK)
        ssh_fatal("error: cannot import public key group.");

      ssh_xfree(a);
      if (verbose) printf("Import/export test succeeded.\n");

      /* Randomizers. */
      if (verbose) printf("Generating randomizers.\n");
      for (i = 0; i < 20; i++)
        {       
          status = ssh_pk_group_generate_randomizer(group);
          if (status) ssh_fatal("Cannot generate randomizers");
        }
      if (ssh_pk_group_count_randomizers(group) != 20)
        ssh_fatal("error: generated incorrect amount of randomizers.");
     
      if (verbose) printf("Exporting randomizers.\n");
   
      if (ssh_pk_group_export_randomizers(group, &a, &a_len) != SSH_CRYPTO_OK)
          ssh_fatal("error: cannot export randomizers.");
   
      if (verbose) printf("Importing randomizers.\n");

      if (ssh_pk_group_import_randomizers(group, a, a_len)
          != SSH_CRYPTO_OK)
        ssh_fatal("error: cannot import randomizers.");
          
      ssh_xfree(a);
      if (verbose) printf("Randomizer test succeeded.\n");

        /* Do precomputation. */
      if (verbose) printf("Precomputation tests.\n");

      status = ssh_pk_group_precompute(group);

      if (status) 
        {printf("status is %d\n" , status);
        ssh_fatal("Error in precomputation");
        }
      if (verbose) printf("Precomputation test succeeded.\n");
      
      /* Diffie-Hellman tests are done in t-dh.c, here just check we can set the scheme. */

      status = ssh_pk_group_select_scheme(group, SSH_PKF_DH, "plain", SSH_PKF_END);
      if (status) 
        ssh_fatal("Cannot set the scheme"); 

      ssh_pk_group_free(group);
      ssh_xfree(group_name);
      tmp_namelist = ssh_name_list_step_forward(tmp_namelist);
   
      /* The IKE 3072, 4096 and 8192 groups are too slow, not really necessary to test 
         anyway, since nobody should consider using them at present. */
      if (size >= 2048)
        break;
    }
  
  /* Remove progress monitoring functions from use. */
  ssh_crypto_library_register_progress_func(NULL_FNPTR, NULL);
  
  ssh_xfree(namelist);
}


void test_pkcs(int flag)
{
  printf(" - selected tests (with timing).\n");
  pkcs_tests();

  printf(" - random tests (with timing).\n");
  pkcs_random_tests();

  printf(" - Test the predefined Diffie-Hellman groups.\n");
  predefined_groups_tests();

}
#endif /* SSHDIST_CRYPT_GENPKCS */
