/*

sshproxykey.h

Author: Vesa Suontama <vsuontam@ssh.fi>
        Patrick Irwin <irwin@ssh.fi> 

Copyright (c) 1999-2002 SSH Communications Security Corp, Helsinki, Finland
              All rights reserved

  Interface for creating proxy keys (and groups) (e.g) creating keys 
  (groups) with a callback that is called when crypto-operation with a 
  key is being performed.

*/


#include "sshcrypt.h"

#ifndef SSH_PROXYKEY_H_DEFINED
#define SSH_PROXYKEY_H_DEFINED


/* A generic freeing function that must be provided to the *_create_proxy 
   functions. Its purpose is to free the context data that is given to 
   the *_create_proxy functions. 
*/
typedef void (*SshProxyFreeOpCB)(void *context);

/* A callback of this type must be called by the key operation callbacks 
   which return data to complete the asynchronous operation. If the operation 
   was successful, status is SSH_CRYPTO_OK and the operated data contains 
   the operated data. Otherwise the operated data should be set to NULL, and
   status details the reason for failure.
 */
typedef void (*SshProxyDataReplyCB)(SshCryptoStatus status,
                                    const unsigned char *operated_data,
                                    size_t data_len,
                                    void *reply_context);

/* A callback of this type must be called by the key operation callback 
   which return a Boolean value to cpmplete the asynchronous operation.
   The callback only returns an SshCryptoStatus and no data. If the 
   operation was successful, status is SSH_CRYPTO_OK. Otherwise status 
   details the reason for failure.
*/ 
typedef void (*SshProxyVerifyReplyCB)(SshCryptoStatus status,
                                      void *reply_context);

/* A callback of this type must be provided for the 
   [r,d]sa_*_key_create_proxy functions. The callback is called to 'operate' 
   on the provided data. When data is operated, the function must call 
   the reply_cb to continue the asynchronous operation. The reply callback 
   must be called with NULL data if the operation could not be completed. 

   If this callback is asynchronous it has to return operation handle, 
   which can be used to cancel the operation. 

   If rgf_name is NULL the callback does not need to pad and hash the 
   data, otherwise rgf_name informs the callback what padding (RGF) 
   and hash algorithm to apply to the data.  
*/
typedef 
SshOperationHandle (*SshProxyKeyOpCB)(const char *rgf_name,
                                      const unsigned char *data,
                                      size_t data_len,
                                      SshProxyDataReplyCB reply_cb,
                                      void *reply_context,
                                      void *context);


/* A callback of this type must be provided for the 
   [d,r]sa_public_key_create_proxy function. The callback is called 
   to verify the provided signature. When data is operated, the function 
   must call the reply_cb to continue the asynchronous operation. 

   If this callback is asynchronous it has to return operation handle, 
   which can be used to cancel the operation. This callback is used when 
   the proxykey operation does the RGF. rgf_name informs the callback 
   what padding and hash algorithm to apply to the data.  
*/
typedef 
SshOperationHandle 
(*SshProxyVerifyOpCB)(const char *rgf_name,
                      const unsigned char *data,
                      size_t data_len,
                      const unsigned char *signature,
                      size_t  signature_len,
                      SshProxyVerifyReplyCB reply_cb,
                      void *reply_context,
                      void *context);

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
/* Create a proxy RSA private key. 

   Calls the operation_cb with the data that is being operated when 
   the library is performing crypto operations with the returned proxy key. 

   The proxy private key is freed with ssh_private_key_free. It is an error 
   to free a key that is currently being used. 

   The Boolean parameter cb_does_rgf if TRUE specifies that the 
   proxykey callbacks will do all padding and hashing operations. 

   If name is supplied (i.e. not NULL), it will set the scheme 
   of the proxy key. It should the the valid name of a RSA key, i.e. 
   obtained from a call to ssh_private_key_name() for an RSA key. 
*/
SshPrivateKey 
ssh_private_rsa_key_create_proxy(Boolean cb_does_rgf,
                                 const char *name, 
                                 SshUInt32 size_in_bits,
                                 SshProxyKeyOpCB sign_op,
                                 SshProxyKeyOpCB decrypt_op,
                                 SshProxyFreeOpCB free_op,  
                                 void *context);

/* Create a proxy RSA public key. 

   Calls the operation_cb with the data that is being operated when 
   the library is performing crypto operations with the returned proxy key. 
   The proxy public key is freed with ssh_public_key_free. It is error 
   to free a key that is currently being used. 

   If cb_does_rgf is TRUE, the verify and encrypt callbacks must pad 
   and hash the data, otherwise the padding and hashing is done in 
   sshproxykey.c. The Verify Operation callback takes different parameters in 
   the two cases, so the create function needs to specify both as callbacks.  

   When cb_does_rgf is TRUE the VerifyWithRGFOpCB verify_rgf_cb is the 
   correct verify callback to use and the KeyOpCB verify_cb should be
   NULL_FNPTR. Otherwise, the VerifyWithRGFOpCB verify_with_rgf_cb should be 
   NULL_FNPTR. 
   
   If name is supplied (i.e. not NULL), it will set the scheme 
   of the proxy key. It should the the valid name of a RSA key, i.e. 
   obtained from a call to ssh_public_key_name() for an RSA key. 

*/
SshPublicKey 
ssh_public_rsa_key_create_proxy(Boolean cb_does_rgf,
                                const char *name, 
                                SshUInt32 size_in_bits,
                                SshProxyVerifyOpCB rgf_verify_op,
                                SshProxyKeyOpCB verify_op,
                                SshProxyKeyOpCB encrypt_op,
                                SshProxyFreeOpCB free_op,    
                                void *context);
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

/* Create a proxy DSA private key. 

   Calls the operation_cb with the data that is being operated when the 
   library is performing crypto operations with the returned proxy key. The 
   proxy private key is freed with ssh_private_key_free. It is an error to 
   free a key that is currently being used. 

   If cb_does_rgf is true, the sign callback must hash the data. 
   If name is supplied (i.e. not NULL), it will set the scheme 
   of the proxy key. It should the the valid name of a DSA key, i.e. 
   obtained from a call to ssh_private_key_name() for a DSA private key. 
*/
SshPrivateKey 
ssh_private_dsa_key_create_proxy(Boolean cb_does_rgf,
                                 const char *name, 
                                 SshUInt32 size_in_bits,
                                 SshProxyKeyOpCB sign_op,
                                 SshProxyFreeOpCB free_op,  
                                 void *context);

/* Create a proxy DSA public key. 

   Calls the operation_cb with the data that is being operated when 
   the library is performing crypto operations with the returned proxy key. 

   The proxy public key is freed with ssh_public_key_free. It is error 
   to free a key that is currently being used. 

   If cb_does_rgf is true, the sign callback must hash the data. 
   If name is supplied (i.e. not NULL), it will set the scheme 
   of the proxy key. It should the the valid name of a DSA key, i.e. 
   obtained from a call to ssh_public_key_name() for a DSA public
   key. 
*/
SshPublicKey 
ssh_public_dsa_key_create_proxy(Boolean cb_does_rgf,
                                const char *name, 
                                SshUInt32 size_in_bits,
                                SshProxyVerifyOpCB verify_op,
                                SshProxyFreeOpCB free_op,    
                                void *context);


/* A callback of this type must be called by the Diffie-Hellman setup 
   operation callback to complete the asynchronous operation. If the 
   operation was successful, status is SSH_CRYPTO_OK and the exchange 
   data contains the Diffie-Hellman exchange value. Otherwise the exchange 
   data should be set to NULL, and status details the reason for failure.
 
   In some cases the secret is not returned, for example if the secret is 
   generated in some hardware cryptographic module that is always kept 
   private to the user. In such cases secret should be set to NULL. 
   Otherwise the secret contains the private Diffie-Hellman exponent. 
*/
typedef void (*SshProxyDHSetupReplyCB)(SshCryptoStatus status,
                                       const unsigned char *exchange,
                                       size_t exchange_len,
                                       const unsigned char *secret,
                                       size_t secret_len,
                                       void *reply_context);

/* A callback of this type must be provided for the 
   ssh_group_create_proxy function. The callback is called to do the 
   Diffie-Hellman setup operation. The callback must generate the 
   Diffie-Hellman private exponent and compute the exchange value. The 
   callback must call the reply_cb to continue the asynchronous operation. 
   The reply callback must be called with exchange set to NULL data if the 
   operation could not be completed.  

   If this callback is asynchronous it has to return operation handle, 
   which can be used to cancel the operation. 
*/
typedef 
SshOperationHandle (*SshProxyDHSetupOpCB)(SshProxyDHSetupReplyCB reply_cb,
                                          void *reply_context,
                                          void *context);

/* A callback of this type must be called by the Diffie-Hellman agree 
   operation callback to complete the asynchronous operation. If the 
   operation was successful, status is SSH_CRYPTO_OK and the shared 
   secret buffer contains the Diffie-Hellman shared secret. Otherwise 
   the shared secret buffer should be set to NULL, and status details 
   the reason for failure.
 */
typedef void (*SshProxyDHAgreeReplyCB)(SshCryptoStatus status,
                                       const unsigned char *shared_secret,
                                       size_t shared_secret_len,
                                       void *reply_context);


/* A callback of this type must be provided for the 
   ssh_group_create_proxy function. The callback is called to operate on 
   the the exchange and secret data. When data is operated, the function 
   must call the reply_cb to continue the asynchronous operation. The reply 
   callback must be called with NULL data if the operation could not be 
   completed. 

   If this callback is asynchronous it has to return operation handle, 
   which can be used to cancel the operation. 

   In some cases the secret is not available to the user, for example if the 
   secret is generated in some hardware cryptographic module that is always 
   kept hidden. In such cases secret should be set to NULL, and secret_len 
   to 0. Otherwise the secret contains the private Diffie-Hellman exponent. 

   The Diffie-Hellman shared secret is returned in the reply_cb if the 
   operation succeeds. 

*/
typedef 
SshOperationHandle (*SshProxyDHAgreeOpCB)(const unsigned char *exchange,
                                          size_t exchange_len,
                                          const unsigned char *secret,
                                          size_t secret_len,
                                          SshProxyDHAgreeReplyCB reply_cb,
                                          void *reply_context,
                                          void *context);


/* Create a proxy group (for Diffie-Hellman). The returned group is of 
   type group_type, which must be prefixed with "proxy:dl-modp".

   Calls the *_operation_cb with the data that is being operated when 
   the library is performing crypto operations with the returned 
   proxy group. 

   The proxy group is freed with ssh_pk_group_free. It is error 
   to free a group that is currently being used. group_type contains 
   the scheme information. If name is supplied (i.e. not NULL), 
   it will set the scheme of the proxy group, it should the the valid 
   name of a Pk group.  
*/
SshPkGroup 
ssh_dh_group_create_proxy(const char *name,
                          SshUInt32 size_in_bits,
                          SshProxyDHSetupOpCB setup_op,
                          SshProxyDHAgreeOpCB agree_op,
                          SshProxyFreeOpCB free_op,    
                          void *context);


#endif /* SSH_PROXYKEY_H_DEFINED */
